# coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import re

from py.common.java_adapter import java_adapter

from py.common.util.connection_util import SshService
from py.common.entity.check_item import CheckItem
from py.common.entity.check_result import CheckResult

PWD_NOT_SUPPORT_CHARACTERS = [" ", "{", "}"]


def execute(env, method):
    return CheckPwdAndEndStrStandard(env).execute(method)


class CheckPwdAndEndStrStandard(CheckItem):
    def __init__(self, env):
        super(CheckPwdAndEndStrStandard, self).__init__(env)
        self.err_info = list()

    @staticmethod
    def get_root_user_pass(dev_node):
        if dev_node.getRootUser():
            return java_adapter.get_base64_encode_util().decode(dev_node.getRootUser().getPassword())
        return dev_node.getLoginUser().getPassword()

    @staticmethod
    def pwd_contains_not_support_char(pwd):
        for character in PWD_NOT_SUPPORT_CHARACTERS:
            if character in pwd:
                return True
        return False

    @staticmethod
    def end_str_not_standard(node_list, end_str):
        error_node_ip_list = list()
        for node in node_list:
            ssh_service = SshService(node)
            who_result = ssh_service.execute_cmd("whoami")
            if end_str not in who_result:
                error_node_ip_list.append(node.getIp())
            ssh_service.release_ssh()
        return error_node_ip_list

    def do_check(self):
        """
        产品脚本限制：
        密码不能包含空格、‘{‘、’}’
        环境登录非root用户结束符必须是$，root用户结束符必须是#
        :return:
        """
        self.check_pwd_not_support_char()
        self.check_whether_end_str_is_standard()
        if self.err_info:
            return CheckResult(CheckResult.NOT_PASS, "", "\n".join(self.err_info))
        return CheckResult(CheckResult.PASS, "", "")

    def check_pwd_not_support_char(self):
        """
        密码不能包含空格、‘{‘、’}’
        """
        error_node_ip_list = list()
        for cluster_node in self.task_env.get_cluster_node_dict().values():
            pwd = self.get_root_user_pass(cluster_node)
            if self.pwd_contains_not_support_char(pwd):
                error_node_ip_list.append(cluster_node.getIp())
        for no_root_cluster_node in self.task_env.get_no_root_cluster_node_dict().values():
            pwd = no_root_cluster_node.getLoginUser().getPassword()
            if self.pwd_contains_not_support_char(pwd):
                error_node_ip_list.append(no_root_cluster_node.getIp())
        error_node_ip_list = list(set(error_node_ip_list))
        if error_node_ip_list:
            self.err_info.append(self.get_msg("pwd.contains.not.support.char", ",".join(error_node_ip_list)))

    def check_whether_end_str_is_standard(self):
        """
        非root用户结束符必须是$，root用户结束符必须是#
        """
        error_root_ip_list = self.end_str_not_standard(self.task_env.get_cluster_node_dict().values(), "#")
        if error_root_ip_list:
            self.err_info.append(self.get_msg("root.user.end_str_not_standard", ",".join(error_root_ip_list)))

        error_not_root_ip_list = self.end_str_not_standard(self.task_env.get_no_root_cluster_node_dict().values(), "$")
        if error_not_root_ip_list:
            self.err_info.append(self.get_msg("not.root.user.end_str_not_standard", ",".join(error_not_root_ip_list)))
