#  coding=utf-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import json

from py.common.entity.check_item import CheckItem
from py.common.entity.check_result import CheckResult
from py.common.util.connection_util import RestService

QUERY_ALARM_URL = "/api/v2/common/alarms?filter=alarmStatus::1"
QUERY_MANAGE_CLUSTER_URL = "/dsware/service/cluster/queryManageCluster"
NETWORK_SERVICE_URL = "/api/v2/network_service/servers"


def execute(env, method):
    return InitPreInfo(env).execute(method)


class InitPreInfo(CheckItem):
    def __init__(self, env):
        super(InitPreInfo, self).__init__(env)
        self.cluster = self.task_env.get_dev_node()
        self.cache_dict = self.task_env.get_task_cache()
        self.dev_node_dict = self.task_env.get_cluster_node_dict()
        self.err_msg = []
        self.ori_info = []
        self.front_ip_dict = {}

    def do_check(self):
        """
        扩容前初始化
        """
        # 建立存储前端ip-管理ip映射并保存
        node_ips_info = json.loads(RestService(self.cluster).execute_post(NETWORK_SERVICE_URL))
        for info in node_ips_info.get("data"):
            ip_address_list = info.get("ip_address_list")
            self.create_front_ip_mapping(ip_address_list)
        self.task_env.get_task_cache()["front_ip_dict"] = self.front_ip_dict
        # 保存原zk集群内节点
        self.save_old_zk_cluster_node()
        # 保存扩容前的告警信息
        self.save_alarm_info()
        if self.err_msg:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(self.ori_info), "\n".join(self.err_msg), True)
        return CheckResult(CheckResult.PASS, "\n".join(self.ori_info), "", True)

    def save_alarm_info(self):
        alarm_sequence_list = []
        res_dict = json.loads(RestService(self.cluster).execute_get(QUERY_ALARM_URL))
        if res_dict and res_dict.get("result").get("code") == 0:
            alarm_info_list = res_dict.get("data")
            for alarm_info in alarm_info_list:
                alarm_sequence_list.append(alarm_info.get("sequence"))
            self.cache_dict["alarmSequence"] = alarm_sequence_list
        else:
            self.err_msg.append(self.get_msg("record.alarm.information.fail"))
            self.ori_info.append(res_dict)

    def save_old_zk_cluster_node(self):
        old_zk_node_list = []
        res_dict = json.loads(RestService(self.cluster).execute_get(QUERY_MANAGE_CLUSTER_URL))
        if res_dict and res_dict.get("result") == 0:
            for node_info in res_dict.get("nodeInfo"):
                # 从所有节点中将原zk节点保存，用于判断是否需要进行fdsaCore扩容
                old_zk_node_list.append(self.dev_node_dict.get(node_info.get("nodeMgrIp")))
            self.cache_dict["oldZkNodes"] = old_zk_node_list
        if not old_zk_node_list:
            self.err_msg.append(self.get_msg("record.old.zk.node.information.fail"))
            self.ori_info.append(res_dict)

    def create_front_ip_mapping(self, ip_address_list):
        management_ip, front_ip = "", ""
        for ip_address in ip_address_list:
            if "management_internal" in ip_address.get("ip_usage"):
                management_ip = ip_address.get("ip_address")
            if "storage_frontend" in ip_address.get("ip_usage"):
                front_ip = ip_address.get("ip_address")
        self.front_ip_dict[front_ip] = management_ip
