#  coding: UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

import re

from py.common.util.connection_util import SshService
from py.common.entity.check_item import CheckItem
from py.common.entity.check_result import CheckResult
from py.common.java_adapter import java_adapter

CLUSTER_TOOL_PATH = "/tmp/smartKitControllerExpansion"
CLUSTER_TOOL_PATH_KEY = "cluster_tool_path"
UN_ZIP_FILE_KEY = "unzip_file"
UN_ZIP_FOLDER_KEY = "unzip_folder_path"
CREAT_DIR_ERROR = "No such file or directory"
# 拼接命令过长会导致命令中的特殊字符有概率被错误拦截（实测超过180条命令出现问题），这里设置每次执行10条
CMD_NUM = 10


def execute(env, method):
    return InitPreHandle(env).execute(method)


class InitPreHandle(CheckItem):
    def __init__(self, env):
        super(InitPreHandle, self).__init__(env)
        self.script_util = java_adapter.get_script_util()
        self.fsm_dev_node = self.task_env.get_master_fsm_dev_node()
        self.expansion_conf_nodes = self.task_env.get_expansion_conf_nodes()
        self.fsm_ssh_service = SshService(self.fsm_dev_node)
        self.cluster = self.task_env.get_dev_node()
        self.err_infos = list()
        self.ori_infos = list()
        self.cache_dict = self.task_env.get_task_cache()

    def do_check(self):
        """
        zk扩容预处理
        1、将扩容脚本压缩包通过scp上传到主FSM节点：/tmp/smartKitControllerExpansion
        2、解压扩容脚本压缩包工具
        3、配置密码和待扩容节点
        :return:
        """
        expand_file = self.script_util.getExpandFile(self.cluster)
        if not expand_file or not expand_file.exists():
            self.logger.error("Initializing the file failed.")
            return CheckResult(CheckResult.NOT_PASS, "", self.get_msg("init.expand.file.fail"))
        self._upload_expand_script_2_fsm(expand_file)
        if len(self.err_infos) > 0:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(self.ori_infos), "\n".join(self.err_infos))
        self.ori_infos.append(self.get_msg("upload.script.success", CLUSTER_TOOL_PATH))

        self._modify_script()
        if len(self.err_infos) > 0:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(self.ori_infos), "\n".join(self.err_infos))
        self._writing_user_info()
        if len(self.err_infos) > 0:
            return CheckResult(CheckResult.NOT_PASS, "\n".join(self.ori_infos), "\n".join(self.err_infos))
        dos_cmd = "dos2unix {}/*".format(self.cache_dict.get(UN_ZIP_FOLDER_KEY))
        self.fsm_ssh_service.execute_cmd(dos_cmd)
        return CheckResult(CheckResult.PASS, self.get_msg("init.script.success"), "")

    def _cache_target_file_info(self, expand_file):
        # 保存待解压文件路径，和解压路径,供后续脚本使用
        self.cache_dict[UN_ZIP_FILE_KEY] = CLUSTER_TOOL_PATH + "/{}".format(expand_file.getName())
        self.cache_dict[UN_ZIP_FOLDER_KEY] = CLUSTER_TOOL_PATH + "/{}".format("expand_zk_cluster")
        self.cache_dict[CLUSTER_TOOL_PATH_KEY] = CLUSTER_TOOL_PATH

    def _upload_expand_script_2_fsm(self, expand_file):
        try:
            if not self._is_mkdir_and_change_owner_success():
                self.fsm_ssh_service.release_ssh()
                self.logger.error("Failed to create the directory. target path: {}", CLUSTER_TOOL_PATH)
                self.err_infos.append(self.get_msg("creat.dir.error", CLUSTER_TOOL_PATH))
                return
            self._cache_target_file_info(expand_file)
            sftp_service = java_adapter.get_connect_factor_instance().createSftpConnector(self.fsm_dev_node)
            sftp_service.putFile(expand_file.getCanonicalPath(), CLUSTER_TOOL_PATH, None)
            self._unzip_expand_file()
            self.logger.info("Upload file success target path: {}".format(CLUSTER_TOOL_PATH))
        except Exception as exp:
            self.fsm_ssh_service.release_ssh()
            self.logger.error("Failed to upload the script. {}", exp)
            self.err_infos.append(self.get_msg("upload.expand.script.error"))

    def _is_mkdir_and_change_owner_success(self):
        self.fsm_ssh_service.execute_cmd("rm -rf {}".format(CLUSTER_TOOL_PATH))
        mkdir_cmd = "mkdir {}".format(CLUSTER_TOOL_PATH)
        execute_result = self.fsm_ssh_service.execute_cmd(mkdir_cmd).replace(mkdir_cmd, "")
        if CREAT_DIR_ERROR in execute_result:
            return False
        return True

    def _unzip_expand_file(self):
        try:
            un_zip_cmd = "unzip -qo {} -d {}".format(self.cache_dict.get(UN_ZIP_FILE_KEY),
                                                     self.cache_dict.get(UN_ZIP_FOLDER_KEY))
            self.fsm_ssh_service.execute_cmd_no_log(un_zip_cmd)
            change_owner_cmd = "chown {}:omm {}".format(self.fsm_dev_node.getLoginUser().getUserName(),
                                                        self.cache_dict.get(UN_ZIP_FOLDER_KEY))
            change_mode_cmd = "chmod 700 {}".format(self.cache_dict.get(UN_ZIP_FOLDER_KEY))
            self.fsm_ssh_service.execute_cmd(change_owner_cmd)
            self.fsm_ssh_service.execute_cmd(change_mode_cmd)
        except Exception as exp:
            self.fsm_ssh_service.release_ssh()
            self.logger.error("Failed to upload the expansion file. {}", exp)
            raise exp

    def _writing_user_info(self):
        cluster_node_list = self.task_env.get_cluster_node_dict().values()
        # 将集群节点按最大10个一批拆分，分别写入密码信息
        for cluster_nodes in [cluster_node_list[i:i + CMD_NUM] for i in range(0, len(cluster_node_list), CMD_NUM)]:
            temp_info = [self._splice_passwd_info_cmd(cluster_node) for cluster_node in cluster_nodes]
            result = self.fsm_ssh_service.execute_cmd_no_log(";".join(temp_info))
            if len(result.splitlines()) != 2:
                for info in temp_info:
                    result = result.replace(info, "******")
                self.logger.error("Write file failed, exec echo cmd result: {}".format(result))
                self.err_infos.append(self.get_msg("write.passwd.error"))
                break

    def _modify_script(self):
        if len(self.expansion_conf_nodes) == 0:
            self.err_infos.append(self.get_msg("zk.expand.node.error"))
            return

        deploy_infos, target_ip, target_passwd = self._modify_deploy_statements()
        deploy_path = "{}/deploy.sh".format(self.cache_dict.get(UN_ZIP_FOLDER_KEY))
        self._execute_modify(deploy_infos, deploy_path)
        deploy_result = self._modify_result(deploy_path, "{}\\|{}".format(target_ip, target_passwd))
        if (target_ip not in deploy_result) or (target_passwd not in deploy_result):
            self.logger.error("Modify deploy failed")
            self.err_infos.append(self.get_msg("modify.management.error"))

        componet_infos, remote_name_info, remote_pw_info = self._modify_componet_statements()
        componet_path = "{}/operateComponetInAgent.sh".format(self.cache_dict.get(UN_ZIP_FOLDER_KEY))
        self._execute_modify(componet_infos, componet_path)
        componet_result = self._modify_result(componet_path, "{}\\|{}".format(remote_name_info, remote_pw_info))
        if (remote_name_info not in componet_result) or (remote_pw_info not in componet_result):
            self.logger.error("Modify operateComponetInAgent failed")
            self.err_infos.append(self.get_msg("modify.operate.componet.error"))

    def _modify_deploy_statements(self):
        deploy_infos = []
        ori_ip = "readonly EXPAND_NODE_MANAGE_IP_LIST=()"
        target_ip = "readonly EXPAND_NODE_MANAGE_IP_LIST=({})".format(" ".join(self._append_ip_list(
            self.expansion_conf_nodes)))
        deploy_infos.append(self._combine_statement(ori_ip, target_ip))

        ori_passwd = "readonly DecryptPasswd=\"\""
        target_passwd = "readonly DecryptPasswd=\"{}\"".format(self.task_env.get_omm_pass())
        deploy_infos.append(self._combine_statement(ori_passwd, target_passwd))

        return deploy_infos, target_ip, target_passwd

    def _execute_modify(self, modify_infos, modify_file_path):
        sed_cmd = "sed -i '{}' {}".format(";".join(modify_infos), modify_file_path)
        self.fsm_ssh_service.execute_cmd_no_log(sed_cmd)

    def _modify_result(self, file_path, target_statement):
        find_target_statement_cmd = "grep \'{}\' {} | cat".format(target_statement, file_path)
        return self.fsm_ssh_service.execute_cmd_no_log(find_target_statement_cmd).replace(find_target_statement_cmd, "")

    def _modify_componet_statements(self):
        componet_infos = []
        org_name = "REMOTE_USER=\"fsadmin\""
        target_name = "REMOTE_USER=\"{}\"".format(self.fsm_dev_node.getLoginUser().getUserName())
        componet_infos.append(self._combine_statement(org_name, target_name))

        org_remote_pw = "REMOTE_PASSWD=\"\""
        target_remote_pw = "REMOTE_PASSWD=\"{}\"".format(self.fsm_dev_node.getLoginUser().getPassword())
        componet_infos.append(self._combine_statement(org_remote_pw, target_remote_pw))

        return componet_infos, target_name, target_remote_pw

    @staticmethod
    def _get_root_user(dev_node):
        if dev_node.getRootUser():
            return dev_node.getRootUser()
        return dev_node.getLoginUser()

    @staticmethod
    def _get_root_user_pass(dev_node):
        if dev_node.getRootUser():
            return java_adapter.get_base64_encode_util().decode(dev_node.getRootUser().getPassword())
        return dev_node.getLoginUser().getPassword()

    @staticmethod
    def _combine_statement(org_info, target_info):
        return "s/^{}/{}/".format(org_info, target_info)

    @staticmethod
    def _append_ip_list(zk_nodes):
        symbol_ip = []
        for zk_node in zk_nodes:
            symbol_ip.append('\"{}\"'.format(zk_node.getIp()))
        return symbol_ip

    def _splice_passwd_info_cmd(self, cluster_node):
        return "echo '{} {} {} {}' >> {}/passwd.ini"\
            .format(cluster_node.getIp(),
                    escape_special_characters(cluster_node.getLoginUser().getUserName()),
                    escape_special_characters(cluster_node.getLoginUser().getPassword()),
                    escape_special_characters(self._get_root_user_pass(cluster_node)),
                    self.cache_dict.get(UN_ZIP_FOLDER_KEY))


def escape_special_characters(string):
    """
    如果是单引号则替换为'“'”'
    """
    escaped_string = re.sub("'", "'" + '"' + "'" + '"' + "'", string)
    return escaped_string
