#!/bin/bash
set +x

CONFIG_FILE="/var/log/Config.txt"
DEVICE_FILE="/var/log/Device.txt"
LOG_FILE="/var/log/inspect.log"
. /opt/inspect/inspect_map.sh
. /opt/node_type_recognize.sh
. /opt/node_devicename.sh

function log
{
    echo [`date`][$$]$@ >> $LOG_FILE
}

function collectPrintFirmWare()
{
    local NodeID=$1
    local FirmWareType=$2
    local FirmWareVer=$3
    echo "" >> $CONFIG_FILE 2>&1
    echo "        NodeId      :${NodeID}" >> $CONFIG_FILE 2>&1
    echo "        FirmwireName:${FirmWareType}" >> $CONFIG_FILE 2>&1
    echo "        FirmwireVer :${FirmWareVer}" >> $CONFIG_FILE 2>&1
}

#获取IB或者Roce卡版本号
function GetIBOrRoceFirmVersion()
{
    local NetList=""
    local iret=0
    local mstinfo=""
    local tmpinfo=""
    local flintInfo=""
    local tmpPSID=""
    local curfwVersion=""
    local ofedConfigPath="/opt/driver/ib-driver"
    local netType="$1"
    local result=""
    local count=0
    mst start >/dev/null
    iret=$?
    if [ ${iret} -ne 0 ]; then
        log "[${FUNCNAME}][${LINENO}]mst start exec failed,iret=$iret."
        return 1
    fi

    #列出当前系统设备上的ROCE与IB卡对应的MST设备
    mstinfo=$(mst status)
    iret=$?
    if [ ${iret} -ne 0 ]; then
        log "[${FUNCNAME}][${LINENO}]mst status exec failed,iret=$iret."
        return 1
    fi

    NetList=$(echo "${mstinfo}" | grep -E "mt4117_pciconf|mt4099_pci_|mt4115_pciconf|mt4119_pciconf" | awk '{print $1}')
    if [ -z "${NetList}" ] ; then
        log "[${FUNCNAME}][${LINENO}]No check is required because no IB/RoCE network adapter exists."
        return 1
    fi

    for idx in ${NetList[@]}; do
        #获取当前网卡固件版本与PSID
        flintInfo=$(flint -d ${idx} q 2>&1)
        iret=$?
        if [ $? -ne 0 ];then
            log "[${FUNCNAME}][${LINENO}] Get network adapter information failed,idx=[${idx}],iret=${iret},flintInfo=[${flintInfo}]"
            continue
        fi

        tmpPSID=$(echo "${flintInfo}" | grep PSID | awk '{print $NF}')
        #区分IB 和Roce
        grep ";${tmpPSID};" ${ofedConfigPath}/ofed*.conf 2>&1 | grep -v '^#' | grep -iq "${netType}"
        if [ $? -eq 0 ];then
            curfwVersion=$(echo "${flintInfo}" | grep "FW Version:" | awk '{print $NF}')
            result="${result} ${curfwVersion}"
            count=$((count+1))
        fi
    done

    if [ ${count} -ge 1 ];then
        echo "${result}"
        return 0
    fi
    return 1
}

#获取固件版本信息
function collectFirmWareInfo()
{
    local NodeID=$(cat /opt/huawei/snas/etc/cm.ini | grep "^NID=" | awk -F'=' '{print $2}')
    local BmcVer=""
    local NvdimmVer=""
    local IbNum=0
    local IbVer=""
    local RoceNum=0
    local RoceVer=""
    local SesVer=""
    local BiosVer=""
    local CpldVer=""
    local FirmWareNum=0
    local TmpMMLCMDFile="/tmp/Mml_Cmd_file"
    local TmpMMLResultFile="/tmp/Mml_Result_file"
    local result=""
    local ArmFlag=1
    #Flag为1表示不是该类型固件
    RHFlag=1
    T3000Flag=1
    C72Flag=1

    IsRHNode
    RHFlag=$?

    IsT3000Node
    T3000Flag=$?

    IsC72Node
    C72Flag=$?

    IsTSNode
    ArmFlag=$?

    echo "Firmware--------------------------------" >> $CONFIG_FILE 2>&1
    echo "    Number of Firmware:${FirmWareNum}" >> $CONFIG_FILE 2>&1
    echo "    Firmware Info--------------------------------" >> $CONFIG_FILE 2>&1

    if [ ${RHFlag} -eq 0 ] || [ ${ArmFlag} -eq 0 ];then
        #获取Nvdimm version
        source /opt/huawei/snas/upd/upd_util/upd_func.sh
        #获取BMC SES Bios version
        source /opt/huawei/snas/upd/upd_fw/rhfw/ses_utility.sh
        source /opt/huawei/snas/upd/upd_fw/rhfw/snas_rhbmc_fw_upd_utility.sh
        source /opt/huawei/snas/upd/upd_fw/rhfw/snas_rhbmc_fw_upd_utility.sh

        BmcVer=$(GetRHBmcFwVer)
        if [ ! -z "${BmcVer}" ];then
            collectPrintFirmWare "${NodeID}" "BMC" "${BmcVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi

        #ARM中无nvdimm
        if [ ${ArmFlag} -eq 0 ];then
            NvdimmVer=""
        else
            NvdimmVer=$(GetRHNvdimmFwVer)
        fi

        if [ ! -z "${NvdimmVer}" ];then
            collectPrintFirmWare "${NodeID}" "NVDIMM" "${NvdimmVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
        #获取IB version
        IbVer=$(GetIBOrRoceFirmVersion IB)
        if [ $? -eq 0 ];then
            collectPrintFirmWare "${NodeID}" "IB" "${IbVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
        #获取RoCE version
        RoceVer=$(GetIBOrRoceFirmVersion RoCE)
        if [ $? -eq 0 ];then
            collectPrintFirmWare "${NodeID}" "ROCE" "${RoceVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi

        #ARM V2中无ses
        if [ ${ArmFlag} -eq 0 ];then
            SesVer=""
        else
            SesVer=$(GetsesFwVer)
        fi

        if [ ! -z "${SesVer}" ];then
            collectPrintFirmWare "${NodeID}" "SES" "${SesVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi

        BiosVer=$(GetBiosFwVer)
        if [ ! -z "${BiosVer}" ];then
            collectPrintFirmWare "${NodeID}" "BIOS" "${BiosVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
        #获取CLDP version
        CpldVer=$(GetCpldFwVer)

        #对于服务器 V5硬件，需要去掉括号
        echo "${CpldVer}" | grep ')' >/dev/null
        if [ $? -eq 0 ];then
            CpldVer=$(echo "${CpldVer}" | awk -F ")" '{print $2}')
        fi

        if [ ! -z "${CpldVer}" ];then
            collectPrintFirmWare "${NodeID}" "CPLD" "${CpldVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
    elif [ ${T3000Flag} -eq 0 ];then
        #获取BMC version
        source /opt/huawei/snas/upd/upd_fw/t3000fw/snas_bmc_fw_upd_utility.sh
        BmcVer=$(GetBmcFwVer)
        if [ ! -z "${BmcVer}" ];then
            collectPrintFirmWare "${NodeID}" "BMC" "${BmcVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
        #获取Nvdimm version
        source /opt/huawei/snas/upd/upd_util/upd_func.sh
        NvdimmVer=$(GetNvdimmFwVer)
        if [ ! -z "${NvdimmVer}" ];then
            collectPrintFirmWare "${NodeID}" "NVDIMM" "${NvdimmVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
        #获取IB version
        IbNum=$(GetIbNum)
        if [ ${IbNum} -ne 0 ];then
            for((index=0; index<${IbNum};index++))
            do
                IbVerTmp=$(GetIbFwVer ${index})
                IbVer="${IbVer} ${IbVerTmp}"
            done
        fi
        if [ ! -z "${IbVer}" ];then
            collectPrintFirmWare "${NodeID}" "IB" "${IbVer}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
    elif [ ${C72Flag} -eq 0 ];then
        echo "mpa scsi 6 0" >${TmpMMLCMDFile}
        echo "bsp getbiosversion" >>${TmpMMLCMDFile}
        echo "mpa sas getmgmtversion 800 0" >>${TmpMMLCMDFile}
        echo "mpa sas getcpldversion 800 0" >>${TmpMMLCMDFile}
        echo "sascbb allinfo" >>${TmpMMLCMDFile}
        echo "exit" >>${TmpMMLCMDFile}

        /OSM/script/mml <${TmpMMLCMDFile} >${TmpMMLResultFile}
        dos2unix ${TmpMMLResultFile}
        #获取BIOS版本号
        result=$(cat ${TmpMMLResultFile} | grep "Bios Version is" | awk -F":" '{print $NF}')
        if [ ! -z "${result}" ];then
            collectPrintFirmWare "${NodeID}" "BIOS" "${result}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
        #获取SES版本号
        result=$(cat ${TmpMMLResultFile} |grep "Mgmt Version" | awk -F":" '{print $NF}')
        if [ ! -z "${result}" ];then
            collectPrintFirmWare "${NodeID}" "SES" "${result}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
        #获取CPLD版本号
        result=$(cat ${TmpMMLResultFile} |grep "Cpld Version" | awk -F":" '{print $NF}')
        if [ ! -z "${result}" ];then
            collectPrintFirmWare "${NodeID}" "CPLD" "${result}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
        #获取SAS版本号
        result=$(cat ${TmpMMLResultFile} |grep "Current Sal Version" | awk -F":" '{print $NF}')
        if [ ! -z "${result}" ];then
            collectPrintFirmWare "${NodeID}" "SAS" "${result}"
            let "FirmWareNum=${FirmWareNum}+1"
        fi
    else
        log "The firmware type is not in C72 or RH or T3000."
    fi
    #设置固件个数，注：请不要随意修改结果过值
    sed -i "s/Number of Firmware:0/Number of Firmware:${FirmWareNum}/g" $CONFIG_FILE
    rm -f ${TmpMMLResultFile}
    rm -f ${TmpMMLCMDFile}
}
#搜集PCICard
function collectPCICardInfo()
{
    local netWorkFile="/tmp/net_isnpect_work_fiel"
    local PCICardNum=""
    local NodeID=$(cat /opt/huawei/snas/etc/cm.ini | grep "^NID=" | awk -F'=' '{print $2}')
    local NetIDList=$(lspci |egrep "Ethernet|Mellanox" |awk '{print $1}')
    local NetIDArray=(${NetIDList})
    local PortName=""
    local PortNum=""
    local PortInfo=""
    local TypeNum=""
    local PCINum=""
    local PortType=""
    local Speed=""

    /usr/sbin/hwinfo --network >${netWorkFile}
    PCICardNum=$(grep "Device File" ${netWorkFile} | grep -v "Device File: lo" | wc -l)

    echo "PCICard--------------------------------" >> $CONFIG_FILE 2>&1
    echo "    Number of PCICard:${PCICardNum}" >> $CONFIG_FILE 2>&1
    echo "    PCICard Info--------------------------------" >> $CONFIG_FILE 2>&1
    for NetID in ${NetIDArray[*]}
    do
        PortName=$(grep -B2 -A10 "${NetID}" ${netWorkFile} | grep "Device File:" | awk -F':' '{print $2}' | tr -d ' ' | tr "\n" " ")
        if [ -z "${PortName}" ];then
            continue
        fi
        for OnePortName in ${PortName[*]}
        do
            PortType=""
            PortInfo=$(lspci | grep "${NetID}" | awk -F":" '{print $3}')
            PortNum=$(cat /opt/huawei/snas/etc/snas.ini | grep ${OnePortName} | awk -F= '{print $1}'|awk -F_ '{print $3}')
            TypeNum=$(cat /opt/huawei/snas/etc/snas.ini | grep subnetwork_id_${PortNum} | awk -F= '{print $2}')
            Speed=$(ethtool $OnePortName|grep Speed|awk -F ' ' '{print $2}')
            if [ ! -z "${PortNum}" ] && [ ! -z "${TypeNum}" ];then
                if [ "1" == "${TypeNum}" ];then
                    PortType="BackPort"
                fi
                if [ "2" == "${TypeNum}" ];then
                    PortType="FrontPort"
                fi
            fi
            echo "" >> $CONFIG_FILE 2>&1
            echo "        NodeId  :${NodeID}" >> $CONFIG_FILE 2>&1
            echo "        PortName:${OnePortName}" >> $CONFIG_FILE 2>&1
            echo "        PortInfo:${PortInfo}" >> $CONFIG_FILE 2>&1
            echo "        PortType:${PortType}" >> $CONFIG_FILE 2>&1
            echo "        PCINum  :${NetID}" >> $CONFIG_FILE 2>&1
            echo "        Speed   :${Speed}" >> $CONFIG_FILE 2>&1
        done
    done
    rm -rf ${netWorkFile}
}

#删除巡检过程中产生的临时文件
function deltmpfile()
{
    rm -f /tmp/tmpfile* > /dev/null 2>&1
    [ -f "${IP_ROLE_FILE}" ] && rm -f "${IP_ROLE_FILE}" 
    [ -f "${NID_ROLE_FILE}" ] && rm -f "${NID_ROLE_FILE}"
    [ -f "${STANDBY_IP_FILE}" ] && rm -f "${STANDBY_IP_FILE}"
    [ -f "/tmp/.ipmifru" ] && rm -f "/tmp/.ipmifru"
    [ -f "/tmp/tmpfilelocaldiskinfo" ] && rm -f "/tmp/tmpfilelocaldiskinfo"
    [ -d "/tmp/InfoInspect" ] && rm -rf "/tmp/InfoInspect"
    [ -f "/tmp/inspect_tmp_bak" ] && rm -rf "/tmp/inspect_tmp_bak"
    rm -f /var/log/message/messages-*.tar.bz2
    #使用disktool工具产生的临时文件
    [ -f "/tmp/disktool_log" ] && rm -f "/tmp/disktool_log"
    IsT3000Node
    if [ $? -eq 0 ] && [ -f "/opt/hardware_firmware/t3000/nvdimm/get_nvdimm_fw_ver_file" ];then
        echo "" > "/opt/hardware_firmware/t3000/nvdimm/get_nvdimm_fw_ver_file"
    fi
}

function Nodegenralinfo()
{
    #node info
    echo "Node--------------------------------" >> $CONFIG_FILE 2>&1
    echo "    Node Info--------------------------------" >> $CONFIG_FILE 2>&1
    NodeID=$(cat /opt/huawei/snas/etc/cm.ini | grep "^NID=" | awk -F'=' '{print $2}')
    NodeName=$(cat /opt/huawei/snas/etc/snas.ini | grep "^Hostname=" | awk -F'=' '{print $2}')
    NodeServiceType=$(cat /opt/huawei/snas/etc/snas.ini | grep "^node_service_type=" | awk -F'=' '{print $2}')
    NodePool=$(cat /opt/huawei/snas/etc/snas.ini | grep "^nodepool=" | awk -F'=' '{print $2}')
    devicetypenumber=$(cat /opt/huawei/deploy/etc/deploy.ini |grep "^devicetype" |awk -F= '{print $2}')
    DeviceType=$(MapNodeType  $devicetypenumber)
    echo "        NodeId         :${NodeID}" >> $CONFIG_FILE 2>&1
    echo "        NodeName       :${NodeName}" >> $CONFIG_FILE 2>&1
    echo "        NodeServiceType:${NodeServiceType}" >> $CONFIG_FILE 2>&1
    echo "        NodePool       :${NodePool}" >> $CONFIG_FILE 2>&1
    echo "        DeviceType     :${DeviceType}" >> $CONFIG_FILE 2>&1

    #SN,ipmitool
    ipmitool fru > /tmp/ipmi_info.txt
    result=`cat /tmp/ipmi_info.txt|sed 's/^ *//g'|sed "/^$/d"| grep -w "Product Name"|awk -F':' '{print $2}'`
    echo "        Product Name   :${result}" >> $CONFIG_FILE 2>&1
    result=`cat /tmp/ipmi_info.txt|sed 's/^ *//g'|sed "/^$/d"| grep -w "Product Serial"|awk -F':' '{print $2}'`
    echo "        Product Serial :${result}" >> $CONFIG_FILE 2>&1

    #memory
    echo "Memory--------------------------------" >> $CONFIG_FILE 2>&1
    echo "    Memory Info--------------------------------" >> $CONFIG_FILE 2>&1
    #MemTotal
    result=`cat /proc/meminfo | grep "MemTotal"`
    echo "        ${result}" >> $CONFIG_FILE 2>&1
    #MemFree
    result=`cat /proc/meminfo | grep "MemFree"`
    echo "        ${result}" >> $CONFIG_FILE 2>&1

    #Version
    echo "Version--------------------------------" >> $CONFIG_FILE 2>&1
    echo "    Version Info--------------------------------" >> $CONFIG_FILE 2>&1
    result=$( cat /opt/huawei/deploy/package/version | grep ExternalVersion | awk -F= '{print $2}' )
    echo "        Spc Version:${result}" >> $CONFIG_FILE 2>&1
    result=$(ls /var/huawei/patch/cur/sph 2>/dev/null|grep -e "OceanStor*" -e "FusionStorage*" | awk -F'_' '{print $NF}')
    echo "        Sph Version:${result}" >> $CONFIG_FILE 2>&1

    rm -rf /tmp/ipmi_info.txt
}

function collectMemInfo()
{
    local isC72=''
    local isARM=''
    local memResult=''
    local totalMemSize=0
    local bak=$IFS
    local loop='0'
    local tmpSize=''
    local tmpSlot=''
    local tmpSpeed=''
    
    #mem info
    echo "PhysicalMemory--------------------------------" >> $CONFIG_FILE 2>&1

    #判断是不是C72,ARM服务器中不存在NVdimm内存，因此DIMM000按照C72的处理
    IsC72Node
    isC72=$?
    IsTSNode
    isARM=$?
    if [ ${isARM} -eq 0 ] || [ $isC72 -eq 0 ];then
        isC72=1
    else
        isC72=0
    fi
    
    IFS=$'\n'
    memResult=$(dmidecode -t memory | grep -P "^\s+(Locator:|Size:|Speed:)")
    #三个一组进行分析，排除掉Nvdimm和无效的回显
    for line in `echo "$memResult"`; do
        if [ $loop == '0' ]; then
            tmpSize=$(echo $line |grep "Size:" |awk '{print $2}')
            loop='1'
            continue;
        fi
        
        if [ $loop == '1' ]; then
            tmpSlot=$(echo $line |grep "Locator:" | awk -F":" '{print $2}' | sed 's/Slot//g' | awk '{print $1}')
            loop='2'
            continue;
        fi
        
        if [ $loop == '2' ]; then
            tmpSpeed=$(echo $line |grep "Speed:" |awk '{print $2}')
            loop='0'
        fi
        
        if [ "${tmpSpeed}X" != "X" ]; then
            if [ "${tmpSize}" == "No" ] || [ "${tmpSlot}" == "" ]; then
                continue
            fi
            if [ "$tmpSlot" == "F0" ] || [ "$tmpSlot" == "DIMM000" -a $isC72 != 1 ]; then
                continue
            fi
            totalMemSize=$(expr $tmpSize + $totalMemSize)
        fi
    done
    IFS=$bak
    echo "    PhysicalMemoryTotalSize:${totalMemSize}MB" >> $CONFIG_FILE 2>&1
}

NID=$(cat /opt/huawei/snas/etc/cm.ini | grep NID | cut -d"=" -f2)
RACK=$(/usr/local/bin/MmlBatch 4016 "cm node" | sed -e 's/\x1B\[0;[3-4][0-9]m//g' -e 's/\x0D//g' -e 's/\x00//g' | grep $NID | awk -F "Frame:" '{print $2}'| cut -d"|" -f1 | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')

log "[${LINENO}] get NID and RACK info. NID=$NID RACK=$RACK"

function collectShelflistInfo()
{
    echo "Shelflist--------------------------------" >> $DEVICE_FILE 2>&1
    BarCode=$(ipmitool fru | grep 'Product Serial' | awk -F ':' '{print $2}' | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
    echo "    BarCode=$BarCode" >> $DEVICE_FILE 2>&1
    echo "    Location=rack=${RACK},shelf=${NID}" >> $DEVICE_FILE 2>&1
    Type=$(ipmitool fru | grep 'Product Name' | awk -F ':' '{print $2}' | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
    echo "    Type=$Type" >> $DEVICE_FILE 2>&1
    echo "    Name=$(hostname)" >> $DEVICE_FILE 2>&1
    
    log [$FUNCNAME][$LINENO] "collect ShelflistInfo. Barcode=$BarCode, Type=$Type, Location=rack=${RACK},shelf=${NID}"
    echo "" >> $DEVICE_FILE 2>&1
    echo "" >> $DEVICE_FILE 2>&1
}

function collectBoardlistInfo()
{
    echo "Boardlist--------------------------------" >> $DEVICE_FILE 2>&1
    echo "    Disk--------------------------------" >> $DEVICE_FILE 2>&1
    /usr/local/bin/MmlBatch 4016 "cm localdiskinfo show" | sed -e 's/\x1B\[0;[3-4][0-9]m//g' -e 's/\x0D//g' -e 's/\x00//g' >> /dev/null
    if [ $? -ne 0 ];then
        log [$FUNCNAME][$LINENO] "cm localdiskinfo show command error"
    fi
    local disk=$(cat /tmp/tmpfilelocaldiskinfo)
    while IFS= read -r line; do
        BarCode=$(echo "${line}" | grep "SN:" | awk -F "SN:" '{print $2}'| cut -d"|" -f1| sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        Description=$(echo "${line}" | grep "Model:" | awk -F "Model:" '{print $2}'| cut -d"|" -f1| sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        VendorName=$(echo "${line}" | awk -F'\\|\\|' '{print $12}' | awk -F':' '{print $2}' | tr -d ' ')
        runningStatus=$(echo "${line}" | awk -F'\\|\\|' '{print $5}' | awk -F':' '{print $2}' | tr -d ' ')
        runningStatus=$(MapRunningStatus $runningStatus)
        diskType=$(echo "${line}" | awk -F'\\|\\|' '{print $6}' | awk -F':' '{print $2}' | tr -d ' ')
        diskType=$(MapDiskType $diskType)
        slot=$(echo "${line}" | grep "location:" | awk -F "location:" '{print $2}'| cut -d"|" -f1| sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        if [ "${runningStatus}" != "NOT_EXIST" ];then
            echo "        BarCode=$BarCode" >> $DEVICE_FILE 2>&1
            echo "        BoardType=${diskType}" >> $DEVICE_FILE 2>&1
            echo "        Description=$Description" >> $DEVICE_FILE 2>&1
            echo "        Location=rack=${RACK},shelf=${NID},subshelf=-1,slotpos=0,slot=${slot}" >> $DEVICE_FILE 2>&1
            echo "        VendorName=${VendorName}" >> $DEVICE_FILE 2>&1
            echo "" >> $DEVICE_FILE 2>&1
        fi
        log [$FUNCNAME][$LINENO] "collect Disk info.BarCode=$BarCode,BoardType=$diskType,slot=$slot,Description=$Description,Location=rack=${RACK},shelf=${NID},subshelf=-1,slotpos=0,slot=${slot}"
    done <<< "$disk"
    echo "" >> $DEVICE_FILE 2>&1
    echo "" >> $DEVICE_FILE 2>&1

    echo "    PowerModule--------------------------------" >> $DEVICE_FILE 2>&1
    local power=$(/usr/local/bin/MmlBatch 4016 "cm power show" | sed -e 's/\x1B\[0;[3-4][0-9]m//g' -e 's/\x0D//g' -e 's/\x00//g' | grep "Power_ID" | grep "$NID")
    if [ $? -ne 0 ];then
        log [$FUNCNAME][$LINENO] "cm power show command error"
    fi
    while IFS= read -r line; do
        BarCode=$(echo "${line}" | grep "SN:" | awk -F "SN:" '{print $2}'| cut -d"|" -f1| sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        Description=$(echo "${line}" | grep "Model:" | awk -F "Model:" '{print $2}'| cut -d"|" -f1| sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        VendorName=$(echo "${line}" | grep "Vendor:" | awk -F "Vendor:" '{print $2}'| cut -d"|" -f1| sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        echo "        BarCode=$BarCode" >> $DEVICE_FILE 2>&1
        echo "        Description=$Description" >> $DEVICE_FILE 2>&1
        echo "        VendorName=$VendorName" >> $DEVICE_FILE 2>&1
        slot=$(echo "${line}" | grep "Power_ID:" | awk -F "Power_ID:" '{print $2}'| cut -d"|" -f1| sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        echo "        Location=rack=${RACK},shelf=${NID},subshelf=-1,slotpos=1,slot=${slot}" >> $DEVICE_FILE 2>&1
        echo "" >> $DEVICE_FILE 2>&1
        log [$FUNCNAME][$LINENO] "collect PowerModule info.BarCode=$BarCode,BoardType=$BoardType,Description=$Description,Location=rack=${RACK},shelf=${NID},subshelf=-1,slotpos=1,slot=${slot}"
    done <<< "$power"
    echo "" >> $DEVICE_FILE 2>&1
    echo "" >> $DEVICE_FILE 2>&1

    echo "    Fan--------------------------------" >> $DEVICE_FILE 2>&1
    local fan=$(/usr/local/bin/MmlBatch 4016 "cm fan show" | sed -e 's/\x1B\[0;[3-4][0-9]m//g' -e 's/\x0D//g' -e 's/\x00//g' | grep "Fan_ID" | grep "${NID}")
    if [ $? -ne 0 ];then
        log [$FUNCNAME][$LINENO] "cm fan show command error"
    fi
    while IFS= read -r line; do
        echo "        BarCode=--" >> $DEVICE_FILE 2>&1
        echo "        BoardType=snas_node" >> $DEVICE_FILE 2>&1
        slot=$(echo "${line}" | grep "Fan_ID:" | awk -F "Fan_ID:" '{print $2}'| cut -d"|" -f1| sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
        echo "        Location=rack=${RACK},shelf=${NID},subshelf=-1,slotpos=1,slot=${slot}" >> $DEVICE_FILE 2>&1
        echo "" >> $DEVICE_FILE 2>&1
        log [$FUNCNAME][$LINENO] "collect Fan info.Location=rack=${RACK},shelf=${NID},subshelf=-1,slotpos=1,slot=${slot}"
    done <<< "$fan"
}

function collectSoftwarelistInfo()
{
    echo "Softwarelist--------------------------------" >> $DEVICE_FILE 2>&1
    version=$(/usr/local/bin/MmlBatch 4016 "cm system" | sed -e 's/\x1B\[0;[3-4][0-9]m//g' -e 's/\x0D//g' -e 's/\x00//g' | grep Cluster_Version | cut -d":" -f2)
    if [ $? -ne 0 ];then
        log [$FUNCNAME][$LINENO] "cm system command error"
    fi
    echo "    Version=$version" >> $DEVICE_FILE 2>&1
    patch=$(ls /var/huawei/patch/cur/sph | awk -F'_' '{print$NF}')
    echo "    Patch=$patch" >> $DEVICE_FILE 2>&1
    log [$FUNCNAME][$LINENO] "collect Softwarelist info.version=$version,patch=$patch"
}

MapDiskType()
{
    case $1 in
        0)
            echo "FC"
            ;;
        1)
            echo "SAS"
            ;;
        2)
            echo "SATA"
            ;;
        3)
            echo "SAS_SSD"
            ;;
        4)
            echo "NL_SAS"
            ;;
        5)
            echo "SLC_SSD"
            ;;
        6)
            echo "MLC_SSD"
            ;;
        15)
            echo "SMR_SATA"
            ;;
        16)
            echo "NVME_SSD"
            ;;
        17)
            echo "SATA_SSD"
            ;;
        *)
            echo "No this type"
            ;;
    esac
}

MapRunningStatus()
{
    case $1 in
    0)
        echo "UNKNOWN"
        ;;
    1)
        echo "NORMAL"
        ;;
    2)
        echo "RUNNING"
        ;;
    3)
        echo "NOT_RUNNING"
        ;;
    4)
        echo "NOT_EXIST"
        ;;
    5)
        echo "HIGH_TEMPERATURE_SLEEP"
        ;;
    6)
        echo "STARTING"
        ;;
    7)
        echo "POWER_FAILURE_PROTECTING"
        ;;
    8)
        echo "SLEEPING"
        ;;
    9)
        echo "SPINGUP"
        ;;
    10)
        echo "LINK_UP"
        ;;
    11)
        echo "LINK_DOWN"
        ;;
    12)
        echo "POWER_ON"
        ;;
    13)
        echo "POWER_OFF"
        ;;
    14)
        echo "PRE_COPY"
        ;;
    15)
        echo "COPYBACK"
        ;;
    16)
        echo "RECONSTRUCTION"
        ;;
    17)
        echo "EXPANSION"
        ;;
    18)
        echo "NOT_FORMAT"
        ;;
    19)
        echo "FORMATTING"
        ;;
    20)
        echo "UNMAPPING"
        ;;
    21)
        echo "INITIAL_SYNCHRONIZING"
        ;;
    22)
        echo "CONSISTENT"
        ;;
    23)
        echo "SYNCHRONIZING"
        ;;
    24)
        echo "SYNCHRONIZED"
        ;;
    25)
        echo "NOT_SYNCHRONIZED"
        ;;
    26)
        echo "SPLIT"
        ;;
    27)
        echo "ONLINE"
        ;;
    28)
        echo "OFFLINE"
        ;;
    29)
        echo "LOCKED"
        ;;
    30)
        echo "ENABLE"
        ;;
    31)
        echo "DISABLED"
        ;;
    32)
        echo "LEVELING"
        ;;
    33)
        echo "TO_BE_RECOVERD"
        ;;
    34)
        echo "INTERRUPTED"
        ;;
    35)
        echo "INVALID"
        ;;
    36)
        echo "NOSTART"
        ;;
    37)
        echo "QUEUING"
        ;;
    38)
        echo "STOP"
        ;;
    39)
        echo "COPYING"
        ;;
    40)
        echo "COMPLETED"
        ;;
    41)
        echo "PAUSE"
        ;;
    42)
        echo "REVSYNCHRONIZING"
        ;;
    43)
        echo "ACTIVATED"
        ;;
    44)
        echo "ROLLBACK"
        ;;
    45)
        echo "INACTIVATED"
        ;;
    46)
        echo "IDLE"
        ;;
    47)
        echo "POWERING_OFF"
        ;;
    48)
        echo "CHARGING"
        ;;
    49)
        echo "CHARGED"
        ;;
    50)
        echo "DISCHARGING"
        ;;
    51)
        echo "UPGRADING"
        ;;
    *)
        echo "No this type"
        ;;
    esac
}

function collect_local_main
{
    #开始搜集,不管文件是否存在，都先删除一把
    rm -rf $CONFIG_FILE >/dev/null 2>&1
    touch $CONFIG_FILE #创建一个Config.txt文件
    chmod 640 $CONFIG_FILE
    rm -rf $DEVICE_FILE >/dev/null 2>&1
    touch $DEVICE_FILE
    chmod 640 $DEVICE_FILE
    #node info
    Nodegenralinfo
    #PCIcard 网卡
    collectPCICardInfo
    #收集FirmWare
    collectFirmWareInfo
    #收集内存信息
    collectMemInfo

    #以下为设备档案信息收集使用的方法
    collectShelflistInfo
    collectBoardlistInfo
    collectSoftwarelistInfo
    
    #此处为删除巡检中产生的临时文件
    deltmpfile
}

collect_local_main
exit 0