﻿# -*- coding: UTF-8 -*-

import traceback

from java.io import File
from java.lang import Exception as JException
from utils import Products

from resource import MESSAGES_DICT
import util
import constant

from cbb.frame.base import baseUtil
from cbb.frame.cli import cliUtil


def execute(context):
    """入口函数
    """
    return InstallPatch(context).execute()


class InstallPatch:
    def __init__(self, context):
        self.logger = context.get('logger')
        self.lang = context.get("lan")
        self.cli = context.get("devSSH")
        self.sftp = context.get("sftp")
        self.dev = context.get("dev")
        self.common_msg = baseUtil.getPyResource(
            self.lang, "install.oem.patch.error", args="",
            resource=MESSAGES_DICT)

    def execute(self):
        self.logger.info("start install oem patch.")
        try:
            if not self._need_install_patch():
                self.logger.info("Do not need install oem patch.")
                return True, ""
            admin_flag, has_super_admin, err_msg = cliUtil.is_super_administrator(
                self.cli, self.dev.getLoginUser().getUserName(), self.lang)
            if not admin_flag:
                self.logger.info("current user execute command failed.")
                return False, err_msg
            if not has_super_admin:
                self.logger.info("current user is not super admin.")
                err_msg = baseUtil.getPyResource(
                    self.lang, "do.not.has.super.admin.rights", args="",
                    resource=MESSAGES_DICT)
                return False, err_msg
            flag, patch_ver, err_msg = util.check_patch_ver(self.cli,
                                                            self.lang)
            if not flag:
                return False, err_msg
            if patch_ver != "" and patch_ver != constant.OEM_PATCH_VER:
                err_msg = baseUtil.getPyResource(
                    self.lang, "uninstall.other.patch",
                    args=patch_ver, resource=MESSAGES_DICT)
                return False, err_msg
            if patch_ver == constant.OEM_PATCH_VER:
                return True, ""

            self._upload_pkg()
            flag, err_msg = self._install_patch()
            if not flag:
                return False, err_msg
            return True, ""
        except Exception:
            self.logger.error("ninstall oem patch exception:" +
                              str(traceback.format_exc()))
            return False, self.common_msg
        finally:
            self._delete_patch_pkg()
            cliUtil.reConnectionCli(self.cli, self.logger)

    def _need_install_patch(self):
        dev_type = str(self.dev.getDeviceType())
        if "OceanProtect" in dev_type or "OceanDisk" in dev_type:
            return True
        dev_version = self.dev.getProductVersion()
        if Products.compareVersion(dev_version, "6.1") < 0:
            return False
        return True

    def _pre_check(self):
        flag, patch_ver, err_msg = util.check_patch_ver(self.cli, self.lang)
        if not flag:
            return False, err_msg

    def _upload_pkg(self):
        flag, pkg_path = util.get_patch_pkg_path()
        self.sftp.putFullPathFile(File(pkg_path),
                                  constant.PATCH_SERVER_PATH,
                                  None)

    def _install_patch(self):
        cmd = "os_upgrade_coldpatch.sh -p %s" % constant.PATCH_SERVER_PATH
        check_ret = cliUtil.excuteCmdInMinisystemModel(
            self.cli, cmd, self.lang)
        if not check_ret[0]:
            return False, check_ret[2]
        if "finish to install coldpatch and ret[0]" in check_ret[1].lower():
            return True, ""
        return False, self.common_msg

    def _delete_patch_pkg(self):
        try:
            self.sftp.deleteFile(constant.PATCH_SERVER_PATH)
        except JException as ex:
            self.logger.error("delete oem patch pkg error:" + str(ex))
