# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2012-. All rights reserved.
from cbb.frame.cli import cliUtil
from cbb.frame.base.exception import UnCheckException
from cbb.frame.ruleConfig import common
from com.huawei.ism.tool.obase.exception import ToolException


RISK_VERSION_PATCH = {
    "V300R002C00SPC100": "V300R002C00SPH105",
    "V300R002C10SPC100": "V300R002C10SPH112",
    "V300R002C20": "V300R002C20SPH002",
}

BLACK_EMERGENCY_PATCH = (
    "V300R002C00SPH171", "V300R002C10SPH171", "V300R002C10SPH173",
    "V300R002C10SPH175", "V300R002C10SPH176", "V300R002C10SPH177",
    "V300R002C10SPH178", "V300R002C10SPH179", "V300R002C10SPH180",
    "V300R002C20SPH070", "V300R002C20SPH072", "V300R002C20SPH075",
)

HOT_PATCH_NOT_EXISTS = ("", "--")


class CofferDiskCheck(object):
    def __init__(self, cli, lang, logger):
        self.cli = cli
        self.lang = lang
        self.logger = logger
        self.all_cli_ret = list()
        self.version = list()

    def execute_check(self):

        try:
            # 检查版本
            if not self.check_risk_version():
                return True, ""
            # 检查引擎内硬盘时间
            return self.check_disk_running_day()
        except UnCheckException as uncheck:
            return cliUtil.RESULT_NOCHECK, uncheck.errorMsg
        except (ToolException, Exception):
            err_msg = "query.result.abnormal"
            return cliUtil.RESULT_NOCHECK, common.getMsg(self.lang, err_msg)

    def check_risk_version(self):
        """
        检查是否有风险
        :return:
        """
        (
            check_ret,
            software_ver,
            hot_patch,
            history_version,
        ) = common.check_ver_hot_his_version(self.cli, self.lang)
        self.version.append(software_ver)
        self.version.append(hot_patch)
        self.all_cli_ret.append(check_ret[1])
        if check_ret[0] is not True:
            raise UnCheckException(check_ret[2])
        if not history_version:
            return False
        if software_ver in RISK_VERSION_PATCH:
            if (
                hot_patch in BLACK_EMERGENCY_PATCH
                or hot_patch < RISK_VERSION_PATCH.get(software_ver)
                or hot_patch in HOT_PATCH_NOT_EXISTS
            ):
                return True
        return False

    def check_disk_running_day(self):
        cmd = r"show disk general |filterRow column=Coffer\sDisk predict=" \
              r"equal_to value=Yes |filterColumn include columnList=I" \
              r"D,Run\sTime(Day)"
        flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_cli_ret.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg)

        cli_ret_list = cliUtil.getHorizontalCliRet(cli_ret)
        disk_dict = {}
        more_one_disk = self.disk_run_time(cli_ret_list, disk_dict)
        if not more_one_disk:
            err_msg = "coffer.disks.does.not.meet.the.specifications"
            return False, common.getMsg(self.lang, err_msg)

        day_list = []
        self.day_difference(disk_dict, day_list)
        if max(day_list) < 7:
            return True, ""
        else:
            err_msg = "coffer.disk.check.warning"
            return False, common.getMsg(self.lang, err_msg, (self.version[0],
                                                             self.version[1]))

    def disk_run_time(self, cli_ret_list, disk_dict):
        # 检查是否为一个保险盘
        more_one_disk = True
        for engine in cli_ret_list:
            engine_id = engine.get("ID", "")
            engine_id = engine_id[:4] if len(engine_id) >= 4 else engine_id
            if engine_id not in disk_dict:
                disk_dict[engine_id] = []
            day_time = engine.get("Run Time(Day)", "")
            if day_time.isdigit():
                disk_dict[engine_id].append(int(day_time))
        for value in disk_dict.values():
            if len(value) == 1:
                more_one_disk = False
                break
        return more_one_disk

    def day_difference(self, disk_dict, day_list):
        # 每个引擎内任意两块硬盘的时间差
        for i in disk_dict:
            num = max(disk_dict[i]) - min(disk_dict[i])
            day_list.append(num)
