# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.
from cbb.business.operate.expansion import common
from cbb.business.operate.expansion import FuncFactory
from cbb.frame.base import funcUtils
from cbb.frame.base.config import TrVersion
from cbb.frame.context import contextUtil
from cbb.frame.tlv import adTlvUtil
from cbb.frame.tlv import tlvUtil
from cbb.frame.tlv import tlvData

# 进度总时间(秒)
TOTAL_LIMIT_TIME_SECS = 720


@funcUtils.fakeProgress(TOTAL_LIMIT_TIME_SECS)
@FuncFactory.handleFailureForRetry
def execute(context):
    """扩控前检查新扩容硬盘认证证书

    :param context:
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        lang = contextUtil.getLang(context)
        tlv = contextUtil.getTlv(context)
        result = {"flag": True, "errMsg": "", "suggestion": ""}
        # 如果原集群与新扩控中任一版本为TR6之前的版本，则不支持本检查项命令，直接通过
        if not check_version_support(context):
            contextUtil.handleSuccess(context)
            logger.logPass()
            return

        # 查询新扩控上的硬盘认证证书
        disk_cert_records = get_certificate_on_new_ctrl(context, tlv)
        logger.logInfo("disk certificate on new controller:"
                       " %s" % disk_cert_records)

        # 下发证书信息到原集群中检查
        err_certificate_label = \
            check_certificate_on_ori_ctrl(context, tlv, logger,
                                          disk_cert_records)

        # 检查不通过的证书
        if err_certificate_label:
            logger.logInfo("check not pass label: %s" % err_certificate_label)
            label_list = [common.getRes(lang, "disk.certificate.label", label)
                          for label in err_certificate_label]
            label_info = "".join(label_list)
            result["flag"] = False
            if contextUtil.isOem(context):
                result["errMsg"], result["suggestion"] = \
                    common.getMsg(lang, "oem.check.disk.certificate.fail",
                                  label_info)
            else:
                result["errMsg"], result["suggestion"] = \
                    common.getMsg(lang, "check.disk.certificate.fail",
                                  label_info)
            contextUtil.handleFailure(context, result)
            logger.logNoPass("check disk certificate fail.")
            return

        contextUtil.handleSuccess(context)
        logger.logPass()
        return

    except Exception as e:
        contextUtil.handleException(context, e)
        logger.logException(e)
        return


def check_version_support(context):
    """ 检查版本是否支持硬盘认证证书相关命令
        TR6之前的版本不支持该检查命令
    :param context:
    :return:
    """
    isDorado = contextUtil.getItem(context, "isDorado")
    # 原集群完整版本号
    full_pdt_ver = contextUtil.getItem(context, "fullProductVersion")
    # 待扩控版本号
    new_pdt_ver = contextUtil.getItem(context, "new_pdt_ver")
    # 待扩控补丁版本号
    new_patch_ver = contextUtil.getItem(context, "new_patch_ver")
    if isDorado:
        if TrVersion.V6_TR5 in [full_pdt_ver, new_pdt_ver]:
            return False
    else:
        new_full_ver = common.getVRCVersion(new_pdt_ver) + new_patch_ver
        if full_pdt_ver < TrVersion.V5_TR6 \
                or new_full_ver < TrVersion.V5_TR6:
            return False
    return True


def get_certificate_on_new_ctrl(context, tlv):
    """ 查询新扩控上的硬盘认证证书信息
        如果第一个节点查询失败，则下发至其他节点查询，全部查询失败，则抛异常
    :param context:
    :param tlv:
    :return:
    """
    new_boards_list = contextUtil.getItem(context, "newBoardsList")
    disk_cert_records = None
    exception = None
    for new_board in new_boards_list:
        try:
            disk_cert_records = adTlvUtil.get_disk_certificate(tlv, new_board)
        except Exception as ex:
            exception = ex
        else:
            if disk_cert_records is not None:
                return disk_cert_records

    if disk_cert_records is None and exception:
        raise exception


def check_certificate_on_ori_ctrl(context, tlv, logger, disk_cert_records):
    """ 将新扩控上的证书信息下发到原集群上检查

    :param context: 上下文
    :param tlv: tlv对象
    :param logger: 日志对象
    :param disk_cert_records: 证书信息
    :return:
    """
    old_boards_list = contextUtil.getItem(context, "oldBoardsList")
    # 检查不通过的证书标签
    err_certificate_label = []
    for record in disk_cert_records:
        # 证书标签
        certificate_label = tlvUtil.getRecordValue(
            record, tlvData.DISK_CERTIFICATE["LABEL_OF_CERTIFICATE"])
        # 证书内容
        certificate_content = tlvUtil.getRecordValue(
            record, tlvData.DISK_CERTIFICATE["CONTENT_OF_CERTIFICATE"])
        result_record = None
        exception = None
        for old_board in old_boards_list:
            try:
                result_record = adTlvUtil.check_disk_certificate(
                    tlv, old_board, certificate_label, certificate_content)
            except Exception as ex:
                exception = ex
                continue
            break
        # 如果在所有节点上都检查失败，则抛出异常
        if result_record is None:
            raise exception
        logger.logInfo("check record: %s" % result_record)
        result = tlvUtil.getRecordValue(result_record,
                                        tlvData.DISK_CERTIFICATE["RESULT"])
        if result == tlvData.DISK_CERTIFICATE["NOTPASS"]:
            err_certificate_label.append(certificate_label)
    return err_certificate_label
