# coding:utf-8
""" 
@version: Toolkit V200R005C00
@time: 2018/06/27 
@file: ctrlExpFlowMgr.py 
@function: 
@modify: 
"""
import time
import traceback

# noinspection PyUnresolvedReferences
import com.huawei.ism.tool.bizpack.wizardparse.entity.item.Item as Item
# noinspection PyUnresolvedReferences
import com.huawei.ism.tool.bizpack.wizardparse.entity.item.ItemResult as ItemResult
# noinspection PyUnresolvedReferences
import java.lang
import java.lang.Exception as JException
# noinspection PyUnresolvedReferences,PyUnresolvedReferences,PyUnresolvedReferences
from java.util import Date

import common
import config
import jsonUtil
from cbb.frame.base import config as basicConfig
from cbb.frame.context import contextUtil
from cbb.frame.rest import commonRestUtil
from cbb.frame.tlv import tlvUtil


class ExpConstants(object):
    """扩容控制器流程常量管理类.

    插入控制器扩控：
        0:初始值;
        1:扩控命令检查参数;
        3:修改和保存原阵列配置文件;
        5:修改原阵列集群配置;
        8:新扩控上电加入集群

    """
    REST_CONN_UN_AUTHORIZED = -401
    # 需要重试一次的错误码
    RETRY_ERROR_CODE = [
        REST_CONN_UN_AUTHORIZED,  # 未授权
        "16797698",  # 连接异常
        "1077949001",  # 消息超时
        "1077949004",  # 消息处理失败
        "1077949005",  # 目标模块消息不可达
        "1077949006",  # 系统繁忙
        "1073949185",
    ]
    COMMON_ERROR_CODE = -1
    STEP_MAP = {0: 'INIT',
                1: 'CHECK_PARA',
                2: 'NEW_CTRL_EXP',
                3: 'OLD_CTRL_MODIFY_BAY_CONFIG',
                4: 'SYNC_VER_TO_NEW_CTRL',
                5: 'OLD_CTRL_MODIFY_SYS_XNET',
                6: 'RST_NEW_CTRL',
                7: 'NEW_CTRL_JOIN_SYS',
                }
    PART_STEP_MAP = {0: 'INIT',
                     1: 'CHECK_PARA',
                     3: 'OLD_CTRL_MODIFY_BAY_CONFIG',
                     5: 'OLD_CTRL_MODIFY_SYS_XNET',
                     7: 'NEW_CTRL_JOIN_SYS',
                     }

    POWER_ON_STEP_MAP = {
        130: 'SYS_LOADING',
        150: 'NODE_POWERING_ON',
        160: 'CLUSTER_POWERING_ON',
        170: 'BUSINESS_RECOVERING',
        188: 'POWER_ON_FINISH',
    }

    FLOW_STATE_MAP = {
        'INIT': 0,
        'EXPANDING': 1,
        'PAUSED': 2,
        'ROLLBACKING': 3,
        'ROLLBACK_FAILED': 4,
    }

    STEP_MAP_REVERSE = {
        'INIT': 0,
        'CHECK_PARA': 1,
        'NEW_CTRL_EXP': 2,
        'OLD_CTRL_MODIFY_BAY_CONFIG': 3,
        'SYNC_VER_TO_NEW_CTRL': 4,
        'OLD_CTRL_MODIFY_SYS_XNET': 5,
        'RST_NEW_CTRL': 6,
        'NEW_CTRL_JOIN_SYS': 7,
    }

    REASON_KEY_MAP = {
        'REASON_PAUSED': 'REASON_PAUSED',
        'REASON_ROLLBACKING': 'REASON_ROLLBACKING',
        'REASON_ROLLBACK_FAILED': 'REASON_ROLLBACK_FAILED',
    }

    STATUS_KEY_MAP = {
        ItemResult.WAITING: 'WAITING',
        ItemResult.EXECUTING: 'EXECUTING',
        ItemResult.PAUSED: 'PAUSED',
        ItemResult.SUCCEED: 'SUCCESS',
        ItemResult.FAILED: 'FAILED',
    }

    @classmethod
    def getStepResource(cls, stepEnum, lang='zh'):
        return common.getExpansionResource(lang, cls.STEP_MAP.get(stepEnum, 0))

    @classmethod
    def getErrMsgAndSuggest(cls, errorKey, lang='zh', *args, **kwargs):
        if not args and not kwargs:
            return common.getMsg(lang, errorKey)

    @classmethod
    def getStatus(cls, reasonKey, lang='zh', *args, **kwargs):
        if not args and not kwargs:
            return common.getExpansionResource(lang, cls.STATUS_KEY_MAP.get(reasonKey, 'WAITING'))


class ExpFlowMgr(object):
    """扩容控制器流程类,包括启动扩控，扩控流程查询，扩控状态刷新，和UI的交互管理等。

    """

    def __init__(self, context, timeOut=2 * 60 * 60, qryFlowInterval=15):
        """扩容流程管理类初始化函数。

        :param context: Java上下文对象。
        :param timeOut:超时时间。
        :param qryFlowInterval:扩容流程进度查询时间间隔。
        """
        self.CTRL_FLOW_TIME_OUT = timeOut
        self.QRY_FLOW_INTERVAL = qryFlowInterval
        self.isFlowRunning = False  # 是否经历过扩控中阶段。
        self.context = context
        self.context["isRollbacking"] = False
        newConfigClustType = contextUtil.getItem(context, "newConfigClustType","")
        self.networkMode = common.getClustType(newConfigClustType)
        self.lang = contextUtil.getLang(context)
        self.logger = common.getLogger(context.get("logger"), __file__)
        self.tlv = contextUtil.getTlv(context)
        self.ui = context['uiObserver']
        self.oldCtrlNum = contextUtil.getItem(context, 'ctrlNum')
        self.newCtrlNum = self.getNewCtrlNum()
        self.ctrlEncHeight = contextUtil.getItem(context, 'ctrlHeight', 2)
        self.reminded = False
        self.needInsertCtrl = self.ctrlEncHeight == 4 and (self.oldCtrlNum, self.newCtrlNum) in [(2, 4), (6, 8)]
        self.expStepMap = ExpConstants.STEP_MAP if not self.needInsertCtrl \
            else ExpConstants.PART_STEP_MAP
        self.STEP_UPPER = max(self.expStepMap.keys()) + 1
        self.expFlowItemsMap = self.initExpansionItems()
        self.expFlowItems = [self.expFlowItemsMap.get(step) for step in sorted(self.expFlowItemsMap.keys())]
        self.context.put('itemList', self.expFlowItems)
        self.lastMillSeconds = 0  # 第一次查到正在执行的步骤的时间.
        self.errCodeMgrObj = contextUtil.getItem(context, 'errCodeMgrObj', {})
        self.msgTimeOut = False
        self.orgBaseIP = contextUtil.getItem(context, 'configBaseIpAddr', '0.0.0.0')
        self.lastExecutingStep = 0
        self.newSnInfoDict = dict()  # 记录新扩容控制器的引擎ID与SN映射关系，报错信息组装使用。

    def refreshConn(self,context):
        """
        刷新内部连接
        :param context:
        :return:
        """
        self.tlv = contextUtil.getTlv(context)

    def initExpansionItems(self):
        """初始化所有扩容执行项。

        :return:
        """
        itemMap = {}
        self.logger.logInfo('first time executing expansion ctrl flow.')

        for step in self.expStepMap:
            itemTitle = ExpConstants.getStepResource(step, self.lang)
            itemResult = ItemResult.WAITING

            expItem = Item(itemTitle, itemResult)
            expItem.setProgress('--')
            expItem.setDate('--')  # 初始化时间.

            itemMap[step] = expItem
        return itemMap

    def updateUI(self):
        """刷新扩容工具界面（执行项状态刷新）

        :return:
        """
        self.ui.postMsg(self.context)

    def getNewCtrlNum(self):
        """获取扩容后控制器个数(6U 2扩容8 时，先2->4,然后4->8)

        :return:
        """
        newCtrlNum = contextUtil.getItem(self.context, "newCtrlNum")
        # newCtrlNum为空，则表示当前扩容为最后一次扩容（比如2->8中的4->8步骤）
        if not newCtrlNum:
            self.logger.logInfo('It is the last step or only one step to '
                                'expand.')
            newCtrlNum = contextUtil.getItem(self.context, 'newConfigCtrlNum')

        newCtrlNum = int(newCtrlNum)
        self.logger.logInfo("newCtrlNum %s" % newCtrlNum)
        return newCtrlNum

    def start(self):
        """下发扩容控制器命令，启动扩容流程。
        6U 2->4: 下发默认基地址IP：172.16.192.2,172.16.193.2

           4->6:  下发的基地址IP:用户输入的IP。
           6->8：从阵列读取内部IP，下发下去。

        :return:
        """
        try:

            newBayIdToEncSnJson = self.constructBayToEncSnPara()
            if not self.tlv:
                self.logger.logInfo('Tlv connection is None, reconnect.')
                self.tlv = contextUtil.getTlv(self.context)
            # 支持C+硬件时，需要下发原控内部型号
            internalModel = ""
            if contextUtil.getItem(self.context, "is_sup_switch_c_plus"):
                internalModel = contextUtil.getItem(
                    self.context, "interProductModel")
                if internalModel.endswith("_2P"):
                    internalModel = internalModel[:-3]

            errCode = commonRestUtil.expandCtrl(
                self.tlv, self.newCtrlNum, self.networkMode,
                newBayIdToEncSnJson, self.logger, internalModel=internalModel)
            # 授权失败或消息超时，则重新下发一次
            if errCode in ExpConstants.RETRY_ERROR_CODE:
                time.sleep(5)
                self.rebuildRestConnection()
                errCode = commonRestUtil.expandCtrl(
                    self.tlv, self.newCtrlNum, self.networkMode,
                    newBayIdToEncSnJson, self.logger,
                    internalModel=internalModel)

            self.logger.logInfo('expansion ctrl error code:%s,' % errCode)
            return errCode
        except Exception as exception:
            self.logger.logInfo("start expansion flow failed.")
            self.logger.logException(exception)
            errorKey = 'start.expansion.flow.failed'
            return errorKey

    def constructBayToEncSnPara(self):
        """构造扩容参数（新扩容控制器bayID到新扩容控制框SN映射的json字符串)

        :return:
        """
        newBoardsList = contextUtil.getItem(self.context, "newBoardsList", [])
        originEnclosureSNs = contextUtil.getItem(self.context, "originEnclosureSNs", [])
        snInfoDict = dict()
        snInfoList = []
        boardSnSet = set()

        if self.needInsertCtrl:
            boardDict = {}
            boardDict["cte"] = 0 if self.newCtrlNum == 4 else 1
            boardDict["sn"] = str(originEnclosureSNs[0]) if self.newCtrlNum == 4 else str(originEnclosureSNs[1])
            snInfoList.append(boardDict)
        else:
            sortedBordlist = sorted(newBoardsList, key=lambda e: e.__getitem__('enclosureSN'))
            bayId = self.oldCtrlNum / common.getCtrlNumPerEnc(self.ctrlEncHeight) - 1
            for board in sortedBordlist:
                if board["enclosureSN"] in boardSnSet:
                    continue
                bayId = bayId + 1
                boardDict = {}
                self.newSnInfoDict['CTE%s' % bayId] = str(board["enclosureSN"])
                boardSnSet.add(board["enclosureSN"])
                boardDict["cte"] = int(bayId)
                boardDict["sn"] = str(board["enclosureSN"])
                snInfoList.append(boardDict)
                self.logger.logInfo('boradDict:%s' % boardDict)

        snInfoDict["sn_info"] = snInfoList
        self.logger.logInfo('snInfoDict:%s' % snInfoDict)
        self.logger.logInfo('newSnInfoDict:%s' % self.newSnInfoDict)

        nestedJson = jsonUtil.dumpsObj(snInfoDict).replace('"', '\\"')
        self.logger.logInfo('bay ID to new ctrl enclosure nestedJson: %s' % nestedJson)
        return nestedJson

    def setItemResult(self, step, itemResult, remainMin=0, ctrlPowerStep='', **kwargs):
        """设置扩容执行项结果，剩余时间，错误消息和修复建议.

        :param step:
        :param itemResult:
        :param remainMin:
        :return:
        """
        tgtItem = self.expFlowItemsMap.get(step)

        remainTimeStr = ((u'（预计剩余:%s分钟）' if self.lang == 'zh' else '(Estimated left time:%s minutes)')
                         % remainMin) if remainMin else ''
        tgtItem.setRemainTime(remainTimeStr)

        tgtItem.setResult(itemResult)

        if itemResult == ItemResult.EXECUTING:
            contextUtil.setItem(self.context, 'executingStep', step)
            self.lastExecutingStep = step

            if step == 0:  # first step special handle.
                tgtItem.setProgress('40%')

            if step == 7:  # Last step: ctrl powering on
                pwrOnDetailLines = []
                for ctrlPowerInfo in ctrlPowerStep.split(';'):
                    if len(ctrlPowerInfo.split(',')) < 2:
                        continue

                    ctrlLoc, pwrOnStep = ctrlPowerInfo.split(',')
                    engId = ctrlLoc.split('.')[0]
                    if engId not in self.newSnInfoDict:  # 原集群的上电状态不用展示.
                        continue

                    sn = self.newSnInfoDict.get(engId, '--')

                    pwrOnMsgKey = ExpConstants.POWER_ON_STEP_MAP.get(int(pwrOnStep))
                    if not pwrOnMsgKey:
                        self.logger.logInfo('power on step:%s' % pwrOnStep)
                        continue

                    pwrOnMsg = common.getExpansionResource(self.lang, pwrOnMsgKey)
                    detailLine = common.getExpansionResource(self.lang, 'NEW_CTRL_POWER_ON_DETAIL_LOC') \
                                 % (ctrlLoc, sn) + pwrOnMsg
                    pwrOnDetailLines.append(detailLine)

                if pwrOnDetailLines:
                    pwrOnDetailLines.append('<br>' * 2)
                    self.context["errMsg"] = common.getExpansionResource(self.lang,
                                                                         'NEW_CTRLS_POWER_ON_DETAIL_TITLE')
                    self.context["suggestion"] = '<br>'.join(pwrOnDetailLines)
                    self.context["showPowerOnInfo"] = True

            if tgtItem.getDate() == '--' or (kwargs and kwargs.get('reSetStartTime')):
                self.lastMillSeconds = java.lang.System.currentTimeMillis()
                tgtItem.setDate(Date(self.lastMillSeconds))

            # 设置执行时间和执行进度
            itemTotMinutes = tgtItem.getTotMinutes()
            if remainMin != 0 and itemTotMinutes != 0:
                elapseMinutes = abs(itemTotMinutes - remainMin)
                percent = int(elapseMinutes * 1.0 / itemTotMinutes * 100)
                tgtItem.setProgress('%(percent)s%%' %
                                    {'percent': percent if percent <= 100 else 99})

            elif remainMin != 0 and itemTotMinutes == 0:  # 设置单项的总执行时间.
                tgtItem.setTotMinutes(remainMin)
                tgtItem.setProgress('0%')

        elif itemResult == ItemResult.SUCCEED:
            tgtItem.setProgress('100%')

        if itemResult != ItemResult.WAITING:
            self.lastExecutingStep = step
            if tgtItem.getDate() == '--':
                self.lastMillSeconds = java.lang.System.currentTimeMillis()
                tgtItem.setDate(Date(self.lastMillSeconds))

        if itemResult in [ItemResult.FAILED, ItemResult.PAUSED]:  # 失败或者暂停时，进度清0.
            tgtItem.setTotMinutes(0)

        if itemResult in [ItemResult.PAUSED, ItemResult.EXECUTING, ItemResult.ROLLBACKING, ItemResult.ROLLBACK_FAILED]:
            self.clearAllPostStepStatus(step)

        self.refreshAllPreviousStepStatus(step)
        self.updateUI()
        return

    def qryCtrlExpansionFlow(self):
        """查询扩容控制器流程状态.

            0:初始值;
            1:扩控命令检查参数;
            3:修改和保存原阵列配置文件;
            5:修改原阵列集群配置; ----此步骤之后弹窗提示用户插入新扩容控制器。（6扩8相同）。
            8:新扩控上电加入集群

        :param:
        :return:
        """
        qryStartTime = time.time()
        timeElapsed = 0
        while timeElapsed < self.CTRL_FLOW_TIME_OUT:
            timeElapsed = time.time() - qryStartTime

            dataDict = {}
            errDict = {}
            # 查询扩容流程,增加重试3次

            try:
                dataDict, errDict = \
                    commonRestUtil.queryExpandCtrlFlow(self.tlv)
            except Exception as ex:
                    # 如果查询返回指定错误码，则重试，最后一次仍异常，则抛出
                self.logger.logInfo('exception ex: %s' % str(ex.args[0]))
                self.rebuild_connection(ex)
                dataDict, errDict = \
                    commonRestUtil.queryExpandCtrlFlow(self.tlv)

            if int(errDict.get('code')) == ExpConstants.REST_CONN_UN_AUTHORIZED:
                self.rebuildRestConnection()
                continue

            # 解析扩容流程查询结果
            curCtrlExpStep, curStepRemainMins, ctrlExpFlowState, ctrlPowerStep, restErrCode, expErrCodeListStr \
                = common.handleCtrlExpFlowItems(self.context, dataDict, errDict)

            # 检查单个扩容执行步骤的状态。
            itemSuccess = self.handleExpItemResult(curCtrlExpStep, curStepRemainMins, ctrlExpFlowState, ctrlPowerStep,
                                                   restErrCode, expErrCodeListStr)
            if itemSuccess:
                self.logger.logInfo('Expansion execution item success.')
            else:
                self.logger.logNoPass('Expansion message or execution item failed.')
                if self.handleFlowRollbacking(curCtrlExpStep, ctrlExpFlowState):  # 扩容回退或者回退失败，界面只能重新扩容。
                    return
                return

            # 6U 2->4 6->8x 提示插入新扩控。
            if self.isFlowExpanding(
                    ctrlExpFlowState) and curCtrlExpStep > 5 and self.needInsertCtrl and not self.reminded:
                newCtrlInserted = self.remindInsertCtrl()
                if not newCtrlInserted:
                    self.logger.logNoPass('New controller not inserted ? return failed.')
                    return

            # 检查扩容流程是否走完（成功）
            isCtrlExpFlowOver = self.checkCtrlFlowOver(ctrlExpFlowState)
            if isCtrlExpFlowOver:
                self.logger.logInfo('Ctrl expansion flow success and over...')
                self.setExpFlowStatus(True)
                break

            time.sleep(self.QRY_FLOW_INTERVAL)
        else:
            raise Exception('QRY_EXP_FLOW_TIME_OUT')

    def rebuild_connection(self, ex):
        if ex and ex.args[0] in ExpConstants.RETRY_ERROR_CODE:
            for i in range(basicConfig.RETRY_TIMES + 1):
                try:
                    time.sleep(5)
                    self.rebuildRestConnection()
                    break
                except (JException, Exception) as ex1:
                    self.logger.logException(ex1)
        else:
            raise ex

    def handleFlowRollbacking(self, curCtrlExpStep, ctrlExpFlowState):
        """处理扩控回退中的场景.

        :param curCtrlExpStep:
        :param ctrlExpFlowState:
        :return:
        """
        if self.isFlowRollbacking(ctrlExpFlowState):  # 扩容回退中，界面只能重新扩容。
            # noinspection PyTypeChecker
            self.context["isRollbacking"] = True
            self.setExpFlowStatus(False, errorCode='EXP_FLOW_ROLLBACKING')
            itemResult = ItemResult.ROLLBACK_FAILED \
                if ctrlExpFlowState == ExpConstants.FLOW_STATE_MAP.get('ROLLBACK_FAILED') else ItemResult.ROLLBACKING
            self.setItemResult(curCtrlExpStep, itemResult)
            self.logger.logInfo('Expansion flow roll backing, tool wizard must stop now.')
            return True
        return False

    def isFlowExpanding(self, curCtrlExpFlowState):
        """扩容流程是否在执行中。

        :param curCtrlExpFlowState:
        :return:
        """
        return curCtrlExpFlowState == ExpConstants.FLOW_STATE_MAP.get('EXPANDING')

    def isFlowRollbacking(self, curCtrlExpFlowState):
        """扩容流程是否在回滚中或回退失败。

        :param curCtrlExpFlowState:
        :return:
        """
        return curCtrlExpFlowState in [ExpConstants.FLOW_STATE_MAP.get('ROLLBACKING'),
                                       ExpConstants.FLOW_STATE_MAP.get('ROLLBACK_FAILED')]

    def rebuildRestConnection(self):
        self.logger.logInfo('Rebuild rest connection begin.')
        self.tlv = contextUtil.getTlv(self.context, reBuild=True)

    def handleExpItemResult(self, curCtrlExpStep, curStepRemainMins, ctrlExpFlowState, ctrlPowerStep,
                            restErrCode, expErrCodeListStr):
        """处理扩容执行项的执行结果，并解析错误码， 若单项执行失败，扩控流程暂停。

        :param curStepRemainMins:
        :param curCtrlExpStep:
        :param ctrlExpFlowState:
        :param restErrCode:
        :param expErrCodeListStr:
        :return: 扩容执行项成功与否。
        """
        try:
            self.refreshAllPreviousStepStatus(curCtrlExpStep)  # 先刷新当前步骤之前所有步骤状态。

            if restErrCode != 0:  # 查询消息动作失败。
                self.logger.logInfo('Query expansion flow rest message failed.')
                self.setItemResult(curCtrlExpStep, ItemResult.FAILED)
                self.setExpFlowStatus(False, restErrCode)
                return False
            if ctrlExpFlowState == ExpConstants.FLOW_STATE_MAP.get('EXPANDING'):
                self.logger.logInfo('Current expansion flow state EXPANDING, power on step:%s' % ctrlPowerStep)
                self.isFlowRunning = True
                self.setItemResult(curCtrlExpStep, ItemResult.EXECUTING, remainMin=curStepRemainMins,
                                   ctrlPowerStep=ctrlPowerStep)
                return True
            elif ctrlExpFlowState == ExpConstants.FLOW_STATE_MAP.get('ROLLBACKING'):
                self.setItemResult(curCtrlExpStep, ItemResult.ROLLBACKING)
                self.logger.logInfo('Current expansion flow state ROOLBACKING.')
                return True
            elif ctrlExpFlowState == ExpConstants.FLOW_STATE_MAP.get('PAUSED'):  # 暂停状态时，查询到的当前正在执行步骤不准确。
                self.lastExecutingStep = curCtrlExpStep if curCtrlExpStep >= self.lastExecutingStep else self.lastExecutingStep
                self.setItemResult(self.lastExecutingStep, ItemResult.PAUSED)

            elif ctrlExpFlowState == ExpConstants.FLOW_STATE_MAP.get('ROLLBACK_FAILED'):
                self.logger.logInfo('Expansion rollback failed.')
                self.setItemResult(curCtrlExpStep, ItemResult.ROLLBACK_FAILED)
            elif ctrlExpFlowState == ExpConstants.FLOW_STATE_MAP.get('INIT'):
                self.logger.logInfo('Expansion flow state come to init, flow may success.')
                return True

            # 非暂停状态，扩控未失败。
            self.logger.logInfo('Expansion flow not expanding, parse error code begin:%s' % str(expErrCodeListStr))
            ctrlIdAndErrCodeList = filter(
                lambda ctrlIdAndErrCodeList: len(ctrlIdAndErrCodeList) == 2 and ctrlIdAndErrCodeList[1] != '0',
                map(lambda ctrlIdAndErrcode: ctrlIdAndErrcode.split(','),
                    expErrCodeListStr.split(';')))
            originalCtrlEncInfoDict = contextUtil.getItem(self.context, 'originalCtrlEncInfoDict')
            expErrCodeList = map(lambda e: int(e[1]), ctrlIdAndErrCodeList)
            self.recodeMessageTimeout(expErrCodeList)

            nodeLocSnAndErrCodeTupList = []
            for ctrlLocAndErrCodeList in ctrlIdAndErrCodeList:
                if len(ctrlLocAndErrCodeList) < 2:
                    continue

                ctrlLoc = ctrlLocAndErrCodeList[0]
                errCode = ctrlLocAndErrCodeList[1]
                engId = ctrlLoc.split('.')[0]
                sn = originalCtrlEncInfoDict.get(engId)
                if not sn:
                    sn = self.newSnInfoDict.get(engId, '--')
                nodeLocSnAndErrCodeTupList.append((ctrlLoc, sn, int(errCode)))

            self.logger.logInfo('Expansion flow error code:%s' % str(nodeLocSnAndErrCodeTupList))
            # 扩控流程失败(暂停）。
            self.setExpFlowStatus(False, expErrorCodeTup=nodeLocSnAndErrCodeTupList)

            return False
        except Exception as e:
            self.logger.logInfo('Parse expansion ctrl flow item error code exception:' + str(e))
            self.logger.logInfo('traceback:' + traceback.format_exc())
            self.setExpFlowStatus(False, -1)
            return False

    def recodeMessageTimeout(self, expErrCodeList):
        """根据错误码判断消息是否超时，若超时则重试时刷新路由信息.

        :param expErrCodeList:
        :return:
        """
        MSG_TIME_OUT_ERR_CODE_LIST = [1073793447, 1073793445]
        for timeOutErrCode in MSG_TIME_OUT_ERR_CODE_LIST:
            if timeOutErrCode in expErrCodeList:
                self.logger.logInfo('Message timeout, record to refresh route info when retrying..')
                self.msgTimeOut = True
                return

    def remindInsertCtrl(self):
        """6U 4控设备插入式扩控需要在中间步骤弹框提示用户插入新扩容控制器。

        :return:
        """
        dialogUtil = self.context['dialogUtil']
        if (self.oldCtrlNum, self.newCtrlNum) == (2, 4):
            msg, suggest = common.getMsg(self.lang, "confirm.items.and.insert.new.ctrls.for.6u2To4")
        else:
            msg, suggest = common.getMsg(self.lang, "confirm.items.and.insert.new.ctrls.for.6u6To8")

        confirmMsg = u'我已完成上述确认，并已插入新扩容控制器' if self.lang == 'zh' \
            else 'I have confirmed the preceding information and inserted the controllers to be added.'

        rec = dialogUtil.showSpecifiedConfirmMsgInfoDialog(msg, confirmMsg)
        if not rec:
            self.setItemResult(7, ItemResult.PAUSED)
            # noinspection PyTypeChecker
            self.setExpFlowStatus(False, errorCode='new.ctrls.not.inserted')
            return False
        else:
            self.reminded = True
        return True

    def checkCtrlFlowOver(self, ctrlExpFlowState):
        """判断当前时刻阵列所处的扩容阶段，如果扩容流程完成，则刷新所有扩容步骤状态。

        :param ctrlExpFlowState:
        :return:
        """
        cur_sys_ctrl_num = tlvUtil.getControllersNum(self.tlv)
        new_config_ctrl_num = contextUtil.getItem(self.context,
                                                  "newConfigCtrlNum")
        # 当前阶段扩容流程完成
        if (cur_sys_ctrl_num == self.newCtrlNum
                and ctrlExpFlowState ==
                ExpConstants.FLOW_STATE_MAP.get('INIT')):
            # 判断当前阶段是否是最后扩容阶段
            if self.newCtrlNum == new_config_ctrl_num:
                contextUtil.setItem(self.context, "all_flow_finish", True)
            self.refreshAllPreviousStepStatus(self.STEP_UPPER)
            self.destroyFlow()
            return True

        return False

    def destroyFlow(self):
        """流程销毁（从上下文移除流程管理对象）。

        :return:
        """
        self.logger.logInfo('Clear ctrl expansion flow in context.')
        contextUtil.setItem(self.context, 'expCtrlFlow', None)
        return

    def recordFlow(self):
        """流程记录（保存流程管理对象到上下文，方便重试）。

        :return:
        """
        self.logger.logInfo('Save ctrl expansion flow in context.')
        contextUtil.setItem(self.context, 'expCtrlFlow', self)
        return

    def refreshAllPreviousStepStatus(self, curExpFlowStep):
        """将当前执行步骤之前的所有步骤（不包括当前执行步骤）的状态刷新为“执行成功”。

        :param curExpFlowStep: 当前步骤
        :return:
        """
        curStepDate = java.lang.System.currentTimeMillis()
        if curExpFlowStep < self.STEP_UPPER:
            curStepDate = self.expFlowItemsMap.get(curExpFlowStep).getDate()

        for step in sorted(self.expFlowItemsMap.keys()):
            if step < curExpFlowStep:
                tgtItem = self.expFlowItemsMap.get(step)
                if tgtItem.getResult != ItemResult.SUCCEED:
                    self.setItemResult(step, ItemResult.SUCCEED)
                if tgtItem.getDate() == '--':
                    tgtItem.setDate(curStepDate)
                tgtItem.setRemainTime('')

    def clearAllPostStepStatus(self, curExpFlowStep):
        """清除当前执行步骤之后的所有步骤（不包括当前执行步骤）的状态（刷为“待执行”)。

        :param curExpFlowStep: 当前步骤
        :return:
        """
        for step in sorted(self.expFlowItemsMap.keys()):
            if step > curExpFlowStep:
                tgtItem = self.expFlowItemsMap.get(step)
                if tgtItem.getResult != ItemResult.WAITING:
                    tgtItem.setResult(ItemResult.WAITING)
                    tgtItem.setDate('--')
                    tgtItem.setTotMinutes(0)
                    tgtItem.setProgress('--')
                tgtItem.setRemainTime('')

        self.logger.logInfo('Clear all post step status over...')

    def getCtrlSn(self, ctrlId):
        """

        :param ctrlId:
        :return:
        """
        orginBoardsList = contextUtil.getItem(self.context, "orginBoardsList")

    def setExpFlowStatus(self, isExpFlowSuccess, errorCode=0, expErrorCodeTup=(), err_msg="", suggestion=""):
        """设置扩容流程整体状态及错误消息和修复建议。

        :param expErrorCodeTup:
        :param isExpFlowSuccess:
        :param errorCode:
        :return:
        """
        if isExpFlowSuccess:
            self.refreshAllPreviousStepStatus(self.STEP_UPPER)  # 所有扩容步骤成功，进度100%
            self.context["succ"] = True
            contextUtil.setItem(self.context, "cifs_status", None)
            return

        self.context["succ"] = False
        if errorCode == "smb_failed":
            self.logger.logInfo(u'errmsg:%s,suggest:%s' % (err_msg, suggestion))
            self.context["errMsg"] = err_msg
            self.context["suggestion"] = suggestion
            return

        if errorCode != 0:
            errCause, suggest = self.errCodeMgrObj.get(errorCode)
            self.logger.logInfo(u'errmsg:%s, suggest:%s' % (errCause, suggest))
            self.context["errMsg"] = errCause
            self.context["suggestion"] = suggest
            return

        self.context["errMsg"] = u'扩容控制器暂停。' if self.lang == 'zh' else 'Controller expansion paused.'
        suggestLines = []

        for ctrlLoc, sn, nodeErrCode in expErrorCodeTup:
            nodeErrCause, nodeSuggest = self.errCodeMgrObj.get(nodeErrCode)
            if nodeErrCause:
                nodeErrCauseDetail = (u'控制器（位置:%s, SN:%s）故障原因：%s' if self.lang == 'zh'
                                      else 'Controller(Location:%s: SN:%s) error cause: %s') \
                                     % (ctrlLoc, sn, nodeErrCause)
                suggestLines.append(nodeErrCauseDetail)

            if nodeSuggest:
                nodeSuggestDetail = (u'修复建议：%s' if self.lang == 'zh' else 'Suggestion: %s') % nodeSuggest
                suggestLines.append(nodeSuggestDetail)

        if len(suggestLines) == 0:
            self.context["errMsg"] = u'操作失败。' if self.lang == 'zh' else 'The operation fails.'
            if self.lang == 'zh':
                suggestDetail = u'请重试，如果问题仍然存在请联系技术支持工程师。'
            else:
                suggestDetail = 'Try again. If the problem persists, contact technical support engineers.'
            suggestLines.append(suggestDetail)

        suggestLines.append('<br>' * 2)
        suggestion = '<br>'.join(suggestLines)
        self.context["suggestion"] = suggestion
        self.logger.logInfo(u'Suggest msg:%s' % suggestion)
        return

    def qrySysCurrentExpFlow(self):
        """查询存储系统当前扩容状态（可能系统发生过复位，需要重建rest连接并鉴权）.

        :return:
        """
        try:
            dataDict, errDict = commonRestUtil.queryExpandCtrlFlow(self.tlv)
        except Exception as e:
            if len(e.args) > 1:
                errCode = e.args[0]
                if errCode in [basicConfig.ERR_CODE_UNAUTHORIZED_REST, basicConfig.ERR_CODE_COMMUNICATION_ABNOEMAL]:
                    self.rebuildRestConnection()
                    dataDict, errDict = commonRestUtil.queryExpandCtrlFlow(self.tlv)
                else:
                    raise e
            else:
                raise e

        curCtrlExpStep, curStepRemainMins, ctrlExpFlowState, ctrlPowerStep, restErrCode, expErrCode \
            = common.handleCtrlExpFlowItems(self.context, dataDict, errDict)

        d = ExpConstants.FLOW_STATE_MAP
        reversedDict = dict(zip(d.values(), d.keys()))
        self.logger.logInfo('Storage system current ctrl expansion flow state:%s, '
                            'expansion step:%s,Rest error code:%s, expansion error code:%s'
                            % (reversedDict.get(ctrlExpFlowState),
                               ExpConstants.STEP_MAP.get(curCtrlExpStep),
                               restErrCode,
                               expErrCode))

        return restErrCode, expErrCode, curCtrlExpStep, ctrlExpFlowState
