﻿# -*- coding: UTF-8 -*-
import os
from defusedxml import ElementTree

from cbb.business.operate.fru.common import BaseFactory
from cbb.business.operate.fru.common import FuncFactory
from cbb.business.operate.fru.common import constData
from cbb.frame.base import baseUtil
from cbb.frame.base import config
from cbb.frame.util import ability_map_util
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData

REPLACED_GUIDE_LINKER = "REPLACED_GUIDE_LINKER"


# FRU工具支持的备件类型，以工具界面为准
class FruType():
    CONTROLLER = "controller"
    MEMORY = "memory"
    SYSTEM_DISK = "system_disk"
    CBU = "cbu"
    POWER = "power"
    DISK = "disk"
    FAN = "fan"
    BBU = "bbu"
    INFT = "interface"
    MGMT_INTF = "mgmt_interface"
    EXP_INTF = "exp_interface"
    ENCLOSURE = "enclosure"
    DISK_DRAWER = "disk_drawer"
    FIBER_MODULE = "fibre_module"
    ELECTRICAL_MODULE = "electrical_module"
    FAN_CTRL_MODULE = "fan_ctrl_module"

    FRONT_END_CABLE = "front_end_cable"
    MGMT_CABLE = "mgmt_cable"
    IB_CABLE = "ib_cable"
    SAS_CABLE = "sas_cable"
    RDMA_CABLE = "rdma_cable"
    PCIE_CABLE = "pcie_cable"
    IP_SCALEOUT_CABLE = "ip_scaleOut_cable"

    DATA_SWITCH = "data_switch"
    GROUND_CABLE = "ground_cable"
    SVP_MGMT_CABLE = "svp_mgmt_cable"

    # 记录在用户行为中选中的模块名称fruType
    CONTROLLER_ENCLOSURE = "controller_enclosure"
    DISK_ENCLOSURE = "disk_enclosure"
    BACK_END_INTF = "back_end_interface"
    SMART_ACC_INTF = "smart_ACC_interface"
    FRONT_END_INTF = "front_end_interface"
    BOX = "box"
    BOX_DISK = "box_disk"
    BOX_INTF = "box_intf"
    OTHER_INTF = "other_interface"
    IP_SCALEOUT_FIBER_MODULE = "ip_scaleOut_fibre_module"
    IP_SCALEOUT_INTERFACE = "ip_scaleOut_interface"
    IP_SCALEOUT_SWITCH = "ip_scaleOut_data_switch"
    PCIE_INTERFACE = "pcie_interface"
    PCIE_SWITCH = "pcie_data_switch"


class DeviceType():
    LOW_END_DEVICE = "LOW_END_DEVICE"
    MID_END_DEVICE = "MID_END_DEVICE"
    HIGH_END_DEVICE = "HIGH_END_DEVICE"
    DORADO_DEVICE = "DORADO_DEVICE"


class GuideItemGroupEnum():
    CTRL_2U_ENCLOSURE = "CTRL_2U_ENCLOSURE"
    CTRL_2U_ENCLOSURE_NVMe = "CTRL_2U_ENCLOSURE_NVMe"
    CTRL_3U_ENCLOSURE = "CTRL_3U_ENCLOSURE"
    CTRL_4U_ENCLOSURE = "CTRL_4U_ENCLOSURE"
    CTRL_6U_ENCLOSURE = "CTRL_6U_ENCLOSURE"
    CTRL_8U_ENCLOSURE = "CTRL_8U_ENCLOSURE"
    COMMON_DISK_ENCLOSURE = "COMMON_DISK_ENCLOSURE"
    IP_DISK_ENCLOSURE = "IP_DISK_ENCLOSURE"
    HIGH_DISK_ENCLOSURE = "HIGH_DISK_ENCLOSURE"
    DISK_4U_ENCLOSURE = "DISK_4U_ENCLOSURE"
    DATA_SWITCH = "DATA_SWITCH"
    CONNECTION_CABLE = "CONNECTION_CABLE"
    FIBRE_MODULE = "FIBRE_MODULE"
    ELECTRICAL_MODULE = "ELECTRICAL_MODULE"


# 每个框组类型的具体包含项
GUIDE_ITEM_GROUP = {
    GuideItemGroupEnum.CTRL_2U_ENCLOSURE: [
        restData.Enum.EnclosureModelEnum.ARM_2U_12,
        restData.Enum.EnclosureModelEnum.ARM_2U_25,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_2U12_EAR,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_2U25_EAR,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_MID,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_MID,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_LOW,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_LOW,
        restData.Enum.EnclosureModelEnum.CTRL_NVMe2U_24,
        restData.Enum.EnclosureModelEnum.CTRL_NVME_2U_25,
        restData.Enum.EnclosureModelEnum.CTRL_NVMe2U_2C_36,
        restData.Enum.EnclosureModelEnum.CTRL_NVMe2U_2C_25_LOW,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_NEW_END,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_NEW_END,
        restData.Enum.EnclosureModelEnum.CTRL_COMPUTE_END,
        restData.Enum.EnclosureModelEnum.CTRL_COMPUTE_END_616,
        restData.Enum.EnclosureModelEnum.CTRL_NVME_2U_36_1610_4_CARD,
        restData.Enum.EnclosureModelEnum.CTRL_NVME_2U_36_1610_6_CARD,
    ],
    GuideItemGroupEnum.CTRL_3U_ENCLOSURE: [
        restData.Enum.EnclosureModelEnum.CTRL_3U,
    ],
    GuideItemGroupEnum.CTRL_4U_ENCLOSURE: [
        restData.Enum.EnclosureModelEnum.CTRL_4U_4C,
    ],
    GuideItemGroupEnum.CTRL_6U_ENCLOSURE: [
        restData.Enum.EnclosureModelEnum.CTRL_6U,
    ],
    GuideItemGroupEnum.CTRL_8U_ENCLOSURE: [
        restData.Enum.EnclosureModelEnum.CTRL_NVME_8U_64,
    ],
    GuideItemGroupEnum.COMMON_DISK_ENCLOSURE: [
        restData.Enum.EnclosureModelEnum.EXPSAS2U_12,
        restData.Enum.EnclosureModelEnum.EXPSAS2U_24,
        restData.Enum.EnclosureModelEnum.EXPSAS2U_25,
        restData.Enum.EnclosureModelEnum.EXP_SAS2U_25,
        restData.Enum.EnclosureModelEnum.EXPSAS2U_25_12GLINK,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_4U24_EAR_12GLINK,
        restData.Enum.EnclosureModelEnum.EXP2U_JBOF,
    ],
    GuideItemGroupEnum.IP_DISK_ENCLOSURE: [
        restData.Enum.EnclosureModelEnum.EXP_IPSAS_2U_25,
        restData.Enum.EnclosureModelEnum.EXP_IPSAS_2U_12,
        restData.Enum.EnclosureModelEnum.EXP_IPNVMe_2U_36
    ],
    GuideItemGroupEnum.DISK_4U_ENCLOSURE: [
        restData.Enum.EnclosureModelEnum.EXPSAS4U_24_NEW,
        restData.Enum.EnclosureModelEnum.EXPSAS4U,
        restData.Enum.EnclosureModelEnum.EXP_SAS4U_24,
    ],
    GuideItemGroupEnum.HIGH_DISK_ENCLOSURE: [
        restData.Enum.EnclosureModelEnum.EXPSAS4U_75,
        restData.Enum.EnclosureModelEnum.EXP_12G_SAS_4U_75,
    ],
    GuideItemGroupEnum.DATA_SWITCH: [
        restData.Enum.EnclosureModelEnum.DSW_PCIe1U,
        FruType.DATA_SWITCH,
    ],

    GuideItemGroupEnum.CONNECTION_CABLE: [
        FruType.FRONT_END_CABLE,
        FruType.SAS_CABLE,
        FruType.RDMA_CABLE,
        FruType.PCIE_CABLE,
        FruType.MGMT_CABLE,
        FruType.IB_CABLE,
        FruType.IP_SCALEOUT_CABLE,
    ],
    GuideItemGroupEnum.FIBRE_MODULE: [
        FruType.FIBER_MODULE,
    ],
    GuideItemGroupEnum.ELECTRICAL_MODULE: [
        FruType.ELECTRICAL_MODULE,
    ],
}

FRONT_PORT_TYPE = {
    restData.Enum.PortEnum.ETH_PORT: "eth_cable",
    restData.Enum.PortEnum.FC_PORT: "fc_cable",
    restData.Enum.PortEnum.FCoE_PORT: "fcoe_cable",
    restData.Enum.PortEnum.IB_PORT: "ib_cable",
}


class EnclosureTypeEnum():
    CTRL_ENC_12 = "2U12"
    CTRL_DISK_ENC_25 = "2U25"
    DISK_ENC_24 = "4U24"
    DISK_ENC_75 = "4U75"
    CTRL_DISK_ENC_36 = "2U36"
    NVME = "NVMe"
    SAS = "SAS"
    SCM = "SCM"


ENCLOSURE_TYPE = {
    EnclosureTypeEnum.CTRL_ENC_12: [
        restData.Enum.EnclosureModelEnum.ARM_2U_12,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_2U12_EAR,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_MID,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_LOW,
        restData.Enum.EnclosureModelEnum.EXP_IPSAS_2U_12,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_NEW_END
    ],
    EnclosureTypeEnum.CTRL_DISK_ENC_25: [
        restData.Enum.EnclosureModelEnum.ARM_2U_25,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_2U25_EAR,
        restData.Enum.EnclosureModelEnum.EXPSAS2U_25,
        restData.Enum.EnclosureModelEnum.EXPSAS2U_25_12GLINK,
        restData.Enum.EnclosureModelEnum.EXP_SAS2U_25,
        restData.Enum.EnclosureModelEnum.EXPSAS2U_24,
        restData.Enum.EnclosureModelEnum.CTRL_NVME_2U_25,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_2U25_EAR_12GLINK,
        restData.Enum.EnclosureModelEnum.EXP2U_JBOF,
        restData.Enum.EnclosureModelEnum.EXP_IPSAS_2U_25,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_LOW,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_MID,
        restData.Enum.EnclosureModelEnum.CTRL_NVMe2U_2C_25_LOW,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_NEW_END,
    ],
    EnclosureTypeEnum.CTRL_DISK_ENC_36: [
        restData.Enum.EnclosureModelEnum.EXP_IPNVMe_2U_36,
        restData.Enum.EnclosureModelEnum.CTRL_NVMe2U_2C_36,
    ],
    EnclosureTypeEnum.DISK_ENC_24: [
        restData.Enum.EnclosureModelEnum.EXPSAS4U_24_NEW,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_4U24_EAR_12GLINK,
        restData.Enum.EnclosureModelEnum.EXP_SAS4U_24,
    ],
    EnclosureTypeEnum.DISK_ENC_75: [restData.Enum.EnclosureModelEnum.EXP_12G_SAS_4U_75,
                                    restData.Enum.EnclosureModelEnum.EXPSAS4U_75
                                    ],
}

ENCLOSURE_TYPE_FOR_DORADO = {
    EnclosureTypeEnum.SCM: [
        restData.Enum.EnclosureModelEnum.CTRL_4U_4C
    ],
    EnclosureTypeEnum.NVME: [
        restData.Enum.EnclosureModelEnum.CTRL_NVME_2U_25,
        restData.Enum.EnclosureModelEnum.EXP2U_JBOF,
        restData.Enum.EnclosureModelEnum.EXP_IPNVMe_2U_36,
        restData.Enum.EnclosureModelEnum.CTRL_NVMe2U_2C_36,
        restData.Enum.EnclosureModelEnum.CTRL_NVMe2U_2C_25_LOW,
        restData.Enum.EnclosureModelEnum.CTRL_NVME_2U_36_1610_4_CARD,
        restData.Enum.EnclosureModelEnum.CTRL_NVME_2U_36_1610_6_CARD,
    ],
    EnclosureTypeEnum.SAS: [
        restData.Enum.EnclosureModelEnum.ARM_2U_12,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_2U12_EAR,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_MID,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_LOW,
        restData.Enum.EnclosureModelEnum.EXP_IPSAS_2U_12,
        restData.Enum.EnclosureModelEnum.ARM_2U_25,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_2U25_EAR,
        restData.Enum.EnclosureModelEnum.EXPSAS2U_25,
        restData.Enum.EnclosureModelEnum.EXPSAS2U_25_12GLINK,
        restData.Enum.EnclosureModelEnum.EXP_SAS2U_25,
        restData.Enum.EnclosureModelEnum.EXPSAS2U_24,
        restData.Enum.EnclosureModelEnum.EXP_IPSAS_2U_25,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_LOW,
        restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_MID,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_2U25_EAR_12GLINK,
        restData.Enum.EnclosureModelEnum.EXPSAS4U_24_NEW,
        restData.Enum.EnclosureModelEnum.CTRL_12GSAS_4U24_EAR_12GLINK,
        restData.Enum.EnclosureModelEnum.EXP_12G_SAS_4U_75,
        restData.Enum.EnclosureModelEnum.EXP_SAS4U_24,
        restData.Enum.EnclosureModelEnum.EXPSAS4U_75,
        restData.Enum.EnclosureModelEnum.CTRL_COMPUTE_END,
        restData.Enum.EnclosureModelEnum.CTRL_COMPUTE_END_616
    ],
}

NET_MODE = {
    restData.Enum.NetModeEnum.DIRECT: "direct",
    restData.Enum.NetModeEnum.SWITCH: "switch",
}


def getPath(filePath, devType, enclType, fruType):
    root = ElementTree.parse(filePath)
    nodeList = root.getiterator("product")

    # 判断设备类型，获取框类型
    devModel = getDownTree(nodeList, devType)

    try:
        # 判断框类型，获取FRU类型
        fruModel = getDownTree(devModel, enclType)

        # 判断FRU类型，获取路径
        if fruModel == None:
            return getDefaultPath(devModel)
        pathNode = getDownTree(fruModel, fruType)

        if pathNode == None:
            return getDefaultPath(devModel)
        path = pathNode.attrib["link"]
        return path

    except:
        return getDefaultPath(devModel)


def getDownTree(nodeList, param):
    """通过参数获取下层节点信息

    :param nodeList: 当前节点列表
    :param param: 当前节点参数
    :return: 下层节点列表
    """
    targetNode = None
    for node in nodeList:
        if node.attrib["id"] == param:
            targetNode = node
            break
    return targetNode


def getDefaultPath(devModel):
    param = "DEFAULT"
    targetNode = getDownTree(devModel, param)
    paramDown = "default"
    pathNode = getDownTree(targetNode, paramDown)
    defaultPath = pathNode.attrib["link"]
    return defaultPath


def getGuideFileDir(context):
    """更换指导目录，归一到resource目录，CI可同步到CRU

    :param context: 上下文
    :return:
    """
    # 初始化更换指导存放路径
    curDir = context.get("importRootDir")
    curDirPar = curDir[:curDir.rfind(os.path.sep)]
    guideDir = os.path.join(curDirPar[:curDirPar.rfind(os.path.sep)],
                            "resource", "guide")
    return guideDir


@ability_map_util.ability_mapping
def getGuideConfFile(context):
    """获取更换指导配置文件

    :param context:
    :return:
    """
    lang = contextUtil.getLang(context)
    product_model = contextUtil.getProductModel(context)
    if baseUtil.isDoradoV6Dev(product_model):
        GUIDE_CONF_FILE = "replacedGuideConf_V6_%s.xml" % lang
    elif baseUtil.is_big_clusters_ai_dev(product_model):
        GUIDE_CONF_FILE = "replacedGuideConf_A800_%s.xml" % lang
    else:
        GUIDE_CONF_FILE = "replacedGuideConf_V5_%s.xml" % lang
    guideDir = getGuideFileDir(context)
    return os.path.join(guideDir, GUIDE_CONF_FILE)


def get_guide_conf_file_for_compute(context):
    """
    获取更换指导配置文件(计算型存储)
    :param context: 上下文
    :return: 更换指导配置文件
    """
    lang = contextUtil.getLang(context)

    guide_conf_file = "replacedGuideConf_compute_{}.xml".format(lang)
    guide_dir = getGuideFileDir(context)
    return os.path.join(guide_dir, guide_conf_file)


def get_guide_conf_file_for_dorado_2000(context):
    """
    获取更换指导配置文件(Dorado 2000、Dorado 2100)
    :param context: 上下文
    :return: 更换指导配置文件
    """
    lang = contextUtil.getLang(context)

    guide_conf_file = "replacedGuideConf_V5_{}.xml".format(lang)
    guide_dir = getGuideFileDir(context)
    return os.path.join(guide_dir, guide_conf_file)


def getDevGuideFileInfo(context, product_model):
    """获取设备类型及对应更换指导文件路径

    :param context: 上下文
    :param product_model: 产品型号
    :return:
    """
    guideFile = ""
    devType = ""
    lang = contextUtil.getLang(context)
    guideDir = getGuideFileDir(context)

    if baseUtil.isDoradoV6Dev(product_model):
        devType, guideFile = get_v6_dev_guide_file_info(context, product_model, lang)
    elif baseUtil.is_big_clusters_ai_dev(product_model):
        devType, guideFile = get_a800_dev_guide_file_info(lang)
    else:
        devType, guideFile = get_v5_dev_guide_file_info(context, product_model, lang)
    return devType, os.path.join(guideDir, guideFile)


def get_a800_dev_guide_file_info(lang):
    """
    获取A800设备类型及指导文档路径
    :param lang: 语言
    :return: 设备类型及指导文档
    """
    return DeviceType.HIGH_END_DEVICE, "OceanStor_A800_%s.chm" % lang


def get_v5_dev_guide_file_info(context, productModel, lang):
    """获取V5设备类型及对应更换指导文件路径

    :param context: context
    :param productModel: 产品型号
    :param lang: 语言类型
    :return:
    """
    if baseUtil.is_kunpeng_high_end(context, productModel):
        devType = DeviceType.HIGH_END_DEVICE
    elif baseUtil.isArmDevLowEnd(context, productModel):
        devType = DeviceType.LOW_END_DEVICE
    else:
        devType = DeviceType.MID_END_DEVICE
    guideFile = "common_v5_kunpeng_%s.chm" % lang
    return devType, guideFile


@ability_map_util.ability_mapping
def get_v6_dev_guide_file_info(context, productModel, lang):
    """获取V6设备类型及对应更换指导文件路径

    :param context: 上下文
    :param productModel: 产品型号
    :param lang: 语言类型
    :return:
    """
    if baseUtil.isDoradoV6HighEnd(productModel):
        devType = DeviceType.HIGH_END_DEVICE
    elif baseUtil.isDoradoV6Mid(productModel):
        devType = DeviceType.MID_END_DEVICE
    else:
        devType = DeviceType.LOW_END_DEVICE
    guideFile = "dorado_v6_%s.chm" % lang
    return devType, guideFile


def get_compute_guide_file_info(context, product_model, lang):
    """
    获取计算型存储设备类型及对应更换指导文件路径
    :param context: 上下文
    :param product_model: 产品型号
    :param lang: 语言类型
    :return: 设备类型， 更换指导文件路径
    """
    if baseUtil.isDoradoV6HighEnd(product_model):
        dev_type = DeviceType.HIGH_END_DEVICE
    elif baseUtil.isDoradoV6Mid(product_model):
        dev_type = DeviceType.MID_END_DEVICE
    else:
        dev_type = DeviceType.LOW_END_DEVICE
    guide_file = "compute_%s.chm" % lang
    return dev_type, guide_file


def get_guide_file_info_for_dorado_2000(context, product_model, lang):
    """
    获取Dorado 2000、Dorado 2100设备类型及对应更换指导文件路径
    :param context: 上下文
    :param product_model: 产品型号
    :param lang: 语言类型
    :return: 设备类型， 更换指导文件路径
    """
    if baseUtil.isDoradoV6HighEnd(product_model):
        dev_type = DeviceType.HIGH_END_DEVICE
    elif baseUtil.isDoradoV6Mid(product_model):
        dev_type = DeviceType.MID_END_DEVICE
    else:
        dev_type = DeviceType.LOW_END_DEVICE
    guide_file = "common_v5_kunpeng_{}.chm".format(lang)
    return dev_type, guide_file


def getFruGroupType(fruType, encModel):
    """获取更换指导备件大类，即第一级目录

    :param fruType: 已选FRU类型
    :param encModel: 框类型
    :return:
    """
    guideItemGroupType = ""

    for (key, value) in GUIDE_ITEM_GROUP.items():
        if fruType in value:
            guideItemGroupType = key
            return guideItemGroupType

    for (key, value) in GUIDE_ITEM_GROUP.items():
        if encModel in value:
            guideItemGroupType = key
            return guideItemGroupType

    return guideItemGroupType


def getFruId(context, fruType, selectedFruType, productModel,
             location, encModel=""):
    """获取待更换FRU ID

    :param context: 上下文
    :param fruType: 备件类型
    :param selectedFruType: 待更换备件类型
    :param productModel: 产品型号
    :param location: 位置
    :param encModel: 框类型枚举
    :return:
    """
    fruId = fruType  # 不需要特殊处理直接返回备件型号

    if fruType == FruType.FRONT_END_CABLE:
        fruId = FRONT_PORT_TYPE.get(int(selectedFruType), "")
        return fruId
    elif fruType == FruType.DISK:
        if baseUtil.is_big_clusters_ai_dev(productModel):
            return "a800_disk"
        enc_type_dict = get_enc_type_dict(productModel)
        for (key, value) in enc_type_dict.items():
            if encModel in value:
                fruId = fruType + "_" + key
                return fruId
    elif fruType in [FruType.BBU, FruType.CONTROLLER, FruType.INFT,
                     FruType.MEMORY, FruType.POWER, FruType.FAN]:
        return get_fru_id(context, fruType, productModel, location)
    elif baseUtil.is_short_enc_low_end(productModel) and fruType == "enclosure":
        return "{}_2000".format(fruType)
    else:
        return fruId


def get_enc_type_dict(product_model):
    """
    获取框类型字典
    @param product_model: 设备型号
    @return: 框类型字典
    """
    if baseUtil.is_short_enc_low_end(product_model):
        return ENCLOSURE_TYPE
    if baseUtil.isDoradoV6Dev(product_model):
        return ENCLOSURE_TYPE_FOR_DORADO
    return ENCLOSURE_TYPE


def get_fru_id(context, fru_type, product_model, location):
    """
    获取待更换的FRU ID（controller, bbu, memory, power）
    :param context: 上下文
    :param fru_type: 备件类型
    :param product_model: 产品型号
    :param location: 位置
    :return:
    """
    # 新硬件更换控制框的电源和风扇时，为电源风扇一体化

    if fru_type in [FruType.POWER, FruType.FAN]:
        if all([baseUtil.is_ultra_low_end(context, product_model),
                "CTE" in location]):
            return fru_type + "_2210"
        elif fru_type == FruType.FAN and "FAN" not in location.upper():
            return get_fru_type_fan_power_bundle(context, fru_type)
        else:
            return fru_type
    # Dorado 2000 更换控制器，BBU,接口卡，设置单独的更换指导
    if fru_type in [FruType.CONTROLLER, FruType.BBU, FruType.INFT] and baseUtil.is_short_enc_low_end(product_model):
        return "{}_2000".format(fru_type)
    # OceanStor Dorado 2000、OceanStor Dorado 2100、OceanStor Dorado 2020 和超低端保持一致
    if product_model in config.ULTRA_LOW_END_NEW or baseUtil.is_short_enc_low_end(product_model):
        return fru_type + "_2210"
    inter_model = FuncFactory.getInternalProductModel(context)
    if inter_model in config.ULTRA_LOW_END_NEW_INTERNAL_PRODUCT_MODEL:
        return fru_type + "_ultra_low"
    return fru_type


def get_fru_type_fan_power_bundle(context, fru_type):
    """
    处理4U硬盘框中电源风扇一体化的场景的问题。如果为一体化返回电源的fru_type.

    :param context: 上下文
    :param fru_type: fru类型
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    sel_row = BaseFactory.json.toDict(context.get(constData.FanKeys.SELECTED_ROW))
    if isinstance(sel_row, list):
        sel_row = sel_row[0]
    fru_id = sel_row[constData.FanKeys.ID]
    location = sel_row[constData.FanKeys.LOCATION]
    logger.logInfo("get_fru_id:%s." % str(fru_id))
    # 由于风扇电源一体化的槽位和电源一致，单风扇槽位包含FAN.
    if FuncFactory.isFruBundle(context, restData.Enum.ObjEnum.FAN, fru_id) and "FAN" not in location:
        return FruType.POWER
    else:
        return fru_type


def getEnclosureModel(context, encName):
    """获取框型号

    :param context: 上下文
    :param encName: 框名
    :return:
    """
    encs = contextUtil.getItem(context, "ENCLOSURE_LIST")
    for enc in encs:
        name = enc.get('name', '')
        model = enc.get('model', '')
        if encName in name:
            return model

    return ""


def getGuideLink(confFile, guideFilePath, devType, fruGroup, fruId, lang="zh"):
    """获取指导链接

    :param confFile:
    :param guideFilePath:
    :param devType:
    :param fruGroup:
    :param fruId:
    :param lang:
    :return:
    """
    link = getPath(confFile, devType, fruGroup, fruId)
    if link.count("%s") == 1:
        link = link % lang
    link = guideFilePath + "::" + link
    return link


@ability_map_util.ability_mapping
def get_enc_name(context, fruLocation):
    """根据位置信息获取狂名称

    :param context: 上下文
    :param fruLocation: 位置
    :return:
    """
    encName = ""
    try:
        encName = fruLocation.split(".")[0]
    except:
        return ''
    return encName


def get_enc_name_for_compute(context, fru_location):
    """
    根据位置信息获取框名称（计算型存储）
    :param context: 上下文
    :param fru_location: 位置
    :return: 框名
    """
    if not fru_location.startswith("DAE"):
        return "CTE0"
    enc_name = ""
    try:
        enc_name = fru_location.split(".")[0]
    except Exception as e:
        BaseFactory.log.error(context, "fail to get enc name: {}".format(str(e)))
        return enc_name
    return enc_name


def initGuideInfo(context, fruType, selectedLocation, selectedFruType=""):
    """初始化更换指导信息

    :param context: 上下文
    :param fruType: 备件类型
    :param selectedLocation: 选择位置
    :param selectedFruType: 待更换备件类型
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    contextUtil.setItem(context, REPLACED_GUIDE_LINKER, None)
    productModel = contextUtil.getProductModel(context)
    try:
        # 获取更换指导文件路径
        devType, guideFilePath = getDevGuideFileInfo(context, productModel)
        logger.logInfo(
            "get guide file result: [product model:%s, info: %s]." % (productModel, str((devType, guideFilePath))))
        # 备件所在框的类型encModel
        logger.logInfo("get selectedLocation:%s." % str(selectedLocation))
        encName = get_enc_name(context, selectedLocation)
        logger.logInfo("get encName:%s." % str(encName))
        encModel = FuncFactory.getEnclosureModel(context, encName)
        # 获取框分组类型
        fruGroup = getFruGroupType(fruType, encModel)
        logger.logInfo("get fru group:%s." % str(fruGroup))
        # 获取具体备件类型key
        logger.logInfo("get fruType:%s" % fruType)
        logger.logInfo("get selectedFruType:%s" % selectedFruType)
        logger.logInfo("get encModel:%s" % encModel)
        fruId = getFruId(context, fruType, selectedFruType, productModel,
                         selectedLocation, encModel)
        logger.logInfo("get fru id:%s." % str(fruId))

        # 获取更换指导连接
        confFile = getGuideConfFile(context)
        logger.logInfo("get guide config file path:%s." % confFile)
        guideLink = getGuideLink(confFile, guideFilePath, devType, fruGroup, fruId)
        logger.logInfo("get guide link:%s" % guideLink)
        contextUtil.setItem(context, REPLACED_GUIDE_LINKER, guideLink)
        guideLink = contextUtil.getItem(context, REPLACED_GUIDE_LINKER)
        return
    except Exception as e:
        contextUtil.initConnection(context)
        logger.logException(e)
        return


def execute(context):
    """执行更换指导

    :param context:
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        guideLink = contextUtil.getItem(context, REPLACED_GUIDE_LINKER)
        logger.logInfo("guide link:%s" % guideLink)
        context["doc_link"] = guideLink
    except Exception as e:
        logger.logException(e)

    finally:
        contextUtil.handleSuccess(context)
        return
