﻿# -*- coding: UTF-8 -*-
import json
import time

from cbb.business.operate.fru.common import common
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
from cbb.frame.oldTlv import tlvData
from cbb.frame.oldTlv import tlvUtil
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil

DEFAULT_TIMEOUT_SECS = 36000  # 预拷贝整体超时时间
START_PRE_COPY_TIME_SECS = 30  # 启动预拷贝时间
PRE_COPY_PROGRESS_TIME_SECS = 150  # 进度开始刷新时间
ISOLATION_DISK_TIME_SECS = 360  # 预拷贝完成后隔离硬盘时间
INTERVAL_TIME_SECS = 30  # 进度查询间隔


def execute(context, select_id):
    """
    :param context: 上下文
    :param select_id: 保持数据的ID
    :return:
    """
    lang = contextUtil.getLang(context)
    log = baseUtil.getLogger(context.get("logger"), __file__)
    disks = json.loads(context.get(select_id))
    try:
        # 故障盘，非成员盘无需预拷贝
        check_disks = get_check_disks(context, disks)
        if not check_disks:
            contextUtil.setResult(context, True, "", "")
            return

        set_start_time(context)
        pre_copy_tasks = [DiskPreCopy(context, d) for d in check_disks]
        start_precopy(pre_copy_tasks)
        while True:
            status, progress, remain_time, err_msg = \
                query_disks_progress(pre_copy_tasks)

            if status == PreCopyStatus.RUNNING:
                common.showUI(context, progress, remain_time)
                baseUtil.safeSleep(INTERVAL_TIME_SECS)
                continue
            elif status == PreCopyStatus.SUCCESS:
                contextUtil.setResult(context, True)
                return
            elif status == PreCopyStatus.FAIL:
                msg, sug = common.getMsg(lang, "pre.copy.fail",
                                         suggestionArgs=err_msg)
                contextUtil.setResult(context, False, msg, sug)
                return

    except Exception as e:
        log.error("precopy error:%s" % str(e))
        msg, sug = common.getMsg(lang, "execute.error")
        contextUtil.setResult(context, False, msg, sug)
        return
    finally:
        common.showUI(context, 100, 0)


def set_start_time(context):
    rest = contextUtil.getRest(context)
    systime_rec = restUtil.CommonRest.getSysUTCTimeRecord(rest)
    systime = restUtil.CommonRest.getRecordValue(systime_rec,
                                                 restData.RestCfg.
                                                 SystemUtcTime.
                                                 CMO_SYS_UTC_TIME)

    contextUtil.setItem(context, "START_SYS_TIME", systime)


def start_precopy(pre_copy_tasks):
    for task in pre_copy_tasks:
        task.precopy()


def query_disks_progress(pre_copy_tasks):
    msg_list = []
    status_list = []
    progress_list = []
    remain_time_list = []
    for task in pre_copy_tasks:
        status, progress, remain_time, msg = \
            task.query_precopy_progress()
        status_list.append(status)
        progress_list.append(progress)
        remain_time_list.append(remain_time)
        if msg:
            msg_list.append(msg)

    last_status = get_precopy_status(status_list)
    last_progress = min(progress_list)
    last_remain_time = max(remain_time_list)
    last_msg = ",".join(msg_list)
    return last_status, last_progress, last_remain_time, last_msg


def get_precopy_status(status_list):
    run_flag = PreCopyStatus.SUCCESS
    for status in status_list:
        if status in [PreCopyStatus.NOT_START, PreCopyStatus.RUNNING]:
            return PreCopyStatus.RUNNING
        if status is PreCopyStatus.FAIL:
            run_flag = PreCopyStatus.FAIL
    return run_flag


def get_disk_info(rest, disk_id, disk_location):
    """
    :param rest:
    :param disk_id:
    :param disk_location:
    :return:
    """
    try:
        rec = restUtil.CommonRest.getDiskById(rest, disk_id)
        return rec
    except Exception as e:
        # 当预拷贝完成硬盘健康状态为FAULT时，通过GET查询不到信息，通过批量查询获取
        recs = restUtil.CommonRest.getBatchDisks(rest)
        for r in recs:
            loc = restUtil.CommonRest.getRecordValue(r,
                                                     restData.RestCfg.
                                                     PublicAttributes.
                                                     LOCATION)
            if loc == disk_location:
                return r

        raise e


def get_check_disks(context, disks):
    """

    :param context:
    :param disks:
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    check_disks = []
    rest = contextUtil.getRest(context)
    for disk in disks:
        dl = disk.get("location", None)
        disk_id = disk.get("id", None)
        rec = get_disk_info(rest, disk_id, dl)
        pool_id = restUtil.CommonRest.getRecordValue(rec, restData.RestCfg.
                                                     Disk.POOLID)
        health_status = restUtil.CommonRest.getRecordValue(rec,
                                                           restData.RestCfg.
                                                           Disk.HEALTH_STATUS)

        # 无硬盘域
        if not pool_id:
            logger.info("The disk[%s] is not the member disk." % dl)
            continue

        # 状态故障的盘不用预拷贝
        if health_status == restData.Enum.HealthStatusEnum.FAULT:
            logger.info("The disk[location:%s, health status:%s] do not need "
                        "precopy." % (dl, health_status))
            continue

        running_status = restUtil.Tlv2Rest.getRecordValue(rec, restData.
                                                          Hardware.Disk.
                                                          RUNNING_STATUS)
        progress = restUtil.Tlv2Rest.getRecordValue(rec,
                                                    restData.Hardware.
                                                    Disk.PROGRESS)
        check_disks.append(
            {"id": disk_id, "location": dl, "runningStatus": running_status,
             "progress": progress, "poolID": pool_id})

    return check_disks


class PreCopyStatus:
    NOT_START = 1
    RUNNING = 2
    FAIL = 3
    SUCCESS = 4


class DiskPreCopy:
    PRE_COPY_START_EVENT = "17592203477004"  # 预拷贝开始的事件ID: 0x1000010A000C
    PRE_COPY_FAIL_EVENT = "17592203477006"  # 预拷贝失败的告警ID: 0x1000010A000E
    PRE_COPY_STOP_EVENT = "17592203477022"  # 预拷贝停止的告警ID: 0x1000010A001E
    HOT_SPARE_NOT_ENOUGH = "17432581"  # 热备空间不足告警: 0x10A0005

    def __init__(self, context, disk):
        self.err_msg = ""
        self.log = baseUtil.getLogger(context.get("logger"), __file__)
        self.lang = contextUtil.getLang(context)
        self.rest = contextUtil.getRest(context)
        self.context = context
        self.disk = disk
        self.pool_name = disk.get("poolID")
        self.status = PreCopyStatus.NOT_START
        self.progress = 0
        self.isolate_time = 0
        self.remain_time = DEFAULT_TIMEOUT_SECS
        self.pre_copy_cur_start_time = time.time()
        self.precopy_start_sys_time = ""
        self.get_pre_copy_begin_time()

    def get_pre_copy_start_event(self):
        """

        :return:
        """
        locations = self.disk.get("location").split(".")
        param_list = [{"paramIndex": 3, "value": locations[0]},
                      {"paramIndex": 4, "value": locations[-1]}]
        condition = {DiskPreCopy.PRE_COPY_START_EVENT: param_list}
        return restUtil.CommonRest.getSysEvent(self.rest, None, None,
                                               condition)

    def get_sys_utc_time(self):
        rec = restUtil.CommonRest.getSysUTCTimeRecord(self.rest)
        self.log.info("get_sys_utc_time: %s" % str(rec))
        return restUtil.CommonRest.getRecordValue(rec, restData.RestCfg.
                                                  SystemUtcTime.
                                                  CMO_SYS_UTC_TIME)

    def get_pre_copy_begin_time(self):
        hs = self.disk.get("healthStatus")
        if hs == restData.Enum.HealthStatusEnum.PRE_FAIL:
            rec = self.get_pre_copy_start_event()
            self.log.info("get_pre_copy_start_event: %s" % str(rec))
            if rec:
                self.precopy_start_sys_time = restUtil.CommonRest.\
                    getRecordValue(rec, restData.RestCfg.Alarm.STARTTIME)
            else:
                self.precopy_start_sys_time = self.get_sys_utc_time()
        else:
            self.precopy_start_sys_time = self.get_sys_utc_time()

        self.log.info(
            "get precopy start system time: %s" % self.precopy_start_sys_time)
        return

    def precopy_tlv(self):
        tlv = contextUtil.getOldTlv(self.context)
        try:
            params = []
            param0 = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E["DISK"])
            param1 = (tlvData.PUB_ATTR['id'], self.disk.get("id"))
            params.append(param0)
            params.append(param1)
            tlvUtil.execCmd(tlv, tlvData.CMD.get("START_DISK_PRECOPY"), params)
        finally:
            contextUtil.destroyOldTlvConnection(self.context)
        return

    def precopy_rest(self):
        param0 = (restData.Hardware.Disk.TYPE, restData.Enum.ObjEnum.DISK)
        param1 = (restData.Hardware.Disk.ID, self.disk.get("id"))
        paramlist = restUtil.Tlv2Rest.getParamList(param0, param1)
        rest = contextUtil.getRest(self.context)
        restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.START_DISK_PRECOPY,
                                  paramlist)
        return

    def precopy(self):
        """
        start to precopy
        :return:
        """
        version = str(contextUtil.getDevObj(self.context).get("version"))
        self.log.info("The current device version:%s" % version)
        if version >= "V300R006":
            self.precopy_rest()
        else:
            self.precopy_tlv()
        self.pre_copy_cur_start_time = time.time()
        return

    def get_pre_copy_stop_alarm(self):
        """
        通过检查告警判断预拷贝是否失败
        :return:
        """
        locations = self.disk.get("location").split(".")
        sys_time = self.get_sys_utc_time()

        param_list = [{"paramIndex": 3, "value": locations[0]},
                      {"paramIndex": 4, "value": locations[-1]}]
        condition = {DiskPreCopy.PRE_COPY_FAIL_EVENT: param_list,
                     DiskPreCopy.PRE_COPY_STOP_EVENT: param_list}

        return restUtil.CommonRest.getSysEvent(self.rest,
                                               self.precopy_start_sys_time,
                                               sys_time, condition)

    def get_hot_spare_alarm(self):
        """

        :return:
        """
        rec = restUtil.CommonRest.getDiskDomainById(
            self.rest, self.disk.get("poolID"))
        self.pool_name = restUtil.CommonRest.getRecordValue(
            rec, restData.RestCfg.PublicAttributes.ID)

        current_alarms = restUtil.CommonRest.getCurrentAlarm(self.rest)
        for alarm in current_alarms:
            alarm_id = restUtil.CommonRest.getRecordValue(
                alarm, restData.RestCfg.Alarm.EVENTID)
            alarm_desc = restUtil.CommonRest.getRecordValue(
                alarm, restData.RestCfg.Alarm.DESCRIPTION)
            self.log.info("alarm_id: %s" % alarm_id)
            self.log.info("alarm_desc: %s" % alarm_desc)
            if alarm_id == DiskPreCopy.HOT_SPARE_NOT_ENOUGH \
                    and self.pool_name in alarm_desc:
                return alarm

        return None

    def check_pre_copy_stop_alarm(self):
        hot_rec = self.get_hot_spare_alarm()
        self.log.info("get_hot_spare_alarm: %s" % str(hot_rec))
        if hot_rec:
            # 当热备空间不足时，进行高危提示，如果确认可更换，该检查通过
            dialog_util = self.context['dialogUtil']
            err_key = "replacingDisk_precopy_hotspace_not_enough"
            msg = common.getRes(self.lang, err_key)
            flag = dialog_util.showWarningDialog(msg)
            if not flag:
                err_msg, __ = common.getMsg(self.lang,
                                            "hot.spare.not.enough",
                                            errMsgArgs=self.pool_name)
                self.set_fail(err_msg)
                return True
            else:
                self.set_succes()
                return False
        return False

    def set_succes(self):
        self.status = PreCopyStatus.SUCCESS
        self.remain_time = 0
        self.progress = 100
        return

    def set_fail(self, err_msg):
        self.err_msg = err_msg
        self.status = PreCopyStatus.FAIL
        self.remain_time = 0
        self.progress = 100
        return

    def set_running(self, progress, used_time):
        if progress == 100:
            self.progress = 99
            self.remain_time = 60

        if self.progress < int(progress):
            self.progress = int(progress)

        if self.progress != 0:
            self.remain_time = int(used_time) / self.progress \
                               * (100 - self.progress)
        else:
            self.remain_time = 5

        self.status = PreCopyStatus.RUNNING
        return

    def get_pre_copy_status(self):
        """
        硬盘预拷贝的状态，进度
        :return:
        """
        pre_copy_time = time.time() - self.pre_copy_cur_start_time

        # 预拷贝超时，预拷贝失败
        if pre_copy_time > DEFAULT_TIMEOUT_SECS:
            err_msg, __ = common.getMsg(self.lang, "pre.copy.time.out",
                                        errMsgArgs=self.disk.get("location"))
            self.set_fail(err_msg)
            return

        # 查询不到硬盘信息说明硬盘故障 预拷贝成功
        try:
            rec = get_disk_info(
                self.rest, self.disk.get("id"), self.disk.get("location"))
        except Exception as ex:
            self.log.error("get disk fail: %s" % self.disk)
            self.log.error(str(ex))
            self.set_succes()
            return

        hs = restUtil.CommonRest.getRecordValue(rec, restData.RestCfg.
                                                PublicAttributes.
                                                HEALTH_STATUS)
        progress = restUtil.CommonRest.getRecordValue(rec, restData.
                                                      RestCfg.Disk.PROGRESS)
        progress = int(progress)

        # 硬盘故障，预拷贝成功
        if hs == restData.RestCfg.HealthStatusEnum.FAULT:
            self.set_succes()
            return

        # 如果30s还未启动预拷贝，则认为预拷贝失败
        if pre_copy_time > START_PRE_COPY_TIME_SECS \
                and self.status == PreCopyStatus.NOT_START:
            err_msg, __ = common.getMsg(self.lang, "pre.copy.start.time.out",
                                        errMsgArgs=self.disk.get("location"))
            self.set_fail(err_msg)
            return

        if hs == restData.RestCfg.HealthStatusEnum.PRE_FAIL:

            # 正常预拷贝
            if self.status == PreCopyStatus.NOT_START or progress > 0:
                self.set_running(progress, pre_copy_time)
                return

            # 如果2分半预拷贝进度为0，且有热备空间不足告警，则弹框确认
            if progress == 0 and self.status == PreCopyStatus.RUNNING \
                    and pre_copy_time >= PRE_COPY_PROGRESS_TIME_SECS \
                    and self.check_pre_copy_stop_alarm():
                err_msg, __ = common.getMsg(self.lang, "hot.spare.not.enough",
                                            errMsgArgs=self.pool_name)
                self.set_fail(err_msg)
                return

            # 超过6分钟，拷贝进度仍然为0，则拷贝失败
            if progress == 0 and self.status == PreCopyStatus.RUNNING \
                    and pre_copy_time > ISOLATION_DISK_TIME_SECS:
                err_msg, __ = common.getMsg(self.lang, "hot.spare.not.enough",
                                            errMsgArgs=self.pool_name)
                self.set_fail(err_msg)
                return

    def query_precopy_progress(self):
        self.get_pre_copy_status()
        return self.status, self.progress, self.remain_time, self.err_msg
