# -*- coding: UTF-8 -*-
import time
import traceback
import functools

from com.huawei.ism.exception import IsmException
from com.huawei.ism.tlv.lang import UnsignedInt32
from java.lang import Exception as JException

from cbb.business.operate.inspection import config
from cbb.frame.base import constants
from cbb.frame.base.exception import ErrorCode, ErrorCodeSet, \
    RestCmdException, TlvCmdException
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest.restUtil import CommonRestService, Tlv2Rest


def transException(fn):
    """ 将ISM异常转换为CliCmdException
        restUtil中的execCmd：
            抛出Exception异常
            CommonRest.REST_CAN_NOT_EXECUTE--
            若response中错误码不为0，则将错误码转为异常抛出
            自带3次重试

    :param fn:
    :return:
    """

    def inner(*params):
        self = params[0]
        try:
            return fn(*params)
        except IsmException as e:
            if e.getErrorId() in [-404, -1, 50334237, 1077949002]:
                return []
        except JException:
            self.logger.logInfo("java excption:%s" % traceback.format_exc())
            raise RestCmdException(ErrorCode(
                ErrorCodeSet.REST_CONNECTION_EXCEPTION))

    return inner


def transException2(fn):
    """ 将ISM异常转换为CliCmdException
        restUtil中的execCmd：
            抛出Exception异常
            CommonRest.REST_CAN_NOT_EXECUTE--
            若response中错误码不为0，则将错误码转为异常抛出
            自带3次重试

    :param fn:
    :return:
    """

    @functools.wraps(fn)
    def inner(*params):
        self = params[0]
        # noinspection PyBroadException
        try:
            return fn(*params)
        except TlvCmdException, e:
            raise e
        except Exception:
            # noinspection PyBroadException
            try:
                return fn(*params)
            except Exception:
                self.logger.error("java excption:%s" % traceback.format_exc())
            raise TlvCmdException(ErrorCode(
                ErrorCodeSet.REST_CONNECTION_EXCEPTION))

    return inner


class RestService:
    def __init__(self, dataDict):
        self.dataDict = dataDict
        self.logger = dataDict.get('logger')

    def excuteCmd(self, cmd, params):
        """
        执行rest命令入口函数
        :param cmd: 待执行的命令
        :param params: 待执行命令的参数
        :return: 执行的结果
        """
        rest = contextUtil.getRest(self.dataDict)
        Tlv2Rest.execCmd(rest, cmd, params)

    def getHost(self):
        """ 查询主机信息 """
        rest = contextUtil.getRest(self.dataDict)
        uri = "host"
        return CommonRestService.get4Page(rest, uri)

    def getHostLink(self, hostId):
        """ 查询指定主机的链路信息"""
        rest = contextUtil.getRest(self.dataDict)
        self.logger.info("gethostlink=%s" % hostId)
        allDatas = []
        for initiatorType in ["223", "222", "16499"]:
            uri = "host_link?INITIATOR_TYPE=%s&PARENTID=%s" \
                  % (initiatorType, hostId)
            datas = CommonRestService.get4Big(rest, uri)
            if datas:
                allDatas.extend(datas)
        return allDatas

    def getFcInitiator(self, hostId):
        """ 查询指定主机的FC启动器信息 """
        rest = contextUtil.getRest(self.dataDict)
        uri = "fc_initiator?PARENTID=%s" % hostId
        return CommonRestService.get(rest, uri)

    def getIscsiInitiator(self, hostId):
        """ 查询指定主机的iSCSI启动器信息 """
        rest = contextUtil.getRest(self.dataDict)
        uri = "iscsi_initiator?PARENTID=%s" % hostId
        return CommonRestService.get(rest, uri)

    @transException
    def getIbInitiator(self, hostId):
        """ 查询指定主机的iB启动器信息 """
        rest = contextUtil.getRest(self.dataDict)
        uri = "ib_initiator/associate/host?ASSOCIATEOBJTYPE=21" \
              "&ASSOCIATEOBJID=%s" % hostId
        return CommonRestService.get(rest, uri)

    def getHostLun(self, hostId):
        """ 查询指定主机的lun信息 """
        rest = contextUtil.getRest(self.dataDict)
        uri = "lun/associate?TYPE=11&ASSOCIATEOBJTYPE=21" \
              "&ASSOCIATEOBJID=%s" % hostId
        return CommonRestService.get4Big(rest, uri)

    def getHostLunCount(self, hostId):
        """ 查询指定主机映射的lun数量"""
        rest = contextUtil.getRest(self.dataDict)
        uri = "lun/count?ASSOCIATEOBJTYPE=21&ASSOCIATEOBJID=%s" % hostId
        return CommonRestService.get(rest, uri)

    @transException
    def getLicenseFeature(self):
        """ 查询所有的增值特性 """
        rest = contextUtil.getRest(self.dataDict)
        uri = "license/feature"
        return CommonRestService.get4Big(rest, uri)

    @transException
    def getReplicationPair(self):
        """ 查询 远程复制PAIR"""
        rest = contextUtil.getRest(self.dataDict)
        uri = "REPLICATIONPAIR"
        return CommonRestService.get4Big(rest, uri)

    @transException
    def getHyperMetroPair(self):
        """ 查询 双活Pair"""
        rest = contextUtil.getRest(self.dataDict)
        uri = "HyperMetroPair"
        return CommonRestService.get4Big(rest, uri)

    @transException
    def getHyperVaultJob(self):
        """ 查询一体化备份"""
        rest = contextUtil.getRest(self.dataDict)
        uri = "HyperVaultJob"
        return CommonRestService.get4Big(rest, uri)

    @transException
    def getMemberLunCopy(self, lunCopyId):
        """ 查询指定luncopy的详细信息"""
        rest = contextUtil.getRest(self.dataDict)
        uri = "LUNCOPYMEMBERLUN/GET_BATCH_NEXT_LUNCOPY_MEMBERLUN?" \
              "LUNCOPYID=%s" % lunCopyId
        return CommonRestService.get4Big(rest, uri)

    def getHostLinkByType(self, hostId, type):
        """
        查询指定主机、类型的链路信息
        :param hostId: 主机ID
        :param type: 链路类型
        :return: 链路信息
        """
        rest = contextUtil.getRest(self.dataDict)
        self.logger.info("gethostlink=%s" % hostId)
        InitiatorTypeDict = dict(FC="223", ISCSI="222", IB="16499")
        initiatorType = InitiatorTypeDict.get(type)

        allDatas = []
        datas = self._getHostLinkByType(rest, initiatorType, hostId)
        if datas:
            allDatas.extend(datas)
        return allDatas

    @transException
    def _getHostLinkByType(self, rest, initiatorType, hostId):
        """
        查询指定主机、类型的链路信息
        :param rest: rest连接
        :param initiatorType:启动器类型
        :param hostId: 主机ID
        :return: 链路信息
        """
        uri = "host_link?INITIATOR_TYPE=%s&PARENTID=%s" \
              % (initiatorType, hostId)
        datas = CommonRestService.get4Big(rest, uri)
        return datas

    def getDataPersistence(self, domaianId, tierType,
                           diskNumInt, diskCapacityInt):
        """
        查询数据持久化信息
        :param domaianId: 域ID
        :param tierType: 层级类型
        :param diskNumInt: 硬盘数量
        :param diskCapacityInt: 硬盘容量
        :return: 数据持久化信息
        """
        rest = contextUtil.getRest(self.dataDict)
        uri = "DiskPool/expansion_data_persistent_check?ID=%s&tierType=%s" \
              "&diskNumInt=%s&diskCapacityInt=%s" \
              % (domaianId, tierType, diskNumInt, diskCapacityInt)
        data = CommonRestService.get(rest, uri)
        return data.get("dataPersistentCheckResult")

    def getHostSnapCount(self, hostId):
        """
        查询快照信息
        :param hostId: 主机ID
        :return: 快照信息
        """
        rest = contextUtil.getRest(self.dataDict)
        uri = "snapshot/count?ASSOCIATEOBJTYPE=21&ASSOCIATEOBJID=%s" % hostId
        return CommonRestService.get(rest, uri)

    def getHostLunAndSnapCount(self, hostId):
        """
        查询指定主机的LUN和快照的总数量
        :param hostId:主机ID
        :return: lun和快照的总数量
        """
        ret = self.getHostLunCount(hostId)
        # noinspection PyBroadException
        try:
            snapRet = self.getHostSnapCount(hostId)
            if snapRet and ret:
                count = int(snapRet.get("COUNT", 0)) + int(ret.get("COUNT", 0))
                return {"COUNT": str(count)}
        except Exception:
            self.logger.error("query host snap lun exception.")
        return ret

    @transException
    def getLunCopy(self):
        """
        查询LUNCopy的信息
        :return: LUNCopy的信息
        """
        rest = contextUtil.getRest(self.dataDict)
        uri = "luncopy"
        return CommonRestService.get4Page(rest, uri)

    # 端口反查检查主机冗余链路用到的接口
    def getFcInitiators(self):
        """ 查询FC启动器 """
        rest = contextUtil.getRest(self.dataDict)
        uri = "fc_initiator"
        return CommonRestService.get4Page(rest, uri)

    def getFcProts(self):
        """ 查询FC端口"""
        rest = contextUtil.getRest(self.dataDict)
        uri = "FC_PORT"
        return CommonRestService.get4Big(rest, uri)

    @transException
    def getFcoeProts(self):
        """ 查询FCOE端口"""
        rest = contextUtil.getRest(self.dataDict)
        uri = "FCoE_PORT"
        return CommonRestService.get4Big(rest, uri)

    def getFcPortLink(self, portId):
        """ 根据端口ID查询FC链路"""
        rest = contextUtil.getRest(self.dataDict)
        uri = "fc_initiator/associate?ASSOCIATEOBJTYPE=212" \
              "&ASSOCIATEOBJID=%s" % portId
        return CommonRestService.get4Page(rest, uri)

    @transException
    def getFcoePortLink(self, portId):
        """ 根据FCoE端口ID查询FC链路"""
        rest = contextUtil.getRest(self.dataDict)
        uri = "fc_initiator/associate?ASSOCIATEOBJTYPE=252" \
              "&ASSOCIATEOBJID=%s" % portId
        return CommonRestService.get4Big(rest, uri)

    def getIscsiInitiators(self):
        rest = contextUtil.getRest(self.dataDict)
        uri = "iscsi_initiator"
        return CommonRestService.get4Page(rest, uri)

    def getEthProts(self):
        rest = contextUtil.getRest(self.dataDict)
        uri = "ETH_PORT"
        return CommonRestService.get4Big(rest, uri)

    def getIscsiPortLink(self, portId):
        rest = contextUtil.getRest(self.dataDict)
        uri = "iscsi_initiator/associate?ASSOCIATEOBJTYPE=213" \
              "&ASSOCIATEOBJID=%s" % portId
        return CommonRestService.get4Page(rest, uri)

    @transException
    def getIbInitiators(self):
        rest = contextUtil.getRest(self.dataDict)
        uri = "ib_initiator"
        return CommonRestService.get4Page(rest, uri)

    @transException
    def getIbProts(self):
        rest = contextUtil.getRest(self.dataDict)
        uri = "ib_port"
        return CommonRestService.get4Big(rest, uri)

    @transException
    def getIbPortLink(self, portId):
        rest = contextUtil.getRest(self.dataDict)
        uri = "ib_initiator/associate?ASSOCIATEOBJTYPE=16500" \
              "&ASSOCIATEOBJID=%s" % portId
        return CommonRestService.get4Page(rest, uri)

    def getMappingViews(self):
        rest = contextUtil.getRest(self.dataDict)
        uri = "mappingview"
        return CommonRestService.get4Page(rest, uri)

    def getPortgroupByMapping(self, mappingviewId):
        rest = contextUtil.getRest(self.dataDict)
        uri = "portgroup/associate/mappingview?TYPE=257&ASSOCIATEOBJTYPE=245" \
              "&ASSOCIATEOBJID=%s" % mappingviewId
        return CommonRestService.get4Big(rest, uri)

    def getHostgroupByMapping(self, mappingviewId):
        rest = contextUtil.getRest(self.dataDict)
        uri = "hostgroup/associate/mappingview?TYPE=14&ASSOCIATEOBJTYPE=245" \
              "&ASSOCIATEOBJID=%s" % mappingviewId
        return CommonRestService.get4Big(rest, uri)

    def getHostByHostGroup(self, hostId):
        rest = contextUtil.getRest(self.dataDict)
        uri = "host/associate?TYPE=21&ASSOCIATEOBJTYPE=14&ASSOCIATEOBJID=%s" \
              % hostId
        return CommonRestService.get4Page(rest, uri)

    def getPortByPortGroup(self, portGroupId):
        uris = ["fc_port/associate/portgroup?TYPE=212&ASSOCIATEOBJTYPE=257"
                "&ASSOCIATEOBJID=%s" % portGroupId,
                "fcoe_port/associate/portgroup?TYPE=252&ASSOCIATEOBJTYPE=257"
                "&ASSOCIATEOBJID=%s" % portGroupId,
                "eth_port/associate/portgroup?TYPE=213&ASSOCIATEOBJTYPE=257"
                "&ASSOCIATEOBJID=%s" % portGroupId,
                "ib_port/associate/portgroup?TYPE=16500&ASSOCIATEOBJTYPE=257"
                "&ASSOCIATEOBJID=%s" % portGroupId]
        ports = []
        for uri in uris:
            tmpData = self._getPortByPortGroup(uri)
            if tmpData:
                ports.extend(tmpData)
        return ports

    @transException
    def _getPortByPortGroup(self, uri):
        rest = contextUtil.getRest(self.dataDict)
        return CommonRestService.get4Big(rest, uri)


class RestAdapter:
    def __init__(self, dataDict):
        self.dataDict = dataDict
        self.logger = dataDict.get('logger')

    @transException
    def excuteDiagnoseCmd(self, command, locationId):
        params = []
        param0 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_MSGTYPE, 0)
        param1 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_NIDLIST, "")
        param2 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_ACTIVETYPE, 0)
        param3 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_BAKPATH, "")
        param4 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_EXENODETYPE,
                  "")
        param5 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_EXEFLOW, "")
        param6 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_EXEFLOWSEGMENT,
                  "")
        param7 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_EXEATOM, "")
        param8 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_SYNCPAR, "")
        param9 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXE_UPD_PKG_TYPE, 0)
        param12 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXC_DIAG_CMD, command)
        param13 = (restData.Upgrade.NotifyExcUpgrade.CMO_EXC_CHECK_MODEL,
                   locationId)
        params.extend(
            [param0, param1, param2, param3, param4, param5, param6, param7,
             param8, param9, param12, param13])

        self.logger.info('Execute cmd [code:%s] [%s]' % (
            restData.TlvCmd.EXC_UPD, params))
        rest = contextUtil.getRest(self.dataDict)
        recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.EXC_UPD, params)
        return recs

    @transException
    def getParamStrValue(self, record, index):

        paramInfo = record.get(str(index), '')
        if paramInfo == '':
            return None
        paramValue = paramInfo.get("value")
        if paramValue == "":
            return None
        return unicode(paramValue)

    @transException
    def getParamIntValue(self, record, index):
        paramInfo = record.get(str(index), '')
        if paramInfo == '':
            return None
        paramValue = paramInfo.get("value")
        if paramValue == "":
            return None
        return int(paramValue)

    @transException2
    def uploadDiskFw(self, pkgName, pkgType, patchSize):
        dev = contextUtil.getDevObj(self.dataDict).getDevNode()
        protocalContext = self.dataDict.get("protocalContext")
        sftp = protocalContext.get("SFTP")

        # 获取补丁包上传路径
        params = []
        param0 = (restData.Upgrade.GetPackageUploadPath.CMO_PACKAGE_TYPE,
                  UnsignedInt32(pkgType))
        param2 = (restData.Upgrade.GetPackageUploadPath.CMO_PACKAGE_SIZE,
                  UnsignedInt32(patchSize))
        params.extend([param0, param2])
        rest = contextUtil.getRest(self.dataDict)
        retRec = Tlv2Rest.execCmd(
            rest, restData.TlvCmd.OM_MSG_OP_GET_PACKAGE_UPLOADPATH,
            params, False)
        self.logger.info('TLV cmd [%d] send[%s] receive[%s]' % (
            constants.TLV_CMD.OM_MSG_OP_GET_PACKAGE_UPLOADPATH,
            str(params), str(retRec)))
        uploadPath = self.getParamStrValue(retRec, 1)

        # 上传升级包
        # noinspection PyBroadException
        try:
            sftp.putFile(dev.getIp(), pkgName, uploadPath,
                         dev.getLoginUser(), None)
        except Exception:
            raise TlvCmdException(ErrorCode(
                ErrorCodeSet.REST_CONNECTION_EXCEPTION))

        # 通知其他控制器同步升级包
        params = []
        self.logger.info("pkg upload ip = %s, uploadpath=%s" % (
            str(dev.getIp()), uploadPath))
        devPkgPath = dev.getIp() + ":" + uploadPath
        param0 = (restData.Upgrade.NotifyPackagePath.CMO_NOTIFY_PACKAGE_PATH,
                  devPkgPath)
        params.append(param0)
        rest = contextUtil.getRest(self.dataDict)
        rec = Tlv2Rest.execCmd(rest,
                               restData.TlvCmd.OM_MSG_OP_NOTIFY_PACKAGE_PATH,
                               params, False)
        self.logger.info('TLV cmd [%d] send[%s] receive[%s]' % (
            constants.TLV_CMD.OM_MSG_OP_NOTIFY_PACKAGE_PATH,
            str(params), str(rec)))

    def deleteUpgradePkg(self, pkgType):
        params = []
        param0 = (restData.Upgrade.GetPackageUploadPath.CMO_PACKAGE_TYPE,
                  UnsignedInt32(pkgType))
        param2 = (restData.Upgrade.GetPackageUploadPath.CMO_PACKAGE_SIZE,
                  UnsignedInt32(0))
        params.extend([param0, param2])
        # noinspection PyBroadException
        try:
            rest = contextUtil.getRest(self.dataDict)
            retRec = Tlv2Rest.execCmd(
                rest, restData.TlvCmd.OM_MSG_OP_GET_PACKAGE_UPLOADPATH,
                params, False)
            self.logger.info('TLV cmd [%d] send[%s] receive[%s]' % (
                constants.TLV_CMD.OM_MSG_OP_GET_PACKAGE_UPLOADPATH,
                str(params), str(retRec)))
        except Exception:
            # noinspection PyBroadException
            try:
                rest = contextUtil.getRest(self.dataDict)
                retRec = Tlv2Rest.execCmd(
                    rest, restData.TlvCmd.OM_MSG_OP_GET_PACKAGE_UPLOADPATH,
                    params, False)
                self.logger.info('TLV cmd [%d] send[%s] receive[%s]' % (
                    constants.TLV_CMD.OM_MSG_OP_GET_PACKAGE_UPLOADPATH,
                    str(params), str(retRec)))
            except Exception:
                self.logger.error("Notify OM delete Package failed.")

    @transException2
    def queryExcuteResult(self):
        params = []
        rest = contextUtil.getRest(self.dataDict)
        rec = Tlv2Rest.execCmd(
            rest, restData.TlvCmd.OM_MSG_OP_UPD_LST_SYS_PROGRESS,
            params, False)
        self.logger.info('TLV cmd [%d] send[%s] receive[%s]' % (
            constants.TLV_CMD.OM_MSG_OP_UPD_LST_SYS_PROGRESS,
            str(params), str(rec)))
        status = self.getParamIntValue(rec, 3)
        return status

    @transException2
    def queryExcuteDetail(self):
        errMsg = ''
        resource = self.dataDict.get("resource")

        # 查询处理详细信息
        params = []
        rest = contextUtil.getRest(self.dataDict)
        recs = Tlv2Rest.execCmd(
            rest, restData.TlvCmd.OM_MSG_OP_UPD_LIST_DETAILINFO,
            params, True)
        self.logger.info('TLV cmd [%d] send[%s] receive[%s]' % (
            constants.TLV_CMD.OM_MSG_OP_UPD_LIST_DETAILINFO,
            str(params), str(recs)))

        # 解析处理详细信息，生成返回信息
        for rec in recs:
            nameKey = self.getParamStrValue(rec, 0)
            errorKey = self.getParamStrValue(rec, 2)
            nodeId = self.getParamStrValue(rec, 3)
            errMsg += "%s--%s--%s\n" \
                      % (nodeId, resource.getString(nameKey),
                         resource.getString(errorKey))
        return errMsg

    def upgradeDisksFw(self, diskUpgradeMode, upgradeType=0, diskIdStr=None):
        # 0-升级方式（1-全盘升级；2-单盘升级；3-多盘升级）
        # 4-硬盘框号（）
        params = []
        param0 = (restData.Upgrade.UpgradeDisk.CMO_UPGRADE_MODE,
                  int(diskUpgradeMode))
        param5 = (restData.Upgrade.UpgradeDisk.CMO_UPGRADE_TYPE, upgradeType)
        params.extend([param0, param5])

        if diskIdStr:
            param4 = (restData.Upgrade.UpgradeDisk.CMO_UPGRADE_DISKS,
                      diskIdStr)
            params.append(param4)
        self.logger.info(
            'Execute cmd [%d] [%s]'
            % (constants.TLV_CMD.OM_MSG_OP_UPD_DISK_FIRMWARE,
               str(params)))
        # 获取补丁包上传路径
        # noinspection PyBroadException
        try:
            rest = contextUtil.getRest(self.dataDict)
            retRec = Tlv2Rest.execCmd(
                rest, restData.TlvCmd.OM_MSG_OP_UPD_DISK_FIRMWARE,
                params, False)
            self.logger.info("invoke tlv disk upgrade response=%s" % retRec)

            # TODO 系统正在进行硬盘升级，该命令会报错；重试升级时，忽略该错（提示正在升级）
        except Exception:
            # noinspection PyBroadException
            try:
                # 重新获取TLV连接，重试一次
                time.sleep(5)
                rest = contextUtil.getRest(self.dataDict)
                retRec = Tlv2Rest.execCmd(
                    rest, restData.TlvCmd.OM_MSG_OP_UPD_DISK_FIRMWARE,
                    params, False)
                self.logger.error("invoke tlv disk upgrade response=%s"
                                  % retRec)
            except Exception:
                self.logger.error(
                    "invoke diskupgrade tlv cmd exception:%s"
                    % traceback.format_exc())
                raise

    def queryDiskUpgradeResult(self, diskUpgradeMode, upgradeType=0):
        # noinspection PyBroadException
        try:
            params = []
            param0 = (restData.Upgrade.UpgradeDisk.CMO_UPGRADE_MODE,
                      int(diskUpgradeMode))
            param5 = (restData.Upgrade.UpgradeDisk.CMO_UPGRADE_TYPE,
                      upgradeType)
            params.extend([param0, param5])

            if diskUpgradeMode == config.DISK_UPGRADE_MODE.MULTI:
                param4 = (restData.Upgrade.UpgradeDisk.CMO_UPGRADE_DISKS,
                          "CTE99.9")
                params.append(param4)
            self.logger.info(
                'Execute cmd [%d] [%s]'
                % (constants.TLV_CMD.OM_MSG_OP_UPD_GET_DISK_RESULT,
                   str(params)))

            rest = contextUtil.getRest(self.dataDict)
            response = Tlv2Rest.execCmd(
                rest, restData.TlvCmd.OM_MSG_OP_UPD_GET_DISK_RESULT, params)

            self.logger.info(
                "cmd=%s response=%s"
                % (constants.TLV_CMD.OM_MSG_OP_UPD_GET_DISK_RESULT,
                   str(response)))
            return response
        except Exception:
            self.logger.error("invoke diskupgrade tlv cmd exception:%s"
                              % traceback.format_exc())
            # noinspection PyBroadException
            try:
                # 重新获取REST连接
                rest = contextUtil.getRest(self.dataDict)
            except Exception:
                # 无法建立连接
                self.logger.error("create tlv con exception:%s"
                                  % traceback.format_exc())
                raise TlvCmdException(ErrorCode(
                    ErrorCodeSet.REST_CONNECTION_EXCEPTION))
            raise TlvCmdException(ErrorCode(ErrorCodeSet.COMMON_CMD_ERROR))
