﻿# -*-coding:utf-8 -*-
import os
import time
from copy import deepcopy

# noinspection PyUnresolvedReferences
from java.awt import Rectangle
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException

from cbb.business.operate.expansion import common
from cbb.frame.context import contextUtil
from cbb.frame.devGraph.devGraphCfg import DEV_CTRL_DEFAULT_START_U
from cbb.frame.devGraph.devGraphCfg import DEV_GRAPH_CFG_FILE
from cbb.frame.devGraph.devGraphCfg import DEV_GRAPH_IMG_PATH
from cbb.frame.devGraph.devGraphCfg import DEV_HD_GRAPH_CFG_FILE
from cbb.frame.devGraph.devGraphCfg import DEV_HD_GRAPH_IMG_PATH
from cbb.frame.rest.restDataConstants import ENCLOSURE_HEIGHT
from cbb.frame.rest.restDataConstants import ENCLOSURE_MODEL
from cbb.frame.tlv import tlvData
from cbb.frame.tlv import tlvUtil

# 机架的宽度 162
RACK_WIDTH = 162
# 机架的高度 462
RACK_HEIGHT = 462
# 机架高度空白
RACK_TOP_BLANK = 22
# 占位CELL的宽度  29
PLACE_HOLDER_CELL_WIDTH = 29
# 具有标题的panel高度
TITLE_PANEL_HEIGHT = 30
# 常量2，用于求某个数值的一半
HALF_TWO = 2


class DeviceStatus:
    # 默认状态
    DEFAULTS = 0
    # 正常状态
    NORMAL = 1
    # 异常状态
    ABNORMAL = 2
    # 选中状态
    SELECTED = 3
    # 点灯状态
    LIGHTING = 4


def drawGraph(context, propDict={}):
    """绘制设备图

    :param context: 上下文对象
    :param propDict: 支持如下key:
            hardwareTypeList: 可选，硬件类型集合（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"），
                               默认支持所有的硬件设备图绘制
            isBack: 可选，值为True表示绘制后视图，值为False表示绘制前视图，默认值为False
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        defaultFrontViewHardwareList = ["BAY", "ENCLOSURE", "CONTROLLER", "FAN_CTRL_MODULE", "FAN", "DISK",
                                        "BACKUP_POWER", "POWER"]

        backViewExtraHardwareList = ['INTF_MODULE',
                                     'EXPBOARD',
                                     'SAS_PORT',
                                     'ETH_PORT',
                                     'PCIE_PORT',
                                     'FCoE_PORT',
                                     'FC_PORT',
                                     'SFP_OPTICAL_TRANSCEIVER']

        hdGraphHardwareList = ['ENCLOSURE', "CONTROLLER", "POWER", 'INTF_MODULE']
        isBack = propDict.get("isBack", False)
        isHdGraph = propDict.get("isHdGraph", False)
        hardwareTypeList = defaultFrontViewHardwareList if not isBack \
            else defaultFrontViewHardwareList + backViewExtraHardwareList
        hardwareTypeList = hdGraphHardwareList if isHdGraph else propDict.get("hardwareTypeList", hardwareTypeList)
        rest = contextUtil.getTlv(context)

        hardwareGraphInfoList = []

        if isDCPower(context) and not isHdGraph:
            # 获取柜信息字典
            cabinetRecords = tlvUtil.getHardwareRecords(rest, "BAY")
            baylist = tlvUtil.getBayNames(cabinetRecords)
            powerEnclist = getPowerEnclosure(baylist)
            hardwareGraphInfoList.extend(powerEnclist)

        for hardwareType in hardwareTypeList:
            try:
                graphInfoList = getHardwareGraphList(rest, hardwareType)
                if isHdGraph:
                    ctrlEngId = propDict.get("ctrlEngId")
                    logger.logInfo("current ctrl enclosure id:%s" % ctrlEngId)
                    if 'ENCLOSURE' == hardwareType:
                        graphInfoList = deepcopy(list(filter(lambda enc: enc.get('name') == ctrlEngId, graphInfoList)))
                    elif hardwareType in ['CONTROLLER', 'POWER', 'INTF_MODULE']:
                        graphInfoList = deepcopy(
                            list(filter(lambda obj: obj.get('location').split('.')[0] == ctrlEngId, graphInfoList)))

                hardwareGraphInfoList.extend(graphInfoList)
                logger.logInfo("get hardware[%s] info successfully." % hardwareType)
            except (JException, Exception) as exception:
                logger.logInfo("get hardware[%s] info fail." % hardwareType)
                logger.logException(exception)

        logger.logInfo("dorado V6 high end hardwareList:%s" % hardwareGraphInfoList)
        if not hardwareGraphInfoList:
            return

        devGraphUtils = context.get("DevGraphUtils")

        cfgMap = {"graphConfigPath": DEV_GRAPH_CFG_FILE if not isHdGraph else DEV_HD_GRAPH_CFG_FILE,
                  "deviceImgCfgPath": DEV_GRAPH_IMG_PATH if not isHdGraph else DEV_HD_GRAPH_IMG_PATH,
                  'isHdGraph': 'true' if isHdGraph else 'false'}

        devGraphUtils.buildGraph(cfgMap, hardwareGraphInfoList, isBack, False)
        contextUtil.setItem(context, "originHardwareGraphInfoList", hardwareGraphInfoList)
        return

    except (JException, Exception) as exception:
        logger.logException(exception)

    return


def highLightHardware(context, hardwareId, hardwareType):
    """高亮硬件

    :param context: 上下文对象
    :param hardwareId: 硬件ID
    :param hardwareType: 硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备
        clearGraph(context)
        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if hardwareId == hardwareInfo["id"] and hardwareType == hardwareInfo["typeName"]:
                devGraphUtils = context.get("DevGraphUtils")
                clearGraph(context)
                devGraphUtils.highlightModule(hardwareInfo, 'selected')
                return True

    except (JException, Exception) as exception:
        logger.logException(exception)

    logger.logInfo("highlight hardware failure(hardwareId:%s, hardwareType:%s)" % (hardwareId, hardwareType))
    return False


def highLightHardwareByLocation(context, hardwareLocation, hardwareType):
    """高亮硬件

    :param context: 上下文对象
    :param hardwareLocation: 硬件location
    :param hardwareType:硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        clearGraph(context)
        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if hardwareLocation == hardwareInfo["location"] and hardwareType == hardwareInfo["typeName"]:
                devGraphUtils = context.get("DevGraphUtils")

                devGraphUtils.highlightModule(hardwareInfo, 'selected')
                return True
    except (JException, Exception) as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        pass

    logger.logInfo(
        "highlight hardware failure(hardwareLocation:%s, hardwareType:%s)" % (hardwareLocation, hardwareType))
    return False


def highLightHardwarePariByName(context, srcLoc, dstLoc):
    """高亮硬件

    :param context: 上下文对象
    :param srcLoc: 始端位置
    :param dstLoc: 末端位置
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        clearGraph(context)
        devGraphUtils = context.get("DevGraphUtils")
        hdInfoList = contextUtil.getItem(context, "hardwareList", [])
        srcHdInfo = filter(lambda hdInfo: hdInfo.get('location', '') == srcLoc, hdInfoList)
        encHdInfo = filter(lambda hdInfo: hdInfo.get('name') == srcLoc.split('.')[0].strip(), hdInfoList)

        dstHdInfo = filter(lambda hdInfo: hdInfo.get('location', '') == dstLoc, hdInfoList)

        dstHdInfoDict = deepcopy(srcHdInfo[0])
        dstHdInfoDict['location'] = dstLoc

        if srcHdInfo:
            logger.logInfo("highlighting source intf module:%s" % srcHdInfo)
            devGraphUtils.highlightEnclosureModule(srcHdInfo[0], encHdInfo[0], 'selected')

        logger.logInfo("highligting dest intf module:%s" % srcHdInfo)
        devGraphUtils.highlightEnclosureModule(dstHdInfoDict, encHdInfo[0], 'target')

    except (JException, Exception) as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        pass

    return False


def highLightHardwarePariByLoc(context, locs):
    """高亮硬件

    :param context:
    :param locs:
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        infos = []
        clearGraph(context)
        devGraphUtils = context.get("DevGraphUtils")
        hdInfoList = contextUtil.getItem(context, "hardwareList", [])
        encHdInfo = filter(lambda hdInfo: hdInfo.get('name') == locs[0].split('.')[0].strip(), hdInfoList)
        interfHdInfo = filter(lambda hdInfo: hdInfo.get('typeName', '') == "INTF_MODULE", hdInfoList)

        for loc in locs:
            info = deepcopy(interfHdInfo[0])
            info['location'] = loc
            infos.append(info)
        devGraphUtils.highlightEnclosureModules(encHdInfo[0], infos, 'target')
        logger.logInfo("highligting dest intf module:%s" % str(infos))

    except (JException, Exception) as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        pass

    return False


def highLightCable(context, port1Id, port2Id, portType):
    """高亮线缆

    :param context: 上下文对象
    :param port1Id: 源端口ID
    :param port2Id: 目的端口ID
    :param portType: 端口类型
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备
        clearGraph(context)

        port1Info = {}
        port2Info = {}
        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if portType != hardwareInfo["typeName"]:
                continue

            if len(port1Info) == 0 and port1Id == hardwareInfo["id"]:
                port1Info = hardwareInfo
                continue

            if len(port2Info) == 0 and port2Id == hardwareInfo["id"]:
                port2Info = hardwareInfo
                continue

        if len(port1Info) > 0 and len(port2Info) > 0:
            devGraphUtils = context.get("DevGraphUtils")
            clearGraph(context)
            devGraphUtils.highlightCable(port1Info, port2Info, 'selected')
            return True

    except (JException, Exception) as exception:
        logger.logException(exception)

    logger.logInfo("highlight cable failure(port1Id:%s, port2Id:%s, portType:%s)" % (port1Id, port2Id, portType))
    return False


def highLightCableByLocation(context, port1Location, port2Location, portType):
    """高亮线缆

    :param context: 上下文对象
    :param port1Location: 源端口位置
    :param port2Location: 目的端口位置
    :param portType: 端口类型
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        port1Info = {}
        port2Info = {}

        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if portType != hardwareInfo["typeName"]:
                continue

            if len(port1Info) == 0 and port1Location == hardwareInfo["location"]:
                port1Info = hardwareInfo
                continue

            if len(port2Info) == 0 and port2Location == hardwareInfo["location"]:
                port2Info = hardwareInfo
                continue
        logger.logInfo("port1Info:%s,port2Info:%s" % (str(port1Info), str(port2Info)))
        if len(port1Info) > 0 and len(port2Info) > 0:
            devGraphUtils = context.get("DevGraphUtils")
            clearGraph(context)
            devGraphUtils.highlightCable(port1Info, port2Info, 'selected')
            return True

    except (JException, Exception) as exception:
        logger.logException(exception)

    logger.logInfo(
        "highlight cable failure(port1Id:%s, port2Id:%s, portType:%s)" % (port1Location, port2Location, portType))
    return False


def clearGraph(context):
    """清除设备图高亮信息
    
    :param context: 上下文对象
    :return: 
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        devGraphUtils = context.get("DevGraphUtils")
        devGraphUtils.clearGraph()
    except (JException, Exception) as exception:
        logger.logException(exception)
    return


def getHardwareGraphList(tlv, hardwareType):
    """获取绘制硬件图的信息集合

    :param tlv: tlv对象
    :param hardwareType: 硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    :return: 所有硬件对象需要包含如下字段：typeName,id,location,modelName,parentId,parentType,healthStatus
             框对象需要包含如下字段：name,SAS_DAE_START_U,heightU,logicTypeName
             端口对象需要包含如下字段：runningStatus,currentPeerId,suggestPeerId
    """
    enclosureHeight = tlvUtil.getCtrlEnclosureHeight(tlv)
    hardwareGraphList = []

    records = tlvUtil.getHardwareRecords(tlv, hardwareType)
    for record in records:
        infoDict = dict()
        infoDict["typeName"] = hardwareType
        infoDict["id"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
        infoDict["name"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"])
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        infoDict["location"] = location
        infoDict["oldLoaction"] = location
        infoDict["modelName"] = getModelName(record, hardwareType, enclosureHeight)
        infoDict["parentId"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["parentID"])
        infoDict["parentType"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["parentType"])
        infoDict["healthStatus"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["healthStatus"])
        infoDict["runningStatus"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        infoDict["startU"] = ""
        infoDict["heightU"] = ""
        infoDict["logicTypeName"] = ""
        infoDict["currentPeerId"] = ""
        infoDict["suggestPeerId"] = ""

        ctrlEncDftStartU = DEV_CTRL_DEFAULT_START_U
        if hardwareType == "ENCLOSURE":
            if "0.0U" in location:
                continue
            logicType = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["logicType"])
            logicTypeName = tlvData.ENCLOSURE_TYPE.get(logicType, "")

            height = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["height"])
            # if height can not be got, the disk enclosure may be fault or pulled out.
            if not height:
                continue

            if location in ['', '--']:
                startU = str(DEV_CTRL_DEFAULT_START_U) if logicTypeName == 'CTRL' \
                    else str(ctrlEncDftStartU - height)
                ctrlEncDftStartU -= height
                location = 'SMB0.%sU' % startU
            else:
                startU = location.split(".")[-1].replace("U", "")

            infoDict["location"] = location
            infoDict["startU"] = startU
            infoDict["heightU"] = height
            infoDict["logicTypeName"] = logicTypeName

        elif hardwareType == "SAS_PORT":
            infoDict["currentPeerId"] = tlvUtil.getRecordValue(record, tlvData.SAS_PORT["current_peer_port_id"])
            infoDict["suggestPeerId"] = tlvUtil.getRecordValue(record, tlvData.SAS_PORT["suggest_peer_port_id"])

        elif hardwareType == "PCIE_PORT":
            infoDict["currentPeerId"] = tlvUtil.getRecordValue(record, tlvData.PCIE_PORT["current_peer_port_id"])
            infoDict["suggestPeerId"] = tlvUtil.getRecordValue(record, tlvData.PCIE_PORT["suggest_peer_port_id"])

        for key in infoDict.keys():
            infoDict[key] = str(infoDict[key])

        hardwareGraphList.append(infoDict.copy())

    return hardwareGraphList


def getModelName(record, hardwareType, enclosureHeight):
    """获取设备图显示需要的modelName

    :param record: tlv记录
    :param hardwareType: 硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    :param enclosureHeight: 框高（2U对应2,3U对应3,6U对应6）
    :return: 对于框对象、接口卡对象、级联板对象，返回model对应的枚举值（2U/3U/6U接口卡特殊处理）
    """
    modelName = ""
    if hardwareType == "ENCLOSURE":
        model = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["model"])
        modelName = tlvData.ENCLOSURE_MODEL.get(model, "")
    elif hardwareType == "INTF_MODULE":
        suffixDict = {2: "_CTRL_2U", 3: "_CTRL_3U", 6: ""}  # 6U取默认值
        suffix = suffixDict.get(enclosureHeight, "")
        model = tlvUtil.getRecordValue(record, tlvData.INTF_MODULE["model"])
        modelName = "%s%s" % (tlvData.INTF_MODEL.get(model, ""), suffix)
    elif hardwareType == "EXPBOARD":
        model = tlvUtil.getRecordValue(record, tlvData.EXPBOARD["model"])
        modelName = tlvData.EXPBOARD_TYPE.get(model, "")
    return modelName


def createBaysGraph(bayNameList):
    """绘制指定数量的机柜

    :param bayNameList: 柜名列表
    :return:
    """
    if not bayNameList:
        return []

    bayInfoList = []

    for bayName in bayNameList:
        infoDict = {}
        infoDict["typeName"] = 'BAY'
        infoDict["id"] = bayName
        infoDict["name"] = bayName
        infoDict["location"] = ''
        infoDict["oldLoaction"] = ''
        infoDict["modelName"] = ''
        infoDict["parentId"] = 'None'
        infoDict["parentType"] = '0'
        infoDict["healthStatus"] = '0'
        infoDict["runningStatus"] = '0'
        infoDict["startU"] = ''
        infoDict["heightU"] = ''
        infoDict["logicTypeName"] = ""
        infoDict["currentPeerId"] = "--"
        infoDict["suggestPeerId"] = "--"

        bayInfoList.append(infoDict.copy())

    return bayInfoList


def createEncGraph(context, encList):
    """在机柜下绘制硬盘框[{"startU":1,"name":"DAE000","bay":"SMB0"},{"startU":9,"name":"DAE010","bay":"SMB0"}]

    :param context: 上下文对象
    :param encList: 框列表
    :return:
    """
    encInfoList = []
    ENC_HEIGHT = {"EXPSAS2U_25": 2, "EXPSAS4U_24_NEW": 4, 118: 2}
    count = 0  # 辅助生成唯一id
    for enclosure in encList:
        count += 1

        encName = enclosure["name"]
        encType = enclosure["type"]
        startU = enclosure["startU"]
        bayName = enclosure["bay"]  # bay的id和name相同
        encParentId = enclosure["bay"]
        height = ENCLOSURE_HEIGHT.get(encType, 2)
        location = '%s.%sU' % (bayName, str(startU))

        infoDict = {}
        infoDict["typeName"] = 'ENCLOSURE'
        infoDict["id"] = str(time.time() + count)
        infoDict["name"] = encName
        infoDict["location"] = location
        infoDict["oldLoaction"] = location
        infoDict["modelName"] = ENCLOSURE_MODEL.get(encType, '')
        infoDict["parentId"] = encParentId
        infoDict["parentType"] = '205'
        infoDict["healthStatus"] = '0'
        infoDict["runningStatus"] = '0'
        infoDict["startU"] = str(startU)
        infoDict["heightU"] = str(height)
        infoDict["logicTypeName"] = "EXP"
        infoDict["currentPeerId"] = "--"
        infoDict["suggestPeerId"] = "--"

        encInfoList.append(infoDict.copy())

    return encInfoList


def createEmptyCtrl(bay, startU):
    encParentId = bay['id']
    bayName = bay['name']
    height = 3
    location = '%s.%sU' % (bayName, str(height + startU))

    infoDict = {}
    infoDict["typeName"] = 'ENCLOSURE'
    infoDict["id"] = str(time.time())
    infoDict["name"] = "CTRL_3U_EMPTY"
    infoDict["location"] = location
    infoDict["oldLoaction"] = location
    infoDict["modelName"] = "CTRL_3U_EMPTY"
    infoDict["parentId"] = encParentId
    infoDict["parentType"] = '205'
    infoDict["healthStatus"] = '0'
    infoDict["runningStatus"] = '0'
    infoDict["startU"] = str(startU)
    infoDict["heightU"] = str(height)
    infoDict["logicTypeName"] = "CTRL"
    infoDict["currentPeerId"] = "--"
    infoDict["suggestPeerId"] = "--"

    return infoDict


def createEncComponent(encInfoList):
    """绘制指定硬盘框下的组件（"FAN","BACKUP_POWER", "POWER", "EXPBOARD", "SAS_PORT"）

    :param encInfoList: 框信息列表
    :return:
    """
    ENCSLOTNAME_DICT = {'FAN': ['FAN 0', 'FAN 1'], 'POWER': ['PSU 0', 'PSU 1'], 'EXPBOARD': ['A', 'B'],
                        'SAS_PORT': ['PRI', 'EXP']}
    encCompmentList = []
    expBoardList = []
    for enc in encInfoList:
        parentId = enc['id']
        encName = enc['name']
        encParentId = enc['parentId']

        for componentType in ENCSLOTNAME_DICT.keys():

            if componentType == 'SAS_PORT':
                continue

            modelName = ''
            if componentType == 'EXPBOARD':
                modelName = 'SAS'

            componentList = ENCSLOTNAME_DICT[componentType]
            for component in componentList:
                componentName = encName + '.' + component
                compnentId = encParentId + '.' + componentName
                currentPeerId = '--'
                suggestPeerId = '--'

                infoDict = {}
                infoDict["id"] = compnentId
                infoDict["typeName"] = componentType
                infoDict["name"] = component
                infoDict["location"] = componentName
                infoDict["oldLoaction"] = componentName
                infoDict["modelName"] = modelName
                infoDict["parentId"] = parentId
                infoDict["parentType"] = '206'
                infoDict["healthStatus"] = '1'
                infoDict["runningStatus"] = '27'
                infoDict["startU"] = ''
                infoDict["heightU"] = ''
                infoDict["logicTypeName"] = ''
                infoDict["currentPeerId"] = currentPeerId
                infoDict["suggestPeerId"] = suggestPeerId

                encCompmentList.append(infoDict.copy())
                if componentType == 'EXPBOARD':
                    expBoardList.append(infoDict.copy())

    # 绘制SAS端口
    if len(expBoardList) != 0:
        for expBoard in expBoardList:
            for sasPortType in ENCSLOTNAME_DICT['SAS_PORT']:
                sasPortParentId = expBoard['id']
                sasPortLocation = expBoard['location'] + '.' + sasPortType
                sasPortNameId = sasPortParentId.split('.')[0] + '.' + sasPortLocation
                currentPeerId = '--'
                suggestPeerId = '--'

                infoDict = {}
                infoDict["id"] = sasPortNameId
                infoDict["typeName"] = 'SAS_PORT'
                infoDict["name"] = sasPortType
                infoDict["location"] = sasPortLocation
                infoDict["oldLoaction"] = sasPortLocation
                infoDict["modelName"] = ''
                infoDict["parentId"] = sasPortParentId
                infoDict["parentType"] = '208'
                infoDict["healthStatus"] = '1'
                infoDict["runningStatus"] = '27'
                infoDict["startU"] = ''
                infoDict["heightU"] = ''
                infoDict["logicTypeName"] = ''
                infoDict["currentPeerId"] = currentPeerId
                infoDict["suggestPeerId"] = suggestPeerId

                encCompmentList.append(infoDict.copy())
    return encCompmentList


def refreshGraph(context, isBack):
    """选择硬盘柜或硬盘框时刷新设备图

    :param context: 上下文对象
    :param isBack: 可选，值为True表示绘制后视图，值为False表示绘制前视图，默认值为False
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除设备图
        logger.logInfo("clearing graph....")
        clearGraph(context)

        originHardwareGraphInfoList = contextUtil.getItem(context, 'originHardwareGraphInfoList', [])
        expBayAndDaeGraphInfoList = contextUtil.getItem(context, 'expBayAndDaeGraphInfoList', [])
        hardware_list = contextUtil.getItem(context, 'hardwareList', [])
        newBayList = contextUtil.getItem(context, "newBayList", [])
        logger.logInfo("newBayList:%s" % str(newBayList))

        originAndExpHdGraphInfoList = []
        originAndExpHdGraphInfoList.extend(originHardwareGraphInfoList)
        originAndExpHdGraphInfoList.extend(expBayAndDaeGraphInfoList)
        originAndExpHdGraphInfoList.extend([item for item in hardware_list if item not in originAndExpHdGraphInfoList])

        if isDCPower(context):
            powerEnclist = getPowerEnclosure(newBayList)
            originAndExpHdGraphInfoList.extend(powerEnclist)
        logger.logInfo("refresh with hardwareList:%s" % str(originAndExpHdGraphInfoList))

        devGraphUtils = context.get("DevGraphUtils")

        cfgMap = {"graphConfigPath": DEV_GRAPH_CFG_FILE,
                  "deviceImgCfgPath": DEV_GRAPH_IMG_PATH}

        devGraphUtils.buildGraph(cfgMap, originAndExpHdGraphInfoList, isBack, False)
        contextUtil.setItem(
            context, "originAndExpHdGraphInfoList",
            originAndExpHdGraphInfoList)

    except (JException, Exception) as exception:
        logger.logException(exception)
    return


def highLightNewDiskBayOrDae(context):
    """高亮硬盘柜

    :param context: 上下文对象
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        clearGraph(context)
        devGraphUtils = context.get("DevGraphUtils")
        expBayAndDaeGraphInfoList = contextUtil.getItem(context, 'expBayAndDaeGraphInfoList', [])
        devGraphUtils.highlightModuleList(expBayAndDaeGraphInfoList, 'selected')
    except (JException, Exception) as exception:
        logger.logException(exception)

    return


def lightDiskEnc(context):
    """点亮所有硬盘框

    :param context: 上下文对象
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备
        clearGraph(context)

        devGraphUtils = context.get("DevGraphUtils")
        hardwareList = contextUtil.getItem(context, "hardwareList")
        hardwareType = "ENCLOSURE"
        logicTypeName = "EXP"
        for hardwareInfo in hardwareList:
            if logicTypeName == hardwareInfo["logicTypeName"]:
                try:
                    devGraphUtils.highlightModule(hardwareInfo, 'selected')
                    logger.logInfo("success to highlight hardware(hardwareId:%s, hardwareType:%s)" % (
                        hardwareInfo["id"], hardwareType))
                except (JException, Exception) as exception:
                    logger.logInfo("fail to highlight hardware(hardwareId:%s, hardwareType:%s)" % (
                        hardwareInfo["id"], hardwareType))
                    logger.logException(exception)

    except (JException, Exception) as exception:
        logger.logException(exception)

    return


def lightNewDiskEnclosures(context):
    """高亮所有新增硬盘框

    :param context: 上下文对象
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 获取新增硬盘框
        newEnc2UList = contextUtil.getItem(context, "newEnc2UList")
        newEnc4UList = contextUtil.getItem(context, "newEnc4UList")
        allnewEncList = newEnc2UList + newEnc4UList

        hardwareList = contextUtil.getItem(context, "hardwareList")
        # 清除高亮设备
        clearGraph(context)

        # 绘制设备图
        devGraphUtils = context.get("DevGraphUtils")

        cfgMap = {"graphConfigPath": DEV_GRAPH_CFG_FILE, "deviceImgCfgPath": DEV_GRAPH_IMG_PATH}

        devGraphUtils.buildGraph(cfgMap, hardwareList, True, False)

        # 高亮新增硬盘框
        hardwareType = "ENCLOSURE"
        encName = ''
        for encInfo in allnewEncList:
            try:
                encName = encInfo[0]
                for hardwareInfo in hardwareList:
                    if encName == hardwareInfo["name"] and hardwareType == hardwareInfo["typeName"]:
                        devGraphUtils.highlightModule(hardwareInfo, 'selected')
                        logger.logInfo(
                            "success to highlight hardware(hardwareName:%s, hardwareType:%s)" % (encName, hardwareType))
                        break
            except (JException, Exception) as exception:
                logger.logInfo("fail to highlight hardware(hardwareName:%s, hardwareType:%s)" % (encName, hardwareType))
                logger.logException(exception)

    except (JException, Exception) as exception:
        logger.logException(exception)
    return


def getBayLocation(index):
    """获取新Bay的位置

    :param index: 在设备图中的相对位置编号
    :return:
    """
    x = index * RACK_WIDTH + PLACE_HOLDER_CELL_WIDTH / HALF_TWO
    y = TITLE_PANEL_HEIGHT

    height = RACK_HEIGHT
    width = RACK_WIDTH - PLACE_HOLDER_CELL_WIDTH + HALF_TWO
    return Rectangle(x, y, width, height)


def getBayIndex(allCabinets, bayName):
    """获取Bay在设备图中的相对位置编号

    :param allCabinets:所有柜
    :param bayName:柜名
    :return:
    """
    # 根据柜名称获取柜的相对位置编号，系统柜：SMB*，硬盘柜DKB*_*
    allCabinets = common.sortBayNames(allCabinets)
    return allCabinets.index(bayName)


def getPowerEnclosure(baylist):
    startLocs = [36, 39]
    encInfoList = []
    count = 0

    for startU in startLocs:
        for bayName in baylist:
            encName = "enc_power_%s" % str(count)
            encType = "POWER_DC_2U_ENCLOSURE"
            encParentId = bayName
            height = 2
            location = '%s.%sU' % (bayName, str(startU))

            count += 1
            infoDict = {}
            infoDict["typeName"] = 'ENCLOSURE'
            infoDict["id"] = str(time.time() + count)
            infoDict["name"] = encName
            infoDict["location"] = location
            infoDict["oldLoaction"] = location
            infoDict["modelName"] = encType
            infoDict["parentId"] = encParentId
            infoDict["parentType"] = '205'
            infoDict["healthStatus"] = '0'
            infoDict["runningStatus"] = '0'
            infoDict["startU"] = str(startU)
            infoDict["heightU"] = str(height)
            infoDict["logicTypeName"] = "POWER_ENCLOSURE"
            infoDict["currentPeerId"] = "--"
            infoDict["suggestPeerId"] = "--"

            encInfoList.append(infoDict.copy())
    return encInfoList


def isDCPower(context):
    powerType = contextUtil.getItem(context, "powerType")
    if powerType == tlvData.POWER_TYPE_E["DC"]:
        return True
    return False


def light_new_disk_enclosure(context):
    """高亮新接入的硬盘框

    :param context: 环境变量
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备
        clearGraph(context)

        dev_graph_utils = context.get("DevGraphUtils")
        hardware_list = contextUtil.getItem(
            context, 'originAndExpHdGraphInfoList', [])
        logger.logInfo("hardware_list is: %s" % str(hardware_list))
        # 新接入的硬盘框列表 在checkPowerOnForDoradoV6HighEnd中保存
        new_enclosure_list = contextUtil.getItem(context, "newInsertedEncList")
        logger.logInfo("new enclosure is: %s" % str(new_enclosure_list))
        hardware_type = "ENCLOSURE"
        logic_type_name = "EXP"
        for hardwareInfo in hardware_list:
            if logic_type_name == hardwareInfo["logicTypeName"] and \
                    hardwareInfo["name"] in new_enclosure_list:
                try:
                    dev_graph_utils.highlightModule(hardwareInfo, 'selected')
                    logger.logInfo(
                        "success to highlight (Id:%s, Type:%s)" % (
                            hardwareInfo["id"], hardware_type))
                except (JException, Exception) as exception:
                    logger.logInfo(
                        "fail to highlight (Id:%s, Type:%s)" % (
                            hardwareInfo["id"], hardware_type))
                    logger.logException(exception)

    except (JException, Exception) as exception:
        logger.logException(exception)

    return
