# -*- coding: utf-8 -*-
"""提供解析表达式语法（Parsing Expression Grammars）的实现。"""

import collections
import copy
import inspect
import pprint
import re
import sre_constants
import string
import sys
import traceback
import types
import warnings
from collections import Iterable
from collections import MutableMapping, Mapping
from collections import OrderedDict as _OrderedDict
from functools import wraps
from operator import itemgetter
from threading import RLock
from weakref import ref as wkref
import __builtin__


class SimpleNamespace(object):
    pass


__compat__ = SimpleNamespace()
__compat__.collect_all_And_tokens = True

__diag__ = SimpleNamespace()

range = xrange


def _ustr(obj):
    if isinstance(obj, unicode):
        return obj

    try:
        return str(obj)
    except UnicodeEncodeError:
        def xml_char_parse_fun(one_char):
            return '\\u' + hex(int(one_char[0][2:-1]))[2:]

        ret = unicode(obj).encode(sys.getdefaultencoding(), 'xmlcharrefreplace')
        xmlcharref = Regex(r'&#\d+;')
        xmlcharref.setParseAction(xml_char_parse_fun)
        return xmlcharref.transformString(ret)


singleArgBuiltins = []

for func_name in ("sum", "len", "sorted", "reversed", "list", "tuple", "set", "any", "all", "min", "max"):
    try:
        singleArgBuiltins.append(getattr(__builtin__, func_name))
    except AttributeError:
        # ignore
        continue

_generatorType = type((y for y in range(1)))


def _escape_regex_range_chars(s):
    # 删除如下的特定字符：^-[]
    for c in r"\^-[]":
        s = s.replace(c, _bslash + c)
    s = s.replace("\n", r"\n")
    s = s.replace("\t", r"\t")
    return _ustr(s)


def _xml_escape(data):
    """清除字符串中的&, <, >, ", ',等特殊字符"""
    from_symbols = '&><"\''
    to_symbols = ('&' + s + ';' for s in "amp gt lt quot apos".split())
    for from_, to_ in zip(from_symbols, to_symbols):
        data = data.replace(from_, to_)
    return data


def _flatten(targets):
    ret = []
    for target in targets:
        if isinstance(target, list):
            ret.extend(_flatten(target))
        else:
            ret.append(target)
    return ret


def conditionAsParseAction(fn, message=None, fatal=False):
    msg = "failed user-defined condition" if message is None else message
    exc_type = ParseFatalException if fatal else ParseException
    fn = _trim_arity(fn)

    @wraps(fn)
    def parse_action(s, l, t):
        if not bool(fn(s, l, t)):
            raise exc_type(s, l, msg)

    return parse_action


nums = "0123456789"
hexnums = nums + "ABCDEFabcdef"
alphas = string.ascii_uppercase + string.ascii_lowercase
alphanums = alphas + nums
printables = "".join(c for c in string.printable if c not in string.whitespace)
_bslash = chr(92)


class ParseBaseException(Exception):
    """
    基础解析异常
    """

    def __init__(self, pstr, loc=0, msg=None, elem=None):
        if msg is not None:
            self.msg = msg
            self.pstr = pstr
        else:
            self.msg = pstr
            self.pstr = ""
        self.loc = loc
        self.args = (pstr, loc, msg)
        self.parser_element = elem

    @classmethod
    def _from_exception(cls, pe):
        return cls(pe.pstr, pe.loc, pe.msg, pe.parser_element)

    def __getattr__(self, aname):
        if aname in ("col", "column"):
            return col(self.loc, self.pstr)
        elif aname == "line":
            return line(self.loc, self.pstr)
        elif aname == "lineno":
            return lineno(self.loc, self.pstr)
        else:
            raise AttributeError(aname)

    def __str__(self):
        if self.pstr:
            if self.loc < len(self.pstr):
                found_str = (', found %r' % self.pstr[self.loc:self.loc + 1]).replace(r'\\', '\\')
            else:
                found_str = ', found end of text'
        else:
            found_str = ''
        return ("%s%s  (at char %d), (line:%d, col:%d)" %
                (self.msg, found_str, self.loc, self.lineno, self.column))

    def __repr__(self):
        return _ustr(self)

    def __dir__(self):
        return "lineno col line".split() + dir(type(self))

    def markInputline(self, markerString=">!<"):
        """
        将异常信息从入参字符串中提取出来，并标记异常位置。
        """
        line_column = self.column - 1
        line_str = self.line
        if markerString:
            line_str = "".join((line_str[:line_column],
                                markerString, line_str[line_column:]))
        return line_str.strip()


class ParseException(ParseBaseException):
    """
    当表达式不满足范式类型时抛出的异常，支持如下属性：

    - lineno - 异常文本的行
    - col - 一场文本的列
    - line - 包含异常文本的行
    """

    @staticmethod
    def explain(exc, depth=16):
        """将Python内部异常堆栈转换为导致异常被引发的表达式的列表。"""

        ret = []
        if isinstance(exc, ParseBaseException):
            ret.append(exc.line)
            ret.append(' ' * (exc.col - 1) + '^')
        ret.append("{0}: {1}".format(type(exc).__name__, exc))

        if depth is None:
            depth = sys.getrecursionlimit()
        if depth > 0:
            callers = inspect.getinnerframes(exc.__traceback__, context=depth)
            seen = set()
            for index, frames in enumerate(callers[-depth:]):
                frame = frames[0]

                frame_self = frame.f_locals.get('self', None)
                if isinstance(frame_self, ParserElement):
                    if frame.f_code.co_name not in ('parseImpl', '_parseNoCache') or frame_self in seen:
                        continue
                    seen.add(frame_self)

                    self_type = type(frame_self)
                    ret.append("{0}.{1} - {2}".format(self_type.__module__,
                                                      self_type.__name__,
                                                      frame_self))
                elif frame_self is not None:
                    self_type = type(frame_self)
                    ret.append("{0}.{1}".format(self_type.__module__,
                                                self_type.__name__))
                else:
                    frame_code = frame.f_code
                    if frame_code.co_name in ('wrapper', '<module>'):
                        continue

                    ret.append("{0}".format(frame_code.co_name))

                depth = depth - 1
                if not depth:
                    break

        return '\n'.join(ret)


class RecursiveGrammarException(Exception):
    """exception thrown by :class:`ParserElement.validate` if the
    grammar could be improperly recursive
    """

    def __init__(self, parseElementList):
        self.parse_element_trace = parseElementList

    def __str__(self):
        return "RecursiveGrammarException: %s" % self.parse_element_trace


class ParseFatalException(ParseBaseException):
    """解析错误异常"""
    pass


class ParseSyntaxException(ParseFatalException):
    """解析语法异常"""
    pass


class _ParseResultsWithOffset(object):
    def __init__(self, p1, p2):
        self.tup = (p1, p2)

    def __repr__(self):
        return repr(self.tup[0])

    def __getitem__(self, i):
        return self.tup[i]

    def setOffset(self, i):
        self.tup = (self.tup[0], i)


class ParseResults(object):
    """结构化的解析结果，提供了多种访问已解析结果的方式：

       - 作为列表访问 (``len(results)``)
       - 通过列表索引访问 (``results[0], results[1]``, 等等)
       - 通过属性访问 (``results.<resultsName>`` - see :class:`ParserElement.setResultsName`)

    示例::

        integer = Word(nums)
        date_str = (integer.setResultsName("year") + '/'
                        + integer.setResultsName("month") + '/'
                        + integer.setResultsName("day"))
        # 等价于:
        # date_str = integer("year") + '/' + integer("month") + '/' + integer("day")

        # parseString返回一个ParseResults对象
        result = date_str.parseString("2000/12/31")

        def test(s, fn=repr):
            print("%s -> %s" % (s, fn(eval(s))))
        test("list(result)")
        test("result['month']")
        test("result[0]")
        test("result.day")
        test("'minutes' in result")
        test("'month' in result")
        test("result.dump()", str)

    结果::

        list(result) -> ['2000', '/', '12', '/', '31']
        result['month'] -> '12'
        result[0] -> '2000'
        result.day -> '31'
        'minutes' in result -> False
        'month' in result -> True
        result.dump() -> ['2000', '/', '12', '/', '31']
        - day: 31
        - month: 12
        - year: 2000
    """

    def __new__(cls, toklist=None, name=None, asList=True, modal=True):
        if not isinstance(toklist, cls):
            ret_obj = object.__new__(cls)
            ret_obj.__doinit = True
            return ret_obj
        else:
            return toklist

    def __init__(self, toklist=None, name=None, asList=True, modal=True, isinstance=isinstance):
        token_list = toklist
        if self.__doinit:
            self.__doinit = False
            self.__name = None
            self.__accumNames = {}
            self.__parent = None
            self.__modal = modal
            self.__asList = asList
            self.__tokdict = dict()
            if token_list is None:
                token_list = []
            if isinstance(token_list, list):
                self.__toklist = token_list[:]
            elif isinstance(token_list, _generatorType):
                self.__toklist = list(token_list)
            else:
                self.__toklist = [token_list]

        if name:
            if not modal:
                self.__accumNames[name] = 0
            if isinstance(name, int):
                name = _ustr(name)  # will always return a str, but use _ustr for consistency
            self.__name = name
            if not (isinstance(token_list, (type(None), basestring, list)) and token_list in (None, '', [])):
                if isinstance(token_list, basestring):
                    token_list = [token_list]
                if not asList:
                    try:
                        self[name] = token_list[0]
                    except (KeyError, TypeError, IndexError):
                        self[name] = token_list
                else:
                    if not isinstance(token_list, ParseResults):
                        self[name] = _ParseResultsWithOffset(ParseResults(token_list[0]), 0)
                    else:
                        self[name] = _ParseResultsWithOffset(ParseResults(token_list.__toklist), 0)
                    self[name].__name = name

    def __setitem__(self, k, v, isinstance=isinstance):
        if isinstance(v, _ParseResultsWithOffset):
            sub = v[0]
            self.__tokdict[k] = self.__tokdict.get(k, list()) + [v]
        elif isinstance(k, (int, slice)):
            sub = v
            self.__toklist[k] = v
        else:
            sub = v
            self.__tokdict[k] = self.__tokdict.get(k, list()) + [_ParseResultsWithOffset(v, 0)]
        if isinstance(sub, ParseResults):
            sub.__parent = wkref(self)

    def __getitem__(self, i):
        if isinstance(i, (int, slice)):
            return self.__toklist[i]
        elif i in self.__accumNames:
            return ParseResults([v[0] for v in self.__tokdict[i]])
        else:
            return self.__tokdict[i][-1][0]

    def __delitem__(self, i):
        if isinstance(i, (int, slice)):
            my_len = len(self.__toklist)
            del self.__toklist[i]

            if isinstance(i, int):
                if i < 0:
                    i += my_len
                i = slice(i, i + 1)
            removed_indices = list(range(*i.indices(my_len)))
            removed_indices.reverse()
            for name, occurrences in self.__tokdict.items():
                for indices in removed_indices:
                    for key, (value, position) in enumerate(occurrences):
                        occurrences[key] = _ParseResultsWithOffset(value, position - (position > indices))
        else:
            del self.__tokdict[i]

    def __len__(self):
        return len(self.__toklist)

    def __contains__(self, k):
        return k in self.__tokdict

    def __bool__(self):
        return not not self.__toklist

    __nonzero__ = __bool__

    def __reversed__(self):
        return iter(self.__toklist[::-1])

    def __iter__(self):
        return iter(self.__toklist)

    # add support for pickle protocol
    def __setstate__(self, state):
        self.__tokdict, par, inAccumNames, self.__name = state[1]
        self.__toklist = state[0]
        self.__accumNames = {}
        self.__accumNames.update(inAccumNames)
        if par is None:
            self.__parent = None
        else:
            self.__parent = wkref(par)

    def __getstate__(self):
        return (self.__toklist, (
            self.__tokdict.copy(), self.__parent is not None and self.__parent() or None, self.__accumNames,
            self.__name))

    def __dir__(self):
        return dir(type(self)) + list(self.keys())

    def __getnewargs__(self):
        return self.__toklist, self.__name, self.__asList, self.__modal

    @classmethod
    def from_dict(cls, other, name=None):
        """
        辅助类方法，用于从字典中构造解析结果对象。如果`name`参数被指定，则会返回一个嵌套的解析结果对象。
        """

        def is_iterable(obj):
            # 判断是否为可迭代对象
            try:
                iter(obj)
            except Exception:
                # 异常，不可迭代
                return False
            else:
                return not isinstance(obj, basestring)

        ret = cls([])
        for key, value in other.items():
            if not isinstance(value, Mapping):
                ret += cls([value], name=key, asList=is_iterable(value))
            else:
                ret += cls.from_dict(value, name=key)
        return ret if name is None else cls([ret], name=name)

    def _itervalues(self):
        return (self[k] for k in self._iterkeys())

    def _iterkeys(self):
        if hasattr(self.__tokdict, "iterkeys"):
            return self.__tokdict.iterkeys()
        else:
            return iter(self.__tokdict)

    def _iteritems(self):
        return ((k, self[k]) for k in self._iterkeys())

    iterkeys = _iterkeys
    """Returns an iterator of all named result keys."""

    iteritems = _iteritems
    """Returns an iterator of all named result key-value tuples."""

    itervalues = _itervalues
    """Returns an iterator of all named result values."""

    def keys(self):
        """Returns all named result keys."""
        return list(self.iterkeys())

    def items(self):
        """Returns all named result key-values."""
        return list(self.iteritems())

    def values(self):
        """Returns all named result values."""
        return list(self.itervalues())

    def haskeys(self):
        """Since keys() returns an iterator, this method is helpful in bypassing
           code that looks for the existence of any defined results names."""
        return bool(self.__tokdict)

    def pop(self, *args, **kwargs):
        if not args:
            args = [-1]
        for key, value in kwargs.items():
            if key != 'default':
                raise TypeError("pop() got an unexpected keyword argument '%s'" % key)
            else:
                args = (args[0], value)
        if not isinstance(args[0], int) and len(args) != 1 and args[0] not in self:
            return args[1]
        else:
            index = args[0]
            ret = self[index]
            del self[index]
            return ret

    def get(self, key, defaultValue=None):
        """
        示例::

            integer = Word(nums)
            date_str = integer("year") + '/' + integer("month") + '/' + integer("day")

            result = date_str.parseString("1999/12/31")
            print(result.get("hour")) # -> None
            print(result.get("hour", "not specified")) # -> 'not specified'
            print(result.get("year")) # -> '1999'
        """
        return defaultValue if key not in self else self[key]

    def append(self, item):
        self.__toklist.append(item)

    def insert(self, index, insStr):
        """
        类似于 ``list.insert()``。

        示例::

            print(OneOrMore(Word(nums)).parseString("0 123 321")) # -> ['0', '123', '321']

            # 自定义parseAction修改结果数据
            def insert_locn(locn, tokens):
                tokens.insert(0, locn)
            # 结果 [0, '0', '123', '321']
            print(OneOrMore(Word(nums)).addParseAction(insert_locn).parseString("0 123 321"))
        """
        self.__toklist.insert(index, insStr)
        for item_name, occurrences in self.__tokdict.items():
            for k, (value, position) in enumerate(occurrences):
                occurrences[k] = _ParseResultsWithOffset(value, position + (position > index))

    def extend(self, itemseq):
        if isinstance(itemseq, ParseResults):
            self.__iadd__(itemseq)
        else:
            self.__toklist.extend(itemseq)

    def clear(self):
        del self.__toklist[:]
        self.__tokdict.clear()

    def __add__(self, other):
        ret = self.copy()
        ret += other
        return ret

    def __getattr__(self, name):
        try:
            return self[name]
        except KeyError:
            return ""

    def __radd__(self, other):
        if isinstance(other, int) and other == 0:
            return self.copy()

        return other + self

    def __iadd__(self, other):
        if other.__tokdict:
            offset = len(self.__toklist)

            def add_offset(value):
                return offset if value < 0 else value + offset

            other_items = other.__tokdict.items()
            other_dict_items = [(key, _ParseResultsWithOffset(value[0], add_offset(value[1])))
                                for key, value_list in other_items for value in value_list]
            for key, value in other_dict_items:
                self[key] = value
                if isinstance(value[0], ParseResults):
                    value[0].__parent = wkref(self)

        self.__toklist += other.__toklist
        self.__accumNames.update(other.__accumNames)
        return self

    def __str__(self):
        return '[' + ', '.join(_ustr(i) if isinstance(i, ParseResults) else repr(i) for i in self.__toklist) + ']'

    def __repr__(self):
        return "(%s, %s)" % (repr(self.__toklist), repr(self.__tokdict))

    def _asStringList(self, sep=''):
        out = []
        for item in self.__toklist:
            if out and sep:
                out.append(sep)
            if not isinstance(item, ParseResults):
                out.append(_ustr(item))
            else:
                out += item._asStringList()
        return out

    def asList(self):
        """
        Returns the parse results as a nested list of matching tokens, all converted to strings.

        示例::

            patt = OneOrMore(Word(alphas))
            result = patt.parseString("abc bac abc")
            # even though the result prints in string-like form, it is actually a expparser ParseResults
            print(type(result), result) # -> <class 'expparser.ParseResults'> ['abc', 'bac', 'abc']

            # Use asList() to create an actual list
            result_list = result.asList()
            print(type(result_list), result_list) # -> <class 'list'> ['abc', 'bac', 'abc']
        """
        return [res.asList() if isinstance(res, ParseResults) else res for res in self.__toklist]

    def copy(self):
        ret = ParseResults(self.__toklist)
        ret.__tokdict = dict(self.__tokdict.items())
        ret.__name = self.__name
        ret.__parent = self.__parent
        ret.__accumNames.update(self.__accumNames)
        return ret

    def asDict(self):
        """
        Returns the named parse results as a nested dictionary.

        示例::

            integer = Word(nums)
            date_str = integer("year") + '/' + integer("month") + '/' + integer("day")

            result = date_str.parseString('12/31/1999')
            # -> <class 'expparser.ParseResults'> (['12', '/', '31', '/', '1999'],
                {'day': [('1999', 4)], 'year': [('12', 0)], 'month': [('31', 2)]})
            print(type(result), repr(result))

            result_dict = result.asDict()
            # 结果 <class 'dict'> {'day': '1999', 'year': '12', 'month': '31'}
            print(type(result_dict), repr(result_dict))

            import json
            # -> Exception: TypeError: ... is not JSON serializable
            print(json.dumps(result))
            # -> {"month": "31", "day": "1999", "year": "12"}
            print(json.dumps(result.asDict()))
        """
        item_fn = self.iteritems

        def to_item(obj):
            if isinstance(obj, ParseResults):
                if obj.haskeys():
                    return obj.asDict()
                else:
                    return [to_item(v) for v in obj]
            else:
                return obj

        return dict((k, to_item(v)) for k, v in item_fn())

    def asXML(self, doctag=None, namedItemsOnly=False, indent="", formatted=True):
        """
        已弃用。(Deprecated) Returns the parse results as XML.
        Tags are created for tokens and lists that have defined results names.
        """
        new_line = "\n"
        out = []
        named_items = dict((v[1], k) for (k, vlist) in self.__tokdict.items()
                           for v in vlist)
        next_level_indent = indent + "  "

        # collapse out indents if formatting is not desired
        if not formatted:
            indent = ""
            next_level_indent = ""
            new_line = ""

        self_tag = None
        if doctag is not None:
            self_tag = doctag
        else:
            if self.__name:
                self_tag = self.__name

        if not self_tag:
            if namedItemsOnly:
                return ""
            else:
                self_tag = "ITEM"

        out += [new_line, indent, "<", self_tag, ">"]

        for i, res in enumerate(self.__toklist):
            if isinstance(res, ParseResults):
                if i in named_items:
                    out += [res.asXML(named_items[i],
                                      namedItemsOnly and doctag is None,
                                      next_level_indent,
                                      formatted)]
                else:
                    out += [res.asXML(None,
                                      namedItemsOnly and doctag is None,
                                      next_level_indent,
                                      formatted)]
            else:
                # individual token, see if there is a name for it
                res_tag = None
                if i in named_items:
                    res_tag = named_items[i]
                if not res_tag:
                    if namedItemsOnly:
                        continue
                    else:
                        res_tag = "ITEM"
                xml_body_text = _xml_escape(_ustr(res))
                out += [new_line, next_level_indent, "<", res_tag, ">",
                        xml_body_text,
                        "</", res_tag, ">"]

        out += [new_line, indent, "</", self_tag, ">"]
        return "".join(out)

    def __lookup(self, sub):
        for key, value_list in self.__tokdict.items():
            for v, loc in value_list:
                if sub is v:
                    return key
        return None

    def dump(self, indent='', full=True, include_list=True, _depth=0):
        """
        Diagnostic method for listing out the contents of a :class:`ParseResults`.
        Accepts an optional ``indent`` argument so that
        this string can be embedded in a nested display of other data.

        示例::

            integer = Word(nums)
            date_str = integer("year") + '/' + integer("month") + '/' + integer("day")

            result = date_str.parseString('20/01/1999')
            print(result.dump())

        结果::

            ['20', '/', '01', '/', '1999']
            - day: 1999
            - month: 01
            - year: 20
        """
        new_line = '\n'
        out = []
        if not include_list:
            out.append('')
        else:
            out.append(indent + _ustr(self.asList()))

        if full:
            if self.haskeys():
                items = sorted((str(k), v) for k, v in self.items())
                for key, value in items:
                    if out:
                        out.append(new_line)
                    out.append("%s%s- %s: " % (indent, ('  ' * _depth), key))
                    if isinstance(value, ParseResults):
                        if not value:
                            out.append(_ustr(value))
                        else:
                            out.append(
                                value.dump(indent=indent, full=full, include_list=include_list, _depth=_depth + 1))
                    else:
                        out.append(repr(value))
            elif any(isinstance(vv, ParseResults) for vv in self):
                value = self
                for index, v_value in enumerate(value):
                    if not isinstance(v_value, ParseResults):
                        out.append("\n%s%s[%d]:\n%s%s%s" % (indent,
                                                            ('  ' * (_depth)),
                                                            index,
                                                            indent,
                                                            ('  ' * (_depth + 1)),
                                                            _ustr(v_value)))
                    else:
                        out.append("\n%s%s[%d]:\n%s%s%s" % (indent,
                                                            ('  ' * (_depth)),
                                                            index,
                                                            indent,
                                                            ('  ' * (_depth + 1)),
                                                            v_value.dump(indent=indent, full=full,
                                                                         include_list=include_list, _depth=_depth + 1)))

        return "".join(out)

    def getName(self):
        r"""
        Returns the results name for this token expression. Useful when several
        different expressions might match at a particular location.

        示例::

            integer = Word(nums)
            expr = Regex(r"\d\d\d-\d\d-\d\d\d\d")
            house_number = Suppress('#') + Word(nums, alphanums)
            user_data = (Group(house_number)("house_number")
                        | Group(expr)("ssn")
                        | Group(integer)("age"))
            user_info = OneOrMore(user_data)

            result = user_info.parseString("11 222-33-4444 #221B")
            for value in result:
                print(value.getName(), ':', value[0])

        结果::

            age : 11
            ssn : 222-33-4444
            house_number : 221B
        """
        if self.__name:
            return self.__name
        elif self.__parent:
            parent = self.__parent()
            if not parent:
                return None
            else:
                return parent.__lookup(self)
        elif (len(self) == 1
              and len(self.__tokdict) == 1
              and next(iter(self.__tokdict.values()))[0][1] in (0, -1)):
            next_value = next(iter(self.__tokdict.keys()))
            return next_value
        else:
            # nothing
            return None

    def pprint(self, *args, **kwargs):
        """
        格式化打印结果，同 `pprint <https://docs.python.org/3/library/pprint.html>`_ 模块.
        接收的参数同 `pprint.pprint <https://docs.python.org/3/library/pprint.html#pprint.pprint>`_ 。
        """
        pprint.pprint(self.asList(), *args, **kwargs)


MutableMapping.register(ParseResults)


def _defaultStartDebugAction(instring, loc, expr):
    """默认开始调试动作"""
    print(("Match " + _ustr(expr) + " at loc " + _ustr(loc) + "(%d,%d)" % (lineno(loc, instring), col(loc, instring))))


def _defaultSuccessDebugAction(instring, startloc, endloc, expr, toks):
    """默认成功调试动作"""
    print("Matched " + _ustr(expr) + " -> " + str(toks.asList()))


def _defaultExceptionDebugAction(instring, loc, expr, exc):
    """默认异常调试动作"""
    print("Exception raised:" + _ustr(exc))


def nullDebugAction(*args):
    """'Do-nothing' debug action, to suppress debugging output during parsing."""
    pass


def lineno(loc, strg):
    return strg.count("\n", 0, loc) + 1


def col(loc, strg):
    s = strg
    return 1 if 0 < loc < len(s) and s[loc - 1] == '\n' else loc - s.rfind("\n", 0, loc)


def _trim_arity(func, maxargs=2):
    if func in singleArgBuiltins:
        return lambda s, l, t: func(t)
    limit = [0]
    found_arity = [False]

    extract_stack = traceback.extract_stack
    extract_tb = traceback.extract_tb

    line_diff = 6
    this_line = extract_stack(limit=2)[-1]
    pa_call_line_synth = (this_line[0], this_line[1] + line_diff)

    def wrapper(*args):
        while 1:
            try:
                ret = func(*args[limit[0]:])
                found_arity[0] = True
                return ret
            except TypeError:
                if not found_arity[0]:
                    try:
                        track_back = sys.exc_info()[-1]
                        if not extract_tb(track_back, limit=2)[-1][:2] == pa_call_line_synth:
                            raise
                    finally:
                        try:
                            del track_back
                        except NameError:
                            pass
                else:
                    raise

                if limit[0] <= maxargs:
                    limit[0] += 1
                    continue
                raise

    fn_name = "<parse action>"
    try:
        fn_name = getattr(func, '__name__', getattr(func, '__class__').__name__)
    except Exception:
        fn_name = str(func)
    wrapper.__name__ = fn_name

    return wrapper


def line(loc, strg):
    """Returns the line of text containing loc within a string, counting newlines as line separators."""
    last_cr = strg.rfind("\n", 0, loc)
    next_cr = strg.find("\n", loc)
    if next_cr >= 0:
        return strg[last_cr + 1:next_cr]
    else:
        return strg[last_cr + 1:]


class ParserElement(object):
    """针对解析元素的基础抽象类。"""
    DEFAULT_WHITE_CHARS = " \n\t\r"
    verbose_stacktrace = False

    @staticmethod
    def inlineLiteralsUsing(cls):
        """
        Set class to be used for inclusion of string literals into a parser.

        示例::

            # default literal class used is Literal
            integer = Word(nums)
            date_str = integer("year") + '/' + integer("month") + '/' + integer("day")

            date_str.parseString("1999/12/31")  # -> ['1999', '/', '12', '/', '31']


            ParserElement.inlineLiteralsUsing(Suppress)
            date_str = integer("year") + '/' + integer("month") + '/' + integer("day")

            date_str.parseString("1999/12/31")  # -> ['1999', '12', '31']
        """
        ParserElement._literalStringClass = cls

    @staticmethod
    def setDefaultWhitespaceChars(chars):
        r"""
        Overrides the default whitespace chars

        示例::

            # default whitespace chars are space, <TAB> and newline
            OneOrMore(Word(alphas)).parseString("abc def\nghi jkl")  # -> ['abc', 'def', 'ghi', 'jkl']

            ParserElement.setDefaultWhitespaceChars(" \t")
            OneOrMore(Word(alphas)).parseString("abc def\nghi jkl")  # -> ['abc', 'def']
        """
        ParserElement.DEFAULT_WHITE_CHARS = chars

    @classmethod
    def _trim_traceback(cls, tb):
        trace_back = tb
        while trace_back.tb_next:
            trace_back = trace_back.tb_next
        return trace_back

    def __init__(self, savelist=False):
        self.parse_action = list()
        self.fail_action = None
        self.str_repr = None
        self.results_name = None
        self.save_as_list = savelist
        self.skip_whitespace = True
        self.white_chars = set(ParserElement.DEFAULT_WHITE_CHARS)
        self.copy_default_white_chars = True
        self.may_return_empty = False  # used when checking for left-recursion
        self.keep_tabs = False
        self.ignore_exprs = list()
        self.debug = False
        self.streamlined = False
        self.may_index_error = True  # used to optimize exception handling for subclasses that don't advance parse index
        self.errmsg = ""
        self.modal_results = True  # used to mark results names as modal (report only last) or cumulative (list all)
        self.debug_actions = (None, None, None)  # custom debug actions
        self.re = None
        self.call_preparse = True  # used to avoid redundant calls to preParse
        self.call_during_try = False

    def copy(self):
        """
        Make a copy of this :class:`ParserElement`.  Useful for defining
        different parse actions for the same parsing pattern, using copies of
        the original parse element.

        示例::

            integer = Word(nums).setParseAction(lambda toks: int(toks[0]))
            integerK = integer.copy().addParseAction(lambda toks: toks[0] * 1024) + Suppress("K")
            integerM = integer.copy().addParseAction(lambda toks: toks[0] * 1024 * 1024) + Suppress("M")

            print(OneOrMore(integerK | integerM | integer).parseString("5K 100 640K 256M"))

        结果::

            [5120, 100, 655360, 268435456]

        Equivalent form of ``expr.copy()`` is just ``expr()``::

            integerM = integer().addParseAction(lambda toks: toks[0] * 1024 * 1024) + Suppress("M")
        """
        copied = copy.copy(self)
        copied.ignore_exprs = self.ignore_exprs[:]
        copied.parse_action = self.parse_action[:]
        if self.copy_default_white_chars:
            copied.white_chars = ParserElement.DEFAULT_WHITE_CHARS
        return copied

    def setResultsName(self, name, listAllMatches=False):
        """
        示例::

            date_str = (integer.setResultsName("year") + '/'
                        + integer.setResultsName("month") + '/'
                        + integer.setResultsName("day"))

            # 等价于：
            date_str = integer("year") + '/' + integer("month") + '/' + integer("day")
        """
        return self._setResultsName(name, listAllMatches)

    def setName(self, name):
        """
        Define name for this expression, makes debugging and exception messages clearer.

        示例::

            Word(nums).parseString("ABC")  # -> Exception: Expected W:(0123...) (at char 0), (line:1, col:1)
            Word(nums).setName("integer").parseString("ABC")  # -> Exception: Expected integer (at char 0), (line:1, col:1)
        """
        self.name = name
        self.errmsg = "Expected " + self.name
        return self

    def _setResultsName(self, name, listAllMatches=False):
        new_self = self.copy()
        if name.endswith("*"):
            name = name[:-1]
            listAllMatches = True
        new_self.results_name = name
        new_self.modal_results = not listAllMatches
        return new_self

    def setParseAction(self, *fns, **kwargs):
        """
        Define one or more actions to perform when successfully
        matching parse element definition.
        Parse action fn is a callable method with 0-3 arguments,
        called as ``fn(s, loc, toks)`` , ``fn(loc, toks)`` ,
            ``fn(toks)`` , or just ``fn()`` , where:

        - s   = the original string being parsed (see note below)
        - loc = the location of the matching substring
        - toks = a list of the matched tokens, packaged as a :class:`ParseResults` object

        If the functions in fns modify the tokens,
        they can return them as the return
        value from fn, and the modified list of tokens will replace the original.
        Otherwise, fn does not need to return any value.

        If None is passed as the parse action,
        all previously added parse actions for this expression are cleared.

        Optional keyword arguments:
        - callDuringTry = (default= ``False``)
          indicate if parse action should be run during lookaheads and alternate testing

        Note: the default parsing behavior is to expand tabs
        in the input string before starting the parsing process.
        See :class:`parseString for more information on parsing
        strings containing ``<TAB>`` s, and suggested methods to
        maintain a consistent view of the parsed string, the parse location,
        and line and column positions within the parsed string.

        示例::

            integer = Word(nums)
            date_str = integer + '/' + integer + '/' + integer

            date_str.parseString("2000/01/31")  # -> ['2000', '/', '01', '/', '31']

            # 通过parseAction在解析时将结果转为int
            integer = Word(nums).setParseAction(lambda toks: int(toks[0]))
            date_str = integer + '/' + integer + '/' + integer

            # note that integer fields are now ints, not strings
            date_str.parseString("2000/01/31")  # -> [2000, '/', 01, '/', 31]
        """
        if list(fns) == [None, ]:
            self.parse_action = []
        else:
            if all(callable(fn) for fn in fns):
                self.parse_action = list(map(_trim_arity, list(fns)))
                self.call_during_try = kwargs.get("callDuringTry", False)
            else:
                raise TypeError("parse actions must be callable")
        return self

    def addCondition(self, *fns, **kwargs):
        """Add a boolean predicate function to expression's list of parse actions.
        See :class:`setParseAction` for function call signatures.
        Unlike ``setParseAction``, functions passed to ``addCondition``
        need to return boolean success/fail of the condition.

        Optional keyword arguments:
        - message = define a custom message to be used in the raised exception
        - fatal   = if True, will raise ParseFatalException to stop
                    parsing immediately; otherwise will raise ParseException

        示例::

            integer_word = Word(nums).setParseAction(lambda toks: int(toks[0]))
            year_int = integer_word.copy()
            year_int.addCondition(lambda toks: toks[0] >= 2000, message="Only support years 2000 and later")
            date_str = year_int + '/' + integer_word + '/' + integer_word

            # -> Exception: Only support years 2000 and later (at char 0), (line:1, col:1)
            result = date_str.parseString("1999/12/31")
        """
        for function in fns:
            self.parse_action.append(conditionAsParseAction(function, message=kwargs.get('message'),
                                                            fatal=kwargs.get('fatal', False)))

        self.call_during_try = self.call_during_try or kwargs.get("callDuringTry", False)
        return self

    def addParseAction(self, *fns, **kwargs):
        """
        Add one or more parse actions to expression's list of parse actions.
        See :class:`setParseAction`.

        See examples in :class:`copy`.
        """
        self.parse_action += list(map(_trim_arity, list(fns)))
        self.call_during_try = self.call_during_try or kwargs.get("callDuringTry", False)
        return self

    def _skipIgnorables(self, instring, loc):
        exprsFound = True
        while exprsFound:
            exprsFound = False
            for expr in self.ignore_exprs:
                try:
                    while True:
                        loc, dummy = expr._parse(instring, loc)
                        exprsFound = True
                except ParseException:
                    pass
        return loc

    def setFailAction(self, fn):
        """Define action to perform if parsing fails at this expression. Fail acton fn
        is a callable function that takes the arguments
        ``fn(s, loc, expr, err)`` where:
        - s = string being parsed
        - loc = location where expression
                match was attempted and failed
        - expr = the parse expression that failed
        - err = the exception thrown
        The function returns no value.
        It may throw :class:`ParseFatalException` if it is desired to stop parsing immediately."""
        self.fail_action = fn
        return self

    def parseImpl(self, instring, loc, doActions=True):
        return loc, []

    def preParse(self, instring, loc):
        if self.ignore_exprs:
            loc = self._skipIgnorables(instring, loc)

        if self.skip_whitespace:
            wt = self.white_chars
            instr_len = len(instring)
            while loc < instr_len and instring[loc] in wt:
                loc += 1

        return loc

    def postParse(self, instring, loc, tokenlist):
        return tokenlist

    def tryParse(self, instring, loc):
        try:
            return self._parse(instring, loc, doActions=False)[0]
        except ParseFatalException:
            raise ParseException(instring, loc, self.errmsg, self)

    def _parseNoCache(self, instring, loc, doActions=True, callPreParse=True):
        TRY, MATCH, FAIL = 0, 1, 2
        debugging = (self.debug)  # and doActions)

        if debugging or self.fail_action:
            if self.debug_actions[TRY]:
                self.debug_actions[TRY](instring, loc, self)
            try:
                if callPreParse and self.call_preparse:
                    preloc = self.preParse(instring, loc)
                else:
                    preloc = loc
                tokensStart = preloc
                if self.may_index_error or preloc >= len(instring):
                    try:
                        loc, tokens = self.parseImpl(instring, preloc, doActions)
                    except IndexError:
                        # raise new exception
                        in_string_len = len(instring)
                        raise ParseException(instring, in_string_len, self.errmsg, self)
                else:
                    loc, tokens = self.parseImpl(instring, preloc, doActions)
            except Exception as err:
                if self.debug_actions[FAIL]:
                    self.debug_actions[FAIL](instring, tokensStart, self, err)
                if self.fail_action:
                    self.fail_action(instring, tokensStart, self, err)
                raise
        else:
            if callPreParse and self.call_preparse:
                preloc = self.preParse(instring, loc)
            else:
                preloc = loc
            tokensStart = preloc
            if self.may_index_error or preloc >= len(instring):
                try:
                    loc, tokens = self.parseImpl(instring, preloc, doActions)
                except IndexError:
                    # raise new exception
                    raise ParseException(instring, len(instring), self.errmsg, self)
            else:
                loc, tokens = self.parseImpl(instring, preloc, doActions)

        tokens = self.postParse(instring, loc, tokens)

        retTokens = ParseResults(tokens, self.results_name, asList=self.save_as_list, modal=self.modal_results)
        if self.parse_action and (doActions or self.call_during_try):
            if debugging:
                try:
                    for action in self.parse_action:
                        try:
                            tokens = action(instring, tokensStart, retTokens)
                        except IndexError as parse_action_exception:
                            exc = ParseException("exception raised in parse action")
                            exc.__cause__ = parse_action_exception
                            raise exc

                        if tokens is not None and tokens is not retTokens:
                            retTokens = ParseResults(tokens,
                                                     self.results_name,
                                                     asList=self.save_as_list and isinstance(tokens,
                                                                                             (ParseResults, list)),
                                                     modal=self.modal_results)
                except Exception as err:
                    if self.debug_actions[FAIL]:
                        self.debug_actions[FAIL](instring, tokensStart, self, err)
                    raise
            else:
                for action in self.parse_action:
                    try:
                        tokens = action(instring, tokensStart, retTokens)
                    except IndexError as parse_action_exception:
                        exc = ParseException("exception raised in parse action")
                        exc.__cause__ = parse_action_exception
                        raise exc

                    if tokens is not None and tokens is not retTokens:
                        retTokens = ParseResults(tokens,
                                                 self.results_name,
                                                 asList=self.save_as_list and isinstance(tokens, (ParseResults, list)),
                                                 modal=self.modal_results)
        if debugging:
            if self.debug_actions[MATCH]:
                self.debug_actions[MATCH](instring, tokensStart, loc, self, retTokens)

        return loc, retTokens

    class _UnboundedCache(object):
        def __init__(self):
            cache = {}
            self.not_in_cache = not_in_cache = object()

            def set(self, key, value):
                cache[key] = value

            def get(self, key):
                return cache.get(key, not_in_cache)

            def cache_len(self):
                return len(cache)

            def clear(self):
                cache.clear()

            self.set = types.MethodType(set, self)
            self.get = types.MethodType(get, self)
            self.__len__ = types.MethodType(cache_len, self)
            self.clear = types.MethodType(clear, self)

    def canParseNext(self, instring, loc):
        try:
            self.tryParse(instring, loc)
        except (ParseException, IndexError):
            # can not parse
            return False
        else:
            return True

    if _OrderedDict is None:
        class _FifoCache(object):
            def __init__(self, size):
                self.not_in_cache = not_in_cache = object()

                cache = {}
                key_fifo = collections.deque([], size)

                def set(self, key, value):
                    cache[key] = value
                    while len(key_fifo) > size:
                        cache.pop(key_fifo.popleft(), None)
                    key_fifo.append(key)

                def get(self, key):
                    return cache.get(key, not_in_cache)

                def cache_len(self):
                    return len(cache)

                def clear(self):
                    cache.clear()
                    key_fifo.clear()

                self.set = types.MethodType(set, self)
                self.get = types.MethodType(get, self)
                self.__len__ = types.MethodType(cache_len, self)
                self.clear = types.MethodType(clear, self)

    else:
        class _FifoCache(object):
            def __init__(self, size):
                cache_dict = _OrderedDict()
                self.not_in_cache = not_in_cache = object()

                def set(self, key, value):
                    cache_dict[key] = value
                    while len(cache_dict) > size:
                        try:
                            cache_dict.popitem(False)
                        except KeyError:
                            pass

                def get(self, key):
                    return cache_dict.get(key, not_in_cache)

                def cache_len(self):
                    return len(cache_dict)

                def clear(self):
                    cache_dict.clear()

                self.set = types.MethodType(set, self)
                self.get = types.MethodType(get, self)
                self.__len__ = types.MethodType(cache_len, self)
                self.clear = types.MethodType(clear, self)

    packrat_cache_lock = RLock()
    packrat_cache_stats = [0, 0]
    packrat_cache = {}  # this is set later by enabledPackrat(); this is here so that resetCache() doesn't fail

    # this method gets repeatedly called during backtracking with the same arguments -
    def _parseCache(self, instring, loc, doActions=True, callPreParse=True):
        lookup = (self, instring, loc, callPreParse, doActions)
        HIT, MISS = 0, 1
        with ParserElement.packrat_cache_lock:
            cache = ParserElement.packrat_cache
            value = cache.get(lookup)
            if value is not cache.not_in_cache:
                ParserElement.packrat_cache_stats[HIT] += 1
                if isinstance(value, Exception):
                    raise value
                return value[0], value[1].copy()
            else:
                ParserElement.packrat_cache_stats[MISS] += 1
                try:
                    value = self._parseNoCache(instring, loc, doActions, callPreParse)
                except ParseBaseException as parse_exc:
                    cache.set(lookup, parse_exc.__class__(*parse_exc.args))
                    raise
                else:
                    cache.set(lookup, (value[0], value[1].copy()))
                    return value

    _parse = _parseNoCache

    @staticmethod
    def resetCache():
        ParserElement.packrat_cache.clear()
        ParserElement.packrat_cache_stats[:] = [0] * len(ParserElement.packrat_cache_stats)

    # clear mark
    _packratEnabled = False

    @staticmethod
    def enablePackrat(cache_size_limit=128):
        """
        Enables "packrat" parsing, which adds memoizing
        to the parsing logic. Repeated parse attempts at the
        same string location (which happens often in many complex grammars)
        can immediately return a cached value,
        instead of re-executing parsing/validating code.
        Memoizing is done of both valid results and parsing exceptions.

           参数:

           - cache_size_limit - (default= ``128``) -
                if an integer value is provided
                will limit the size of the packrat cache; if None is passed,
                then the cache size will be unbounded; if 0 is passed,
                the cache will be effectively disabled.

           This speedup may break existing programs that use parse actions that
           have side-effects.  For this reason, packrat parsing is disabled when
           you first import expparser.  To activate the packrat feature, your
           program must call the class method :class:`ParserElement.enablePackrat`.
           For best results, call ``enablePackrat()`` immediately after
           importing expparser.

           示例::

               import expparser
               expparser.ParserElement.enablePackrat()
        """
        if not ParserElement._packratEnabled:
            ParserElement._packratEnabled = True
            if cache_size_limit is not None:
                ParserElement.packrat_cache = ParserElement._FifoCache(cache_size_limit)
            else:
                ParserElement.packrat_cache = ParserElement._UnboundedCache()
            ParserElement._parse = ParserElement._parseCache

    def parseString(self, instring, parseAll=False):
        """
        Execute the parse expression with the given string.
        This is the main interface to the client code,
        once the complete expression has been built.

        Returns the parsed data as a :class:`ParseResults` object,
        which may be accessed as a list,
        or as a dict or object with attributes if the given parser includes results names.

        If you want the grammar to require that the entire input string be
        successfully parsed, then set ``parseAll`` to True
        (equivalent to ending the grammar with ``StringEnd()``).

        Note: ``parseString`` implicitly calls ``expandtabs()`` on the input string,
        in order to report proper column numbers in parse actions. If the input string contains tabs and
        the grammar uses parse actions that use the ``loc`` argument to index into the string being parsed,
        you can ensure you have a consistent view of the input string by:

        - calling ``parseWithTabs`` on your grammar before calling ``parseString`` (see :class:`parseWithTabs`)
        - define your parse action using the full ``(s, loc, toks)`` signature,
            and reference the input string using the parse action's ``s`` argument
        - explictly expand the tabs in your input string before calling ``parseString``

        示例::

            # -> Exception: Expected end of text
            Word('a').parseString('aaaaabaaa', parseAll=True)
            # -> ['aaaaa']
            Word('a').parseString('aaaaabaaa')
        """
        ParserElement.resetCache()
        if not self.streamlined:
            self.streamline()
        for e in self.ignore_exprs:
            e.streamline()
        if not self.keep_tabs:
            instring = instring.expandtabs()
        try:
            loc, tokens = self._parse(instring, 0)
            if parseAll:
                loc = self.preParse(instring, loc)
                empty_string_end = Empty() + StringEnd()
                empty_string_end._parse(instring, loc)
        except ParseBaseException as exc:
            if ParserElement.verbose_stacktrace:
                raise
            else:
                # catch and re-raise exception from here, clearing out expparser internal stack trace
                if getattr(exc, '__traceback__', None) is not None:
                    exc.__traceback__ = self._trim_traceback(exc.__traceback__)
                raise exc
        else:
            return tokens

    def scanString(self, instring, maxMatches=sys.maxint, overlap=False):
        """
        Scan the input string for expression matches.
        Each match will return the matching tokens, start location, and end location.
        May be called with optional ``maxMatches`` argument, to clip scanning after 'n' matches are found.
        If ``overlap`` is specified, then overlapping matches will be reported.

        Note that the start and end locations are reported relative to the string being parsed.
        See :class:`parseString` for more information on parsing
        strings with embedded tabs.

        示例::

            source = "abcde123bacdefg345cdsjak879jskial987"
            print(source)
            for tokens, start, end in Word(alphas).scanString(source):
                print(' '*start + '^'*(end-start))
                print(' '*start + tokens[0])

        prints::

            abcde123bacdefg345cdsjak879jskial987
            ^^^^^
            abcde
                    ^^^^^^^
                    bacdefg
                              ^^^^^^
                              cdsjak
                                       ^^^^^^
                                       jskial
        """
        if not self.streamlined:
            self.streamline()
        for e in self.ignore_exprs:
            e.streamline()

        if not self.keep_tabs:
            instring = _ustr(instring).expandtabs()
        in_str_len = len(instring)
        loc = 0
        preparse_fn = self.preParse
        parse_fn = self._parse
        ParserElement.resetCache()
        matches = 0
        try:
            while loc <= in_str_len and matches < maxMatches:
                try:
                    pre_loc = preparse_fn(instring, loc)
                    next_loc, tokens = parse_fn(instring, pre_loc, callPreParse=False)
                except ParseException:
                    loc = pre_loc + 1
                else:
                    if next_loc <= loc:
                        loc = pre_loc + 1
                    else:
                        matches += 1
                        yield tokens, pre_loc, next_loc
                        if not overlap:
                            loc = next_loc
                        else:
                            next_loc = preparse_fn(instring, loc)
                            if next_loc > loc:
                                loc = next_loc
                            else:
                                loc += 1
        except ParseBaseException as exc:
            if ParserElement.verbose_stacktrace:
                raise
            else:
                # catch and re-raise exception from here, clearing out expparser internal stack trace
                if getattr(exc, '__traceback__', None) is not None:
                    exc.__traceback__ = self._trim_traceback(exc.__traceback__)
                raise exc

    def transformString(self, instring):
        """
        Extension to :class:`scanString`,
        to modify matching text with modified tokens that may be returned from a parse action.
        To use ``transformString``,
        define a grammar and attach a parse action to it that modifies the returned token list.
        Invoking ``transformString()`` on a target string
        will then scan for matches, and replace the matched text patterns
        according to the logic in the parse action.
        ``transformString()`` returns the resulting transformed string.

        示例::

            wd = Word(alphas)
            wd.setParseAction(lambda toks: toks[0].title())
            print(wd.transformString("now is the winter of our discontent made glorious summer by this sun of york."))

        打印::

            Now Is The Winter Of Our Discontent Made Glorious Summer By This Sun Of York.
        """
        out = []
        last_e = 0
        # force preservation of <TAB>s, to minimize unwanted transformation of string, and to
        # keep string locs straight between transformString and scanString
        self.keep_tabs = True
        try:
            for tokens, pre_loc, next_loc in self.scanString(instring):
                out.append(instring[last_e:pre_loc])
                if tokens:
                    if isinstance(tokens, ParseResults):
                        out += tokens.asList()
                    elif isinstance(tokens, list):
                        out += tokens
                    else:
                        out.append(tokens)
                last_e = next_loc
            out.append(instring[last_e:])
            out = [o for o in out if o]
            return "".join(map(_ustr, _flatten(out)))
        except ParseBaseException as exception:
            if ParserElement.verbose_stacktrace:
                raise
            else:
                # catch and re-raise exception from here, clearing out expparser internal stack trace
                if getattr(exception, '__traceback__', None) is not None:
                    exception.__traceback__ = self._trim_traceback(exception.__traceback__)
                raise exception

    def searchString(self, instring, maxMatches=sys.maxint):
        """
        Another extension to :class:`scanString`,
        simplifying the access to the tokens found to match the given parse expression.
        May be called with optional ``maxMatches`` argument, to clip searching after 'n' matches are found.

        示例::

            # a capitalized word starts with an uppercase letter,
            #   followed by zero or more lowercase letters
            cap_word = Word(alphas.upper(), alphas.lower())

            print(cap_word.searchString("More than Iron, more than Lead, more than Gold I need Electricity"))

            # the sum() builtin can be used to merge results
            #   into a single ParseResults object
            print(sum(cap_word.searchString("More than Iron, more than Lead, more than Gold I need Electricity")))

        结果::

            [['More'], ['Iron'], ['Lead'], ['Gold'], ['I'], ['Electricity']]
            ['More', 'Iron', 'Lead', 'Gold', 'I', 'Electricity']
        """
        try:
            return ParseResults([tokens for tokens, s, e in self.scanString(instring, maxMatches)])
        except ParseBaseException as exception:
            if ParserElement.verbose_stacktrace:
                raise
            else:
                # catch and re-raise exception from here, clearing out expparser internal stack trace
                if getattr(exception, '__traceback__', None) is not None:
                    exception.__traceback__ = self._trim_traceback(exception.__traceback__)
                raise exception

    def split(self, instring, maxsplit=sys.maxint, includeSeparators=False):
        """
        Generator method to split a string using the given expression as a separator.
        May be called with optional ``maxsplit`` argument, to limit the number of splits;
        and the optional ``includeSeparators`` argument (default= ``False``), if the separating
        matching text should be included in the split results.

        示例::

            punc = oneOf(list(".,;:/-!?"))
            print(list(punc.split("This, this?, this sentence, is badly punctuated!")))

        结果::

            ['This', ' this', '', ' this sentence', ' is badly punctuated', '']
        """
        last = 0
        for tokens, pre_loc, next_loc in self.scanString(instring, maxMatches=maxsplit):
            yield instring[last:pre_loc]
            if includeSeparators:
                yield tokens[0]
            last = next_loc
        yield instring[last:]

    def __radd__(self, other):
        if other is Ellipsis:
            return SkipTo(self)("_skipped*") + self

        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        if isinstance(other, ParserElement):
            return other + self
        else:
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None

    def __add__(self, other):
        if other is Ellipsis:
            return _PendingSkip(self)

        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        if isinstance(other, ParserElement):
            return And([self, other])
        else:
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None

    def __rsub__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        if isinstance(other, ParserElement):
            return other - self
        else:
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None

    def __sub__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        if isinstance(other, ParserElement):
            return self + And._ErrorStop() + other
        else:
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None

    def __rmul__(self, other):
        return self.__mul__(other)

    def __mul__(self, other):
        if other is Ellipsis:
            other = (0, None)
        elif isinstance(other, tuple) and other[:1] == (Ellipsis,):
            other = ((0,) + other[1:] + (None,))[:2]

        if isinstance(other, int):
            min_elements, opt_elements = other, 0
        elif isinstance(other, tuple):
            other = tuple(other_value if other_value is not Ellipsis else None for other_value in other)
            other = (other + (None, None))[:2]
            if other[0] is None:
                other = (0, other[1])
            # update
            if other[1] is None and isinstance(other[0], int):
                if other[0] == 0:
                    return ZeroOrMore(self)
                if other[0] != 1:
                    return self * other[0] + ZeroOrMore(self)
                else:
                    return OneOrMore(self)
            elif isinstance(other[0], int) and isinstance(other[1], int):
                min_elements, opt_elements = other
                opt_elements -= min_elements
            else:
                raise TypeError("cannot multiply 'ParserElement' and ('%s', '%s') objects", type(other[0]),
                                type(other[1]))
        else:
            raise TypeError("cannot multiply 'ParserElement' and '%s' objects", type(other))

        if min_elements < 0:
            raise ValueError("cannot multiply ParserElement by negative value")
        if opt_elements < 0:
            raise ValueError("second tuple value must be greater or equal to first tuple value")
        if min_elements == opt_elements == 0:
            raise ValueError("cannot multiply ParserElement by 0 or (0, 0)")

        if opt_elements:
            def makeOptionalList(n):
                if n <= 1:
                    return Optional(self)
                else:
                    return Optional(self + makeOptionalList(n - 1))

            if min_elements:
                if min_elements == 1:
                    ret = self + makeOptionalList(opt_elements)
                else:
                    ret = And([self] * min_elements) + makeOptionalList(opt_elements)
            else:
                ret = makeOptionalList(opt_elements)
        else:
            if min_elements != 1:
                ret = And([self] * min_elements)
            else:
                ret = self
        return ret

    def __ror__(self, other):
        """
        Implementation of | operator when left operand
        is not a :class:`ParserElement`
        """
        if isinstance(other, basestring):
            # convert to standard
            other = self._literalStringClass(other)
        if isinstance(other, ParserElement):
            return other | self
        else:
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None

    def __or__(self, other):
        if other is Ellipsis:
            return _PendingSkip(self, must_skip=True)

        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        if isinstance(other, ParserElement):
            return MatchFirst([self, other])
        else:
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None

    def __rxor__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        if not isinstance(other, ParserElement):
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None
        else:
            return other ^ self

    def __xor__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        if not isinstance(other, ParserElement):
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None
        else:
            return Or([self, other])

    def __rand__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        if not isinstance(other, ParserElement):
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None
        else:
            return other & self

    def __and__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        if not isinstance(other, ParserElement):
            warnings.warn("Cannot combine element of type %s with ParserElement" % type(other),
                          SyntaxWarning, stacklevel=2)
            return None
        else:
            return Each([self, other])

    def __iter__(self):
        raise TypeError('%r object is not iterable' % self.__class__.__name__)

    def __invert__(self):
        return NotAny(self)

    def __call__(self, name=None):
        if name is None:
            return self.copy()
        else:
            return self._setResultsName(name)

    def __getitem__(self, key):
        try:
            if isinstance(key, str):
                # convert to tuple
                key = (key,)
            # iterate key
            iter(key)
        except TypeError:
            key = (key, key)

        if len(key) > 2:
            warnings.warn("only 1 or 2 index arguments supported ({0}{1})"
                          .format(key[:5], '... [{0}]'.format(len(key)) if len(key) > 5 else ''))

        ret = self * tuple(key[:2])
        return ret

    def leaveWhitespace(self):
        """
        Disables the skipping of whitespace before matching
        the characters in the :class:`ParserElement`'s defined pattern.
        This is normally only used internally by the expparser module,
        but may be needed in some whitespace-sensitive grammars.
        """
        self.skip_whitespace = False
        return self

    def suppress(self):
        """
        Suppresses the output of this :class:`ParserElement`;
        useful to keep punctuation from cluttering up returned output.
        """
        return Suppress(self)

    def parseWithTabs(self):
        """
        Overrides default behavior to expand ``<TAB>``s
        to spaces before parsing the input string.
        Must be called before ``parseString``
        when the input grammar contains elements that match ``<TAB>`` characters.
        """
        self.keep_tabs = True
        return self

    def setWhitespaceChars(self, chars):
        """
        Overrides the default whitespace chars
        """
        self.copy_default_white_chars = False
        self.white_chars = chars
        self.skip_whitespace = True
        return self

    def ignore(self, other):
        """
        Define expression to be ignored (e.g., comments)
        while doing pattern matching;
        may be called repeatedly, to define multiple comment or other ignorable patterns.

        示例::

            patt = OneOrMore(Word(alphas))
            patt.parseString('abcde /* comment */ bacde') # -> ['abcde']

            patt.ignore(cStyleComment)
            patt.parseString('abcde /* comment */ bacde') # -> ['abcde', 'bacde']
        """
        if isinstance(other, basestring):
            other = Suppress(other)

        if not isinstance(other, Suppress):
            self.ignore_exprs.append(Suppress(other.copy()))
        else:
            if other not in self.ignore_exprs:
                self.ignore_exprs.append(other)
        return self

    def setDebug(self, flag=True):
        """
        Enable display of debugging messages while doing pattern matching. Set ``flag`` to True to enable,
        False to disable.

        示例::

            integer = Word(nums).setName("numword")
            wd = Word(alphas).setName("alphaword")
            term = wd | integer

            # turn on debugging for wd
            wd.setDebug()

            OneOrMore(term).parseString("abc 123 xyz 890")

        The output shown is that produced by
        the default debug actions - custom debug actions can be specified using :class:`setDebugActions`.
        Prior to attempting to match the ``wd`` expression,
        the debugging message ``"Match <exprname> at loc <n>(<line>,<col>)"`` is shown.
        Then if the parse succeeds, a ``"Matched"`` message is shown, or an ``"Exception raised"``
        message is shown.
        Also note the use of :class:`setName` to assign a human-readable name to the expression,
        which makes debugging and exception messages easier to understand - for instance,
        the default name created for the :class:`Word` expression without calling ``setName`` is ``"W:(ABCD...)"``.
        """
        if not flag:
            self.debug = False
        else:
            self.setDebugActions(_defaultStartDebugAction, _defaultSuccessDebugAction, _defaultExceptionDebugAction)
        return self

    def setDebugActions(self, startAction, successAction, exceptionAction):
        """
        Enable display of debugging messages while doing pattern matching.
        """
        self.debug_actions = (
            startAction or _defaultStartDebugAction,
            successAction or _defaultSuccessDebugAction,
            exceptionAction or _defaultExceptionDebugAction)
        self.debug = True
        return self

    def __repr__(self):
        return _ustr(self)

    def __str__(self):
        return self.name

    def checkRecursion(self, parseElementList):
        pass

    def streamline(self):
        self.str_repr = None
        self.streamlined = True
        return self

    def parseFile(self, file_or_filename, parseAll=False):
        """
        Execute the parse expression on the given file or filename.
        If a filename is specified (instead of a file object), the entire file is
        opened, read, and closed before parsing.
        """
        try:
            file_contents = file_or_filename.read()
        except AttributeError:
            with open(file_or_filename, "r") as file_handle:
                file_contents = file_handle.read()
        try:
            return self.parseString(file_contents, parseAll)
        except ParseBaseException as exception:
            if ParserElement.verbose_stacktrace:
                raise
            else:
                # catch and re-raise exception from here, clearing out expparser internal stack trace
                if getattr(exception, '__traceback__', None) is not None:
                    exception.__traceback__ = self._trim_traceback(exception.__traceback__)
                raise exception

    def validate(self, validateTrace=None):
        """
        Check defined expressions for valid structure,
        check for infinite recursive definitions.
        """
        self.checkRecursion([])

    def __eq__(self, other):
        if self is other:
            return True
        if isinstance(other, basestring):
            return self.matches(other)
        if isinstance(other, ParserElement):
            return vars(self) == vars(other)
        return False

    def __hash__(self):
        return id(self)

    def __ne__(self, other):
        return not (self == other)

    def __rne__(self, other):
        return not (self == other)

    def __req__(self, other):
        return self == other

    def matches(self, testString, parseAll=True):
        """
        Method for quick testing of a parser against a test string.
        Good for simple inline microtests of sub expressions while building up larger parser.

        Parameters:
         - testString - to test against this expression for a match
         - parseAll - (default= ``True``) - flag to pass to :class:`parseString`
                                            when running tests

        示例::

            assert Word(nums).matches("100")
        """
        try:
            in_str = _ustr(testString)
            self.parseString(in_str, parseAll=parseAll)
            return True
        except ParseBaseException:
            return False


class _PendingSkip(ParserElement):
    def __init__(self, expr, must_skip=False):
        super(_PendingSkip, self).__init__()
        self.str_repr = str(expr + Empty()).replace('Empty', '...')
        self.name = self.str_repr
        self.anchor = expr
        self.must_skip = must_skip

    def __repr__(self):
        return self.str_repr

    def __add__(self, other):
        skipper = SkipTo(other).setName("...")("_skipped*")
        if self.must_skip:
            def show_skip(tokens):
                if tokens._skipped.asList()[-1:] == ['']:
                    tokens['_skipped'] = 'missing <' + repr(self.anchor) + '>'

            def must_skip(tokens):
                if not tokens._skipped or tokens._skipped.asList() == ['']:
                    del tokens[0]
                    tokens.pop("_skipped", None)

            return (self.anchor + skipper().addParseAction(must_skip)
                    | skipper().addParseAction(show_skip)) + other

        return self.anchor + skipper + other

    def parseImpl(self, *args):
        raise Exception("use of `...` expression without following SkipTo target expression")


class Token(ParserElement):
    """定义匹配范式，是解析元素抽象的子类。"""

    def __init__(self):
        super(Token, self).__init__(savelist=False)


class Empty(Token):
    """空的匹配范式，总是匹配成功。"""

    def __init__(self):
        super(Empty, self).__init__()
        self.name = "Empty"
        self.may_return_empty = True
        self.may_index_error = False


class NoMatch(Token):
    """永远不会匹配成功的范式。"""

    def __init__(self):
        super(NoMatch, self).__init__()
        self.name = "NoMatch"
        self.may_return_empty = True
        self.may_index_error = False
        self.errmsg = "Unmatchable token"

    def parseImpl(self, instring, loc, doActions=True):
        raise ParseException(instring, loc, self.errmsg, self)


class Literal(Token):
    """精确匹配给定字符串的范式，可以匹配包含给定字符串的其他字符串（即匹配字串）。

    例子::

        Literal('blah').parseString('blah')  # -> ['blah']
        Literal('blah').parseString('blahfooblah')  # -> ['blah']
        Literal('blah').parseString('bla')  # -> Exception: Expected "blah"

    对于大小写不敏感的字符串匹配，请使用 :class:`CaselessLiteral`.

    对于关键字匹配（即完整单词匹配），请使用 :class:`Keyword` or :class:`CaselessKeyword`.
    """

    def __init__(self, matchString):
        super(Literal, self).__init__()
        self.match = matchString
        self.match_len = len(matchString)
        try:
            self.first_match_char = matchString[0]
        except IndexError:
            warnings.warn("null string passed to Literal; use Empty() instead",
                          SyntaxWarning, stacklevel=2)
            self.__class__ = Empty

        self.may_return_empty = False
        self.name = '"%s"' % _ustr(self.match)
        self.may_index_error = False
        self.errmsg = "Expected " + self.name

        if self.match_len == 1 and type(self) is Literal:
            self.__class__ = _SingleCharLiteral

    def parseImpl(self, instring, loc, doActions=True):
        if instring[loc] == self.first_match_char and instring.startswith(self.match, loc):
            return loc + self.match_len, self.match
        raise ParseException(instring, loc, self.errmsg, self)


class _SingleCharLiteral(Literal):
    def parseImpl(self, instring, loc, doActions=True):
        if instring[loc] == self.first_match_char:
            return loc + 1, self.match
        raise ParseException(instring, loc, self.errmsg, self)


ParserElement._literalStringClass = Literal


class Keyword(Token):
    """作为关键字精确匹配指定字符串，即必须紧跟非关键字字符。"""
    DEFAULT_KEYWORD_CHARS = alphanums + "_$"

    def __init__(self, matchString, identChars=None, caseless=False):
        super(Keyword, self).__init__()
        if identChars is None:
            identChars = Keyword.DEFAULT_KEYWORD_CHARS
        self.match_len = len(matchString)
        self.match = matchString
        try:
            self.first_match_char = matchString[0]
        except IndexError:
            warnings.warn("null string passed to Keyword; use Empty() instead",
                          SyntaxWarning, stacklevel=2)

        self.may_index_error = False
        self.may_return_empty = False
        self.name = '"%s"' % self.match
        self.errmsg = "Expected " + self.name

        if caseless:
            self.caselessmatch = matchString.upper()
            identChars = identChars.upper()
        self.ident_chars = set(identChars)
        self.caseless = caseless

    def parseImpl(self, instring, loc, doActions=True):
        if not self.caseless:
            if instring[loc] == self.first_match_char:
                if ((self.match_len == 1 or instring.startswith(self.match, loc))
                        and (loc >= len(instring) - self.match_len
                             or instring[loc + self.match_len] not in self.ident_chars)
                        and (loc == 0 or instring[loc - 1] not in self.ident_chars)):
                    return loc + self.match_len, self.match
        else:
            if ((instring[loc:loc + self.match_len].upper() == self.caselessmatch)
                    and (loc >= len(instring) - self.match_len
                         or instring[loc + self.match_len].upper() not in self.ident_chars)
                    and (loc == 0
                         or instring[loc - 1].upper() not in self.ident_chars)):
                return loc + self.match_len, self.match

        raise ParseException(instring, loc, self.errmsg, self)

    def copy(self):
        c = super(Keyword, self).copy()
        c.ident_chars = Keyword.DEFAULT_KEYWORD_CHARS
        return c

    @staticmethod
    def setDefaultKeywordChars(chars):
        """覆盖默认的关键字字符"""
        Keyword.DEFAULT_KEYWORD_CHARS = chars


class CaselessLiteral(Literal):
    """匹配指定字符串，忽略字母的大小写。"""

    def __init__(self, matchString):
        super(CaselessLiteral, self).__init__(matchString.upper())
        self.name = "'%s'" % self.return_string
        self.errmsg = "Expected " + self.name
        self.return_string = matchString

    def parseImpl(self, instring, loc, doActions=True):
        if instring[loc:loc + self.match_len].upper() == self.match:
            return loc + self.match_len, self.return_string
        raise ParseException(instring, loc, self.errmsg, self)


class CaselessKeyword(Keyword):
    """Caseless version of :class:`Keyword`."""

    def __init__(self, matchString, identChars=None):
        super(CaselessKeyword, self).__init__(matchString, identChars, caseless=True)


class CloseMatch(Token):
    """Literal的变体类，即最多n个不匹配字符的字符串。"""

    def __init__(self, match_string, maxMismatches=1):
        super(CloseMatch, self).__init__()
        self.name = match_string
        self.max_mismatches = maxMismatches
        self.match_string = match_string
        self.may_return_empty = False
        self.may_index_error = False
        self.errmsg = "Expected %r (with up to %d mismatches)" % (self.match_string, self.max_mismatches)

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        start = loc
        instr_len = len(instring)
        max_loc = start + len(self.match_string)

        if max_loc <= instr_len:
            match_string = self.match_string
            match_stringloc = 0
            mismatches = []
            max_mismatches = self.max_mismatches

            for match_stringloc, s_m in enumerate(zip(instring[loc:max_loc], match_string)):
                src, mat = s_m
                if src != mat:
                    mismatches.append(match_stringloc)
                    if len(mismatches) > max_mismatches:
                        break
            else:
                # 当没有匹配到成功时，计算时需要把当前位置加1。
                loc = match_stringloc + 1
                results = ParseResults([instring[start:loc]])
                # 需要保存原始的匹配信息
                results['mismatches'] = mismatches
                results['original'] = match_string
                return loc, results

        raise ParseException(instring, loc, self.errmsg, self)


class Word(Token):
    """
    匹配单测由给定字符组成的文本。

    参数::

    - ``initChars`` - 首字符范围
    - ``bodyChars`` - 首字符以后的范围，留空则等同于 ``initChars``
    - ``min``, ``max``, ``extract`` - 匹配的 最小、最大和精确长度
    - ``asKeyword`` - 完整匹配单词边界
    - ``excludeChars`` - 排除字符

    示例::

        # 数字
        integer = Word(nums) # equivalent to Word("0123456789") or Word(srange("0-9"))

        # 首字母大写，其余字母小写
        capital_word = Word(alphas.upper(), alphas.lower())

        # 罗马数字字符范围
        roman = Word("IVXLCDM")

        # 除了逗号以外的非空白字符
        csv_value = Word(printables, excludeChars=",")
    """

    def __init__(self, initChars, bodyChars=None, min=1, max=0, exact=0, asKeyword=False, excludeChars=None):
        super(Word, self).__init__()
        if excludeChars:
            exclude_chars = set(excludeChars)
            # 替换initChars和bodyChars
            initChars = ''.join(c for c in initChars if c not in exclude_chars)
            if bodyChars:
                bodyChars = ''.join(c for c in bodyChars if c not in exclude_chars)
        self.init_chars_original = initChars
        self.init_chars = set(initChars)
        if bodyChars:
            self.body_chars_original = bodyChars
            self.body_chars = set(bodyChars)
        else:
            self.body_chars_original = initChars
            self.body_chars = set(initChars)

        if min < 1:
            raise ValueError(
                "cannot specify a minimum length < 1; use Optional(Word()) if zero-length word is permitted")

        self.min_len = min

        self.max_specified = max > 0

        if max <= 0:
            self.max_len = sys.maxint
        else:
            self.max_len = max

        if exact > 0:
            self.max_len = exact
            self.min_len = exact

        self.name = _ustr(self)
        self.errmsg = "Expected " + self.name
        self.may_index_error = False
        self.as_keyword = asKeyword

        if ' ' not in self.init_chars_original + self.body_chars_original and (min == 1 and max == 0 and exact == 0):
            if self.body_chars_original == self.init_chars_original:
                self.re_string = "[%s]+" % _escape_regex_range_chars(self.init_chars_original)
            elif len(self.init_chars_original) == 1:
                self.re_string = "%s[%s]*" % (re.escape(self.init_chars_original),
                                              _escape_regex_range_chars(self.body_chars_original),)
            else:
                self.re_string = "[%s][%s]*" % (_escape_regex_range_chars(self.init_chars_original),
                                                _escape_regex_range_chars(self.body_chars_original),)
            if self.as_keyword:
                self.re_string = r"\b" + self.re_string + r"\b"

            try:
                self.re = re.compile(self.re_string)
                self.re_match = self.re.match
                self.__class__ = _WordRegex
            except Exception:
                self.re = None

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的字符串。"""
        if instring[loc] not in self.init_chars:
            raise ParseException(instring, loc, self.errmsg, self)

        start = loc
        loc += 1
        instrlen = len(instring)
        body_chars = self.body_chars
        max_loc = min(start + self.max_len, instrlen)
        while loc < max_loc and instring[loc] in body_chars:
            loc += 1

        need_throw_exc = False
        if loc - start < self.min_len:
            need_throw_exc = True
        elif self.max_specified and loc < instrlen and instring[loc] in body_chars:
            need_throw_exc = True
        elif self.as_keyword:
            if (start > 0 and instring[start - 1] in body_chars
                    or loc < instrlen and instring[loc] in body_chars):
                need_throw_exc = True

        if need_throw_exc:
            raise ParseException(instring, loc, self.errmsg, self)
        else:
            return loc, instring[start:loc]

    def __str__(self):
        try:
            return super(Word, self).__str__()
        except Exception:
            # ignored
            pass

        self._reset_str_repr()
        return self.str_repr

    def _reset_str_repr(self):
        if self.str_repr is None:

            def chars_as_str(s):
                if len(s) > 4:
                    return s[:4] + "..."

                return s

            if self.init_chars_original != self.body_chars_original:
                self.str_repr = "W:(%s, %s)" % (
                    chars_as_str(self.init_chars_original), chars_as_str(self.body_chars_original))
            else:
                self.str_repr = "W:(%s)" % chars_as_str(self.init_chars_original)


class _WordRegex(Word):
    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式"""
        result = self.re_match(instring, loc)
        if not result:
            raise ParseException(instring, loc, self.errmsg, self)

        # 找到结尾的位置
        loc = result.end()
        return loc, result.group()


class Char(_WordRegex):
    """``Word(characters, exact=1)``的单字符版本。"""

    def __init__(self, charset, asKeyword=False, excludeChars=None):
        super(Char, self).__init__(charset, exact=1, asKeyword=asKeyword, excludeChars=excludeChars)
        self.re_string = "[%s]" % _escape_regex_range_chars(''.join(self.init_chars))
        if asKeyword:
            self.re_string = r"\b%s\b" % self.re_string
        self.re = re.compile(self.re_string)
        self.re_match = self.re.match


class Regex(Token):
    """
    匹配给定的正则表达式，正则表达式使用标准的python模块 `re module <https://docs.python.org/3/library/re.html>`_.
    命名捕获组会被转换为命名结果。
    """

    def __init__(self, pattern, flags=0, asGroupList=False, asMatch=False):
        """参数 ``pattern`` 和 ``flags`` 会被传递给 ``re.compile()`` 方法。"""
        super(Regex, self).__init__()
        self._set_re_info(pattern, flags)

        self.re_match = self.re.match

        self.name = _ustr(self)
        self.errmsg = "Expected " + self.name
        self.may_index_error = False
        self.may_return_empty = self.re_match("") is not None
        self.as_match = asMatch
        self.as_group_list = asGroupList
        if self.as_group_list:
            self.parseImpl = self.parseImplAsGroupList
        if self.as_match:
            self.parseImpl = self.parseImplAsMatch

    def parseImplAsGroupList(self, instring, loc, doActions=True):
        result = self.re_match(instring, loc)
        if result:
            loc = result.end()
            ret = result.groups()
            return loc, ret
        else:
            raise ParseException(instring, loc, self.errmsg, self)

    def parseImplAsMatch(self, instring, loc, doActions=True):
        result = self.re_match(instring, loc)
        if result:
            loc = result.end()
            ret = result
            return loc, ret
        else:
            raise ParseException(instring, loc, self.errmsg, self)

    def parseImpl(self, instring, loc, doActions=True):
        result = self.re_match(instring, loc)
        if result:
            loc = result.end()
            ret = ParseResults(result.group())
            group_dict = result.groupdict()
            if group_dict:
                for k, v in group_dict.items():
                    ret[k] = v
            return loc, ret
        else:
            raise ParseException(instring, loc, self.errmsg, self)

    def __str__(self):
        try:
            return super(Regex, self).__str__()
        except Exception:
            if self.str_repr is None:
                self.str_repr = "Re:(%s)" % repr(self.pattern)

            return self.str_repr

    def sub(self, repl):
        """返回带有附加解析操作的Regex以转换解析结果，就像使用re.sub(expr, repl, string)调用一样"""
        if self.as_group_list:
            warnings.warn("cannot use sub() with Regex(asGroupList=True)",
                          SyntaxWarning, stacklevel=2)
            raise SyntaxError()

        if self.as_match and callable(repl):
            warnings.warn("cannot use sub() with a callable with Regex(asMatch=True)",
                          SyntaxWarning, stacklevel=2)
            raise SyntaxError()

        if not self.as_match:
            def pa(tokens):
                return self.re.sub(repl, tokens[0])
        else:
            def pa(tokens):
                return tokens[0].expand(repl)

        return self.addParseAction(pa)

    def _set_re_info(self, pattern, flags):
        if isinstance(pattern, basestring):
            if not pattern:
                warnings.warn("null string passed to Regex; use Empty() instead", SyntaxWarning, stacklevel=2)

            self.flags = flags
            self.pattern = pattern

            try:
                # compile regex
                self.re = re.compile(self.pattern, self.flags)
                self.re_string = self.pattern
            except sre_constants.error:
                warnings.warn("invalid pattern (%s) passed to Regex" % pattern, SyntaxWarning, stacklevel=2)
                raise
        elif hasattr(pattern, 'pattern') and hasattr(pattern, 'match'):
            self.flags = flags
            self.re = pattern
            self.pattern = self.re_string = pattern.pattern
        else:
            raise TypeError("Regex may only be constructed with a string or a compiled RE object")


class QuotedString(Token):
    """用于匹配由引号字符分隔的字符串。"""

    def __init__(self, quoteChar, escChar=None, escQuote=None, multiline=False,
                 unquoteResults=True, endQuoteChar=None, convertWhitespaceEscapes=True):
        super(QuotedString, self).__init__()

        quota_char = quoteChar.strip()
        if not quota_char:
            warnings.warn("quoteChar cannot be the empty string", SyntaxWarning, stacklevel=2)
            raise SyntaxError()

        if endQuoteChar is None:
            end_quota_char = quota_char
        else:
            end_quota_char = endQuoteChar.strip()
            if not end_quota_char:
                warnings.warn("endQuoteChar cannot be the empty string", SyntaxWarning, stacklevel=2)
                raise SyntaxError()

        self.quote_char = quota_char
        self.quote_char_len = len(quota_char)
        self.first_quote_char = quota_char[0]
        self.end_quote_char = end_quota_char
        self.end_quote_char_len = len(end_quota_char)
        self.esc_char = escChar
        self.esc_quote = escQuote
        self.convert_whitespace_escapes = convertWhitespaceEscapes
        self.unquote_results = unquoteResults

        if multiline:
            self.flags = re.MULTILINE | re.DOTALL
            self.pattern = r'%s(?:[^%s%s]' % (re.escape(self.quote_char),
                                              _escape_regex_range_chars(self.end_quote_char[0]),
                                              (escChar is not None and _escape_regex_range_chars(escChar) or ''))
        else:
            self.flags = 0
            self.pattern = r'%s(?:[^%s\n\r%s]' % (re.escape(self.quote_char),
                                                  _escape_regex_range_chars(self.end_quote_char[0]),
                                                  (escChar is not None and _escape_regex_range_chars(escChar) or ''))
        if len(self.end_quote_char) > 1:
            self.pattern += (
                    '|(?:' + ')|(?:'.join("%s[^%s]" % (re.escape(self.end_quote_char[:i]),
                                                       _escape_regex_range_chars(self.end_quote_char[i]))
                                          for i in range(len(self.end_quote_char) - 1, 0, -1)) + ')')

        if escQuote:
            self.pattern += (r'|(?:%s)' % re.escape(escQuote))
        if escChar:
            self.pattern += (r'|(?:%s.)' % re.escape(escChar))
            self.esc_charReplacePattern = re.escape(self.esc_char) + "(.)"
        self.pattern += (r')*%s' % re.escape(self.end_quote_char))
        self._set_re_info()

        self.name = _ustr(self)
        self.errmsg = "Expected " + self.name
        self.may_index_error = False
        self.may_return_empty = True

    def parseImpl(self, instring, loc, doActions=True):
        result = instring[loc] == self.first_quote_char and self.re_match(instring, loc) or None
        if not result:
            raise ParseException(instring, loc, self.errmsg, self)

        loc = result.end()
        ret = result.group()

        if self.unquote_results:

            ret = ret[self.quote_char_len: -self.end_quote_char_len]

            if isinstance(ret, basestring):
                if '\\' in ret and self.convert_whitespace_escapes:
                    ws_map = {
                        r'\t': '\t',
                        r'\n': '\n',
                        r'\f': '\f',
                        r'\r': '\r',
                    }
                    for ws_lit, ws_char in ws_map.items():
                        ret = ret.replace(ws_lit, ws_char)

                if self.esc_char:
                    ret = re.sub(self.esc_charReplacePattern, r"\g<1>", ret)

                if self.esc_quote:
                    ret = ret.replace(self.esc_quote, self.end_quote_char)

        return loc, ret

    def __str__(self):
        try:
            return super(QuotedString, self).__str__()
        except Exception:
            if self.str_repr is None:
                self.str_repr = "quoted string, starting with %s ending with %s" % (
                    self.quote_char, self.end_quote_char)

            return self.str_repr

    def _set_re_info(self):
        try:
            self.re = re.compile(self.pattern, self.flags)
            self.re_match = self.re.match
            self.re_string = self.pattern
        except sre_constants.error:
            warnings.warn("invalid pattern (%s) passed to Regex" % self.pattern,
                          SyntaxWarning, stacklevel=2)
            raise


class CharsNotIn(Token):
    """匹配不包含给定字符串中字符的文本。

    ``min``和``max``定义匹配结果内容的最小和最大长度，``min``默认为1，``max``默认为0；
    ``exact``定义精确长度，默认为0。

    示例::

        csv_value = CharsNotIn(',')
        print(delimitedList(csv_value).parseString("dkls,lsdkjf,s12 34,@!#,213"))

    得到::

        ['dkls', 'lsdkjf', 's12 34', '@!#', '213']
    """

    def __init__(self, notChars, min=1, max=0, exact=0):
        super(CharsNotIn, self).__init__()
        self.skip_whitespace = False
        self.exclude_chars = notChars

        if min < 1:
            raise ValueError("cannot specify a minimum length < 1; use "
                             "Optional(CharsNotIn()) if zero-length char group is permitted")

        self.min_len = min

        if max > 0:
            self.max_len = max
        else:
            self.max_len = sys.maxint

        if exact > 0:
            self.max_len = exact
            self.min_len = exact

        self.name = _ustr(self)
        self.errmsg = "Expected " + self.name
        self.may_return_empty = (self.min_len == 0)
        self.may_index_error = False

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        if instring[loc] in self.exclude_chars:
            raise ParseException(instring, loc, self.errmsg, self)

        start = loc
        loc += 1
        max_len = min(start + self.max_len, len(instring))
        exclude_chars = self.exclude_chars
        while loc < max_len and instring[loc] not in exclude_chars:
            loc += 1

        if start > loc - self.min_len:
            raise ParseException(instring, loc, self.errmsg, self)
        else:
            return loc, instring[start:loc]

    def __str__(self):
        try:
            # just call super method
            return super(CharsNotIn, self).__str__()
        except Exception:
            pass

        self._reset_str_repr()
        return self.str_repr

    def _reset_str_repr(self):
        if self.str_repr is None:
            if len(self.exclude_chars) <= 4:
                self.str_repr = "!W:(%s)" % self.exclude_chars
            else:
                self.str_repr = "!W:(%s...)" % self.exclude_chars[:4]


class White(Token):
    """
    匹配空白字符（空白字符一般会被忽略），默认的空白字符为 ``"\\t\\r\\n"``，
    接受可选的 ``min`` 、 ``max`` 和 ``exact`` 参数。
    """
    white_strs = {
        u'\u00A0': '<NBSP>',
        u'\u1680': '<OGHAM_SPACE_MARK>', u'\u180E': '<MONGOLIAN_VOWEL_SEPARATOR>',
        u'\u2000': '<EN_QUAD>', u'\u2001': '<EM_QUAD>',
        u'\u2002': '<EN_SPACE>', u'\u2003': '<EM_SPACE>',
        u'\u2004': '<THREE-PER-EM_SPACE>', u'\u2005': '<FOUR-PER-EM_SPACE>',
        u'\u2006': '<SIX-PER-EM_SPACE>', u'\u2007': '<FIGURE_SPACE>',
        u'\u2008': '<PUNCTUATION_SPACE>', u'\u2009': '<THIN_SPACE>',
        u'\u200A': '<HAIR_SPACE>', u'\u200B': '<ZERO_WIDTH_SPACE>',
        u'\u202F': '<NNBSP>', u'\u205F': '<MMSP>',
        u'\u3000': '<IDEOGRAPHIC_SPACE>',
        ' ': '<SP>',
        '\t': '<TAB>',
        '\n': '<LF>',
        '\r': '<CR>',
        '\f': '<FF>',
    }

    def __init__(self, ws=" \t\r\n", min=1, max=0, exact=0):
        super(White, self).__init__()
        self.match_white = ws
        self.setWhitespaceChars("".join(c for c in self.white_chars if c not in self.match_white))
        self.name = ("".join(White.white_strs[c] for c in self.match_white))
        self.may_return_empty = True
        self.errmsg = "Expected " + self.name

        self.min_len = min

        if max > 0:
            self.max_len = max
        else:
            self.max_len = sys.maxint

        if exact > 0:
            self.max_len = exact
            self.min_len = exact

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的字符串"""
        if instring[loc] not in self.match_white:
            raise ParseException(instring, loc, self.errmsg, self)
        start = loc
        loc += 1
        max_loc = start + self.max_len
        max_loc = min(max_loc, len(instring))
        while loc < max_loc and instring[loc] in self.match_white:
            loc += 1

        if loc - start < self.min_len:
            raise ParseException(instring, loc, self.errmsg, self)

        return loc, instring[start:loc]


class _PositionToken(Token):
    def __init__(self):
        super(_PositionToken, self).__init__()
        self.may_index_error = False
        self.may_return_empty = True
        self.name = self.__class__.__name__


class LineStart(_PositionToken):
    """匹配当前位置是否位于解析字符串中某行的开头。"""

    def __init__(self):
        super(LineStart, self).__init__()
        self.errmsg = "Expected start of line"

    def parseImpl(self, instring, loc, doActions=True):
        if col(loc, instring) != 1:
            raise ParseException(instring, loc, self.errmsg, self)

        return loc, []


class LineEnd(_PositionToken):
    """如果当前位置在解析字符串中的某一行的末尾，则匹配。"""

    def __init__(self):
        super(LineEnd, self).__init__()
        self.setWhitespaceChars(ParserElement.DEFAULT_WHITE_CHARS.replace("\n", ""))
        self.errmsg = "Expected end of line"

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        if loc == len(instring):
            return loc + 1, []
        elif loc < len(instring):
            if instring[loc] == "\n":
                return loc + 1, "\n"
            else:
                raise ParseException(instring, loc, self.errmsg, self)

        raise ParseException(instring, loc, self.errmsg, self)


class GoToColumn(_PositionToken):
    """用于前进到输入文本的特定列。"""

    def __init__(self, colno):
        super(GoToColumn, self).__init__()
        self.col = colno

    def preParse(self, instring, loc):
        if col(loc, instring) != self.col:
            len_of_instr = len(instring)
            if self.ignore_exprs:
                loc = self._skipIgnorables(instring, loc)
            while loc < len_of_instr and instring[loc].isspace() and col(loc, instring) != self.col:
                loc += 1
        return loc

    def parseImpl(self, instring, loc, doActions=True):
        this_col = col(loc, instring)
        if this_col > self.col:
            raise ParseException(instring, loc, "Text not in expected column", self)

        now_loc = loc + self.col - this_col
        ret = instring[loc: now_loc]
        return now_loc, ret


class StringStart(_PositionToken):
    """如果当前位置位于解析字符串的开头，则匹配。"""

    def __init__(self):
        super(StringStart, self).__init__()
        self.errmsg = "Expected start of text"

    def parseImpl(self, instring, loc, doActions=True):
        if loc != 0:
            if loc != self.preParse(instring, 0):
                raise ParseException(instring, loc, self.errmsg, self)
        return loc, []


class StringEnd(_PositionToken):
    """如果当前位置位于解析字符串的末尾，则匹配。"""

    def __init__(self):
        super(StringEnd, self).__init__()
        # setup errmsg
        self.errmsg = "Expected end of text"

    def parseImpl(self, instring, loc, doActions=True):
        if loc == len(instring):
            return loc + 1, []
        if loc < len(instring):
            raise ParseException(instring, loc, self.errmsg, self)
        else:
            return loc, []


class WordStart(_PositionToken):
    """按照单词开始匹配的方式进行处理。"""

    def __init__(self, wordChars=printables):
        super(WordStart, self).__init__()
        self.word_chars = set(wordChars)
        self.errmsg = "Not at the start of a word"

    def parseImpl(self, instring, loc, doActions=True):
        if loc != 0:
            if (instring[loc - 1] in self.word_chars
                    or instring[loc] not in self.word_chars):
                raise ParseException(instring, loc, self.errmsg, self)
        return loc, []


class WordEnd(_PositionToken):
    """按照单词结尾匹配的方式进行处理。"""

    def __init__(self, wordChars=printables):
        super(WordEnd, self).__init__()
        self.word_chars = set(wordChars)
        self.errmsg = "Not at the end of a word"
        self.skip_whitespace = False

    def parseImpl(self, instring, loc, doActions=True):
        len_of_instr = len(instring)
        if len_of_instr > 0 and loc < len_of_instr:
            if (instring[loc] in self.word_chars or
                    instring[loc - 1] not in self.word_chars):
                raise ParseException(instring, loc, self.errmsg, self)
        return loc, []


class ParseExpression(ParserElement):
    """ParserElement的抽象子类，用于对解析后的token进行合并和后处理。"""

    def __init__(self, exprs, savelist=False):
        super(ParseExpression, self).__init__(savelist)
        if isinstance(exprs, _generatorType):
            exprs = list(exprs)

        self.call_preparse = False

        if isinstance(exprs, ParserElement):
            self.exprs = [exprs]
        elif isinstance(exprs, basestring):
            self.exprs = [self._literalStringClass(exprs)]
        elif isinstance(exprs, Iterable):
            exprs = list(exprs)
            if any(isinstance(expr, basestring) for expr in exprs):
                exprs = (self._literalStringClass(expr) if isinstance(expr, basestring) else expr for expr in exprs)
            self.exprs = list(exprs)
        else:
            try:
                # update as list
                self.exprs = list(exprs)
            except TypeError:
                # error
                self.exprs = [exprs]

    def ignore(self, other):
        if isinstance(other, Suppress):
            if other not in self.ignore_exprs:
                super(ParseExpression, self).ignore(other)
                for e in self.exprs:
                    e.ignore(self.ignore_exprs[-1])
        else:
            super(ParseExpression, self).ignore(other)
            for e in self.exprs:
                e.ignore(self.ignore_exprs[-1])
        return self

    def append(self, other):
        self.exprs.append(other)
        self.str_repr = None
        return self

    def leaveWhitespace(self):
        self.skip_whitespace = False
        self.exprs = [expr.copy() for expr in self.exprs]
        for expr in self.exprs:
            expr.leaveWhitespace()
        return self

    def __str__(self):
        try:
            return super(ParseExpression, self).__str__()
        except Exception:
            if self.str_repr is None:
                self.str_repr = "%s:(%s)" % (self.__class__.__name__, _ustr(self.exprs))
            return self.str_repr

    def streamline(self):
        super(ParseExpression, self).streamline()

        for e in self.exprs:
            e.streamline()

        if len(self.exprs) == 2:
            other = self.exprs[0]
            if (isinstance(other, self.__class__)
                    and not other.parse_action
                    and other.results_name is None
                    and not other.debug):
                self.exprs = other.exprs[:] + [self.exprs[1]]
                self.str_repr = None
                self.may_return_empty |= other.may_return_empty
                self.may_index_error |= other.may_index_error

            other = self.exprs[-1]
            if (isinstance(other, self.__class__)
                    and not other.parse_action
                    and other.results_name is None
                    and not other.debug):
                self.exprs = self.exprs[:-1] + other.exprs[:]
                self.str_repr = None
                self.may_return_empty |= other.may_return_empty
                self.may_index_error |= other.may_index_error

        self.errmsg = "Expected " + _ustr(self)

        return self

    def copy(self):
        ret = super(ParseExpression, self).copy()
        ret.exprs = [expr.copy() for expr in self.exprs]
        return ret

    def validate(self, validateTrace=None):
        tmp_trace = (validateTrace if validateTrace is not None else [])[:] + [self]
        for expr in self.exprs:
            expr.validate(tmp_trace)
        self.checkRecursion([])

    def _setResultsName(self, name, listAllMatches=False):
        return super(ParseExpression, self)._setResultsName(name, listAllMatches)


class And(ParseExpression):
    """按照逻辑与的方式处理表达式。"""

    class _ErrorStop(Empty):
        def __init__(self, *args, **kwargs):
            super(And._ErrorStop, self).__init__(*args, **kwargs)
            # init selft
            self.leaveWhitespace()
            self.name = '-'

    def __init__(self, exprs, savelist=True):
        converted_exprs = self._convert_exprs(exprs)
        super(And, self).__init__(converted_exprs, savelist)
        self.may_return_empty = all(e.may_return_empty for e in self.exprs)
        self.setWhitespaceChars(self.exprs[0].white_chars)
        self.skip_whitespace = self.exprs[0].skip_whitespace
        self.call_preparse = True

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        loc, result_list = self.exprs[0]._parse(instring, loc, doActions, callPreParse=False)
        error_stop = False
        for e in self.exprs[1:]:
            if isinstance(e, And._ErrorStop):
                error_stop = True
                continue
            if error_stop:
                try:
                    loc, expr_tokens = e._parse(instring, loc, doActions)
                except ParseSyntaxException:
                    raise
                except ParseBaseException as ex:
                    # 擦除异常堆栈
                    ex.__traceback__ = None
                    raise ParseSyntaxException._from_exception(ex)
                except IndexError:
                    raise ParseSyntaxException(instring, len(instring), self.errmsg, self)
            else:
                loc, expr_tokens = e._parse(instring, loc, doActions)
            if expr_tokens or expr_tokens.haskeys():
                result_list += expr_tokens
        return loc, result_list

    def streamline(self):
        if self.exprs:
            if any(isinstance(expr, ParseExpression) and expr.exprs and isinstance(expr.exprs[-1], _PendingSkip)
                   for expr in self.exprs[:-1]):
                for index, expr in enumerate(self.exprs[:-1]):
                    if expr is None:
                        continue
                    if (isinstance(expr, ParseExpression)
                            and expr.exprs
                            and isinstance(expr.exprs[-1], _PendingSkip)):
                        expr.exprs[-1] = expr.exprs[-1] + self.exprs[index + 1]
                        self.exprs[index + 1] = None
                self.exprs = [expr for expr in self.exprs if expr is not None]

        super(And, self).streamline()
        self.may_return_empty = all(e.may_return_empty for e in self.exprs)
        return self

    def __iadd__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        return self.append(other)

    def checkRecursion(self, parseElementList):
        subRecCheckList = parseElementList[:] + [self]
        for e in self.exprs:
            e.checkRecursion(subRecCheckList)
            if not e.may_return_empty:
                break

    def __str__(self):
        if hasattr(self, "name"):
            return self.name

        if self.str_repr is None:
            self.str_repr = "{" + " ".join(_ustr(e) for e in self.exprs) + "}"

        return self.str_repr

    def _convert_exprs(self, exprs):
        exprs = list(exprs)
        if exprs and Ellipsis in exprs:
            n = len(exprs)
            tmp_exprs = []
            for i, expr in enumerate(exprs):
                if expr is not Ellipsis:
                    tmp_exprs.append(expr)
                    continue

                if i < n - 1:
                    skipto_args = (Empty() + exprs[i + 1]).exprs[-1]
                    tmp_exprs.append(SkipTo(skipto_args)("_skipped*"))
                else:
                    raise Exception("cannot construct And with sequence ending in ...")

            exprs[:] = tmp_exprs

        return exprs


class Or(ParseExpression):
    """针对表达式执行或的逻辑运算操作。"""

    def __init__(self, exprs, savelist=False):
        super(Or, self).__init__(exprs, savelist)
        if not self.exprs:
            self.may_return_empty = True
        else:
            self.may_return_empty = any(e.may_return_empty for e in self.exprs)

    def streamline(self):
        super(Or, self).streamline()
        if __compat__.collect_all_And_tokens:
            self.save_as_list = any(e.save_as_list for e in self.exprs)
        return self

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        max_exc_loc = -1
        max_exception = None
        matched = []
        for e in self.exprs:
            try:
                loc2 = e.tryParse(instring, loc)
            except ParseException as err:
                err.__traceback__ = None
                if err.loc > max_exc_loc:
                    max_exception = err
                    max_exc_loc = err.loc
            except IndexError:
                if len(instring) > max_exc_loc:
                    max_exception = ParseException(instring, len(instring), e.errmsg, self)
                    max_exc_loc = len(instring)
            else:
                matched.append((loc2, e))

        if matched:
            matched.sort(key=itemgetter(0), reverse=True)

            if not doActions:
                best_expr = matched[0][1]
                return best_expr._parse(instring, loc, doActions)

            longest = -1, None
            for loc1, expr1 in matched:
                if loc1 <= longest[0]:
                    return longest

                try:
                    loc2, toks = expr1._parse(instring, loc, doActions)
                except ParseException as err:
                    err.__traceback__ = None
                    if err.loc > max_exc_loc:
                        max_exception = err
                        max_exc_loc = err.loc
                else:
                    if loc2 >= loc1:
                        return loc2, toks

                    elif loc2 > longest[0]:
                        longest = loc2, toks

            if longest != (-1, None):
                return longest

        if max_exception is not None:
            max_exception.msg = self.errmsg
            raise max_exception
        else:
            raise ParseException(instring, loc, "no defined alternatives to match", self)

    def __ixor__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        return self.append(other)

    def __str__(self):
        if hasattr(self, "name"):
            return self.name

        if self.str_repr is None:
            self.str_repr = "{" + " ^ ".join(_ustr(e) for e in self.exprs) + "}"

        return self.str_repr

    def checkRecursion(self, parseElementList):
        subRecCheckList = parseElementList[:] + [self]
        for e in self.exprs:
            e.checkRecursion(subRecCheckList)

    def _setResultsName(self, name, listAllMatches=False):
        return super(Or, self)._setResultsName(name, listAllMatches)


class MatchFirst(ParseExpression):
    """找到第一个满足条件的表达式。"""

    def __init__(self, exprs, savelist=False):
        super(MatchFirst, self).__init__(exprs, savelist)
        if self.exprs:
            self.may_return_empty = any(e.may_return_empty for e in self.exprs)
        else:
            self.may_return_empty = True

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        max_exc_loc = -1
        max_exception = None
        for e in self.exprs:
            try:
                ret = e._parse(instring, loc, doActions)
                return ret
            except ParseException as err:
                if err.loc > max_exc_loc:
                    max_exception = err
                    max_exc_loc = err.loc
            except IndexError:
                if len(instring) > max_exc_loc:
                    max_exception = ParseException(instring, len(instring), e.errmsg, self)
                    max_exc_loc = len(instring)

        else:
            if max_exception is not None:
                max_exception.msg = self.errmsg
                raise max_exception
            else:
                raise ParseException(instring, loc, "no defined alternatives to match", self)

    def streamline(self):
        super(MatchFirst, self).streamline()
        if __compat__.collect_all_And_tokens:
            self.save_as_list = any(e.save_as_list for e in self.exprs)
        return self

    def __str__(self):
        if hasattr(self, "name"):
            return self.name

        if self.str_repr is None:
            self.str_repr = "{" + " | ".join(_ustr(e) for e in self.exprs) + "}"

        return self.str_repr

    def __ior__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        return self.append(other)

    def checkRecursion(self, parseElementList):
        subRecCheckList = parseElementList[:] + [self]
        for e in self.exprs:
            e.checkRecursion(subRecCheckList)

    def _setResultsName(self, name, listAllMatches=False):
        return super(MatchFirst, self)._setResultsName(name, listAllMatches)


class Each(ParseExpression):
    """要求找到所有给定的表达式，但顺序不限。表达式可以用空格分隔。"""

    def __init__(self, exprs, savelist=True):
        super(Each, self).__init__(exprs, savelist)
        self.may_return_empty = all(e.may_return_empty for e in self.exprs)
        self.skip_whitespace = True
        self.init_expr_groups = True
        self.save_as_list = True

    def streamline(self):
        super(Each, self).streamline()
        self.may_return_empty = all(e.may_return_empty for e in self.exprs)
        return self

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式"""
        if self.init_expr_groups:
            self.opt1map = dict((id(e.expr), e) for e in self.exprs if isinstance(e, Optional))
            opt1 = [e.expr for e in self.exprs if isinstance(e, Optional)]
            opt2 = [e for e in self.exprs if e.may_return_empty and not isinstance(e, (Optional, Regex))]
            self.optionals = opt1 + opt2
            self.multioptionals = [e.expr for e in self.exprs if isinstance(e, ZeroOrMore)]
            self.multirequired = [e.expr for e in self.exprs if isinstance(e, OneOrMore)]
            self.required = [e for e in self.exprs if not isinstance(e, (Optional, ZeroOrMore, OneOrMore))]
            self.required += self.multirequired
            self.init_expr_groups = False
        tmp_loc = loc
        tmp_reqd = self.required[:]
        tmp_opt = self.optionals[:]
        match_order = []

        keep_matching = True
        while keep_matching:
            tmpExprs = tmp_reqd + tmp_opt + self.multioptionals + self.multirequired
            failed = []
            for e in tmpExprs:
                try:
                    tmp_loc = e.tryParse(instring, tmp_loc)
                except ParseException:
                    failed.append(e)
                else:
                    match_order.append(self.opt1map.get(id(e), e))
                    if e in tmp_reqd:
                        tmp_reqd.remove(e)
                    elif e in tmp_opt:
                        tmp_opt.remove(e)
            if len(failed) == len(tmpExprs):
                keep_matching = False

        if tmp_reqd:
            missing = ", ".join(_ustr(e) for e in tmp_reqd)
            raise ParseException(instring, loc, "Missing one or more required elements (%s)" % missing)

        match_order += [e for e in self.exprs if isinstance(e, Optional) and e.expr in tmp_opt]

        ret = []
        for e in match_order:
            loc, results = e._parse(instring, loc, doActions)
            ret.append(results)

        final_results = sum(ret, ParseResults([]))
        return loc, final_results

    def __str__(self):
        if hasattr(self, "name"):
            return self.name

        if self.str_repr is None:
            self.str_repr = "{" + " & ".join(_ustr(e) for e in self.exprs) + "}"

        return self.str_repr

    def checkRecursion(self, parseElementList):
        subRecCheckList = parseElementList[:] + [self]
        for e in self.exprs:
            e.checkRecursion(subRecCheckList)


class ParseElementEnhance(ParserElement):
    """:class:`ParserElement` 的抽象子类，用于组装或后置处理匹配范式。"""

    def __init__(self, expr, savelist=False):
        super(ParseElementEnhance, self).__init__(savelist)
        if isinstance(expr, basestring):
            if not issubclass(self._literalStringClass, Token):
                expr = self._literalStringClass(Literal(expr))
            else:
                expr = self._literalStringClass(expr)
        self.expr = expr
        self.str_repr = None
        if expr is not None:
            self.skip_whitespace = expr.skip_whitespace
            self.save_as_list = expr.save_as_list
            self.may_index_error = expr.may_index_error
            self.may_return_empty = expr.may_return_empty
            self.call_preparse = expr.call_preparse
            self.ignore_exprs.extend(expr.ignore_exprs)
            self.setWhitespaceChars(expr.white_chars)

    def __str__(self):
        try:
            return super(ParseElementEnhance, self).__str__()
        except Exception:
            pass

        if self.str_repr is None and self.expr is not None:
            self.str_repr = "%s:(%s)" % (self.__class__.__name__, _ustr(self.expr))
        return self.str_repr

    def parseImpl(self, instring, loc, doActions=True):
        if self.expr is not None:
            return self.expr._parse(instring, loc, doActions, callPreParse=False)
        else:
            raise ParseException("", loc, self.errmsg, self)

    def checkRecursion(self, parseElementList):
        if self in parseElementList:
            raise RecursiveGrammarException(parseElementList + [self])
        sub_rec_check_list = parseElementList[:] + [self]
        if self.expr is not None:
            self.expr.checkRecursion(sub_rec_check_list)

    def leaveWhitespace(self):
        self.skip_whitespace = False
        self.expr = self.expr.copy()
        if self.expr is not None:
            self.expr.leaveWhitespace()
        return self

    def streamline(self):
        super(ParseElementEnhance, self).streamline()
        expr = self.expr
        if expr is not None:
            # call expr first
            expr.streamline()
        return self

    def ignore(self, other):
        if isinstance(other, Suppress):
            if other not in self.ignore_exprs:
                super(ParseElementEnhance, self).ignore(other)
                if self.expr is not None:
                    self.expr.ignore(self.ignore_exprs[-1])
        else:
            super(ParseElementEnhance, self).ignore(other)
            if self.expr is not None:
                self.expr.ignore(self.ignore_exprs[-1])
        return self

    def validate(self, validateTrace=None):
        if validateTrace is None:
            validate_trace = []
        else:
            validate_trace = validateTrace
        tmp = validate_trace[:] + [self]
        if self.expr is not None:
            self.expr.validate(tmp)
        self.checkRecursion([])


class FollowedBy(ParseElementEnhance):
    """给定解析表达式的前缀匹配。"""

    def __init__(self, expr):
        super(FollowedBy, self).__init__(expr)
        self.may_return_empty = True

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        _, ret = self.expr._parse(instring, loc, doActions=doActions)
        del ret[:]

        return loc, ret


class NotAny(ParseElementEnhance):
    """不允许与给定的解析表达式进行匹配。"""

    def __init__(self, expr):
        super(NotAny, self).__init__(expr)
        self.errmsg = "Found unwanted token, " + _ustr(self.expr)
        self.may_return_empty = True
        self.skip_whitespace = False

    def __str__(self):
        if hasattr(self, "name"):
            return self.name

        if self.str_repr is None:
            self.str_repr = "~{" + _ustr(self.expr) + "}"

        return self.str_repr

    def parseImpl(self, instring, loc, doActions=True):
        if self.expr.canParseNext(instring, loc):
            raise ParseException(instring, loc, self.errmsg, self)
        return loc, []


class PrecededBy(ParseElementEnhance):
    """查找给定解析表达式的匹配。"""

    def __init__(self, expr, retreat=None):
        super(PrecededBy, self).__init__(expr)
        self.may_index_error = False
        self.may_return_empty = True
        self.expr = self.expr().leaveWhitespace()
        self.exact = False
        if isinstance(expr, str):
            retreat = len(expr)
            self.exact = True
        elif isinstance(expr, (Literal, Keyword)):
            retreat = expr.match_len
            self.exact = True
        elif isinstance(expr, (Word, CharsNotIn)) and expr.max_len != sys.maxint:
            retreat = expr.max_len
            self.exact = True
        elif isinstance(expr, _PositionToken):
            retreat = 0
            self.exact = True
        self.retreat = retreat
        self.skip_whitespace = False
        self.errmsg = "not preceded by " + str(expr)
        self.parse_action.append(lambda s, l, t: t.__delitem__(slice(None, None)))

    def parseImpl(self, instring, loc=0, doActions=True):
        """解析给定的表达式。"""
        if self.exact:
            if loc < self.retreat:
                raise ParseException(instring, loc, self.errmsg)

            # 起始偏移需要减去repeat的值
            _, ret = self.expr._parse(instring, loc - self.retreat)
        else:
            test_expr = self.expr + StringEnd()
            instring_slice = instring[max(0, loc - self.retreat):loc]
            last_expr = ParseException(instring, loc, self.errmsg)
            for offset in range(1, min(loc, self.retreat + 1) + 1):
                try:
                    _, ret = test_expr._parse(instring_slice, len(instring_slice) - offset)
                except ParseBaseException as ex:
                    # 需要保留最后一个表达式产生的异常
                    last_expr = ex
                else:
                    break
            else:
                raise last_expr
        return loc, ret


class _MultipleMatch(ParseElementEnhance):
    def __init__(self, expr, stopOn=None):
        super(_MultipleMatch, self).__init__(expr)
        self.save_as_list = True
        if isinstance(stopOn, basestring):
            ender = self._literalStringClass(stopOn)
        else:
            ender = stopOn

        self.stopOn(ender)

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        self_expr_parse = self.expr._parse
        self_skip_ignorable = self._skipIgnorables
        check_ender = self.not_ender is not None
        if check_ender:
            self.not_ender.tryParse(instring, loc)

        loc, tokens = self_expr_parse(instring, loc, doActions, callPreParse=False)
        try:
            has_ignore_exprs = (not not self.ignore_exprs)
            while 1:
                if check_ender:
                    self.not_ender.tryParse(instring, loc)
                if has_ignore_exprs:
                    pre_loc = self_skip_ignorable(instring, loc)
                else:
                    pre_loc = loc
                loc, tmp_tokens = self_expr_parse(instring, pre_loc, doActions)
                if tmp_tokens or tmp_tokens.haskeys():
                    tokens += tmp_tokens
        except (ParseException, IndexError):
            pass

        return loc, tokens

    def stopOn(self, ender):
        if isinstance(ender, basestring):
            ender = self._literalStringClass(ender)
        self.not_ender = ~ender if ender is not None else None
        return self

    def _setResultsName(self, name, listAllMatches=False):
        return super(_MultipleMatch, self)._setResultsName(name, listAllMatches)


class OneOrMore(_MultipleMatch):
    """重复给定的匹配范式一次或多次（即至少匹配一次）。

    参数:
     - expr - 需要重复匹配的表达式
     - stopOn - 终止表达式

    示例::

        data_word = Word(alphas)
        label = data_word + FollowedBy(':')
        attr_expr = Group(label + Suppress(':') + OneOrMore(data_word).setParseAction(' '.join))

        text = "shape: SQUARE posn: upper left color: BLACK"
        OneOrMore(attr_expr).parseString(text).pprint()
        # Fail! -> [['shape', 'SQUARE color']]

        # 设置 stopOn 属性避免额外匹配
        attr_expr = Group(label + Suppress(':') + OneOrMore(data_word, stopOn=label).setParseAction(' '.join))
        OneOrMore(attr_expr).parseString(text).pprint()
        # Better -> [['shape', 'SQUARE'], ['posn', 'upper left'], ['color', 'BLACK']]

        # 同
        (attr_expr * (1,)).parseString(text).pprint()
    """

    def __str__(self):
        if hasattr(self, "name"):
            return self.name

        if self.str_repr is None:
            self.str_repr = "{" + _ustr(self.expr) + "}..."

        return self.str_repr


class ZeroOrMore(_MultipleMatch):
    """重复给定的匹配范式零次或多次（即匹配任意次）。

    Parameters:
     - expr - 需要重复匹配的表达式
     - stopOn - 终止表达式
    """

    def __init__(self, expr, stopOn=None):
        super(ZeroOrMore, self).__init__(expr, stopOn=stopOn)
        self.may_return_empty = True

    def __str__(self):
        if hasattr(self, "name"):
            return self.name

        if self.str_repr is None:
            self.str_repr = "[" + _ustr(self.expr) + "]..."

        return self.str_repr

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        try:
            return super(ZeroOrMore, self).parseImpl(instring, loc, doActions)
        except (ParseException, IndexError):
            return loc, []


class _NullToken(object):
    """标识空的可选符号"""

    def __str__(self):
        return ""

    def __bool__(self):
        return False

    __nonzero__ = __bool__


class Optional(ParseElementEnhance):
    """对给定的表达式执行可选匹配"""

    __optionalNotMatched = _NullToken()

    def __init__(self, expr, default=__optionalNotMatched):
        super(Optional, self).__init__(expr, savelist=False)
        self.default_value = default
        self.save_as_list = self.expr.save_as_list
        self.may_return_empty = True

    def __str__(self):
        if hasattr(self, "name"):
            return self.name

        if self.str_repr is None:
            self.str_repr = "[" + _ustr(self.expr) + "]"

        return self.str_repr

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        try:
            loc, tokens = self.expr._parse(instring, loc, doActions, callPreParse=False)
        except (ParseException, IndexError):
            if self.default_value is not self.__optionalNotMatched:
                if not self.expr.results_name:
                    tokens = [self.default_value]
                else:
                    tokens = ParseResults([self.default_value])
                    tokens[self.expr.results_name] = self.default_value
            else:
                tokens = []
        return loc, tokens


class SkipTo(ParseElementEnhance):
    """用于跳过所有未定义的文本直到找到匹配的表达式的令牌。"""

    def __init__(self, other, include=False, ignore=None, failOn=None):
        super(SkipTo, self).__init__(other)
        self.may_return_empty = True
        self.ignore_expr = ignore
        self.may_index_error = False
        self.save_as_list = False
        self.include_match = include
        if isinstance(failOn, basestring):
            self.fail_on = self._literalStringClass(failOn)
        else:
            self.fail_on = failOn
        self.errmsg = "No match found for " + _ustr(self.expr)

    def parseImpl(self, instring, loc, doActions=True):
        """解析给定的表达式。"""
        start_loc = loc
        instr_len = len(instring)
        expr_parse = self.expr._parse
        self_fail_on_can_parse_next = self.fail_on.canParseNext if self.fail_on is not None else None
        self_ignore_expr_try_parse = self.ignore_expr.tryParse if self.ignore_expr is not None else None

        tmp_loc = loc
        while tmp_loc <= instr_len:
            if self_fail_on_can_parse_next is not None:
                if self_fail_on_can_parse_next(instring, tmp_loc):
                    break

            if self_ignore_expr_try_parse is not None:
                while 1:
                    try:
                        tmp_loc = self_ignore_expr_try_parse(instring, tmp_loc)
                    except ParseBaseException:
                        break

            try:
                expr_parse(instring, tmp_loc, doActions=False, callPreParse=False)
            except (ParseException, IndexError):
                tmp_loc += 1
            else:
                break

        else:
            raise ParseException(instring, loc, self.errmsg, self)

        loc = tmp_loc
        skiptext = instring[start_loc:loc]
        skipresult = ParseResults(skiptext)

        if self.include_match:
            loc, mat = expr_parse(instring, loc, doActions, callPreParse=False)
            skipresult += mat

        return loc, skipresult


class Forward(ParseElementEnhance):
    """表达式的前向声明-用于递归文法，如代数中缀表示法。"""

    def __init__(self, other=None):
        super(Forward, self).__init__(other, savelist=False)

    def __lshift__(self, other):
        if isinstance(other, basestring):
            other = self._literalStringClass(other)
        self.str_repr = None
        self.expr = other
        self.may_return_empty = self.expr.may_return_empty
        self.may_index_error = self.expr.may_index_error
        self.skip_whitespace = self.expr.skip_whitespace
        self.save_as_list = self.expr.save_as_list
        self.setWhitespaceChars(self.expr.white_chars)
        self.ignore_exprs.extend(self.expr.ignore_exprs)
        return self

    def __ilshift__(self, other):
        return self << other

    def leaveWhitespace(self):
        self.skip_whitespace = False
        return self

    def validate(self, validateTrace=None):
        if validateTrace is None:
            validate_trace = []
        else:
            validate_trace = validateTrace

        if self not in validate_trace:
            tmp = validate_trace[:] + [self]
            if self.expr is not None:
                self.expr.validate(tmp)
        self.checkRecursion([])

    def streamline(self):
        if self.streamlined:
            return self

        self.streamlined = True
        if self.expr is not None:
            self.expr.streamline()
        return self

    def __str__(self):
        if hasattr(self, "name"):
            return self.name
        if self.str_repr is not None:
            return self.str_repr

        self.str_repr = ": ..."

        rest_str = '...'
        try:
            if self.expr is not None:
                rest_str = _ustr(self.expr)[:1000]
            else:
                rest_str = "None"
        finally:
            self.str_repr = self.__class__.__name__ + ": " + rest_str
        return self.str_repr

    def copy(self):
        if self.expr is not None:
            return super(Forward, self).copy()

        ret = Forward()
        ret <<= self
        return ret

    def _setResultsName(self, name, listAllMatches=False):
        return super(Forward, self)._setResultsName(name, listAllMatches)


class TokenConverter(ParseElementEnhance):
    """:class:`ParseExpression`的抽象子类，用于解析结果转换。
    """

    def __init__(self, expr, savelist=False):
        super(TokenConverter, self).__init__(expr)  # , savelist)
        self.save_as_list = False


class Combine(TokenConverter):
    """将给定的匹配范式组合拼接为单字符串。

    示例::

        real = Word(nums) + '.' + Word(nums)
        print(real.parseString('3.1416')) # -> ['3', '.', '1416']
        # 等同于
        print(real.parseString('3. 1416')) # -> ['3', '.', '1416']

        real = Combine(Word(nums) + '.' + Word(nums))
        print(real.parseString('3.1416')) # -> ['3.1416']
        # 内部存在空格时不匹配
        print(real.parseString('3. 1416')) # -> Exception: Expected W:(0123...)
    """

    def __init__(self, expr, joinString="", adjacent=True):
        super(Combine, self).__init__(expr)
        self.adjacent = adjacent
        if adjacent:
            self.leaveWhitespace()
        self.call_preparse = True
        self.join_string = joinString
        self.skip_whitespace = True

    def ignore(self, other):
        if not self.adjacent:
            super(Combine, self).ignore(other)
        else:
            ParserElement.ignore(self, other)
        return self

    def postParse(self, instring, loc, tokenlist):
        ret_tokens = tokenlist.copy()
        del ret_tokens[:]
        ret_tokens += ParseResults(["".join(tokenlist._asStringList(self.join_string))], modal=self.modal_results)

        if self.results_name and ret_tokens.haskeys():
            return [ret_tokens]
        else:
            return ret_tokens


class Group(TokenConverter):
    """将给定的匹配范式转换为列表，适合与 :class:`ZeroOrMore` 和 :class:`OneOrMore` 搭配使用。"""

    def __init__(self, expr):
        super(Group, self).__init__(expr)
        self.save_as_list = True

    def postParse(self, instring, loc, tokenlist):
        return [tokenlist]


class Dict(TokenConverter):
    """以列表或字典形式返回重复表达式"""

    def __init__(self, expr):
        super(Dict, self).__init__(expr)
        self.save_as_list = True

    def postParse(self, instring, loc, tokenlist):
        for i, token in enumerate(tokenlist):
            if len(token) == 0:
                continue

            key = token[0]
            if isinstance(key, int):
                key = _ustr(token[0]).strip()
            if len(token) == 1:
                tokenlist[key] = _ParseResultsWithOffset("", i)
            elif len(token) == 2 and not isinstance(token[1], ParseResults):
                tokenlist[key] = _ParseResultsWithOffset(token[1], i)
            else:
                dictvalue = token.copy()
                del dictvalue[0]
                if len(dictvalue) != 1 or (isinstance(dictvalue, ParseResults) and dictvalue.haskeys()):
                    tokenlist[key] = _ParseResultsWithOffset(dictvalue, i)
                else:
                    tokenlist[key] = _ParseResultsWithOffset(dictvalue[0], i)

        return [tokenlist] if self.results_name else tokenlist


class Suppress(TokenConverter):
    """忽略已解析的表达式结果。

    示例::

        source = "a, b, c,d"
        wd = Word(alphas)
        wd_list1 = wd + ZeroOrMore(',' + wd)
        print(wd_list1.parseString(source))

        # 分隔符用于分割定位，但是结果中可以忽略分隔符
        wd_list2 = wd + ZeroOrMore(Suppress(',') + wd)
        print(wd_list2.parseString(source))

    结果::

        ['a', ',', 'b', ',', 'c', ',', 'd']
        ['a', 'b', 'c', 'd']
    """

    def postParse(self, instring, loc, tokenlist):
        return []

    def suppress(self):
        return self
