#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.

import logging
import os
import pwd
import shlex
import subprocess
import sys
import re
import signal
import yaml


# Configure logging
logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')

# Specify configuration file path
CONFIG_FILE = "/OSM/conf/app_data/ftds/product.ini"


def parse_cpu_list(cpu_list_str, list_name):
    """
    Parse a CPU list string, supporting single CPU values and ranges, 
    and convert it into an expanded list of integers.
    
    Args:
        cpu_list_str (str): CPU list string, e.g., "10,11,12,17" or "10-12,17"
        list_name (str): Name of the list for logging purposes

    Returns:
        list: A list containing all CPU IDs
        int: Status code (0 for success, 1 for error)
    """

    cpu_list_expanded = []
    for part in cpu_list_str.split(','):
        # Check for range format (only allows numbers and a single hyphen)
        if '-' in part:
            if not re.match(r'^\d+-\d+$', part):
                logging.error("Error parsing %s: invalid range format '%s'", list_name, part)
                return [], 1
            # Handle range, e.g., 10-12
            try:
                start, end = map(int, part.split('-'))
                cpu_list_expanded.extend(range(start, end + 1))
            except ValueError as e:
                logging.error("Error parsing %s range %s: %s", list_name, part, str(e))
                return [], 1
        else:
            # Check for single integer format
            if not part.isdigit():
                logging.error("Error parsing %s: invalid CPU value '%s'", list_name, part)
                return [], 1
            # Single CPU, e.g., 17
            try:
                cpu_list_expanded.append(int(part))
            except ValueError as e:
                logging.error("Error parsing %s CPU value %s: %s", list_name, part, str(e))
                return [], 1
    return cpu_list_expanded, 0


def get_fault_core_cpu_list():
    """
    Retrieve faultCore CPU list from the configuration file.
    """

    if not os.path.isfile(CONFIG_FILE):
        logging.info("Configuration file %s does not exist", CONFIG_FILE)
        return [], 1

    try:
        with open(CONFIG_FILE, 'r') as f:
            content = f.read()
    except Exception as e:
        logging.error("Error reading configuration file %s: %s", CONFIG_FILE, str(e))
        return [], 1

    match = re.search(r'^faultCore=(.*)', content, re.MULTILINE)
    if match:
        config_cpu_list_str = match.group(1).replace(' ', '')  # Remove whitespace
        return parse_cpu_list(config_cpu_list_str, "faultCore")
    else:
        logging.info("No faultCore entry found in %s, assuming no CPU isolation info", CONFIG_FILE)
        return [], 0


def timeout_handler(signum, frame):
    raise Exception("Timed Out")


def get_offline_cpu_list():
    """
    Retrieve system off-line CPU list using lscpu command.
    """

    try:
        # Set the signal handler for the timeout
        signal.signal(signal.SIGALRM, timeout_handler)
        signal.alarm(10)  # Set the timeout to 10 seconds

        # Call the lscpu command using its absolute path
        process = subprocess.Popen(['/usr/bin/lscpu'], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        
        # Manually handle output
        lscpu_output = process.stdout.read()  # Read standard output
        error = process.stderr.read()  # Read standard error

        # Wait for the process to complete
        process.wait()
        
        lscpu_output = lscpu_output.decode('utf-8')  # Decode the output in Python 2.x

        # Disable the alarm after successful completion
        signal.alarm(0)
    except Exception as e:
        # Handle timeout error manually
        if "Timed Out" in str(e):
            process.kill()  # Terminate the process if it exceeds the timeout
            logging.error("lscpu command timed out and was terminated")
            return [], 1
        # Check for the exception when the lscpu command is not found
        elif "No such file or directory" in str(e) or "command not found" in str(e):
            logging.info("Lscpu command: %s", str(e))
            return [], 0
        else:
            logging.error("Unexpected error executing lscpu command: %s", str(e))
            return [], 1

    match = re.search(r'Off-line CPU\(s\) list:\s*(.*)', lscpu_output)
    if match:
        system_cpu_list_str = match.group(1).replace(' ', '')  # Remove whitespace
        return parse_cpu_list(system_cpu_list_str, "off-line CPU")
    else:
        logging.info("No off-line CPU list found, assuming no CPU isolation info")
        return [], 0


def compare_cpu_lists(config_cpu_list_expanded, system_cpu_list_expanded):
    """
    Compare the two CPU lists and return the appropriate status.
    """

    if not config_cpu_list_expanded and not system_cpu_list_expanded:
        return 0
    elif sorted(config_cpu_list_expanded) == sorted(system_cpu_list_expanded):
        return 0
    else:
        return 1


def run():

    # Get faultCore CPU list
    config_cpu_list_expanded, status = get_fault_core_cpu_list()
    if status != 0:
        return 1

    # Get system off-line CPU list
    system_cpu_list_expanded, status = get_offline_cpu_list()
    if status != 0:
        return 1

    # Compare CPU lists
    return compare_cpu_lists(config_cpu_list_expanded, system_cpu_list_expanded)


if __name__ == "__main__":
    # Run the main function and exit with the returned code
    try:
        exit_code = run()
        if exit_code == 0:
            print("True")
            sys.exit(0)
        else:
            print("False")
            sys.exit(0)
    except Exception as main_except:
        logging.info("cpu_online_fault: exception %s", str(main_except))
        print("False")
        sys.exit(0)
