#!/usr/bin/python
# -*- coding: utf-8 -*-
import os
import re
import sys
import logging
from time import sleep

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(filename)s, %(lineno)d]: %(message)s',
                    datefmt='%Y-%m-%d %H:%M:%S')


def is_ip_encl():
    info = execute_cmd('cat /proc/osp/bsp | grep "Product Name is"')
    return 'EXP' in info


def get_user_name():
    for _ in range(3):
        admin_user = 'admin'
        user_exist_cmd = 'id -u {} > /dev/null 2>&1'.format(admin_user)
        ctrl_user_cmd = r"cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:root:'"
        ip_user_cmd = r"cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:SuperAdministrator:'"
        user_cmd = ip_user_cmd if is_ip_encl() else ctrl_user_cmd
        if os.system(user_exist_cmd) == 0:
            return admin_user
        else:
            output = execute_cmd(user_cmd)
            if output:
                return output.strip().split(':')[0]
        sleep(2)
    logging.warning('DMI_ALARM_CHECK:get user failed')
    return ''


def execute_cli_cmd(cmd):
    user_name = get_user_name()
    if not user_name:
        print('True')
        exit()
    cli_format = "/ISM/cli/ismcli -u {} -c '{}'"
    return execute_cmd(cli_format.format(user_name, cmd))


def execute_cmd(cmd):
    cmd_fd = os.popen(cmd)
    data = cmd_fd.read().strip()
    cmd_fd.close()
    return data


def execute_diagnose21_cmd(cmd):
    cli_format = 'diagsh --attach=*_21 --cmd="{}"'
    return execute_cmd(cli_format.format(cmd))


def is_mask_alarm():
    cmd = 'show alarm_mask'
    res = execute_cli_cmd(cmd)
    if re.search(r'0xF00C9015D\s+On', res) and re.search(r'0xF00C9015E\s+On', res):
        logging.info('DMI_ALARM_CHECK: alarm is mask, check success')
        return True
    return False


def is_have_ip_encl():
    cmd = 'show enclosure'
    res = execute_cli_cmd(cmd)
    for line in res.splitlines():
        if 'smart' in line.lower() and 'offline' not in line.lower():
            return True
    return False


def is_routine_thread_status_ok():
    cmd = 'dpl routine show'
    info = ''
    for _ in range(3):
        info = execute_diagnose21_cmd(cmd)
        if 'RoutineThread' in info:
            break
        sleep(2)
    if 'RoutineThread' not in info:
        logging.warning('DMI_ALARM_CHECK: query thread status failed, info %s', info)
        return True
    flag = False
    for line in info.splitlines():
        if line.startswith('RoutineThread'):
            flag = True
            continue
        if not flag:
            continue
        match = re.search(r'\S+\s+(-?\d+)\s+\d+\s+(\d+)', line)
        if not match:
            break
        period, sleep_time = int(match.group(1)), int(match.group(2))
        if period == -1 and sleep_time > 1000:  # 1s
            logging.warning('DMI_ALARM_CHECK: thread check failed,info %s', line)
            return False
        if period != -1 and sleep_time > (60 * 60 * 1000):  # 1h
            logging.warning('DMI_ALARM_CHECK: thread check failed,info %s', line)
            return False
    logging.info('DMI_ALARM_CHECK: thread check success')
    return True


def get_product_version():
    cmd = 'show system general'
    info = execute_cli_cmd(cmd)
    match = re.search(r'Product Version\s+:\s+(\S+)', info)
    if match:
        return match.group(1)
    logging.warning('DMI_ALARM_CHECK:get product version failed,info %s', info)
    return ''


def is_product_version(product_version):
    return product_version in version


def is_install_specified_patch(num):
    match = re.search(r'SPH(\d+)', version)
    if match:
        patch_version = int(match.group(1))
        if patch_version >= num:
            return True
    return False


def main():
    patch_map = {'6.1.3': 10, '6.1.5': 3, '1.2.0': 3}
    logging.info('DMI_ALARM_CHECK: cur version is %s', version)
    try:
        for product_version, patch_version in patch_map.items():
            if is_product_version(product_version) and is_install_specified_patch(patch_version):
                logging.info('DMI_ALARM_CHECK: patch version check success')
                print('True')
                return 0
        if is_mask_alarm():
            print('True')
        elif is_have_ip_encl():
            logging.warning('DMI_ALARM_CHECK: cur env have ip encl, check fail')
            print('False')
        elif is_routine_thread_status_ok():
            print('True')
        else:
            print('False')
        return 0
    except Exception as exception:
        logging.exception("DMI_ALARM_CHECK:check failed: %s", exception)
        print("False")
        return 1


if __name__ == '__main__':
    version = get_product_version()
    if not version:
        print('True')
        exit()
    sys.exit(main())
