#!/usr/bin/python
# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import logging
import sys
import subprocess
import shlex
import re
import os
import time
import yaml
 
# 定义返回值
RETURN_OK = 0
RETURN_ERROR = 1
# 版本信息
IMAGE_DISK = '/startup_disk/image'
PATCH_MANIFEST = 'patch.yml'
# 当前版本信息
CUR_MANIFEST_PATH = '/OSM/conf/manifest.yml'
# 当前内核版本
CUR_KERNEL_VERSION = '/OSM/script/upgrade.sh'
# 问题涉及的版本范围
VERSION_MIN_NO = "7600511219"
VERSION_MAX_NO = "7600516210"
# 最大泄漏内存512MB和平均泄漏内存10MB/天
MAX_LEAK_MEM = 512 * 1024 / 4 * 507
AVERAGE_LEAK_MEM = 10 * 1024 / 4 * 507
# 初始化系统日志
logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')
 
 
# 直接执行命令
def execute_cmd(cmd):
    cmd_fd = os.popen(cmd)
    data = cmd_fd.read().strip()
    cmd_fd.close()
    return data
 
 
# 获得配置文件中的系统版本
def get_sys_version(manifest_path):
    if not os.path.exists(manifest_path):
        logging.warning("sys file(%s) does not exist.", manifest_path)
        return "", ""
    with open(manifest_path) as file_handle:
        cfg_yml = yaml.safe_load(file_handle)
        try:
            target_version = str(cfg_yml.get("SYS")["Version"])
            target_spc_version = str(cfg_yml.get("SYS")["SpcVersion"])
            return target_version, target_spc_version
        except Exception:
            logging.warning("CHECK_FUNC: Failed to get version.")
            return "", ""
 
 
# 从补丁配置文件中获得补丁版本
def get_patch_version(conf_file):
    if not os.path.exists(conf_file):
        logging.warning("patch conf file(%s) does not exist.", conf_file)
        return ""
    file_handle = open(conf_file)
    patch_conf = yaml.safe_load(file_handle)
    file_handle.close()
    ver_conf = patch_conf.get("PatchConf", None)
    if not ver_conf:
        logging.warning("Patch Conf file does not exist.")
        return ""
    return ver_conf.get('patch_version', "")
 
 
# 检查内核版本，仅A5开始会出现内存泄漏
def check_kernel_version():
    cmd = "sh {} kernel showversion"
    kernel_ver = execute_cmd(cmd.format(CUR_KERNEL_VERSION))
    logging.info("%s", kernel_ver)
    digit_kernel_ver = re.findall(r"kernel version:\s+\d+.\d+.(\d+).\d+", kernel_ver, re.S)
    if len(digit_kernel_ver) > 0:
        if digit_kernel_ver[0] < "5":
            return True
    return False
 
 
def check_software_and_patch_version(manifest_path):
    upgrade_patch_cfg = {
        '7600511219': 'SPH36',   # 6.1.5版本
        '7600513292': 'SPH16',   # 6.1.6版本
        '7600516210': 'SPH6'    # 6.1.7版本
    }
    digit_ver, sys_ver = get_sys_version(manifest_path)
    if not digit_ver:
        logging.warning("Get system version failed.")
        return False
    # 此版本问题已解决或无问题，检查通过
    if digit_ver > VERSION_MAX_NO or digit_ver < VERSION_MIN_NO:
        return True
 
    # 获得该系统版本的补丁版本目录
    patch_path = os.path.join(IMAGE_DISK, str(digit_ver), 'hotpatch', 'patch_cur', PATCH_MANIFEST)
    # 获得补丁版本
    patch_ver = get_patch_version(patch_path)
 
    if not patch_ver:
        patch_digit_ver = 0
    else:
        patch_digit_ver = int(re.split('(\\d+)', patch_ver)[1])
    logging.info("CHECK_VERSION: sys_ver(%s), patch_ver(%s)", sys_ver, patch_ver)
 
    # 检查补丁版本问题是否解决
    if digit_ver in upgrade_patch_cfg:
        require_patch_digit_ver = int(re.split('(\\d+)', upgrade_patch_cfg[digit_ver])[1])
        if patch_digit_ver >= require_patch_digit_ver:
            return True
        else:
            return False
    else:
        logging.error("Digital version(%s) is invalid.", digit_ver)
        return False
 
 
def check_current_and_target_version():
    logging.info("Start check current version.")
    ret = check_kernel_version()
    if ret:
        return True
    ret = check_software_and_patch_version(CUR_MANIFEST_PATH)
    if ret:
        return True
    return False
 
 
# 检查泄漏的内存
def check_leak_memory():
    cmd = "hinicadm3 info |grep TIOE"
    card_info = str(execute_cmd(cmd))
    card_list = re.findall(r"hinic(\d+).*?TIOE", card_info)
    card_len = len(card_list)
    if card_len == 0:
        logging.info("The 1823 DTOE card does not exist.")
        return True
    total_container_num = 0
 
    # 获取系统运行时间
    cmd = "uptime |grep -o \'up .* days\'|awk '{{print $2}}'"
    uptime = execute_cmd(cmd)
    if uptime:
        uptime = int(uptime)
    else:
        uptime = 1
 
    # 检查所有卡的泄漏内存
    for i in range(0, card_len):
        card = card_list[i]
        cmd = "hinicadm3 counter -i hinic{} -t 1 -x toe_s32 |grep toe_srq_alloc_container_num |awk '{{print $2}}'"
        container_num = execute_cmd(cmd.format(card))
        if container_num:
            container_num = int(container_num)
        else:
            container_num = 0
        logging.info("Size of the leaked memory of the hinic%s: %s", card, container_num)
        total_container_num += container_num
        if total_container_num >= MAX_LEAK_MEM or total_container_num / uptime >= AVERAGE_LEAK_MEM:
            return False
 
    return True
 
 
def main():
    try:
        logging.info("Start check dtoe leaked memory.")
        # 检查版本
        check_result = check_current_and_target_version()
        if check_result:
            logging.info("Check version successfully.")
            print("True")
            return RETURN_OK
        check_result = check_leak_memory()
        if check_result:
            logging.info("No leaked memory exists.")
            print("True")
            return RETURN_OK
        
        logging.error("There are a lot of memory leaks")
        print("False")
        return RETURN_OK
 
    except Exception as err:
        logging.exception("check_leak_mem_for_dtoe: %s", err)
        print("False")
        return RETURN_ERROR
 
 
if __name__ == '__main__':
    sys.exit(main())
