#!/usr/bin/python
# coding=utf-8
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import getopt
import logging
import shlex
import subprocess
import re
import sys
import os
from comm_check_func import get_upg_sys_version, get_patch_version
import io

from check_item.om_check_audit_log import get_product_model

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')

VERSION_DORADO_V6_1_5 = "7600511219"
VERSION_DORADO_V6_1_6 = "7600513292"
VERSION_DORADO_V6_1_7 = "7600516210"

VERSION_OCEAN_PROTECT_1_2_0 = "7600509200"
VERSION_OCEAN_PROTECT_1_3_0 = "7600513292"
VERSION_OCEAN_PROTECT_1_5_0 = "7600516210"

VERSION_OCEAN_DISK_1_2_0 = "7610106170"

VERSION_2910_6_1_5 = "7600511102"
VERSION_2910_6_1_6 = "7600513180"
VERSION_2910_6_1_7 = "7600516210"

PRODUCT_MODEL_OCEAN_PROTECT = "OceanProtect"
PRODUCT_MODEL_OCEAN_DISK = "OceanDisk"
PRODUCT_MODEL_OCEAN_STOR_2910 = "OceanStor 2910"

IMAGE_DISK = '/startup_disk/image'
PATCH_MANIFEST = 'patch.yml'

SPH_30 = 'SPH30'
SPH_7 = 'SPH7'


def is_file_open(file):
    try:
        with io.open(file, 'r') as f:
            f.read(1)
    except Exception as exp:
        logging.info("open file(%s) failed", file)
        return False
    return True


# 检查到非法文件，会抛出异常
def is_files_check_ok():
    check_file_list = [
        '/etc/passwd',
        '/etc/shadow',
        '/etc/group',
        '/etc/passwd_default',
        '/etc/shadow_default',
        '/etc/group_default',
        '/startup_disk/conf/conf_local/passwd',
        '/startup_disk/conf/conf_local/shadow',
        '/startup_disk/conf/conf_local/group',
        '/startup_disk/conf/conf_local/passwd.bak',
        '/startup_disk/conf/conf_local/shadow.bak',
        '/startup_disk/conf/conf_local/group.bak'
    ]
    for file in check_file_list:
        if os.path.exists(file) and not is_file_open(file):
            return False
    return True


def check_software_version():
    res, upg_sys_ver = get_upg_sys_version()
    if not res:
        logging.warning("CHECK_NOF_UPGRADE_VERSION: get system version failed.")
        return False
    upg_patch_path = os.path.join(IMAGE_DISK, str(upg_sys_ver), 'hotpatch', 'patch_cur', PATCH_MANIFEST)
    upg_patch_ver = get_patch_version(upg_patch_path)
    if not upg_patch_ver:
        patch_digit_ver = 0
    else:
        patch_digit_ver = int(re.split('(\\d+)', upg_patch_ver)[1])
    product_model = get_product_model()
    logging.info("CHECK_NOF_UPGRADE_VERSION: upg_sys_ver(%s), upg_patch_ver(%s), product model(%s)",
                 upg_sys_ver, upg_patch_ver, product_model)

    if PRODUCT_MODEL_OCEAN_PROTECT in product_model:
        upgrade_patch_list = {
            VERSION_OCEAN_PROTECT_1_2_0: SPH_30,  # 1.2.0版本
            VERSION_OCEAN_PROTECT_1_3_0: SPH_7,  # 1.3.0版本
        }
        return check_version(upg_sys_ver, patch_digit_ver, VERSION_OCEAN_PROTECT_1_5_0, upgrade_patch_list)
    elif PRODUCT_MODEL_OCEAN_DISK in product_model:
        return check_version(upg_sys_ver, patch_digit_ver, VERSION_OCEAN_DISK_1_2_0, {})
    elif PRODUCT_MODEL_OCEAN_STOR_2910 in product_model:
        upgrade_patch_list = {
            VERSION_2910_6_1_5: SPH_30,  # 6.1.5版本
            VERSION_2910_6_1_6: SPH_7,  # 6.1.6版本
        }
        return check_version(upg_sys_ver, patch_digit_ver, VERSION_2910_6_1_7, upgrade_patch_list)
    else:
        upgrade_patch_list = {
            VERSION_DORADO_V6_1_5: SPH_30,  # 6.1.5版本
            VERSION_DORADO_V6_1_6: SPH_7,  # 6.1.6版本
        }
        return check_version(upg_sys_ver, patch_digit_ver, VERSION_DORADO_V6_1_7, upgrade_patch_list)


def check_version(upg_sys_ver, patch_digit_ver, skip_version, upgrade_patch_list):
    """
    upg_sys_ver: 目标升级版本
    patch_digit_ver: 目标升级补丁版本
    skip_version: 适配版本
    upgrade_patch_list: 适配补丁版本列表

    return: 是否通过版本检查
    """
    if upg_sys_ver > skip_version:
        return True
    elif upg_sys_ver in upgrade_patch_list:
        require_patch_digit_ver = int(re.split('(\\d+)', upgrade_patch_list[upg_sys_ver])[1])
        return patch_digit_ver >= require_patch_digit_ver
    return False


def main(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        # -c:升级前检查; -e:升级前评估;升级前评估不判断是否大于等于617版本
        opts, _ = getopt.getopt(argv[1:], "ce", ["check_nof_software_version", "check_nof_upgrade_version"])
        for opt, _ in opts:
            if opt in ['-c', '--check_nof_software_version'] and check_software_version():
                print("True")
                return 0
        if not is_files_check_ok():
            print("False")
            return 0
        # 检查完成，未抛出异常或超时，返回检查通过，print为True表示通过，False表示不通过
        print("True")
        return 0
    except Exception as exp:
        print("False")
        return 0


if __name__ == '__main__':
    sys.exit(main())
