#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.
"""
# @brief   : 2023/04/28
# @File    : check_smart_io_eth_shard_card.py
# @Software: Software management infrastructure
"""
import getopt
import os
import re
import sys
from infra.debug.log import swm_logger as log
from check_item.check_util import util


class CheckSmartIOEthShardCard(object):
    """
    背景： 高端双控的环境，ETH共享卡在C/D控的槽位上，由于ETH共享卡属于单归属，且C/D控没有控制器所以未做升级，导致C/D控上的ETH共享卡未升级。
    检查方式：
        1、判断是高端双控环境
        2、读取ETH共享卡所在的槽位是否在 L7-L13 或 H7-L13
        3、如果存在则检查项不通过，否则通过。
    """
    LOG_PREFIX = "CheckSmartIOEthShardCard"
    SLOT_RANGE = (7, 13)  # L7-13,H7-H13 为C/D控的槽位号
    IS_HVS_CMD = [["grep", "^IsHvs=", "/OSM/conf/devm/devm.ini"], ["sed", "s/^IsHvs=//"]]
    CTRL_TOTAL_NUM_CMD = [["grep", "^CtrlTotal=", "/OSM/conf/bay_config.ini"], ["sed", "s/^CtrlTotal=//"]]
    SHOW_ETH_INFO_CMD = "show port general logic_type=Host_Port physical_type=ETH"
    _CMD_PREFIX = ['python'] if os.path.exists('/.dockerenv') or os.path.exists('/crash/permitdir') else []
    _CLI_CMD_LIST_TEMPLATE = _CMD_PREFIX + ["/ISM/cli/ismcli", "-u", "admin", "-c", SHOW_ETH_INFO_CMD]
    HIGH_DEVICE = [
        'PANGEA_V6R1C00_HI1620_4U4C', 'PANGEA_V6R3C10_HI1620_4U4C', 'PANGEA_V6R3C00_HI1620_4U4C'
    ]

    @classmethod
    def is_high_env(cls):
        with open('/OSM/script/proc_osp_bsp.info', 'rb') as f:
            bsp_info = f.read()
        board_type = re.search(b'Product Name is (\S+)', bsp_info).groups()[0].decode()
        return board_type in cls.HIGH_DEVICE


    @classmethod
    def _check_ctrl_total_num(cls):
        ret, output = util.call_system_cmd_list(cls.CTRL_TOTAL_NUM_CMD)
        if ret:
            log.warning("%s: Execute cmd(%s) failed, ret(%s), output(%s).",
                        cls.LOG_PREFIX, cls.CTRL_TOTAL_NUM_CMD, ret, output)
            return False
        log.info("%s: Execute cmd(%s) success, output(%s).", cls.LOG_PREFIX, cls.CTRL_TOTAL_NUM_CMD, output)
        return int(output.strip("\n")) == 2  # 双控

    @classmethod
    def _check_smart_io_eth_shard_card(cls):
        ret, output = util.call_system_cmd(cls._CLI_CMD_LIST_TEMPLATE)
        if ret:
            log.warning("%s: Execute cmd(%s) failed, ret(%s), output(%s).",
                        cls.LOG_PREFIX, cls._CLI_CMD_LIST_TEMPLATE, ret, output)
            return False
        log.info("%s: Execute cmd(%s) success.", cls.LOG_PREFIX, cls._CLI_CMD_LIST_TEMPLATE)
        eth_slots = []
        for line in output.split("\n"):
            slot_info = line.split()
            if not slot_info:
                continue
            match = re.search(r"\.(L|H)(\d+)", slot_info[0])
            if match:
                slot_num = int(match.group(2))
                if cls.SLOT_RANGE[0] <= slot_num <= cls.SLOT_RANGE[1]:
                    eth_slots.append(slot_info[0])
        if not eth_slots:
            log.info("%s: Eth shard card not in C/D controller slot.", cls.LOG_PREFIX)
            return True
        log.warning("%s: Eth shard card(%s) in C/D controller slot.", cls.LOG_PREFIX, str(eth_slots))
        return False

    @classmethod
    def check(cls):
        if not os.path.exists('/OSM/script/proc_osp_bsp.info'):
            log.error("The file proc_osp_bsp.info not exists")
            return False
        if cls.is_high_env() and cls._check_ctrl_total_num():
            return cls._check_smart_io_eth_shard_card()
        return True


def show_help():
    usage = (
        "This is used for upgrade pre-check.\n"
        "  -h, --help show        the cmd help info.\n"
        "  -c, --check_smart_io_eth_shard_card   check smart io eth shard card in C/D controller.\n"
    ).format(os.path.basename(__file__))
    print(usage)


def check_smart_io_eth_shard_card():
    print('True' if CheckSmartIOEthShardCard.check() else 'False')


def main(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        opts, args = getopt.getopt(argv[1:], "hc", ["help", "check_smart_io_eth_shard_card"])
        for o, _ in opts:
            if o in ("-h", "--help"):
                show_help()
                return 0

            if o in ['-c', '--check_smart_io_eth_shard_card']:
                check_smart_io_eth_shard_card()
                return 0

    except Exception as e:
        print("exception")
        print("exception")
        print("exception")
        log.exception(e)
        return 1
    return 0


if __name__ == "__main__":
    sys.exit(main())
