#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.
"""
# @brief   : 2021/12/20
# @File    : check_sysdisk.py
# @Software: Software management infrastructure
"""
import getopt
import os
import platform
import random
import re
import shlex
import sys
import time

from check_item.check_util import util
from check_item.check_util.util import verify_by_sha256sum, check_mem_is_enough, get_pkg_version, get_patch_version
from infra.debug.log import swm_logger as log
from infra.util import shell


class SysDiskCheck(object):
    RAID_CMD = "mdadm -D /dev/md1"
    MOUNT_CMD = "mount -l | grep {path:s}"
    CHECK_PATH = ["/startup_disk/image", "/OSM/coffer_data"]
    CHECK_DIR = "/startup_disk/image"
    CUR_PKG_MANIFEST = "/startup_disk/image/pkg_cur/manifest.yml"
    CUR_PATCH_YML = "/startup_disk/image/pkg_cur/../hotpatch/patch_cur/patch.yml"
    PROC_OSP_BSP = '/proc/osp/bsp'
    BOARD_TYPE_FILE = '/OSM/script/proc_osp_bsp.info'
    BOOT_DIR = os.path.join(CHECK_DIR, "boot")
    ARM = (platform.machine() == 'aarch64')
    SIMU = (os.path.exists('/.dockerenv') or os.path.exists('/crash/permitdir')
            or os.path.exists('/.swm_llt_stub'))
    NO_NEED_CHECK_MIN_PATCH_DICT = {
        7600506206: 60,  # 6.1.2SPH60
        7600509200: 30,  # 6.1.3SPH30
        7600511219: 30,  # 6.1.5SPH30
    }

    @classmethod
    def _check_raid(cls):
        # exec cmd
        if os.path.islink("/dev/md1"):
            log.info("SysDiskCheck: Execute cmd success, /dev/md1 is Symbolic link.")
            return True
        ret, outputs = shell.call_system_cmd(cls.RAID_CMD)
        if ret != 0:
            log.warning("SysDiskCheck: Execute cmd failed, ret(%s), output(%s).", ret, outputs)
            return False
        outputs = outputs.split("\n")
        # check raid
        state_line = None
        for line in outputs:
            line = line.strip()
            if "State :" in line:
                state_line = line
                break
        if not state_line:
            log.error("SysDiskCheck: The echo is error, output(%s).", outputs)
            return False

        if "degraded" in state_line:
            log.error("SysDiskCheck: The raid state is wrong, state(%s).", state_line)
            return False

        log.info("SysDiskCheck: Check raid success, the disks are active, "
                 "info(%s).", state_line)
        return True

    @classmethod
    def _check_mount(cls):
        for path in cls.CHECK_PATH:
            mount_cmd = cls.MOUNT_CMD.format(path=path)
            ret, outputs = shell.call_system_cmd(mount_cmd)
            if ret != 0:
                log.error("SysDiskCheck: Check mount failed, ret(%s), path(%s), outputs(%s).", ret, path, outputs)
                return False
        log.info("SysDiskCheck: Check mount success.")
        return True

    @classmethod
    @shell.remount_image
    def _check_write(cls):
        for path in cls.CHECK_PATH:
            random_str = str(random.Random(os.urandom(8)).randint(0, 999999))
            write_test_file = "".join(("write_test", str(time.time()), random_str))
            test_file = os.path.join(path, write_test_file)
            try:
                if not os.path.exists(path):
                    log.error("SysDiskCheck: The dir is not existed, %s.", path)
                    return False

                if os.path.exists(test_file):
                    os.remove(test_file)
                with open(test_file, "w") as fp:
                    fp.write("SysDiskCheck: WriteTest.")
            except Exception:
                log.exception("SysDiskCheck: Unknown error.")
                return False
            finally:
                if os.path.exists(test_file):
                    os.remove(test_file)
        log.info("SysDiskCheck: Check write success.")
        return True

    @classmethod
    def check(cls):
        return (cls._check_raid() and cls._check_mount() and cls._check_write() and cls.check_hidden_partition() and
                cls._check_read())

    @classmethod
    def need_check_read(cls):
        """
        # 补丁版本如果是已经合入了整改问题的版本，就不执行检查了
        6.1.2SPH60/6.1.5SPH30/6.1.3SPH30
        """
        sys_version, product = get_pkg_version(cls.CUR_PKG_MANIFEST)
        patch_version = get_patch_version(cls.CUR_PATCH_YML)
        if not sys_version:
            return True

        if product and product.startswith('Micro'):
            # 微存储版本号和其他系列不一样
            no_need_check_ver = 7610107138  # 1.3.RC1
        else:
            no_need_check_ver = 7600518238

        if int(sys_version) >= no_need_check_ver:
            return False

        if not patch_version:
            patch_version = 0
        else:
            patch_version = int(patch_version.strip('SPH'))
        if int(sys_version) in cls.NO_NEED_CHECK_MIN_PATCH_DICT and \
                patch_version >= cls.NO_NEED_CHECK_MIN_PATCH_DICT.get(int(sys_version)):
            log.info('SysDiskCheck: cur base:%s, cur patch: %s, no need to check image file.',
                     sys_version, patch_version)
            return False
        return True

    @classmethod
    def _check_read(cls):
        if not check_mem_is_enough(500):
            return True

        if not cls.need_check_read():
            return True
        log.info('SysDiskCheck: Start check image file.')
        check_result = True
        for root, _, files in os.walk(cls.CHECK_DIR):
            filepaths = [os.path.join(root, filename) for filename in files]
            check_result = check_result if cls._check_can_be_read(filepaths) else False

        log.info('SysDiskCheck: check result is %s', check_result)
        return check_result

    @classmethod
    def _check_can_be_read(cls, filepath_list):
        check_result = True
        for filepath in filepath_list:
            check_result = check_result if cls._check_file_can_be_read(filepath) else False
        return check_result

    @classmethod
    def _check_file_can_be_read(cls, filepath):
        max_read_bytes = 10 * 1024 * 1024
        try:
            with open(filepath, 'rb') as filehandler:
                while filehandler.read(max_read_bytes):
                    pass
        except Exception:
            log.exception("SysDiskCheck: Read %s exception.", filepath)
            return False
        return True

    @classmethod
    def check_hidden_partition(cls):
        if not cls.ARM or cls.SIMU:
            return True
        disk_list = cls.system_disk_list()
        check_pass = False
        for partition in disk_list:
            mount_point = '/home/temp_mount_point'
            mount_cmd = "mkdir -p {1};mount -o ro {0} {1}".format(partition, mount_point)
            end_cmd = "umount -l {0}; rm -d {1}".format(partition, mount_point)
            ret_code, mount_result = cls.check_cmd_output(mount_cmd, ['unknown filesystem type', 'wrong fs type'])
            if ret_code:
                log.warning("Exec cmd(%s) failed ret(%s).", mount_cmd, ret_code)
            if mount_result:
                os.system(end_cmd)
                log.error("Unknown filesystem type, check hidden partition failed, partition:%s", partition)
                continue
            find_cmd = 'find /home/temp_mount_point/'
            ret_code_find, find_result = cls.check_cmd_output(find_cmd, ['output error'])
            if ret_code_find:
                log.warning("Exec cmd(%s) failed ret(%s).", find_cmd, ret_code_find)
            if find_result:
                os.system(end_cmd)
                log.error("Input/output error, check hidden partition failed, partition:%s", partition)
                continue
            grub_path_boot = '/home/temp_mount_point/EFI/BOOT/grub.cfg'
            grub_path_grub2 = '/home/temp_mount_point/EFI/grub2/grub.cfg'
            if os.path.exists(grub_path_boot) and os.path.exists(grub_path_grub2):
                if os.path.getsize(grub_path_boot) == 0 \
                        or os.path.getsize(grub_path_grub2) == 0:
                    log.error("The grub.cfg file size is 0, please check it.")
                    os.system(end_cmd)
                    continue
            os.system(end_cmd)
            check_pass = True
        if disk_list and not check_pass:
            return False
        log.info('SysDiskCheck: check hidden partition success.')
        return True

    @classmethod
    def system_disk_list(cls):
        partition_type = cls.get_partition_type()
        if not partition_type:
            return []
        partition_conf = os.path.join(cls.BOOT_DIR, 'conf/partition_%s.conf' % partition_type)
        with open(partition_conf, "r") as f:
            content = f.read()
        pat = r"clearpart.*drivers=(\w+),?(\w+)?"
        ret = re.search(pat, content)
        if not ret:
            log.error("The disk partition config has no 'clearpart' section.")
            return []
        # 1 is hidden partition
        dev_list = [disk + '1' for disk in ret.groups() if not disk]
        dev_list = [('/dev/' + disk) for disk in dev_list if os.path.exists('/dev/' + disk)]
        return dev_list

    @classmethod
    def get_partition_type(cls):
        board_type = cls.get_board_type()
        cmd_list = [['cat', cls.BOARD_TYPE_FILE], ['grep', "'Disk State is'"]]
        ret, output = util.call_system_cmd_list(cmd_list)
        if ret:
            return board_type
        disk_state = output.strip().split(" ")[-1]
        return board_type + '_' + disk_state

    @classmethod
    def get_board_type(cls):
        if os.path.exists(cls.BOARD_TYPE_FILE):
            cmd_list = [['cat', cls.BOARD_TYPE_FILE], ['grep', "'Product Name'"], ['awk', '{print $4}']]
        else:
            cmd_list = [['cat', cls.PROC_OSP_BSP], ['grep', "'Product Name'"], ['awk', '{print $4}']]
        ret, output = util.call_system_cmd_list(cmd_list)
        return output.rstrip()

    @classmethod
    def check_cmd_output(cls, cmd, check_str_list):
        find_result = False
        ret_code, output = util.call_system_cmd(shlex.split(cmd))
        for check_str in check_str_list:
            if output.find(check_str) >= 0:
                log.info("Find string in output(%s).", check_str)
                find_result = True
        return ret_code, find_result


def show_help():
    usage = (
        "This is used for upgrade pre-check.\n"
        "  -h, --help show        the cmd help info.\n"
        "  -s, --check_sys_disk   check system disk.\n"
    ).format(os.path.basename(__file__))
    print(usage)


def check_sys_disk():
    print('True' if SysDiskCheck.check() else 'False')
    return 0


def main(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        opts, args = getopt.getopt(argv[1:], "hs", ["check_sys_disk"])
        for opt, _ in opts:
            if opt in ("-h", "--help"):
                show_help()
                return 0

            if opt in ['-s', '--check_sys_disk']:
                return check_sys_disk()

    except Exception as e:
        print("exception")
        print("exception")
        print("exception")
        log.exception(e)
        return 1
    return 0


if __name__ == "__main__":
    sys.exit(main())
