#!/usr/bin/python
# -*- coding:utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
import sys
import subprocess
import shlex
import time
import logging
import os

USER_EXIST = r'id -u admin > /dev/null 2>&1'
USER_CMD = r"cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:root:'"
# 定义返回值
RETURN_OK = 0
RETURN_ERROR = 1
OSM_CONF_YAML = '/OSM/conf/manifest.yml'
TIMEOUT = 5 * 60
DORADO_MAP_OP = {'6.1.5': ['1.2.1', '1.2.1SP'], '6.1.5RC1': ['1.2.RC1'], '6.1.5RC2': ['1.2.RC2'],
                 '6.1.5SP': ['1.2.1SP'],
                 '6.1.6': ['1.3.0', '1.3.0SP'], '6.1.6RC1': ['1.3.RC1'], '6.1.6RC2': ['1.3.RC2'],
                 '6.1.6SP': ['1.3.0SP'],
                 '6.1.7': ['1.5.0', '1.5.0SP'], '6.1.7RC1': ['1.5.RC1'], '6.1.7RC2': ['1.5.RC2'],
                 '6.1.7SP': ['1.5.0SP'],
                 '6.1.8': ['1.6.0', '1.6.0SP'], '6.1.8RC1': ['1.6.RC1'], '6.1.8RC2': ['1.6.RC2'],
                 '6.1.8SP': ['1.6.0SP'],
                 '1.7.RC1': ['1.6.RC2', '1.6.0'], 'V700R001C00': ['1.6.RC2', '1.6.0'], '1.7.0': ['1.6.RC2', '1.6.0']}
WHITE_LIST = ['6.1.5.SPH20', '1.6.0', '1.7.RC1.SPH5']
OP_C_VERSION_LIST = ['1.0', '1.1', '1.2.1', '6.1.5']
# 初始化系统日志
logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')


def cli(command):
    sub_cmds = command.split(" | ")
    process = None
    for sub_cmd in sub_cmds:
        if process:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdin=process.stdout, stdout=subprocess.PIPE)
        else:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdout=subprocess.PIPE)
    out = process.communicate(timeout=TIMEOUT)
    return out[0].decode("utf-8")


def get_user_name():
    ret = os.system(USER_EXIST)
    if not ret:
        return 'admin'
    else:
        output = cli(USER_CMD)
        if not output:
            logging.error('Get user failed')
            return ''
        return output.strip().split(':')[0]


def get_helm_chart_version(user_name, spc_version):
    # 1.3后将app name更新为 DataBackup，之前的版本保持原应用名
    app_name = "'OceanProtect-DataProtect'"
    if all(_ not in spc_version for _ in OP_C_VERSION_LIST):
        app_name = "'DataBackup'"
    cmd = r"echo -e 'show helm_chart general' | /ISM/cli/ismcli -u " + user_name + r" | grep -E " + app_name
    cli_result = cli(cmd)
    if not cli_result:
        logging.info("CHECK_VERSION_MATCH: Did not get any valid hel chart version from cli")
        print('False')
        return RETURN_ERROR
    results = cli_result.splitlines(False)
    version_time_list = []
    for line in results:
        time_array = time.strptime(line.split(maxsplit=2)[2].strip(), "%Y-%m-%d %H:%M:%S")
        time_stamp = int(time.mktime(time_array))
        version_time_list.append([line.split(maxsplit=2)[1], time_stamp])
    version_time_list = sorted(version_time_list, key=lambda x: x[1], reverse=True)
    res = version_time_list[0][0]
    # 针对容器SPC版本
    if "SPC" in res:
        tmp_chart_version = res.rsplit(".", 1)
        # 拼接提取容器版本和SPC字段
        cur_chart_version = tmp_chart_version[0] + tmp_chart_version[-1][:2]
    else:
        cur_chart_version = res
    return cur_chart_version


def get_product_version(user_name):
    cmd = r"echo -e 'show system general' | /ISM/cli/ismcli -u " + user_name + r" | grep -E 'Product Version'"
    cli_result = cli(cmd)
    if not cli_result:
        logging.info("CHECK_VERSION_MATCH: Did not get any valid product version from cli")
        print('False')
        return RETURN_ERROR
    return cli_result.split()[-1]


def main():
    try:
        # 首先要判断用户情况
        user_name = get_user_name()
        if not user_name:
            logging.error("Not found user.")
            print("False")
            return RETURN_ERROR

        # 绿色通道
        dorado_version = get_product_version(user_name)
        if dorado_version in WHITE_LIST:
            print("True")
            return RETURN_OK
        # 针对底座SPH版本
        if "SPH" in dorado_version:
            tmp_version = dorado_version.rsplit(".", 1)
            # 拼接提取底座版本和SPH字段
            product_version = tmp_version[0] + tmp_version[-1][:2]
        else:
            product_version = dorado_version
        if not product_version:
            logging.info("CHECK_VERSION_MATCH: Did not find the spc version from OSM_CONF_YAML: %s", OSM_CONF_YAML)
            print("False")
            return RETURN_ERROR
        # 二级存储C版本信息
        ocean_protect_c_version = get_helm_chart_version(user_name, product_version)
        if ocean_protect_c_version == RETURN_ERROR:
            print("False")
            return RETURN_ERROR
        if product_version in DORADO_MAP_OP.keys():
            dorado_map_op_list = DORADO_MAP_OP[product_version]
            if ocean_protect_c_version not in dorado_map_op_list:
                print('False')
                return RETURN_ERROR
        else:
            if product_version != ocean_protect_c_version:
                logging.info("CHECK_VERSION_MATCH: The spc version: %s is not equal to ocean protect version: "
                             "%s the check is not ok.", product_version, ocean_protect_c_version)
                print('False')
                return RETURN_ERROR
        logging.info("CHECK_VERSION_MATCH: check is ok.")
        # 打印回显给swm作为脚本执行检查结果，False失败，True成功
        print("True")
        return RETURN_OK
    except Exception as err:
        logging.exception("CHECK_VERSION_MATCH: %s", err)
        print("False")
        return RETURN_ERROR


if __name__ == '__main__':
    sys.exit(main())