#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.
import re
import logging
import os
import shutil
import xml.dom.minidom
import yaml

CUR_MANIFEST_PATH = '/OSM/conf/manifest.yml'
IMAGE_DISK = '/startup_disk/image'
PATCH_MANIFEST = 'patch.yml'
COFFER_DATA_DIR = '/OSM/coffer_data'

log = logging.getLogger("check_func")
formatter = logging.Formatter(fmt="[%(asctime)s][%(levelname)s][%(message)s]"
                                  "[%(filename)s, %(lineno)d]",
                              datefmt="%b %d %Y %H:%M:%S")
file_handler = logging.FileHandler(filename="/OSM/log/cur_debug/messages")
file_handler.setFormatter(formatter)
log.addHandler(file_handler)
log.setLevel(logging.INFO)
log.propagate = False


def get_cur_sys_version():
    if not os.path.exists(CUR_MANIFEST_PATH):
        log.info("CHECK_FUNC: can not get the src version, %s does"
                 " not exist." % CUR_MANIFEST_PATH)
        return "", "", ""
    with open(CUR_MANIFEST_PATH) as fd:
        cfg_yml = yaml.safe_load(fd)
        try:
            target_version = str(cfg_yml.get("SYS")["Version"])
            target_spc_version = str(cfg_yml.get("SYS")["SpcVersion"])
            target_apollo_version = str(cfg_yml.get("ApolloVersion", ''))
            return target_version, target_spc_version, target_apollo_version
        except Exception:
            log.exception("CHECK_FUNC: Failed to get version.")
            return "", "", ""


def get_upg_sys_version():
    current_base_version = get_cur_sys_version()[0]
    if not current_base_version:
        return False, ''
    dirs = os.listdir(IMAGE_DISK)
    upg_sys_ver = None
    for per_dir in dirs:
        if per_dir.isdigit() and per_dir.startswith('76') and per_dir != current_base_version:
            upg_sys_ver = per_dir
            break
    if not upg_sys_ver:
        return False, ''
    return True, upg_sys_ver


def get_patch_version(conf_file):
    """

    :param conf_file:补丁配置文件
    :return:
    """
    if not os.path.exists(conf_file):
        return ""
    fd = open(conf_file)
    patch_conf = yaml.safe_load(fd)
    fd.close()
    ver_conf = patch_conf.get("PatchConf", None)
    if not ver_conf:
        return ""
    return ver_conf.get('patch_version', "")


def get_hotpatch_version():
    """
    :return:
    """
    version = get_cur_sys_version()[0]
    cur_patch_path = os.path.join(IMAGE_DISK, str(version),
                                  'hotpatch/patch_cur/', PATCH_MANIFEST)
    cur_sph = get_patch_version(cur_patch_path)
    bak_patch_path = os.path.join(IMAGE_DISK, str(version),
                                  'hotpatch/patch_bak', PATCH_MANIFEST)
    bak_sph = get_patch_version(bak_patch_path)
    return cur_sph, bak_sph


def get_conf_product(conf_file, conf_type):
    if not os.path.exists(conf_file):
        log.info("CHECK_FUNC: %s not exist.", conf_file)
        return ""

    with open(conf_file) as fd:
        conf_yml = yaml.safe_load(fd)
        if not conf_yml:
            return ""
        label_map = dict(sys="SYS", patch="PatchConf")
        product_conf = conf_yml.get(label_map.get(conf_type), None)
        if not product_conf:
            return ""
        return product_conf.get('Product', "")


def check_sys_and_patch_product(sys_conf_file, patch_conf_file):
    sys_product = get_conf_product(sys_conf_file, 'sys')
    patch_product = get_conf_product(patch_conf_file, 'patch')
    log.info("CHECK_FUNC: System product type: %s, patch product type: %s.", sys_product, patch_product)
    if patch_product != sys_product:
        log.error("CHECK_FUNC: Patch product type not match system product type.")
        return False
    return True


def check_sys_and_hotpatch_version(resolve_patch):
    """
    :param resolve_patch:
    :return:
    """
    sys_version = get_cur_sys_version()[0]
    hotpatch_version = get_hotpatch_version()[0]
    if not hotpatch_version or not sys_version:
        return False
    if sys_version in resolve_patch:
        hotpatch_version_num = int(filter(lambda ch: ch in '0123456789',
                                          hotpatch_version))
        resolve_hotpatch_version_num = int(filter(lambda ch: ch in '0123456789',
                                                  resolve_patch[sys_version]))
        if hotpatch_version_num >= resolve_hotpatch_version_num:
            log.info("CHECK_FUNC: The %s has been installed.", hotpatch_version)
            return True
    return False


def get_patch_version_cfg(tag_name):
    patch_xml_path = os.path.join(os.path.dirname(__file__), "hotpatch_version.xml")
    dom = xml.dom.minidom.parse(patch_xml_path)
    if dom is None:
        return None
    root = dom.documentElement
    patch_item = root.getElementsByTagName(tag_name)
    version_items = patch_item[0].getElementsByTagName('version')
    version_dict = dict()
    for item in version_items:
        version_dict[item.getAttribute('base_ver')] = item.getAttribute('patch_ver')
    return version_dict


def show_verbose(result, error_code, cur_patch, required_path):
    print(result)
    print(error_code)
    print(';'.join([cur_patch, required_path]))
    return


def check_patch_version(patch_cfg, base_version, patch_version, error_code):
    if base_version not in patch_cfg:
        log.info("CHECK_FUNC: Version(%s) does not configured.", base_version)
        show_verbose('True', '', '', '')
        return 0
    require_patch_ver = patch_cfg[base_version]
    if not patch_version:
        log.error("CHECK_FUNC: Required patch ver(%s), but no patch version.", require_patch_ver)
        show_verbose('False', error_code, 'NA', require_patch_ver)
        return 0
    try:
        require_patch_digit_ver = int(re.split('(\\d+)', require_patch_ver)[1])
        patch_digit_ver = int(re.split('(\\d+)', patch_version)[1])
        if patch_digit_ver >= require_patch_digit_ver:
            log.info("CHECK_FUNC: The %s has been installed, required patch(%s).", patch_version, require_patch_ver)
            show_verbose('True', '', '', '')
            return 0
        log.error("CHECK_FUNC: The patch ver(%s) is lower than required(%s).", patch_version, require_patch_ver)
        show_verbose('False', error_code, patch_version, require_patch_ver)
        return 0
    except Exception:
        log.exception("CHECK_FUNC: Check patch version failed")
        show_verbose('False', '', '', '')
    return 1


def check_current_patch_version():
    error_code = '10008'
    current_patch_cfg = get_patch_version_cfg("current_patch_version")
    cur_base_version = get_cur_sys_version()[0]
    cur_patch_version = get_hotpatch_version()[0]
    if not cur_base_version:
        show_verbose('False', '', '', '')
        return 1
    return check_patch_version(current_patch_cfg, cur_base_version, cur_patch_version, error_code)


def check_upgrade_patch_version():
    error_code = '10009'
    upgrade_patch_cfg = get_patch_version_cfg("upgrade_patch_version")
    res, upg_sys_ver = get_upg_sys_version()
    if not res:
        show_verbose('False', '', '', '')
        return 1
    upg_patch_path = os.path.join(IMAGE_DISK, str(upg_sys_ver), 'hotpatch', 'patch_cur', PATCH_MANIFEST)
    upg_patch_ver = get_patch_version(upg_patch_path)
    return check_patch_version(upgrade_patch_cfg, upg_sys_ver, upg_patch_ver, error_code)


def check_upgrade_patch_product():
    res, upg_sys_ver = get_upg_sys_version()
    if not res:
        show_verbose('False', 'upg version not exist', '', '')
        return 1
    upg_sys_path = os.path.join(IMAGE_DISK, str(upg_sys_ver), 'system', 'manifest.yml')
    if not os.path.exists(upg_sys_path):
        show_verbose('False', 'upg sys conf not exist', '', '')
        return 1
    upg_patch_path = os.path.join(IMAGE_DISK, str(upg_sys_ver), 'hotpatch', 'patch_cur', PATCH_MANIFEST)
    if not os.path.exists(upg_patch_path):
        show_verbose('True', 'patch check passed', '', '')
        return 0
    if not check_sys_and_patch_product(upg_sys_path, upg_patch_path):
        show_verbose('False', 'product type consistence check failed', '', '')
        return 0
    show_verbose('True', 'patch check passed', '', '')
    return 0


def clean_patch_backup():
    # 6.0.0 - 6.0.1版本删除残留的补丁备份文件
    try:
        cur_version = get_cur_sys_version()[0]
        if not cur_version:
            show_verbose("False", 'Failed to read the configuration file', '', '')
            return 1
        for dirname in os.listdir(COFFER_DATA_DIR):
            if dirname == cur_version or not dirname.startswith(cur_version[:2]):
                continue
            shutil.rmtree(os.path.join(COFFER_DATA_DIR, dirname))
        log.info("Clean patch backup files in %s success", COFFER_DATA_DIR)
        show_verbose('True', 'patch backup clear success', '', '')
        return 0
    except Exception:
        log.exception("Clean patch backup files in %s exception!", COFFER_DATA_DIR)
    show_verbose('False', 'patch backup clear failed', '', '')
    return 1
