#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import sys
import subprocess
import re
import os
import shlex
import logging
import yaml

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')

PORT_ID_PATTERN = r'0x11[0-9a-fA-F]{2}0[0-3]'
v120_port_list = []
VERSION_6_1_8 = '7600519193'
TARGET_CONFIG_PATH = "/startup_disk/image/pkg_upd/manifest.yml"


def get_pkg_version(manifest_file):
    if not os.path.exists(manifest_file):
        return ""

    with open(manifest_file) as file_desc:
        cfg_yml = yaml.safe_load(file_desc)
    try:
        pkg_version = str(cfg_yml.get("SYS")["Version"])
        return pkg_version
    except Exception:
        return ""


def is_target_env():
    get_fc_port_info_cmd = 'drvunf showportinfo'
    port_info = diagnose_cmd(get_fc_port_info_cmd)
    for line in port_info.splitlines():
        array = line.lstrip()
        if '0x110' in array:
            return True
    return False


def diagnose_cmd(command):
    diag_cmd = ['diagsh', '--attach=*_12', "--cmd=%s" % command]
    diag_process = subprocess.Popen(diag_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    stdout, stderr = diag_process.communicate(timeout=20)
    return stdout.decode("utf-8")


def get_fc_v120_card_port():
    get_card_cmd = 'drvunf showversion'
    card_type = 'ioc_v3r1'
    card_info = diagnose_cmd(get_card_cmd)
    # 发现1822 v120 FC前端接口卡
    if re.search(card_type, card_info, re.IGNORECASE):
        for line in card_info.splitlines():
            if card_type in line:
                array = line.lstrip()
                v120_port_list.extend(re.findall(PORT_ID_PATTERN, array))
        return True
    return False


def find_port_in_list(port_info):
    for line in port_info.splitlines():
        array = line.lstrip()
        if 'FC-NVMe' in array:
            port_id = re.findall(PORT_ID_PATTERN, array)
            if port_id[0] in v120_port_list:
                return True
    return False


def find_nvme_port():
    get_fc_port_info_cmd = 'drvunf showportinfo'
    protocol_type = 'FC-NVMe'
    port_info = diagnose_cmd(get_fc_port_info_cmd)
    if not re.search(protocol_type, port_info, re.IGNORECASE):
        return False
    else:
        if find_port_in_list(port_info):
            return True
    return False


def main():
    try:
        logging.info("Start to check fc port protocol.")
        # 只在python3版本执行检查
        if not sys.version_info.major == 3:
            logging.info("No need check.")
            print("True")
            return 0
        # 判断是否中低端环境，高端直接跳过
        if not is_target_env():
            logging.info("Check skip on this env.")
            print("True")
            return 0    
        target_ver = get_pkg_version(TARGET_CONFIG_PATH)
        if not target_ver:
            logging.warning("Get target version failed.")
            print("False")
            return 0
        # 获取1822 V120 FC前端接口卡的端口集合，若集合为空，则检查通过
        if not get_fc_v120_card_port():
            logging.info("No risk of the fc card version.")
            print("True")
            return 0
        # 判断回退版本是否支持FC-NVMe
        if target_ver > VERSION_6_1_8:
            logging.info("No need check for this version.")
            print("True")
            return 0
        if find_nvme_port():
            logging.warning("Find the FC-NVMe port during the check, check failed.")
            print("False")
            return 0
        # 完成检查
        logging.info("Check completed.")
        print("True")
        return 0
    except Exception as e:
        logging.exception("Check fc port protocol failed: %s", e)
        print("False")
        print(e)
        return 1

if __name__ == '__main__':
    sys.exit(main())   
