#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2023. All rights reserved.

import os
import random
import sys
import subprocess
import logging
import yaml

# 版本信息文件
SRC_CONFIG_PATH = "/startup_disk/image/pkg_cur/manifest.yml"
TARGET_CONFIG_PATH = "/startup_disk/image/pkg_upd/manifest.yml"

# 微存储615发布版本，版本号
VER_615_REALEASE = '7610101'
# 版本号说明：
# 7610101    186
# 大版本    B版本
# ver_micro_615_realease '7610101186'
# ver_615 '7610101'

# 直接执行diagnose命令的字符串格式：
# {0}表示attach的进程id,比如app_data为12，
# {1}表示需要执行的diagnose命令。
DIAGNOSE_CHECK_CMD = "diagsh,--attach=*_{0},--cmd={1}"

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][log_qos][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')


def get_pkg_version(manifest_file):
    """
    :param manifest_file:
    :return:
    """
    if not os.path.exists(manifest_file):
        logging.warning("Can not get the new version, config %s not existing.", manifest_file)
        return ""

    with open(manifest_file) as file_desc:
        cfg_yml = yaml.safe_load(file_desc)
    try:
        pkg_version = str(cfg_yml.get("SYS")["Version"])
        logging.debug('Get Version(%s).', pkg_version)
        return pkg_version
    except Exception:
        logging.exception("Failed to get version.")
        return ""


def downgrade_check(src_ver, target_ver):
    """
    通过调用diagnose或者CLI命令方式，实现降级检查逻辑，如果需要通过执行shell命令
    检查，建议直接按本脚本逻辑通过shell脚本实现。
    此脚本根据实际检查情况，打印检查结果、错误码和错误信息，
    升级模块解析打印回显，将检查结果、错误码和错误信息上报到升级工具
    检查结果说明：
    True或False
    错误码规则说明：
    十六进制，前四位为模块id，后四位为模块错误码，模块ID参考模块PID，
    错误码由模块自行分配管理，如：0x01B50001表示SWM模块（PID十进制为437）的001
    号错误码。
    错误信息说明：
    表示检查不通过的详细信息，用于匹配升级工具资源文件中的关键字，
    比如资源文件描述“硬盘利用率检查不通过，当前最大利用率为%s（检查阈值50%），
    检查不通过的硬盘为(%s)。”，说明资源文件需要匹配两个关键字，
    这种情况错误信息打印格式为：当前最大利用率;硬盘id列表，如90;DAE000.0,DAE000.1,
    关键字之间使用英文分号“;”分隔，参数内部如果需要返回列表使用英文逗号“,”分隔
    :param src_ver: 当前版本，如7600501032
    :param target_ver:降级的目标版本，如7600500099
    :return:
    """
    # 如果降级目标版本大于6.1.5，检查通过
    if (target_ver[:7] > VER_615_REALEASE):
        print("True")
        return
    diag_cmd_str = "qos show downgrade_check read_write_smartqos"
    diag_cmd_str = DIAGNOSE_CHECK_CMD.format(12, diag_cmd_str)
    try:
        diag_process = subprocess.Popen(diag_cmd_str.split(','), stdout=subprocess.PIPE, shell=False)
        diag_out = diag_process.communicate(timeout=30)[0].decode('utf-8')  # 设置超时时间等于30s
    except Exception as error:
        print("False")
        logging.exception(error)
        logging.error("DOWNGRADE_CHECK: Cmd(%s) exec failed.", diag_cmd_str)
        return

    for line in diag_out.split('\n'):
        # 解析每行字符串，获取需要的信息进行判断，
        # 如果需要解析某行某列更详细的信息或者需要进行更详细的计算，
        # 请根据检查逻辑实现，这里只列举解析行信息的方法
        if "False" in line:
            print("False")
            return
    print("True")
    return


def main():
    """
    检查主函数，一般不需要修改
    :return:
    """
    try:
        # 当前版本，如果采用shell脚本实现，需要自行解析
        # /startup_disk/image/pkg_cur/manifest.yml中SYS下的version字段值
        src_ver = get_pkg_version(SRC_CONFIG_PATH)
        # 降级/升级目标版本，如果采用shell脚本实现，需要自行解析
        # /startup_disk/image/pkg_upd/manifest.yml中SYS下的version字段值，
        # 此配置文件只有升级开始并完成软件包上传后才有
        target_ver = get_pkg_version(TARGET_CONFIG_PATH)
        logging.info("DOWNGRADE_CHECK: System Version: %s target version: %s.",
                     src_ver, target_ver)
        if not src_ver or not target_ver:
            logging.error("DOWNGRADE_CHECK: The src or target version is empty.")
            print("False")
            return 1
        downgrade_check(src_ver, target_ver)
        return 0
    except Exception as e:
        print("False")
        logging.exception(e)
        return 1


if __name__ == '__main__':
    sys.exit(main())
