#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.
import argparse
import os
import shutil
import sys
import time

from check_item.check_util import util
from config import env, path
from config import pkg as pkg_cfg
from infra.debug.log import swm_logger as log
from infra.util import pkg
from infra.util.pkg import get_cur_version
from infra.util.shell import remount_image
from plat.fw.firmware_mgr import FirmwareMgr, FirmwareMgrObjEnum
from plat.fw.fw_base.base_enum import UpgradeTypeEnum
from plat.fw.fw_base.fw_interface import FwInterface
from plat.host.cluster_mgr import ClusterMgr

try:
    from plat.fw.fw_base.card_fru_type import CardFruType
    from plat.fw.fw_base.card_fru_type import CPlusCardFruType
except ImportError:
    log.error('ImportError: this version is not support CardFruType and CPlusCardFruType.')


def init_thrift_interface():
    fw_mgr = FirmwareMgr()
    fw_mgr.interface = FwInterface()
    if hasattr(fw_mgr.interface.dmi_adapter, 'open_relate_so'):
        fw_mgr.interface.dmi_adapter.open_relate_so()
    ret_code = fw_mgr.interface.start()
    if ret_code != 0:
        log.error("FW_CHECK: interface init failed.")
        return None
    return fw_mgr


def apollo_copy_pkg(src_plat, dst_plat, dst_default_plat):
    dst_cmpt_list = [x.split('-')[0] for x in os.listdir(dst_plat)]
    for cmpt_pkg in os.listdir(src_plat):
        cmpt_name = cmpt_pkg.split('-')[0]
        if cmpt_name not in dst_cmpt_list:
            if cmpt_name.startswith("his_kernel"):  # his_kernel 不拷贝 降级场景会残留
                continue
            log.info("WEAK_MATCH: The dst plat has no cmpt(%s), copy to dst.", cmpt_name)
            if cmpt_name == 'firmware' and dst_default_plat != os.path.realpath(dst_plat):
                cmd = 'cp -arf {0}/{1} {2}/'.format(dst_default_plat, cmpt_pkg, dst_plat)
            else:
                cmd = 'cp -arf {0}/{1} {2}/'.format(src_plat, cmpt_pkg, dst_plat)
            log.info("cmd:%s", cmd)
            ret, _ = util.call_system_cmd(cmd.split(), timeout=5)
            if ret:
                log.error("WEAK_MATCH: Copy weak match cmpt(%s) to dst failed.", cmpt_name)
                return False
            log.info("WEAK_MATCH: Copy weak match cmpt(%s) to dst successfully.", cmpt_name)
    return True


def copy_his_kernel(src_plat, dst_plat):
    """
    拷贝源包中的 kernel 组件到目的包中
    从版本开始支持his_kernel，不支持的版本不拷贝

    :param:src_plat:源apollo目录
    :param:dst_plat:目的apollo目录
    """
    src_his_kern_path = get_cmpt_path(src_plat, "his_kernel")
    log.info("CO_KERNEL: src_his_kern_path %s.", src_his_kern_path)
    # 源目录已经存在 his_kernel 直接拷贝到目的目录
    if src_his_kern_path:
        dst_cmpt_path = os.path.join(dst_plat, os.path.basename(src_his_kern_path))
        shutil.copyfile(src_his_kern_path, dst_cmpt_path)
        return
    # 否则拷贝源目录下的 kernel 组件到目的目录作为 his_kernel
    src_cmpt_path = get_cmpt_path(src_plat, "kernel")
    if not src_cmpt_path:
        log.error("CO_KERNEL: Src kernel path is not exist.")
        return
    dst_cmpt_path = os.path.join(dst_plat, "his_" + os.path.basename(src_cmpt_path))
    shutil.copyfile(src_cmpt_path, dst_cmpt_path)
    return


def get_apollo_version(apollo_link):
    if not os.path.islink(apollo_link):
        return ""
    apollo_path = os.path.realpath(apollo_link)
    return os.path.basename(apollo_path).split('-')[1]


def get_cmpt_path(pkg_dir, cmpt_name):
    if not os.path.exists(pkg_dir):
        return ""
    for pkg_name in os.listdir(pkg_dir):
        if pkg_name.startswith(cmpt_name + "-"):
            return os.path.join(pkg_dir, pkg_name)
    return ""


def complete_weak_match_plat(src_plat, dst_plat):
    if not os.path.islink(src_plat):
        log.warning("WEAK_MATCH: The src plat(%s) is not link.", src_plat)
        return True
    if not os.path.islink(dst_plat):
        log.warning("WEAK_MATCH: The dst plat(%s) is not link.", dst_plat)
        return True
    dst_default_plat = os.path.realpath(dst_plat)
    src_apollo_version = get_apollo_version(src_plat)
    dst_default_apollo_version = get_apollo_version(dst_plat)
    if os.path.realpath(src_plat) != os.path.realpath(dst_plat):
        log.info("WEAK_MATCH: The src plat(%s) diff with dst plat(%s), change dst plat link.", src_plat, dst_plat)
        os.unlink(dst_plat)
        os.symlink(os.readlink(src_plat), dst_plat)
    if src_apollo_version != dst_default_apollo_version:  # 原版本和目的版本的默认apollo不一致拷贝 his_kernel
        log.info("WEAK_MATCH: The src_apollo_version(%s) diff with dst_default_apollo_version(%s), copy his kernel.",
                 src_apollo_version, dst_default_apollo_version)
        copy_his_kernel(src_plat, dst_plat)
    return apollo_copy_pkg(src_plat, dst_plat, dst_default_plat)


def complete_weak_pkg_with_retry(dst_pkg_root, retry_time=3, interval=2):
    for i in range(retry_time):
        if complete_weak_pkg(dst_pkg_root):
            return True
        if (i+1) == retry_time:
            break
        time.sleep(interval)
        log.warning("WEAK_MATCH: Complete weak pkg failed retry.")
    return False


@remount_image
def complete_weak_pkg(dst_pkg_root):
    """
    Complete weak match package
    :param dst_pkg_root:
    :return:
    """
    src_pkg_root = os.path.realpath(path.PKG_CUR_DIR)
    for plat in pkg_cfg.WEAK_MATCH_PLAT_LIST:
        src_plat = os.path.join(src_pkg_root, plat)
        dst_plat = os.path.join(dst_pkg_root, plat)
        ret = complete_weak_match_plat(src_plat, dst_plat)
        if not ret:
            log.error("WEAK_MATCH: Complete weak match plat(%s) failed.", plat)
            return False
        log.info("WEAK_MATCH: Complete weak match plat(%s) successfully.", plat)
    log.info("WEAK_MATCH: Complete weak match pkg successfully.")
    return True


def check_fast_func():
    # 1822 不做检查，只检查bios和控制框
    # 6.1.RC2(7600501218)之后增加对1822卡检查
    if not os.path.exists(path.TARGET_CONFIG_PATH):
        log.info("FW_CHECK: Can not get the new version,pkg %s does not exist." % path.UPLOAD_SOFTWARE_PATH)
        return False
    pkg_ver = pkg.get_pkg_version(path.TARGET_CONFIG_PATH)[0]
    if not pkg_ver:
        return False
    complete_res = complete_weak_pkg_with_retry(os.path.join(path.IMAGE_DISK, pkg_ver, 'system'))
    if not complete_res:
        return False
    fw_dir_temp = path.UPG_FW_PATH.format(pkg_ver)
    fw_mgr = init_thrift_interface()
    if fw_mgr is None:
        return False
    if not env.ARM:
        return True
    fw_path = os.path.join(fw_dir_temp, env.BOARD_TYPE)
    bios_res = fw_mgr.check_support_fast_upgrade(FirmwareMgrObjEnum.BIOS, fw_path)
    if not bios_res.upg_res:
        log.info("FW_CHECK: bios not support fast upgrade.")
        return False
    ctrl_res = fw_mgr.check_support_fast_upgrade(FirmwareMgrObjEnum.ENCLOSURE, fw_path)
    if not ctrl_res.upg_res:
        log.info("FW_CHECK: enclosure not support fast upgrade.")
        return False

    if env.DATA_NODE:
        return True

    cur_version = get_cur_version()[0]
    cur_version = cur_version if cur_version else 0
    if int(cur_version) < int('7600501218'):
        return True

    front_not_share_cards = CardFruType.front_not_share_cards() + CPlusCardFruType.front_not_share_cards()
    front_not_share_cards_res = fw_mgr.check_support_fast_upgrade(FirmwareMgrObjEnum.CARD_1822,
                                                                  fw_path,
                                                                  front_not_share_cards)

    if not front_not_share_cards_res.upg_res:
        log.info("FW_CHECK: front cards not support fast upgrade.")
        return False

    if not ClusterMgr.is_local_upgrade_share_engine_min_node():
        return True

    front_share_cards = CardFruType.front_share_cards() + CPlusCardFruType.front_share_cards()
    front_share_cards_res = fw_mgr.check_support_fast_upgrade(FirmwareMgrObjEnum.SHARE_CARD_1822,
                                                              fw_path,
                                                              front_share_cards)
    if not front_share_cards_res.upg_res:
        log.info("FW_CHECK: front share cards not support fast upgrade.")
        return False

    return True


def check_is_support_apollo_switch():
    # 0: 可以切换
    # 1：不需要切换，已经是最新
    # 2：内部错误
    # 获取pkg_cur指向的包
    # 存在两个apollo包，1.当前指向的不是最大apollo version链接，   返回0
    #                 2.指向是最大apollo version链接，固件不匹配，返回0
    #                 3.指向是最大apollo version链接，固件匹配，  返回1
    # 存在一个apollo包，1.固件匹配  返回1
    #                 2.固件不匹配 返回0
    cur_pkg_root = os.path.realpath(path.PKG_CUR_DIR)
    dst_list = [x for x in os.listdir(cur_pkg_root) if
                x.startswith("apollo-")]
    if len(dst_list) == 0:
        log.error("APOLLO_PRE_CHECK: system dir has error.")
        return 2
    cur_apollo_ver_dir = os.path.basename(
        os.path.realpath(os.path.join(cur_pkg_root, 'apollo')))
    max_apollo_ver_dir = pkg.get_max_apollo_ver_dir(dst_list)
    if max_apollo_ver_dir != "" and max_apollo_ver_dir != cur_apollo_ver_dir:
        log.info("APOLLO_PRE_CHECK: cur not the max apollo version.")
        return 0
    # check firmware
    fw_mgr = init_thrift_interface()
    if fw_mgr is None:
        return 2
    fw_path = os.path.join(path.FW_DIR, env.BOARD_TYPE)
    bios_res = fw_mgr.check(FirmwareMgrObjEnum.BIOS, fw_path,
                            UpgradeTypeEnum.ONLINE_UPGRADE)
    if not bios_res.upg_res:
        log.info("APOLLO_PRE_CHECK: bios ver not same, can switch apollo.")
        return 0
    encl_res = fw_mgr.check(FirmwareMgrObjEnum.ENCLOSURE, fw_path,
                            UpgradeTypeEnum.ONLINE_UPGRADE)
    if not encl_res.upg_res:
        log.info("APOLLO_PRE_CHECK: ctrl ver not same, can switch apollo.")
        return 0
    log.info("APOLLO_PRE_CHECK: latest, no need switch apollo link.")
    return 1


def main():
    parser = argparse.ArgumentParser(
        description="This is a fast upgrade check script.")
    parser.add_argument('-c', '--check', dest='check_fast',
                        action="store_true",
                        help='check is support fast upgrade.')
    parser.add_argument('-a', '--apollo', dest='check_apollo',
                        action="store_true",
                        help='check is support apollo switch.')
    args = parser.parse_args()
    check_fast = getattr(args, "check_fast", False)
    if check_fast:
        check_result = check_fast_func()
        if not check_result:
            return 1

    check_apollo = getattr(args, "check_apollo", False)
    if check_apollo:
        return check_is_support_apollo_switch()
    return 0


if __name__ == '__main__':
    sys.exit(main())
