#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.
import abc
import inspect
import os
import re
import sys
from xml.etree import ElementTree as ET

from check_item.check_util import util
from check_item.host_check.check_exception import CmdExecError, UnExpectValueError
from infra.debug.log import swm_logger as log

_LOG_PREFIX = "HOST_COMPATIBLE_CHECK"


class HostOnArray(object):
    """
    阵列侧配置的主机对象
    """

    def __init__(self, host_id, wwn_iqn, status):
        self.host_id = host_id
        self.wwn_iqn = wwn_iqn
        self.status = status

    def __eq__(self, other):
        if self.host_id == other.host_id:
            return True
        else:
            return False

    def __hash__(self):
        return hash(self.host_id)

    def __str__(self):
        return "host id(%s), wwn_iqn(%s), status(%s)." % \
               (self.host_id, self.wwn_iqn, self.status)


class CheckResultType(object):
    PASS = 0
    NOT_PASS = 1
    CAN_NOT_CHECK = 2


class FailedReason(object):
    INNER_ERROR = "1077949019"
    HBA_NOT_COMPATIBLE = "0x01b50005"
    SYS_VER_NOT_COMPATIBLE = "0x01b50006"
    ULTRA_PATH_NOT_COMPATIBLE = "0x01b50007"
    SYS_VER_UNAVAILABLE = "0x01b50008"
    HBA_INFO_UNAVAILABLE = "0x01b50009"
    ULTRA_PATH_INFO_UNAVAILABLE = "0x01b5000a"
    ULTRA_PATH_NOT_INSTALL = "0x01b5000b"
    LOAD_HOST_INFO_FAILED = '0x01b5000c'
    SYSTEM_INFO_NOT_COMPATIBLE = '0x01b5000e'


class CheckResultInfo(object):
    """
    检查结果详细信息
    """

    def __init__(self, host_id, detail_info,
                 check_res_type=CheckResultType.PASS,
                 failed_reason=None):
        self.host_id = host_id
        self.detail_info = detail_info
        self.check_res_type = check_res_type
        self.failed_reason = failed_reason


class FileParser(object):
    _CMD_END_STR = "EndCmd:"
    _CMD_START_STR = "StartCmd:"
    _PYTHON_2 = '2.7' in sys.version

    def __init__(self, host_info_file_path):
        self.host_info_file_path = host_info_file_path
        self.start_cmd = ""
        self.end_cmd = ""
        self.start_line = 0
        self.end_line = 0

    def load_info_by_col_num(self, cmd, line_filter_regex,
                             col_split_str=None, col_index=0):
        """
        根据命令及列序号返回多行信息
        :param cmd: 在主机信息文件中的命令
        :param line_filter_regex: 过滤行的正则表达式
        :param col_split_str: 用于分隔行中列信息的关键字，默认为""
        :param col_index: 返回信息在行分隔后的列位置
        :return:
        """
        return_list = set()
        try:
            lines = self.load_info_by_cmd(cmd)
            if not lines:
                log.warning("%s: No any info for cmd(%s).", _LOG_PREFIX, cmd)
                return set()

            lines = [str(line).strip() for line in lines if
                     re.findall(line_filter_regex, line)]

            for line in lines:
                log.debug("%s: Line (%s) info.", _LOG_PREFIX, line)
                if col_split_str:
                    col_info = line.split(col_split_str)[col_index]
                else:
                    col_info = line.split()[col_index]
                if col_info:
                    return_list.add(col_info)
            return return_list
        except Exception as e:
            log.exception(e)
            return set()

    def load_info_by_col_regex(self, cmd, line_filter_regex, col_regex_str):
        """
        根据命令和列信息匹配正则表达式返回多行信息
        :param cmd: 在主机信息文件中的命令
        :param line_filter_regex: 过滤行的正则表达式
        :param col_regex_str: 用于获取行中关键信息的正则表达式
        :return:
        """
        return_list = set()
        try:
            lines = self.load_info_by_cmd(cmd)
            if not lines:
                log.warning("%s: No any info for cmd(%s).", _LOG_PREFIX, cmd)
                return []

            lines = [str(line).strip() for line in lines if
                     re.findall(line_filter_regex, line)]

            for line in lines:
                log.debug("%s: Line (%s) info.", _LOG_PREFIX, line)
                col_info = re.findall(col_regex_str, line)
                if not col_info:
                    continue
                return_list.add(col_info[0])
            return return_list
        except Exception as e:
            log.exception(e)
            return []

    def load_key_info(self, cmd_dict):
        """
        :param cmd_dict:
        :return:
        """
        output_info = self.load_info_by_cmd(cmd_dict.get("Cmd"))
        if not output_info:
            log.warning("%s: Cmd(%s) output is empty.",
                        _LOG_PREFIX, cmd_dict)
            return ""
        key_line_info = ""
        for line in output_info:
            line_info = line.strip()
            line_filter_str = cmd_dict.get("line_filter_str", "")
            if not line_filter_str or re.findall(line_filter_str, line):
                key_line_info = line_info
                break
        if not key_line_info:
            log.error("%s: Can not find key line for(%s).",
                      _LOG_PREFIX, cmd_dict.get("line_filter_str", ""))
            return ""
        key_info_filter_regex = cmd_dict.get("key_info_find_regex", "")
        log.debug("%s: Going to get key info by regex string(%s).",
                  _LOG_PREFIX, key_info_filter_regex)
        if not key_info_filter_regex:
            log.debug("%s: Load key info successfully, key info(%s).",
                      _LOG_PREFIX, key_line_info)
            return key_line_info
        key_line_info = re.findall(key_info_filter_regex, key_line_info)
        log.debug("%s: Load key info successfully, key info(%s).",
                  _LOG_PREFIX, key_line_info)
        if key_line_info:
            return key_line_info[0]
        return ''

    def load_info_by_cmd(self, cmd):
        """
        通过命令获取命令执行的回显信息
        :param cmd:
        :return:
        """
        if not cmd:
            return []
        self.start_cmd = "{0}{1}".format(self._CMD_START_STR, cmd)
        start_line_cmd = [["grep", "-n", "-E", "^{obj.start_cmd}$".format(obj=self), str(self.host_info_file_path)],
                          ["cut", "-d", ":", "-f", "1"]]
        cmd_output = self.exec_cmd_with_output(start_line_cmd)
        if not cmd_output or not str(cmd_output[0]).isdigit():
            return []

        self.start_line = int(cmd_output[0]) + 1

        self.end_cmd = "{0}{1}".format(self._CMD_END_STR, cmd)
        end_line_cmd = [["grep", "-n", "-E", "^{obj.end_cmd}$".format(obj=self), str(self.host_info_file_path)],
                        ["cut", "-d", ":", "-f", "1"]]
        cmd_output = self.exec_cmd_with_output(end_line_cmd)
        if not cmd_output or not str(cmd_output[0]).isdigit():
            return []
        self.end_line = int(cmd_output[0]) - 1

        # 开始行和结束行相同，说明命令返回内容为空
        # 如果只有一行内容，self.start_line+1 == self.end_line-1
        if self.start_line > self.end_line:
            log.warning("%s: cmd(%s) have no any info.", _LOG_PREFIX, cmd)
            return []

        load_info_cmd = [["sed", "-n", "{obj.start_line},{obj.end_line}p".format(obj=self),
                          "{obj.host_info_file_path}".format(obj=self)]]
        cmd_output = self.exec_cmd_with_output(load_info_cmd)

        return cmd_output

    def exec_cmd_with_output(self, shell_cmd):
        """
        执行shell命令，并返回回显信息
        :param shell_cmd:
        :return:
        """
        ret_code, info = util.call_system_cmd_list(shell_cmd, 30)
        if ret_code != 0:
            log.error('%s: Command(%s) exec failed, '
                      'code (%d), info(%s).', _LOG_PREFIX,
                      shell_cmd, ret_code, info)
            raise CmdExecError(shell_cmd)
        info = info.splitlines()
        if info is None or len(info) < 1:
            log.warning('%s: Command(%s) exec output is empty.',
                        _LOG_PREFIX, shell_cmd)
            return []
        info = [self._decode_str(line) for line in info]
        log.debug('%s: Command(%s) exec result(%s)',
                  _LOG_PREFIX, shell_cmd, info)
        return info

    def _decode_str(self, value):
        if not self._PYTHON_2 and isinstance(value, bytes):
            return value.decode()
        return value


class Host(object):
    """
    Host base class
    """
    __metaclass__ = abc.ABCMeta
    _OS_TYPE = ""
    _IP_REGEX = r"^(\d+.\d+.\d+.\d+)"
    _IPV6_REGEX = r'^(\[.*:.*\])'
    _SYS_VER_CMD = {"Cmd": "", "line_filter_str": "",
                    "key_info_find_regex": ""}
    _SYS_KERNEL_VER_CMD = {"Cmd": "uname -r", "line_filter_str": "",
                           "key_info_find_regex": ""}
    _HBA_CARD_TYPE_CMD = {"Cmd": "",
                          "line_filter_str": "",
                          "key_info_find_regex": ""}
    _HBA_CARD_FW_VER_CMD = {"Cmd": "",
                            "line_filter_str": "",
                            "key_info_find_regex": ""}
    _HBA_CARD_DRI_VER_CMD = {"Cmd": "",
                             "line_filter_str": "",
                             "key_info_find_regex": ""}
    _HBA_CARD_DRIVER_CMD = {"Cmd": "",
                            "line_filter_str": "",
                            "key_info_find_regex": ""}
    _ULTRA_PATH_VER_CMD = {"Cmd": "upadm show version",
                           "line_filter_str": r".*Software Version.*",
                           "key_info_find_regex": r"\d+.\d+.\d+"}
    _INITIATOR_INFO_CMD = {"Cmd": "upadm show path",
                           "line_filter_str": r".*{0}.*",
                           "key_info_find_regex": ""}
    _FILE_SYSTEM_MOUNT_IPV4_CMD = {"Cmd": "mount",
                                   "line_filter_str": r".*" + _IP_REGEX + ".*",
                                   "key_info_find_regex": _IP_REGEX
                                   }
    _FILE_SYSTEM_MOUNT_IPV6_CMD = {"Cmd": "mount",
                                   "line_filter_str": r".*" + _IPV6_REGEX + ".*",
                                   "key_info_find_regex": _IPV6_REGEX}
    _SHOW_VLUN_CMD = {"Cmd": "upadm show vlun", "line_filter_str": "{0}",
                      "key_info_find_regex": ""}
    _RISK_HOST_VERSION_TUPLE = ()
    _RISK_KERNEL_VERSION_DICT = {"Min": "", "Max": ""}

    def __init__(self, host_info_file, array_name, array_hosts,
                 array_logic_ports, array_lun_wwns):
        self.host_info_file_path = host_info_file
        self.array_name = array_name
        self.array_hosts = array_hosts
        self.array_logic_ports = array_logic_ports
        self.array_lun_wwns = array_lun_wwns
        self.initiator_list = set()
        self.host_id = -1
        self.system_ver = ""
        self.system_kernel_ver = ""
        self.hba_card_type = ""
        self.hba_card_fw_ver = ""
        self.hba_card_driver_ver = ""
        self.ultra_path_ver = ""
        self.have_huawei_lun = False
        self.have_huawei_file_system = False
        self.file_parser = FileParser(host_info_file)
        self._load_initiator_and_host_id()

    def check(self):
        try:
            self._load_host_info()
            #  如果当前主机未关联华为LUN或文件系统，则不需要检查
            if not self.have_huawei_lun and not self.have_huawei_file_system:
                log.info("%s: No LUN or file system is associated with current"
                         " array system on the host(%s) which os type(%s). "
                         "No check is required.", _LOG_PREFIX, self.host_id,
                         self._OS_TYPE)
                return CheckResultInfo(self.host_id, "")

            check_result = self._check_hba_compatible()
            if check_result.check_res_type != CheckResultType.PASS:
                log.warning("%s: Hba card compatible of host(%s) os type(%s) "
                            "check not passed.", _LOG_PREFIX, self.host_id,
                            self._OS_TYPE)
                return check_result

            check_result = self._check_system_compatible()
            if check_result.check_res_type != CheckResultType.PASS:
                log.warning("%s: System compatible of host(%s) os type(%s) "
                            "check not passed.", _LOG_PREFIX, self.host_id,
                            self._OS_TYPE)
                return check_result

            check_result = self._check_ultra_path_compatible()
            return check_result
        except CmdExecError as e:
            log.exception(e)
            return CheckResultInfo(self.host_id, self.host_id,
                                   CheckResultType.CAN_NOT_CHECK,
                                   FailedReason.HBA_INFO_UNAVAILABLE)
        except UnExpectValueError as e:
            log.exception(e)
            return CheckResultInfo(self.host_id, self.host_id,
                                   CheckResultType.CAN_NOT_CHECK,
                                   FailedReason.HBA_INFO_UNAVAILABLE)
        except Exception as e:
            log.exception(e)
            return CheckResultInfo(self.host_id, self.host_id,
                                   CheckResultType.CAN_NOT_CHECK,
                                   FailedReason.INNER_ERROR)

    @abc.abstractmethod
    def _check_system_compatible(self):
        return CheckResultInfo(self.host_id, "")

    @abc.abstractmethod
    def _check_hba_compatible(self):
        return CheckResultInfo(self.host_id, "")

    @abc.abstractmethod
    def _check_ultra_path_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _load_host_info(self):
        self.system_ver = self.file_parser.load_key_info(self._SYS_VER_CMD)
        self.system_kernel_ver = self.file_parser.load_key_info(
            self._SYS_KERNEL_VER_CMD)
        self.hba_card_type = self.file_parser.load_key_info(
            self._HBA_CARD_TYPE_CMD)
        self.hba_card_fw_ver = self.file_parser.load_key_info(
            self._HBA_CARD_FW_VER_CMD)
        self.hba_card_driver_ver = self.file_parser.load_key_info(
            self._HBA_CARD_DRIVER_CMD)
        self.ultra_path_ver = self.file_parser.load_key_info(
            self._ULTRA_PATH_VER_CMD)
        self._load_lun_and_file_system_info()

    def _load_initiator_and_host_id(self):
        if not self._load_initiator_from_host_info():
            log.error('%s: Load initiator from host info failed.', _LOG_PREFIX)
            raise UnExpectValueError()

        if not self._generate_host_id():
            log.error('%s: Generate host id failed.', _LOG_PREFIX)
            raise UnExpectValueError()

    def _load_initiator_from_host_info(self):
        """

        :return:
        upadm show path回显：
        --------------------------------------------------------------------
        Path ID   Initiator Port        Array Name       Controller
          0      21000024ff17e0bc  Huawei.storage217_92      0A
          1     21000024ff17e0bc  Huawei.storage217_92      0B
        --------------------------------------------------------------------
        """
        try:
            cmd = self._INITIATOR_INFO_CMD.get("Cmd")
            line_filter_str = self._INITIATOR_INFO_CMD. \
                get("line_filter_str").format(self.array_name)
            initiator_col_index = 1
            initiator_list = self.file_parser.load_info_by_col_num(
                cmd, line_filter_regex=line_filter_str,
                col_index=initiator_col_index)

            for initiator_info_line in initiator_list:
                log.debug("%s: Initiator line info.", _LOG_PREFIX)
                if initiator_info_line:
                    self.initiator_list.add(
                        initiator_info_line.strip().split("::")[0])
            return True
        except Exception as e:
            log.exception(e)
            return False

    def _gen_id_from_initiator(self):
        for array_host in self.array_hosts:
            for initiator in self.initiator_list:
                if initiator == array_host.wwn_iqn:
                    self.host_id = int(array_host.host_id)
                    log.debug("%s: Generate a host(%s).",
                              _LOG_PREFIX, self.host_id)
                    return True
        log.error("%s: Failed to generate a host ID because no "
                  "initiator is associated with the host and "
                  "storage array.", _LOG_PREFIX)
        return False

    def _generate_host_id(self):
        try:
            return self._gen_id_from_initiator()
        except Exception as e:
            log.exception(e)
            return False

    def _load_lun_and_file_system_info(self):
        self._load_huawei_lun_info()
        self._load_huawei_file_system()

    def _load_huawei_lun_info(self):
        """
        获取vlun信息
        :return: True or False
        upadm show vlun命令回显格式：
        ---------------------------------------------------------------------
         Vlun ID   Disk         Name                    Lun WWN
            0     Disk4     vr-sql-tutor    6407d0f100f4d2672451c81e0000005f
            1     Disk5    vr-sql-doctrix   6407d0f100f4d26756bcbfcf00000061
        --------------------------------------------------------------------
        """
        host_vlun_infos = self.file_parser.load_info_by_cmd(
            self._SHOW_VLUN_CMD.get("Cmd"))
        if not host_vlun_infos:
            log.info("%s: No any vlun info in %s from host(%s).",
                     _LOG_PREFIX,
                     self.host_info_file_path, self.host_id)
            return

        # 通过LUN WWN匹配是否有映射本阵列的LUN
        for lun_line in host_vlun_infos:
            lun_line = lun_line.strip()
            # 过滤标题行
            if not re.findall(r"^\d", lun_line):
                continue
            if lun_line.split()[3].strip() in self.array_lun_wwns:
                self.have_huawei_lun = True
                log.debug(
                    "%s: The host(%s) have at least one LUN associated "
                    "with the storage array.", _LOG_PREFIX, self.host_id)
                return

        log.info("%s: The host(%s) does not have any LUN associated "
                 "with the storage array.", _LOG_PREFIX, self.host_id)

    @abc.abstractmethod
    def _load_huawei_file_system(self):
        """
        检查主机是否已关联华为文件系统：通过主机mount回显查询类似
        192.168.14.73:/fcsxwj_csfs_dwjxio_fs0000/dtree1/和回显，并获取IP部分，
        然后通过与阵列侧的逻辑端口进行匹配，如果能够匹配到至少一个IP，
        则认为有关联华为文件系统进行关联判断。
        除Windows系统外，基本都可以通过mount命令查看关联的文件系统，
        特殊情况请重写此函数
        :return:
        """
        mounted_ips = self._get_mounted_ips_from_host()
        if not mounted_ips:
            log.warning("%s: No any file system mounted on current "
                        "host(%s), os type(%s).", _LOG_PREFIX,
                        self.host_id, self._OS_TYPE)
            return

        intersection_ips = list(
            set(mounted_ips).intersection(set(self.array_logic_ports)))
        if not intersection_ips:
            log.warning("%s: No any file system on host(%s) mounted from "
                        "current array, mounted ips(%s), logical port on "
                        "array are(%s).", _LOG_PREFIX, self.host_id,
                        mounted_ips, self.array_logic_ports)
            return
        log.debug(
            "%s: There have at least one file system share on host(%s).",
            _LOG_PREFIX, self.host_id)
        self.have_huawei_file_system = True

    def _get_mounted_ips_from_host(self):
        """
        通过mount命令查看主机关联的文件系统共享
        :return:
        mount回显格式：
        192.168.2.150:/151vs2_fs4 on /mnt/151vs2_fs4 type nfs (rw,relatime,vers=3,
        rsize=262144,wsize=262144,namlen=255,hard,proto=tcp,timeo=600,retrans=2,
        sec=sys,mountaddr=192.168.2.150,mountvers=3,mountport=2050,mountproto=udp,
        local_lock=none,addr=192.168.2.150)
        """
        all_version_cmd = {"ipv4": self._FILE_SYSTEM_MOUNT_IPV4_CMD,
                           "ipv6": self._FILE_SYSTEM_MOUNT_IPV6_CMD}
        system_file_ips = set()
        for version, file_system_mount_cmd in all_version_cmd.items():
            cmd = file_system_mount_cmd.get("Cmd")
            key_line_regex = file_system_mount_cmd.get("line_filter_str", "")
            ip_filter_regex = file_system_mount_cmd.get("key_info_find_regex", "")
            ips = self.file_parser.load_info_by_col_regex(cmd, key_line_regex, ip_filter_regex)
            if version == "ipv6":
                for item in ips:
                    ips.discard(item)
                    ips.add(str(item).lstrip('[').rstrip(']'))
            system_file_ips = system_file_ips.union(ips)

        mounted_ips = []
        for mounted_ip in system_file_ips:
            if mounted_ip:
                mounted_ips.append(mounted_ip)
        log.info("%s: File system share ips on host(%s) of os type(%s) are(%s) "
                 ".", _LOG_PREFIX, self.host_id, self._OS_TYPE, mounted_ips)
        return mounted_ips


class Linux(Host):
    _OS_TYPE = "Linux"
    _SYS_VER_CMD = {"Cmd": r"cat /etc/\*-release", "line_filter_str": "",
                    "key_info_find_regex": ""}
    _SYS_KERNEL_VER_CMD = {"Cmd": r"uname -r", "line_filter_str": "",
                           "key_info_find_regex": ""}
    _INITIATOR_INFO_CMD = {"Cmd": "upadmin show path",
                           "line_filter_str": r".*{0}.*",
                           "key_info_find_regex": ""}
    _ULTRA_PATH_VER_CMD = {"Cmd": "upadmin show version",
                           "line_filter_str": r".*Software Version.*",
                           "key_info_find_regex": r"\d+.\d+.\d+"}
    _SHOW_VLUN_CMD = {"Cmd": "upadmin show vlun",
                      "line_filter_str": "{0}",
                      "key_info_find_regex": ""}

    def _check_system_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_hba_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_ultra_path_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_sys_ver_compatible_for_cent_os_and_red_hat(self):
        if not self.have_huawei_file_system:
            log.info("%s: No file system is associated between the host(%s) "
                     "and array, no need to check the system compatibility.",
                     _LOG_PREFIX, self.host_id)
            return CheckResultInfo(self.host_id, "")

        if not self.system_ver or not self.system_kernel_ver:
            log.error("%s: System version or kernel version is empty on "
                      "host(%s), can not check the system compatibility. ",
                      _LOG_PREFIX, self.host_id)
            return CheckResultInfo(self.host_id,
                                   self.host_id,
                                   CheckResultType.CAN_NOT_CHECK,
                                   FailedReason.SYS_VER_UNAVAILABLE)

        if self.system_ver not in self._RISK_HOST_VERSION_TUPLE:
            log.info("%s: System version(%s) of host(%s) is not in the risk "
                     "host system version(%s), system compatible check pass.",
                     _LOG_PREFIX, self.system_ver, self.host_id,
                     self._RISK_HOST_VERSION_TUPLE)
            return CheckResultInfo(self.host_id, "")

        if not self.risk_kernel_version_cmp():
            log.info("%s: Kernel version(%s) of host(%s) is not in the risk "
                     "host kernel version(%s), system compatible check pass.",
                     _LOG_PREFIX, self.system_kernel_ver, self.host_id,
                     self._RISK_KERNEL_VERSION_DICT)
            return CheckResultInfo(self.host_id, "")

        log.warning("%s: Host(%s), os type(%s) system version check not pass, "
                    "system version(%s), kernel version(%s).", _LOG_PREFIX,
                    self.host_id, self._OS_TYPE, self.system_ver,
                    self.system_kernel_ver)
        return_detail_info = "{0}|{1}|{2}".format(self.host_id,
                                                  self.system_ver,
                                                  self.system_kernel_ver)
        return CheckResultInfo(self.host_id,
                               return_detail_info,
                               CheckResultType.NOT_PASS,
                               FailedReason.SYS_VER_NOT_COMPATIBLE)

    def _load_huawei_file_system(self):
        """
        :return:
        """
        return super(Linux, self)._load_huawei_file_system()

    def risk_kernel_version_cmp(self):
        """
        比较当前内核版本是否在对应的内核版本区间内。正则匹配分割区间以所有非字母、非数字、非汉字、非_。
        如果在_RISK_KERNEL_VERSION_DICT变量所标识的区间则返回True；否则返回False,区间比较的位数以最小区间的位数为准。
        :return: True or False
        """
        min_versions = [int(v) for v in re.split(r'\W', self._RISK_KERNEL_VERSION_DICT.get("Min")) if v.isdigit()]
        max_versions = [int(v) for v in re.split(r'\W', self._RISK_KERNEL_VERSION_DICT.get("Max")) if v.isdigit()]
        kernel_ver = [int(v) for v in re.split(r'\W', self.system_kernel_ver) if v.isdigit()][0:len(min_versions)]
        return all([min_versions[i] <= v <= max_versions[i] for i, v in enumerate(kernel_ver)])


class SUSE(Linux):
    _OS_TYPE = "SUSE"
    _SYS_VER_CMD = {"Cmd": r"cat /etc/\*-release",
                    "line_filter_str": r"^\s*VERSION.*",
                    "key_info_find_regex": r"VERSION\s*=\s*(\d+)"}
    _RISK_KERNEL_VERSION_DICT = {"Min": "3.0.76", "Max": "3.0.100"}

    def _check_system_compatible(self):
        cmp_len = len(self._RISK_KERNEL_VERSION_DICT.get('Min').split('.'))
        kernel_ver = re.split(r'\W', self.system_kernel_ver)[0:cmp_len]
        if len(kernel_ver) < cmp_len:
            log.error("%s: System version or kernel version is empty on host(%s), "
                      "can not check the system compatibility. ", _LOG_PREFIX, self.host_id)
            return CheckResultInfo(self.host_id, self.host_id, CheckResultType.CAN_NOT_CHECK,
                                   FailedReason.SYS_VER_UNAVAILABLE)

        if not self.risk_kernel_version_cmp():
            return CheckResultInfo(self.host_id, "")

        log.error("%s: System kernel version (%s) is in [(%s), (%s)], the system incompatible. ", _LOG_PREFIX,
                  kernel_ver, self._RISK_KERNEL_VERSION_DICT.get('Min'),
                  self._RISK_KERNEL_VERSION_DICT.get('Max'))

        return CheckResultInfo(self.host_id, '{0}|{1}'.format(str(self.host_id), self.system_kernel_ver),
                               CheckResultType.NOT_PASS, FailedReason.SYSTEM_INFO_NOT_COMPATIBLE)


class CentOS(Linux):
    _OS_TYPE = "CentOS"
    _SYS_VER_CMD = {"Cmd": r"cat /etc/\*-release",
                    "line_filter_str": r".*CentOS Linux release.*|"
                                       r".*CentOS release.*",
                    "key_info_find_regex": r"\d+.\d+"}
    _RISK_HOST_VERSION_TUPLE = ("7.6",)
    _RISK_KERNEL_VERSION_DICT = {
        "Min": "3.10.0-957.0.0.el7", "Max": "3.10.0-957.5.1.el7"
    }

    def _check_system_compatible(self):
        return self._check_sys_ver_compatible_for_cent_os_and_red_hat()


class RedHat(Linux):
    _OS_TYPE = "RedHat"
    _SYS_VER_CMD = {"Cmd": r"cat /etc/\*-release",
                    "line_filter_str": r".*VERSION_ID.*",
                    "key_info_find_regex": r"\d+.\d+"}
    _RISK_HOST_VERSION_TUPLE = ("7.6",)
    _RISK_KERNEL_VERSION_DICT = {
        "Min": "3.10.0-957.0.0.el7", "Max": "3.10.0-957.5.1.el7"
    }

    def _check_system_compatible(self):
        return self._check_sys_ver_compatible_for_cent_os_and_red_hat()


class VMware(Host):
    _OS_TYPE = "ESXi"
    _SYS_VER_CMD = {"Cmd": "vmware \-v",
                    "line_filter_str": "",
                    "key_info_find_regex": r"\d+.\d+"}
    _HBA_CARD_VENDOR_CMD = {"Cmd": "esxcfg-scsidevs \-a",
                            "line_filter_str": "",
                            "key_info_find_regex": ""}
    _HBA_CARD_DRI_VER_CMD = {"Cmd": "/usr/lib/vmware/vmkmgmt_keyval/vmkmgmt_keyval"
                                    " \-l \-i {0}/Emulex \| grep \-i \"Emulex LightPulse\"",
                             "line_filter_str": "",
                             "key_info_find_regex": ""}
    _HBA_CARD_FW_VER_CMD = {"Cmd": "/usr/lib/vmware/vmkmgmt_keyval/vmkmgmt_keyval"
                                   " \-l \-i {0}/Emulex \| grep \-i \"FW Version\"",
                            "line_filter_str": "",
                            "key_info_find_regex": ""}
    _HBA_CARD_TYPE_CMD = {"Cmd": "/usr/lib/vmware/vmkmgmt_keyval/"
                                 "vmkmgmt_keyval \-l \-i {0}/Emulex \| grep \-i "
                                 "\"Fibre Channel Adapter\"",
                          "line_filter_str": "",
                          "key_info_find_regex": ""}
    _HBA_CARD_DRIVER_CMD = {"Cmd": "", "line_filter_str": "",
                            "key_info_find_regex": ""}
    _ULTRA_PATH_VER_CMD = {"Cmd": "esxcli upadm show version",
                           "line_filter_str": "Software Version",
                           "key_info_find_regex": "r\\d+.\\d+.\\d+"}
    _INITIATOR_INFO_CMD = {"Cmd": "esxcli upadm show path",
                           "line_filter_str": r".*{0}.*",
                           "key_info_find_regex": ""}
    _SHOW_VLUN_CMD = {"Cmd": "esxcli upadm show vlun -t all",
                      "line_filter_str": r".*{0}.*",
                      "key_info_find_regex": ""}

    _RISK_HOST_VERSION_TUPLE = ()
    _RISK_KERNEL_VERSION_DICT = {}
    # blacklist xml file path
    _HBA_BLACK_LIST_FILE_PATH = os.path.join(os.path.dirname(
        os.path.abspath(__file__)), "hba_blacklist.xml")
    # hba vendor
    _VENDOR = ['Emulex']

    def __init__(self, host_info_file, array_name,
                 array_hosts, array_logic_ports, array_lun_wwns):
        # name of the hba in the os
        self._hba_card_name = []
        self._hba_name_vendor = []
        self._hba_fw_dri_ver = []
        self._hba_type = []
        self.hba_name_vendor = {}
        self.hba_fw_dri_ver = {}
        self.hba_type = {}
        super(VMware, self).__init__(host_info_file, array_name,
                                     array_hosts, array_logic_ports,
                                     array_lun_wwns)

    def _load_host_info(self):
        self.system_ver = self.file_parser.load_key_info(
            self._SYS_VER_CMD)
        self.ultra_path_ver = self.file_parser.load_key_info(
            self._ULTRA_PATH_VER_CMD)
        self._load_lun_and_file_system_info()

    def _get_hba_info(self):
        """
        get hba name、vendor、firmware version、driver version、type,
        some of these values may be ''
        eg: {'vendor': 'Hewlett',
             'fw_version': '',
             'dri_version': '',
             'type': ''}

        :return: {key: value}
        key is hba name
        value = {'vendor': vendor,
                 'fw_version': firmware version,
                 'dri_version': driver version,
                 'type': type}
        """
        all_hba_info = {}

        self._get_hba_name_vendor()
        self._get_hba_fw_dri_ver()
        self._get_hba_type()

        # Constructing Data Structure
        for k, v in self.hba_name_vendor.items():
            fw_dri_ver = self.hba_fw_dri_ver.get(k)
            hba_info = {'vendor': v,
                        'fw_version': fw_dri_ver[0],
                        'dri_version': fw_dri_ver[1],
                        'type': self.hba_type.get(k)}
            all_hba_info.update({k: hba_info})
        return all_hba_info

    def _get_hba_name_vendor(self):
        """
        :return: {key: value}
        key is hba name,value is vendor of hba
        """
        cmd = self._HBA_CARD_VENDOR_CMD.get("Cmd")
        name_vendor = self.file_parser.load_info_by_cmd(cmd)
        log.debug("{0} exec: {1} to get hba name and "
                  "vendor: {2}".format(_LOG_PREFIX, cmd, name_vendor))

        if not name_vendor:
            log.warning("{0}: No hba card.".format(_LOG_PREFIX))
            return self.hba_name_vendor

        for hba in name_vendor:
            fc_info = hba.find(r'Fibre Channel')
            is_fc = False if not fc_info else True
            if not is_fc:
                continue

            # get hba card name and vendor
            info = str(hba).split()
            if info[2] != 'link-up':
                continue
            name = info[0]

            pat = '{0[0]}'.format(self._VENDOR)
            vendor = re.search(pat, info[5])
            vendor = '' if not vendor else vendor.group(0)
            if not vendor:
                continue

            self.hba_name_vendor.update({name: vendor})
            self._hba_card_name.append(name)
        return self.hba_name_vendor

    def _get_hba_fw_dri_ver(self):
        """
        get hba card Firmware and Driver versions
        :return:{key: value}
        key is hba name,
        value = [Firmware version, Driver version]
        """
        for hba_name in self._hba_card_name:
            dri_cmd = self._HBA_CARD_DRI_VER_CMD.get('Cmd').format(hba_name)
            dri_ver = self.file_parser.load_info_by_cmd(dri_cmd)
            log.debug("{0} exec: {1} to get {2} firmware and driver "
                      "version: {3}".format(_LOG_PREFIX, dri_cmd,
                                            hba_name, dri_ver))

            if dri_ver:
                info = dri_ver[0].split()[-1]
                hba_dri_ver = info.split("-")[0]
            else:
                log.warning("{0}: Failed to detect the HBA "
                            "version.".format(_LOG_PREFIX))
                raise UnExpectValueError()

            fw_cmd = self._HBA_CARD_FW_VER_CMD.get('Cmd').format(hba_name)
            fw_ver = self.file_parser.load_info_by_cmd(fw_cmd)
            log.debug("{0} exec: {1} to get {2} firmware and driver "
                      "version: {3}".format(_LOG_PREFIX, fw_cmd,
                                            hba_name, hba_dri_ver))

            if fw_ver:
                info = re.sub(r'\s+', '', fw_ver[0]).split(':')
                hba_fw_ver = info[-1]
            else:
                log.warning("{0}: Failed to detect the HBA "
                            "version.".format(_LOG_PREFIX))
                raise UnExpectValueError()

            self.hba_fw_dri_ver.update({hba_name: [hba_fw_ver, hba_dri_ver]})

    def _get_hba_type(self):
        """
        get hba type
        :return: {key:value}
        key is hba name, value is type
        """
        for hba_name in self._hba_card_name:
            cmd = self._HBA_CARD_TYPE_CMD.get('Cmd').format(hba_name)
            type_info = self.file_parser.load_info_by_cmd(cmd)
            log.debug("{0} exec: {1} to get {2} type: {3}".
                      format(_LOG_PREFIX, cmd, hba_name, type_info))

            if type_info:
                type_info = type_info[0].split()[1]
                type_name = type_info.split("-")[0]
            else:
                log.warning("{0}: Failed to detect the HBA "
                            "type.".format(_LOG_PREFIX))
                raise UnExpectValueError()
            self.hba_type.update({hba_name: type_name})

    def _get_blacklist(self):
        """
        parsing blacklist
        :return:{key1: {key2:value}}
        key1 is No.
        key2 is para
        eg: Operating_System: VMware
            Operating_System_version: 5.5||6.0
            Host_Bus_Adapter:  LPe15000||LPe16000
            HBA_driver_version: 0--10.2.455.0
            Solution_zh: 问题原因： HBA卡驱动存在已知风险，会导致目标器注册无法完全建立
                                  连接或无法向上层多路径软件(NMP)触发链路失效。
                                  HBA驱动版本必须为11.1.145.18及以上。
            Solution_en: xxx
        """
        blacklist = {}
        tree = ET.parse(self._HBA_BLACK_LIST_FILE_PATH)
        root = tree.getroot()
        if len(root) > 0:
            for child in root:
                child_info = {"Operating_System": child[1].text,
                              "Operating_System_version": child[2].text,
                              "Host_Bus_Adapter": child[3].text,
                              "HBA_driver_version": child[4].text,
                              "Solution_zh": child[5].text,
                              "Solution_en": child[6].text}
                blacklist.update({child[0].text: child_info})
        return blacklist

    def _exec_hba_check(self, all_hba_info, blacklist):
        failed_hba = []
        compatible = True
        for k, v in all_hba_info.items():
            hba_type = v.get('type')
            dri_version = list(map(int, v.get('dri_version').split('.')))

            # Blacklist-based one-by-one verification
            for rule_id, rule in blacklist.items():
                # adapter
                adapter_list = rule.get('Host_Bus_Adapter').split('||')
                if not hba_type:
                    continue
                if hba_type not in adapter_list:
                    log.debug("{0}: {1} not in {2}".format(_LOG_PREFIX,
                                                           hba_type,
                                                           adapter_list))
                    continue

                # os version
                rule_sys_ver = rule.get('Operating_System_version')
                if rule_sys_ver != 'all':
                    rule_sys_ver = rule_sys_ver.split('||')
                if not self.system_ver:
                    continue
                if rule_sys_ver != 'all' and \
                        self.system_ver not in rule_sys_ver:
                    log.debug("{0}: {1} not in {2}".format(_LOG_PREFIX,
                                                           self.system_ver,
                                                           rule_sys_ver))
                    continue

                # hba driver version
                rule_driver_version = rule.get('HBA_driver_version')
                driver_version = rule_driver_version.split('--')
                start_dri_ver = list(map(int, driver_version[0].split('.')))
                end_dri_ver = list(map(int, driver_version[1].split('.')))
                if not dri_version:
                    continue
                if dri_version < start_dri_ver or dri_version > end_dri_ver:
                    log.debug("{0}: {1} not in {2}".
                              format(_LOG_PREFIX,
                                     dri_version,
                                     rule_driver_version))
                    continue

                failed_hba.append({'name': k, 'type': hba_type,
                                   'dri_version': v.get('dri_version'),
                                   'rule_id': rule_id,
                                   'fw_version': v.get('fw_version')})
                compatible = False
        return failed_hba, compatible

    def _check_hba_compatible(self):
        """
        check hba compatible base on the blacklist
        Based on the type、driver version and system version、 parameters
        Operating System default is VMware Time complexity: n*n, to be optimized later
        :return: True or False
        """
        try:
            log.debug('{0}: checking hba compatible'.format(_LOG_PREFIX))

            blacklist = self._get_blacklist()
            if not blacklist:
                return CheckResultInfo(self.host_id, "")

            all_hba_info = self._get_hba_info()
            if not all_hba_info:
                return CheckResultInfo(self.host_id, "")

            failed_hba, compatible = self._exec_hba_check(all_hba_info, blacklist)

            return self._ret_res_info(compatible, failed_hba, blacklist)
        except CmdExecError as e:
            log.error(e)
            return CheckResultInfo(self.host_id, self.host_id, CheckResultType.CAN_NOT_CHECK,
                                   FailedReason.HBA_INFO_UNAVAILABLE)
        except UnExpectValueError as e:
            log.error(e)
            return CheckResultInfo(self.host_id, self.host_id, CheckResultType.CAN_NOT_CHECK,
                                   FailedReason.HBA_INFO_UNAVAILABLE)
        except IOError as io_e:
            log.error("{0}: failed to parse blacklist file: {1}, reason: "
                      "{2}".format(_LOG_PREFIX,
                                   self._HBA_BLACK_LIST_FILE_PATH, io_e))
            return CheckResultInfo(self.host_id, self.host_id, CheckResultType.CAN_NOT_CHECK, FailedReason.INNER_ERROR)
        except IndexError as index_e:
            log.error("{0}: failed to parse blacklist file: {1}, reason: "
                      "{2}".format(_LOG_PREFIX,
                                   self._HBA_BLACK_LIST_FILE_PATH, index_e))
            return CheckResultInfo(self.host_id, self.host_id, CheckResultType.CAN_NOT_CHECK, FailedReason.INNER_ERROR)
        except Exception as e:
            log.exception(e)
            return CheckResultInfo(self.host_id, self.host_id, CheckResultType.CAN_NOT_CHECK, FailedReason.INNER_ERROR)

    @staticmethod
    def _print_log(hba, blacklist):
        log.warning("{0} the check failed hba, name: {1}, type: {2}, "
                    "driver version: {3}".format(_LOG_PREFIX,
                                                 hba.get('name'),
                                                 hba.get('type'),
                                                 hba.get('dri_version')
                                                 )
                    )

        matched_rule = blacklist.get(hba.get('rule_id'))
        operating_system = matched_rule.get('Operating_System')
        operating_sys_ver = matched_rule.get('Operating_System_version')
        host_bus_adapter = matched_rule.get('Host_Bus_Adapter')
        hba_driver_version = matched_rule.get('HBA_driver_version')
        solution = matched_rule.get('Solution_en')
        log.debug('{0}: rule of the matched blacklist, '
                  'Operating_System:{1}, '
                  'Operating_System_version:{2}, '
                  'Host_Bus_Adapter:{3}, '
                  'HBA_driver_version: {4}'.format(_LOG_PREFIX,
                                                   operating_system,
                                                   operating_sys_ver,
                                                   host_bus_adapter,
                                                   hba_driver_version))
        log.debug('{0}: {1}'.format(_LOG_PREFIX, solution))

    def _ret_res_info(self, compatible, failed_hba, blacklist):
        if compatible:
            return CheckResultInfo(self.host_id, "")
        else:
            log.warning("{0} compatibility check failed".format(_LOG_PREFIX))

            ret_detail_info = {'type': [],
                               'fw_version': [],
                               'dri_version': []}
            for hba in failed_hba:
                self._print_log(hba, blacklist)

                hba_type = hba.get('type')
                ret_hba_type = ret_detail_info.get('type')
                if hba_type not in ret_hba_type:
                    fw_version = hba.get('fw_version')
                    dri_version = hba.get('dri_version')

                    ret_hba_fw_ver = ret_detail_info.get('fw_version')
                    ret_hba_dri_ver = ret_detail_info.get('dri_version')

                    ret_hba_type.append(hba_type)
                    ret_hba_fw_ver.append(fw_version)
                    ret_hba_dri_ver.append(dri_version)

                    ret_detail_info.update({'type': ret_hba_type,
                                            'fw_version': ret_hba_fw_ver,
                                            'dri_version': ret_hba_dri_ver})

            all_hba_type = ','.join(ret_detail_info.get('type'))
            all_hba_fw_version = ','.join(ret_detail_info.get('fw_version'))
            all_hba_dri_version = ','.join(ret_detail_info.get('dri_version'))
            return_detail_info = '{0}|{1}|{2}|{3}'. \
                format(self.host_id,
                       all_hba_type,
                       all_hba_fw_version,
                       all_hba_dri_version)
            return CheckResultInfo(self.host_id,
                                   return_detail_info,
                                   CheckResultType.NOT_PASS,
                                   FailedReason.HBA_NOT_COMPATIBLE)

    def _load_huawei_file_system(self):
        """
        检查主机是否已关联华为文件系统, 通过主机mount回显和
        阵列的show logic port的逻辑IP进行关联判断。
        因为VMware目前无NFS兼容性问题，默认返回无文件系统
        :return:
        """
        self.have_huawei_file_system = False
        return True

    def _check_system_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_ultra_path_compatible(self):
        return CheckResultInfo(self.host_id, "")


class AIX(Host):
    _OS_TYPE = "AIX"
    _SYS_VER_CMD = {"Cmd": "oslevel -s",
                    "line_filter_str": "Red Hat",
                    "key_info_find_regex": "r\\d+.\\d+"}

    _INITIATOR_INFO_CMD = {"Cmd": "upadm show phypath",
                           "line_filter_str": ".*{0}.*",
                           "key_info_find_regex": ""
                           }

    def _check_system_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_hba_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_ultra_path_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _load_huawei_file_system(self):
        """
        检查主机是否已关联华为文件系统, 通过主机mount回显和
        阵列的show logic port的逻辑IP进行关联判断。
        因为AIX目前无NFS兼容性问题，默认返回无文件系统
        :return:
        """
        self.have_huawei_file_system = False
        return True


class Solaris(Host):
    _OS_TYPE = "Solaris"
    _SYS_VER_CMD = {"Cmd": "cat /etc/release",
                    "line_filter_str": "Oracle Solaris",
                    "key_info_find_regex": "\\d+.\\d+|\\d+"}
    _SYS_KERNEL_VER_CMD = {"Cmd": r"uname -a", "line_filter_str": "",
                           "key_info_find_regex": "\\d+.\\d+"}

    def _check_system_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_hba_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_ultra_path_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _load_huawei_file_system(self):
        """
        检查主机是否已关联华为文件系统, 通过主机mount回显和
        阵列的show logic port的逻辑IP进行关联判断。
        因为Solaris目前无NFS兼容性问题，默认返回无文件系统
        :return:
        """
        self.have_huawei_file_system = False
        return True


class Windows(Host):
    _OS_TYPE = "Windows"
    _SYS_KERNEL_VER_CMD = {"Cmd": "", "line_filter_str": "",
                           "key_info_find_regex": ""}
    _ULTRA_PATH_VER_CMD = {"Cmd": "upadm show version",
                           "line_filter_str": ".*Software Version.*",
                           "key_info_find_regex": "\\d+.\\d+.\\d+"}

    def _check_system_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_hba_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _check_ultra_path_compatible(self):
        return CheckResultInfo(self.host_id, "")

    def _load_huawei_file_system(self):
        """
        检查主机是否已关联华为文件系统, 通过主机mount回显和
        阵列的show logic port的逻辑IP进行关联判断。
        因为Windows目前无NFS兼容性问题，默认返回无文件系统
        :return:
        """
        self.have_huawei_file_system = False
        return True


class HostGenerator(object):
    """
    根据主机类型名称，生成主机对象
    """

    @classmethod
    def gen_host_instance(cls, host_type, host_info_file_path, array_name,
                          array_hosts, array_logic_ports, array_lun_wwns):
        try:
            this_module = sys.modules.get(__name__)
            host_classes = inspect.getmembers(this_module, inspect.isclass)
            for host_class in host_classes:
                if host_class[0] == host_type:
                    host_instance = host_class[1](host_info_file_path,
                                                  array_name,
                                                  array_hosts,
                                                  array_logic_ports,
                                                  array_lun_wwns)
                    log.info("%s: Generate a host, host id(%s).",
                             _LOG_PREFIX, host_instance.host_id)
                    return host_instance
            return None
        except Exception as e:
            log.exception(e)
            return None
