#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.
"""
# @brief   : 2024/5/31
# @File    : mgr_cpu_usage.py
# @Software: Software management infrastructure
"""
import os
import logging
import re
import sys
import time
from itertools import chain

PARTITION_FILE = "/OSM/conf/app_data/ftds/product.ini"
PROC_STAT_FILE = "/proc/stat"
DATA_NODE_FLAG = "/startup_disk/conf/conf_local/data_node"
LOG_PREFIX = "CPU_USAGE"
PARAM_NUM = 7  # 前7个参数
IDEL_NUM = 3  # 第四个元素表示 idle  从系统启动开始累计到当前时刻
"""
cpu0 205289 1703 317970 1678028 10759 53338 20039 0 0 0
参数解释：
•	user (205289) 从系统启动开始累计到当前时刻，用户态的CPU时间（单位：jiffies） ，不包含 nice值为负进程。1jiffies=0.01秒
•	nice (1703) 从系统启动开始累计到当前时刻，nice值为负的进程所占用的CPU时间（单位：jiffies）
•	system (317970) 从系统启动开始累计到当前时刻，核心时间（单位：jiffies）
•	idle (1678028) 从系统启动开始累计到当前时刻，除硬盘IO等待时间以外其它等待时间（单位：jiffies）
•	iowait (10759) 从系统启动开始累计到当前时刻，硬盘IO等待时间（单位：jiffies） ，
•	irq (53338) 从系统启动开始累计到当前时刻，硬中断时间（单位：jiffies）
•	softirq (20039) 从系统启动开始累计到当前时刻，软中断时间（单位：jiffies）
计算方式为 10s后的上述参数 - 当前的参数 使用如下公式计算利用率
CPU0核的利用率=1 - 1678028 / (205289 + 1703 + 317970 + 1678028 +10759 + 53338 + 20039)=1 - 0.73368 = 0.26632
"""


def mgr_cpu_list():
    if not os.path.exists(PARTITION_FILE):
        log.info("%s: File (%s) not exist .", LOG_PREFIX, PARTITION_FILE)
        return []
    try:
        with open(PARTITION_FILE, 'r') as part_file:
            file_info = part_file.readlines()
        cpu_list = []
        for line in file_info:
            matcher = re.search(r'partition.*DEFAULT:.*:(\S+)', line)
            if not matcher:
                continue
            result = matcher.group(1).strip().split(',')
            cpu_nums = [cpus.split('-') for cpus in result]
            cpu_list.extend(list(chain(*[list(range(int(cpu_num[0]), int(cpu_num[-1]) + 1)) for cpu_num in cpu_nums])))
        return cpu_list
    except Exception:
        log.exception("%s: Get manager cpu num info from %s exception!", LOG_PREFIX, PARTITION_FILE)
    return []


def get_cpu_info(cpu_num_list):
    if not os.path.exists(PROC_STAT_FILE):
        log.info("%s: File (%s) not exist .", LOG_PREFIX, PROC_STAT_FILE)
        return {}
    cpu_dict = {}
    try:
        with open(PROC_STAT_FILE, 'r') as proc_file:
            file_info = proc_file.readlines()
        for line in file_info:
            matcher = re.search(r'cpu(\d+)', line)
            if not matcher:
                continue
            cpu_num = matcher.group(1)
            if int(cpu_num) not in cpu_num_list:
                continue
            num_infos = [int(num) for num in line.split() if num.isdigit()]
            if len(num_infos) < PARAM_NUM:
                continue
            cpu_dict[int(cpu_num)] = num_infos
    except Exception:
        log.exception("%s: Get manager cpu info from %s execption.", LOG_PREFIX, PROC_STAT_FILE)
        return {}
    log.info("%s: Get manager cpu dict %s success.", LOG_PREFIX, cpu_dict)
    return cpu_dict


def mgr_cpu_usage():
    cpu_num_list = mgr_cpu_list()
    interval = 10
    log.info("%s: Manager cpu num list(%s).", LOG_PREFIX, str(cpu_num_list))
    if not cpu_num_list:
        return 0
    # 计算10s内的CPU利用率
    cpu_first_info = get_cpu_info(cpu_num_list)
    if not cpu_first_info:
        return 0
    time.sleep(interval)
    cpu_second_info = get_cpu_info(cpu_num_list)
    if not cpu_second_info:
        return 0
    cpu_loads = []
    try:
        for cpu_num in cpu_num_list:
            diffs = [cpu_second_info.get(cpu_num)[index] - cpu_first_info.get(cpu_num)[index]
                     for index in range(PARAM_NUM)]
            cpu_loads.append((1 - float(diffs[IDEL_NUM]) / float(sum(diffs[:PARAM_NUM]))) * 100)
        log.info("%s: Manager cpu load info list(%s).", LOG_PREFIX, str(cpu_loads))
        if len(cpu_loads) == 0:
            return 0
        return int(sum(cpu_loads) / len(cpu_loads))
    except Exception:
        log.exception("%s: Get manager cpu load from %s execption.", LOG_PREFIX, PROC_STAT_FILE)
    return 0


if __name__ == '__main__':
    log = logging.getLogger("cpu_usage")
    formatter = logging.Formatter(fmt="[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]",
                                  datefmt="%b %d %Y %H:%M:%S")
    file_handler = logging.FileHandler(filename="/OSM/log/cur_debug/message_euler")
    file_handler.setFormatter(formatter)
    log.addHandler(file_handler)
    log.setLevel(logging.INFO)
    if os.path.exists(DATA_NODE_FLAG):
        sys.exit(0)
    cpu_load = mgr_cpu_usage()
    print(cpu_load)
