#!/usr/bin/python
# -*- coding: utf-8 -*-
import logging
import os
import pwd
import sys
import re
import subprocess
import shlex
import yaml

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')

RESULT_FILE_PATH = "tempResult"

# 直接执行CLI命令的字符串格式：
# {0}表示需要执行的CLI命令，
# {1}表示执行的CLI命令的用户名，
# {2}表示命令执行回显重定向文件，用于命令执行后的结果解析
USER_CMD = "cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:root:'"
CLI_CHECK_CMD = "echo -e '{0}' | /ISM/cli/ismcli -u {1}  > '{2}'"
DEV_CHECK_CMD = "sudo /ISM/cli/start.sh -u {0} -c 'change user_mode current_mode " \
                "user_mode=developer;y;y;{1}' > '{2}'"


# 直接执行diagnose命令的字符串格式：
# {0}表示attach的进程id,比如app_data为12，
# {1}表示需要执行的diagnose命令，
# {2}表示命令执行回显重定向文件，用于命令执行后的结果解析
DIAGNOSE_CHECK_CMD = "diagsh --attach=*_{0} --cmd='{1}' > '{2}'"

# 最大循环次数
MAX_LOOP_CNT = 1000

# 涉及的大版本
PRODUCT_SOFTWARE_VERSION_MAP = {
    '6.1.3': '6.1.3',
    '6.1.5': '6.1.5',
}

# 涉及大版本对应的补丁版本
PRODUCT_HOTPATCH_VERSION_MAP = {
    '6.1.3': 16,
    '6.1.5': 10,
}
# 新融合相关设备的Product Model
PRODUCT_HYBRID_V6 = ["OceanStor 5310", "OceanStor 5510", "OceanStor 5510S", "OceanStor 5610",
                     "OceanStor 6810", "OceanStor 18510", "OceanStor 18500K",
                     "OceanStor 18810", "OceanStor 5810-HS", "OceanStor 5320",
                     "OceanStor 5300K", "OceanStor 5500K", "OceanStor 2200", "OceanStor 2220", "OceanStor 5120",
                     "OceanStor 2600", "OceanStor 2620", "OceanStor 5210", "OceanStor 5220"]


def cli(command):
    sub_cmds = command.split(" | ")
    process = None
    for sub_cmd in sub_cmds:
        if process:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdin=process.stdout, stdout=subprocess.PIPE)
        else:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdout=subprocess.PIPE)
    out = process.communicate()
    return out[0].decode("utf-8")


def get_user_name():
    try:
        pwd.getpwnam('admin')
        return 'admin'
    except Exception as exp:
        logging.warning("not find admin: %s", exp)
        output = cli(USER_CMD)
        if not output:
            logging.error('Get user failed')
            return ''
        return output.strip().split(':')[0]


# 获取设备信息（包括大版本号、补丁版本号、控制器名称）
def get_system_info():
    controller_name_s = []
    software_version = ''
    hotpatch_version = ''
    count = 0
    cli_cmd_str = "show upgrade package"
    cli_cmd_str = CLI_CHECK_CMD.format(cli_cmd_str, get_user_name(), RESULT_FILE_PATH)
    result = os.system(cli_cmd_str)
    if result:
        logging.warning("not find the controller.")
        shell_cmd_str = "rm -f " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return software_version, hotpatch_version, controller_name_s

    # 获取cli回显的每一行的信息
    with open(RESULT_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        for i, line in enumerate(lines_array):
            if 'Software Version' in line:
                i += 4
                software_version = lines_array[i].split()[3][0:5]
                while count < MAX_LOOP_CNT:
                    controller_name_s.append(lines_array[i].split()[1])
                    i += 1
                    count += 1
                    if 'HotPatch Version' in lines_array[i]:
                        break

            if 'HotPatch Version' in line:
                hotpatch_version = lines_array[i + 4].split()[3]
    if hotpatch_version == "--":
        hotpatch_version = "0"
    # 回显格式: 6.1.5 SPH10 ['0A', '0B']
    return software_version, hotpatch_version, controller_name_s


# 判断产品型号，是否在检查范围内
def check_product_model():
    cli_cmd_str = "show system general"
    cli_cmd_str = CLI_CHECK_CMD.format(cli_cmd_str, get_user_name(), RESULT_FILE_PATH)
    result = os.system(cli_cmd_str)
    if result:
        logging.warning("not find the controller.")
        shell_cmd_str = "rm -f " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return True
    with open(RESULT_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        for line in lines_array:
            if 'Product Model' in line:
                product_model = line.split(':')[1].strip()
                if product_model not in PRODUCT_HYBRID_V6:
                    return True
    return False


# 判断版本是否在属于检查范围内
def check_version(software_version, hot_patch_version):
    if software_version not in PRODUCT_SOFTWARE_VERSION_MAP:
        return True

    if int(re.findall(r"\d+", hot_patch_version)[0]) < PRODUCT_HOTPATCH_VERSION_MAP.get(software_version):
        return True

    return False


def get_disk_domain_id_s():
    disk_domain_id_s = []
    cli_cmd_str = "show disk_domain general"
    cli_cmd_str = CLI_CHECK_CMD.format(cli_cmd_str, get_user_name(), RESULT_FILE_PATH)
    result = os.system(cli_cmd_str)

    with open(RESULT_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        # 获取id
        for line in lines_array:
            if ("-" in line) or ("Health Status" in line) or ("#" in line):
                continue
            if ("." in line) and ("B" in line):
                dd_id = re.findall(r"\d+.?\d*", line)[0]
                # 去除空格等非数字
                dd_id = ''.join([c for c in dd_id if c in '0123456789'])
                disk_domain_id_s.append(dd_id)
    return disk_domain_id_s


def check_kvs_count(controller_name_s):
    leak_num = 0
    for controller_name in controller_name_s:
        cli_cmd_str = "show kvs count controller={}".format(controller_name)
        cli_cmd_str = DEV_CHECK_CMD.format(get_user_name(), cli_cmd_str, RESULT_FILE_PATH)
        result = os.system(cli_cmd_str)
        if result:
            logging.warning("not find the kvs count.")
            shell_cmd_str = "rm -f " + RESULT_FILE_PATH
            os.system(shell_cmd_str)
            return True
        with open(RESULT_FILE_PATH) as lines_of_file:
            lines_array = lines_of_file.readlines()
            for line in lines_array:
                if 'KVS_DEFRAG_UPDATE_KVM' in line:
                    leak_num += int(line.split()[1])
    if leak_num > 0:
        return False
    return True


def get_cap_info(disk_domain_id):
    is_have_capacity_tier = ''
    perf_layer_cap_ratio = ''
    perf_layer_cap_ratio_threshold = ''
    dia_cmd_str = "dop showDataFlowParam {}".format(disk_domain_id)
    dia_cmd_str = DIAGNOSE_CHECK_CMD.format(12, dia_cmd_str, RESULT_FILE_PATH)
    result = os.system(dia_cmd_str)
    if result:
        logging.warning("not find the disk domain(%s).", disk_domain_id)
        shell_cmd_str = "rm -f " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return True
    with open(RESULT_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        for line in lines_array:
            if 'is have capacity tier' == line.split(':')[0]:
                is_have_capacity_tier = line.split()[-1]
            if 'perf layer cap ratio' == line.split(':')[0]:
                perf_layer_cap_ratio = line.split()[-1]
            if 'perf layer cap ratio threshold' == line.split(':')[0]:
                perf_layer_cap_ratio_threshold = line.split()[-1]
    return int(is_have_capacity_tier), int(perf_layer_cap_ratio), int(perf_layer_cap_ratio_threshold)


def get_average_free_ratio_of_first_four(free_ratio):
    free_ratio_1 = free_ratio[0][0] * 1000 * free_ratio[0][1] / 100
    free_ratio_2 = free_ratio[1][0] * 1000 * free_ratio[1][1] / 100
    free_ratio_3 = free_ratio[2][0] * 1000 * free_ratio[2][1] / 100
    free_ratio_4 = free_ratio[3][0] * 1000 * free_ratio[3][1] / 100
    free_ratio_average = (free_ratio_1 + free_ratio_2 + free_ratio_3 + free_ratio_4) / 4
    return free_ratio_average


def get_free_ratio_by_cul(ssd_free_ratio, hdd_free_ratio, is_have_capacity_tier,
                          perf_layer_cap_ratio, perf_layer_cap_ratio_threshold):
    if is_have_capacity_tier == 0:  # 不存在容量层
        free_ratio = get_average_free_ratio_of_first_four(ssd_free_ratio)
        return 'ssd', free_ratio
    elif is_have_capacity_tier > 0:  # 存在容量层
        if perf_layer_cap_ratio < perf_layer_cap_ratio_threshold:
            free_ratio = get_average_free_ratio_of_first_four(hdd_free_ratio)
            return 'hdd', free_ratio
        else:
            if get_average_free_ratio_of_first_four(ssd_free_ratio) \
                    > get_average_free_ratio_of_first_four(hdd_free_ratio):
                free_ratio = get_average_free_ratio_of_first_four(hdd_free_ratio)
                return 'hdd', free_ratio
            elif get_average_free_ratio_of_first_four(ssd_free_ratio) \
                    < get_average_free_ratio_of_first_four(hdd_free_ratio):
                free_ratio = get_average_free_ratio_of_first_four(ssd_free_ratio)
                return 'ssd', free_ratio
            else:
                free_ratio = get_average_free_ratio_of_first_four(ssd_free_ratio)
                return 'equal', free_ratio
    else:
        return 'ssd', -1


def get_free_ratio(disk_domain_id):
    ssd_free_ratio = []
    hdd_free_ratio = []
    is_have_capacity_tier, perf_layer_cap_ratio, perf_layer_cap_ratio_threshold \
        = get_cap_info(disk_domain_id)
    dia_cmd_str = "bm vdbmgr show diskcap -p {}".format(disk_domain_id)
    dia_cmd_str = DIAGNOSE_CHECK_CMD.format(12, dia_cmd_str, RESULT_FILE_PATH)
    result = os.system(dia_cmd_str)
    if result:
        logging.warning("not find the disk domain(%s).", disk_domain_id)
        shell_cmd_str = "rm -f " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return '', -1
    with open(RESULT_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        for i, disk_cap_info in enumerate(lines_array):
            # 获取SSD的freeRatio
            if 'ShareDataVdb' in disk_cap_info:
                i += 1
                count = 0
                while count <= MAX_LOOP_CNT:
                    if i >= len(lines_array) - 1:
                        break

                    count += 1
                    i += 1
                    next_disk_cap_info = lines_array[i]
                    if 'DpMgrDiskInfo' in lines_array[i]:
                        break
                    next_disk_cap_info_split = next_disk_cap_info.split('|')
                    if len(next_disk_cap_info_split) > 12:
                        share_data_vdb_free_ratio = \
                            float(next_disk_cap_info_split[12].lstrip().rstrip().rstrip('%'))
                        ssd_free_ratio.append([share_data_vdb_free_ratio, float(next_disk_cap_info_split[14])])
                ssd_free_ratio.sort(key=(lambda x: x[0]))

            # 获取HDD的freeRatio
            if 'DpMgrDiskInfo' in disk_cap_info:
                i += 1
                count = 0
                while count <= MAX_LOOP_CNT:
                    if i >= len(lines_array) - 1:
                        break

                    count += 1
                    i += 1
                    next_disk_cap_info = lines_array[i]
                    next_disk_cap_info_split = next_disk_cap_info.split('|')
                    if len(next_disk_cap_info_split) > 12:
                        dp_mgr_disk_info_free_ratio = \
                            float(next_disk_cap_info_split[12].lstrip().rstrip().rstrip('%'))

                        hdd_free_ratio.append([dp_mgr_disk_info_free_ratio, float(next_disk_cap_info_split[14])])
                hdd_free_ratio.sort(key=(lambda x: x[0]))
        free_ratio_type, free_ratio = get_free_ratio_by_cul(ssd_free_ratio, hdd_free_ratio, is_have_capacity_tier,
                                                            perf_layer_cap_ratio, perf_layer_cap_ratio_threshold)
    return free_ratio_type, free_ratio


def get_startup_watermark_by_cul(ssd_info, hdd_info, free_ratio_type):
    if 'ssd' in free_ratio_type:
        startup_watermark = 5000 + ssd_info[0] * ssd_info[1] / 100 / ssd_info[2]
    elif 'hdd' in free_ratio_type:
        startup_watermark = 5000 + hdd_info[0] * hdd_info[1] / 100 / hdd_info[2]
    else:
        startup_watermark = min(5000 + ssd_info[0] * ssd_info[1] / 100 / ssd_info[2],
                                5000 + hdd_info[0] * hdd_info[1] / 100 / hdd_info[2])
    return startup_watermark


def get_startup_watermark(free_ratio_type, disk_domain_id):
    ssd_info = []
    hdd_info = []
    dia_cmd_str = "bm vdbmgr show vdbmgrobj -s {}".format(disk_domain_id)
    dia_cmd_str = DIAGNOSE_CHECK_CMD.format(12, dia_cmd_str, RESULT_FILE_PATH)
    result = os.system(dia_cmd_str)
    if result:
        logging.warning("not find the disk domain(%s).", disk_domain_id)
        shell_cmd_str = "rm -f " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return ''
    with open(RESULT_FILE_PATH) as lines_of_file:
        vdb_mgr_info_s = lines_of_file.readlines()
        for i, vdb_mgr_info in enumerate(vdb_mgr_info_s):
            if 'DpVmPartCapInfo' in vdb_mgr_info:
                i += 1
                count = 0
                while count <= MAX_LOOP_CNT:
                    if i >= len(vdb_mgr_info_s) - 1:
                        break

                    count += 1
                    i += 1
                    next_vdb_mgr_info_split = vdb_mgr_info_s[i].split('|')
                    if len(next_vdb_mgr_info_split) < 13:
                        break
                    # 获取SSD的freeRatio
                    if 'ShareDataVdb' in next_vdb_mgr_info_split[1]:
                        ssd_info.append(int(next_vdb_mgr_info_split[6], 16))  # ssd_hot_spare_quota
                        ssd_info.append(int(next_vdb_mgr_info_split[13]))  # ssd_amplify
                        ssd_info.append(int(next_vdb_mgr_info_split[3], 16))  # ssd_total_size

                    # 获取HDD的freeRatio
                    if 'DpData' in next_vdb_mgr_info_split[1]:
                        hdd_info.append(int(next_vdb_mgr_info_split[6], 16))
                        hdd_info.append(int(next_vdb_mgr_info_split[13]))
                        hdd_info.append(int(next_vdb_mgr_info_split[3], 16))

        startup_watermark = get_startup_watermark_by_cul(ssd_info, hdd_info, free_ratio_type)
        return startup_watermark


def check_free_ratio(startup_watermark, free_ratio):
    if free_ratio < 2000:
        return True
    elif free_ratio < startup_watermark:
        return False
    else:
        return True


def get_result_by_check_free_ratio():
    res = check_product_model()  # 检查产品型号
    if res:
        return True
    software_version, hotpatch_version, controller_name_s = get_system_info()  # 获取软件版本，补丁版本，控制器名称
    res = check_version(software_version, hotpatch_version)  # 检查版本
    logging.info("(mkv_check_rec_free_ratio)check_version %s:%s", software_version, hotpatch_version)
    if res:
        return True
    # 获取dd id
    disk_domain_s = get_disk_domain_id_s()
    if not disk_domain_s:
        return True

    # 检查kvs count是否存在KVS_DEFRAG_UPDATE_KVM的计数
    check_kvs_count_res = check_kvs_count(controller_name_s)
    if check_kvs_count_res:
        return True

    for disk_domain_id in disk_domain_s:
        # 获取空间回收率
        free_ratio_type, free_ratio = get_free_ratio(disk_domain_id)
        logging.info("(mkv_check_rec_free_ratio)get_free_ratio:%s", free_ratio)
        if free_ratio < 0:
            return True

        # 获取启动水位
        startup_watermark = get_startup_watermark(free_ratio_type, disk_domain_id)
        logging.info("(mkv_check_rec_free_ratio)get_startup_watermark:%s", startup_watermark)
        if startup_watermark < 0:
            return True

        # 判断是否有风险
        check_result = check_free_ratio(startup_watermark, free_ratio)
        if not check_result:
            return False
    return True


def main():
    try:
        res = get_result_by_check_free_ratio()
        if res:
            print("True")
        else:
            print("False")
        shell_cmd_str = "rm " + RESULT_FILE_PATH
        os.system(shell_cmd_str)
        return 0

    except Exception as exp:
        logging.exception("mkv_check_rec_free_ratio: %s", exp)
        print("False")
        return 1


if __name__ == '__main__':
    sys.exit(main())
