#!/usr/bin/python
# -*- coding: utf-8 -*-
import json
import logging
import os
import pwd
import sys
import re
import subprocess
import shlex
import yaml

# 功能：判定mkv fpi rc 开关状态是否打开，打开情况下则升级检查不通过，因为有可能导致介质错误
# 实现：阵列上，进入CLI模式获取diskDomain的id列表，遍历ddId列表，获取fpi rc开关的状态，若为打开状态则检查不通过。

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')

RESULT_CLI_FILE_PATH = "tempCheckCLIFpiResult"
RESULT_DIA_FILE_PATH = "tempCheckDiaFpiResult"

# 直接执行diagnose命令的字符串格式：
# {0}表示attach的进程id,比如app_data为12，
# {1}表示需要执行的diagnose命令，
# {2}表示命令执行回显重定向文件，用于命令执行后的结果解析
DIAGNOSE_CHECK_CMD = "diagsh --attach=*_{0} --cmd='{1}' > '{2}'"

# 直接执行CLI命令的字符串格式：
# {0}表示需要执行的CLI命令，
# {1}表示执行的CLI命令的用户名，
# {2}表示命令执行回显重定向文件，用于命令执行后的结果解析
USER_CMD = "cat /etc/passwd | grep '/ISM/cli/ismcli' | grep ':0:root:'"
CLI_CHECK_CMD = "echo -e '{0}' | /ISM/cli/ismcli -u {1}  > '{2}'"

# 存放当前版本信息所在的路径：
CUR_MANIFEST_PATH = '/startup_disk/image/pkg_cur/manifest.yml'
TARGET_VERSION_PATH = '/tmp/check_extend_param'

RETURN_TRUE = 0
RETURN_CONTINUE = 1

TARGET_PRODUCT_VERSION = {
    "6.0.0": 0,
    "6.0.1": 25,
    "6.1.0": 15,
    "6.1.3": 5
}

CUR_PRODUCT_VERSION = [
    "6.0.0",
    "6.0.1",
    "6.1.0",
    "6.1.2",
    "6.1.3"
]


def cli(command):
    sub_cmds = command.split(" | ")
    process = None
    for sub_cmd in sub_cmds:
        if process:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdin=process.stdout, stdout=subprocess.PIPE)
        else:
            process = subprocess.Popen(shlex.split(sub_cmd.strip()), stdout=subprocess.PIPE)
    out = process.communicate()
    return out[0].decode("utf-8")


def get_user_name():
    try:
        pwd.getpwnam('admin')
        return 'admin'
    except Exception as exp:
        logging.warning("not find admin: %s", exp)
        output = cli(USER_CMD)
        if not output:
            logging.error('Get user failed')
            return ''
        return output.strip().split(':')[0]


# 获取diskDomainId列表，字符串处理函数
def get_disk_domain_id_from_txt():
    list_of_diskdomain_id = []
    with open(RESULT_CLI_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        # 获取id
        for line in lines_array:
            if ("-" in line) or ("Health Status" in line) or ("#" in line):
                continue
            if ("." in line) and ("B" in line):
                dd_id = re.findall(r"\d+.?\d*", line)[0]
                # 去除空格等非数字
                dd_id = ''.join([c for c in dd_id if c in '0123456789'])
                list_of_diskdomain_id.append(dd_id)
    return list_of_diskdomain_id


# 从CLI中获取diskDomainId列表
def mkv_check_fpi_rc_stats_get_cli_ddid():
    cli_cmd_str = "show disk_domain general"
    cli_cmd_str = CLI_CHECK_CMD.format(cli_cmd_str, get_user_name(), RESULT_CLI_FILE_PATH)
    result = os.system(cli_cmd_str)
    list_of_disk_domains = get_disk_domain_id_from_txt()
    return list_of_disk_domains


# 获取rc字符串中对应开关是否打开
def find_first_str(given, targets):
    for tar in targets:
        index = given.find(tar, 10, 45)
        if index != -1:
            if tar == '0x0':
                return False
            else:
                return True
    return True


def is_fpi_rc_stats_on():
    targets = ['0x0', '0x1', '0xff']

    # 从文本中获取fpi rc开关对应的位置，并判定是否打开
    with open(RESULT_DIA_FILE_PATH) as lines_of_file:
        lines_array = lines_of_file.readlines()
        for line in lines_array:
            if "rc" not in line:
                continue
            fpi_rc_stats = find_first_str(line, targets)
            if fpi_rc_stats is False:
                return False

    return True


# 从dia中获取rc开关状态
def mkv_check_fpi_rc_stats_dia(diag_cmd_str):
    diag_cmd_str = DIAGNOSE_CHECK_CMD.format(12, diag_cmd_str, RESULT_DIA_FILE_PATH)
    result = os.system(diag_cmd_str)
    return is_fpi_rc_stats_on()


def clear_temp_file():
    shell_cmd_str = "rm -f " + RESULT_CLI_FILE_PATH
    os.system(shell_cmd_str)
    shell_cmd_str = "rm -f " + RESULT_DIA_FILE_PATH
    os.system(shell_cmd_str)
    return


# 获取升级前工勘的目标版本与目标补丁版本
def get_target_sys_version_new():
    target_product_version = ""
    target_patch_version = 0
    try:
        with open(TARGET_VERSION_PATH) as file_handle:
            cfg_json = json.loads(str(file_handle.read()).replace('\'', '"'))
            target_spc_version = cfg_json['targetVersion']
            if len(target_spc_version.split('.SPC')) > 1:
                target_product_version = target_spc_version.split('.SPC')[0]
                target_patch_version = int(target_spc_version.split('.SPC')[1])
            else:
                target_product_version = target_spc_version
            return target_product_version, target_patch_version
    except Exception as exp:
        logging.exception("CHECK_FUNC: Failed to get version.(%s)", exp)
        return target_product_version, target_patch_version


# 获取当前版本，找不到路径及对应字段时返回空字符
def get_cur_sys_version_new():
    try:
        with open(CUR_MANIFEST_PATH) as file_handle:
            cfg_yml = yaml.safe_load(file_handle)
            cur_spc_version = str(cfg_yml.get("SYS")["SpcVersion"])
            return cur_spc_version
    except Exception as exp:
        logging.exception("CHECK_FUNC: Failed to get version.(%s)", exp)
        return ""


# 检查当前版本是否需要判断开关状态
def check_cur_version():
    product_version, patch_version = get_target_sys_version_new()
    cur_version = get_cur_sys_version_new()

    # 如果获取不到目标版本，则使用当前版本判断
    # 目标版本 小于或等于 当前版本，则使用当前版本判断
    if product_version \
            and (product_version > cur_version):
        if product_version in TARGET_PRODUCT_VERSION and TARGET_PRODUCT_VERSION[product_version] > patch_version:
            return RETURN_CONTINUE
        return RETURN_TRUE
    logging.info("mkv check target version, product_version(%s), patch_version(%d).", product_version, patch_version)
    logging.info("mkv check cur version, cur_version(%s).", cur_version)
    if cur_version in CUR_PRODUCT_VERSION or not cur_version:
        return RETURN_CONTINUE
    return RETURN_TRUE


def mkv_check_fpi_rc_stats():
    # 首先检查当前版本
    check_version_result = check_cur_version()
    if check_version_result == RETURN_TRUE:
        print("True")
        return True

    list_of_diskdomain_id = mkv_check_fpi_rc_stats_get_cli_ddid()
    # 没有diskDomain的情况下不会有开关，检查通过
    if len(list_of_diskdomain_id) == 0:
        print("True")
        return True

    # 遍历diskDomain，查看fpi rc开关的状态
    for dd_id in list_of_diskdomain_id:
        diag_cmd_str = "pmgr showPoolTaskSwitch -d " + dd_id
        stats = mkv_check_fpi_rc_stats_dia(diag_cmd_str)
        if stats is False:
            clear_temp_file()
            print("False")
            return False

    clear_temp_file()
    print("True")
    return True


def main():
    try:
        mkv_check_fpi_rc_stats()
        return 0

    except Exception as exp:
        logging.exception("mkv_check_fpi_rc_stats: %s", exp)
        print("False")
        return 1


if __name__ == '__main__':
    sys.exit(main())
