#!/usr/bin/python
# coding=utf-8

import subprocess
import re
import datetime
import time
import logging
import sys
import os

logging.getLogger().setLevel(logging.INFO) # 设置根logger的级别
console = logging.StreamHandler(stream=sys.stdout)
console.setLevel(logging.INFO)
logging.getLogger().addHandler(console)


# 获取系统时间，单位（秒）
def get_sys_time():
    with open('/proc/uptime', 'r') as f:
        uptime_seconds = float(f.readline().split()[0])
    return uptime_seconds


class PreupgradeCheck:
    def __init__(self):
        self.notify = 'dpl notifypreupgradecheck'
        self.get = 'dpl getpreupgradecheck'

    @classmethod
    def diagnose_usr(cls, command, timeout=30):
        starttime = get_sys_time()
        devm_file_path = "/OSM/script/get_devm_pro_num.sh"
        if os.path.exists(devm_file_path):
            # 执行sh脚本查询devm进程号 | 在shell脚本中第一行必须指定脚本解释器，如：#!/bin/sh。否则必须使用sh命令显示调用
            dev = subprocess.Popen(["/bin/sh", devm_file_path, ], shell=False, stdout=subprocess.PIPE)
            # 为了兼容python2，因此保留poll的调用
            while dev.poll() is None:
                time.sleep(0.1)
                endtime = get_sys_time()
                # 如果达到超时时间，则退出
                if (endtime - starttime) > timeout:
                    return "False"
            stdout = dev.communicate()
            devm_info = stdout[0].decode("utf-8")
            match = re.search("Devm Pro num is ([0-9]+)", devm_info)
            if match:
                devm_num = match.group(1)
            else:
                devm_num = "21"
        else:
            devm_num = "21"
        # 执行命令：'diagsh --attach=*_%s --cmd="%s" | grep -v "diagsh>"' % (devm_num, command)
        diagsh_cmd = ['diagsh', '--attach=*_{}'.format(devm_num), ] + ['--cmd={}'.format(command)]
        grep_cmd = ['grep', '-v', 'diagsh>']
        cmd_list = [diagsh_cmd, grep_cmd]
        stdout = PreupgradeCheck.execute_muti_command(cmd_list)
        cmd_info = stdout[0].decode("utf-8").split('\n')
        return cmd_info

    @classmethod
    def execute_muti_command(cls, command_list, timeout=50):
        '''
        功能描述：
                1. 作为excute_command的补充，执行多个组合命令。即通过此种方式替换shell=True场景下，使用管道符存在安全的问题
                2. 注意：所传入的命令不能包含特殊字符，包括: *, $ 等在shell中有特殊意义的符号。
                3. command_list: [命令1的列表形式, 命令2的列表形式, ...]
        '''
        pre_popen = None
        starttime = get_sys_time()
        for command in command_list:
            if pre_popen is None:
                fd = subprocess.Popen(command, shell=False, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                pre_popen = fd
            else:
                fd = subprocess.Popen(command, shell=False, stdin=pre_popen.stdout,
                     stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                pre_popen = fd
        while pre_popen.poll() is None:
            time.sleep(0.1)
            endtime = get_sys_time()
            # 如果达到超时时间，则退出
            if (endtime - starttime) > timeout:
                raise Exception('run commands timout, the commands is :{}'.format(command_list))
        out = pre_popen.communicate()
        return out

    # 通知升级前检查
    def notify_pre_upgrade_check(self):
        ret = "False"
        stdouts = PreupgradeCheck.diagnose_usr(self.notify)
        # 通过回显判断是否通知完成
        if re.search('finished', stdouts[0], re.IGNORECASE):
            ret = "True"
        elif re.search("found", stdouts[0], re.IGNORECASE):
            ret = 'commond_invalid'
        return ret

    # 获取升级前检查结果
    def get_pre_upgrade_check(self):
        ret = "False"
        stdouts = PreupgradeCheck.diagnose_usr(self.get)
        if re.search('True', stdouts[0]):
            ret = "True"
        elif re.search('Fault', stdouts[0], re.IGNORECASE):
            return stdouts[0]
        elif re.search('Doing', stdouts[0], re.IGNORECASE):
            return "Doing"
        return ret

    # 通知升级前检查重试，如果结果不为ok，则重试。
    # 每次重试为50s，重试间隔为10s，重试次数为5次
    def notify_pre_upgrade_check_retry(self, timeout=60):
        invalid_retry = 0
        for _ in range(5):
            start = get_sys_time()
            ret = self.notify_pre_upgrade_check()
            # 如果DMI不支持，则返回成功
            if ret == 'commond_invalid':
                # 进程故障时，接口返回不支持。所以需要进行一次重试来覆盖
                if invalid_retry > 0:
                    logging.info('True\n\nNot Support')
                    return False
                invalid_retry += 1
            # 如果通知结果为ok，则返回成功
            elif ret == "True":
                return True
            end = get_sys_time()
            # 如果未达到超时时间（60s），则休眠;超过则继续重试
            if (end - start) < timeout:
                time.sleep(timeout - (end - start))
        # 如果重试后依然失败，则返回失败并打印
        logging.info('False\n\nError')
        return False

    # 获取升级前检查结果重试，如果结果不为全ok或者故障，则重试。
    # 每次重试为50s，重试间隔为10s，重试次数为5次
    def get_pre_upgrade_check_result_retry(self, timeout=60):
        check = False
        for _ in range(5):
            start = get_sys_time()
            ret = self.get_pre_upgrade_check()
            if ret == "True":
                logging.info('True\n\n')
                check = True
                return check
            else: # 失败和正在检查都重试；检查618的bmc时，超过20s bmc仍返回检查中则返回fault，重试可以成功，所以失败也需要重试
                end = get_sys_time()
                # 如果未达到超时时间（60s），则休眠;超过则继续重试
                if (end - start) < timeout:
                    time.sleep(timeout - (end - start))
        # 返回失败,则打印失败
        logging.info('False\n\nError')
        return False


if __name__ == "__main__":
    # 1）下发通知，通知失败则退出，内部有打印
    result = PreupgradeCheck().notify_pre_upgrade_check_retry()
    # 2)下发通知成功，则获取通知检查结果
    # 下发后，驱动可能不能立即返回结果，所以这里等待5s后再去获取结果
    if result:
        time.sleep(5)
        PreupgradeCheck().get_pre_upgrade_check_result_retry()
