#!/bin/bash
#包含头脚本
source /OSM/script/os_load_base.sh

if [ -f /usr/bin/root_proxy ]; then
  ROOT_PROXY="/usr/bin/root_proxy --proxy-ignore-insecure 1 --proxy-shname $0 --proxy-lineno $LINENO"
fi

UPGRADE_SH=upgrade_preck.sh
G_CUR_YML_FILE=/startup_disk/image/pkg_cur/manifest.yml
G_PKG_YML_FILE=/startup_disk/image/pkg_upd/manifest.yml
G_DISK_UTIL=/tmp/diskusagetmp
G_DISK_ID=/tmp/diskidtmp
G_DISK_INFO=/tmp/diskinformation
G_ULTRA_CHECK=/tmp/ultraCheck
G_CPU_USAGE=/tmp/cpuusagetmp
G_IO_STATE=/tmp/iostatetmp
G_BSP_INFO=/OSM/script/proc_osp_bsp.info
CUR_VERSION=$(cat /OSM/conf/manifest.yml |grep " Version:" | ${ROOT_PROXY} awk '/Version:/ {print $2}')
QOS_CHECK_CPU_USAGE_VERSION=7600500136
D2000_ONE_PACKAGE_FIRST_VERSION=7600518068
PRODUCT_MODEL_SCRIPT=/OSM/script/product_model.py
MGR_CPU_USAGE=/OSM/coffer_bin/upg_check/check_item/mgr_cpu_usage.py
declare -a G_DISK_ID_LIST

OS_ExecDiagnose()
{
    local L_PID=$1
    local L_CMD=$2
    local L_FILE=$3

    if [ "$L_FILE" == "" ]; then
        diagsh --attach="$L_PID" --cmd="$L_CMD"
    else
        diagsh --attach="$L_PID" --cmd="$L_CMD" > ${L_FILE}
        ${ROOT_PROXY} chmod 640 ${L_FILE}
    fi
    return 0
}

# **************************************************************************** #
# Function Name: OS_CHECK_CPU_USAGE_EAM
# Description: 使用 EAM 命令检查 cpu 使用率
# Parameter:
# Return: 0表示成功，1表示失败
# **************************************************************************** #
OS_CHECK_CPU_USAGE_EAM()
{
    [ -f ${G_CPU_USAGE} ] && ${ROOT_PROXY} rm ${G_CPU_USAGE}

    OS_ExecDiagnose "*_21" "eam showcpuusage" ${G_CPU_USAGE}

    local cmd_not_find=`cat ${G_CPU_USAGE} | grep  "Can't Find The Diagnose Cmd."`
    if [ -n "${cmd_not_find}" ];then
        OS_echo "[${UPGRADE_SH}]: command not exist." >/dev/null
        echo "0"
        echo ""
        echo ""
        ${ROOT_PROXY} rm ${G_CPU_USAGE} &>/dev/null
        return 0
    fi

    local l_cur_cpu_using=`cat ${G_CPU_USAGE} | grep -e '^[0-9]\{1,3\}$'`
    if [ -z "${l_cur_cpu_using}" ];then
        OS_echo "[${UPGRADE_SH}]: command execute failed." >/dev/null
        echo "${l_cur_cpu_using}"
        echo ""
        echo ""
        ${ROOT_PROXY} rm ${G_CPU_USAGE} &>/dev/null
        exit 1
    fi

    OS_echo "[${UPGRADE_SH}]: the cpu usage is ${l_cur_cpu_using}." >/dev/null
    echo "${l_cur_cpu_using}"
    echo ""
    echo ""
    ${ROOT_PROXY} rm $G_CPU_USAGE &>/dev/null
    return 0
}
# **************************************************************************** #
# Function Name: OS_CHECK_CPU_USAGE_QOS
# Description: 使用 QOS 命令检查 cpu 使用率
# Parameter:
# Return: 0表示成功，1表示失败
# **************************************************************************** #
OS_CHECK_CPU_USAGE_QOS()
{
    local data_node=`cat ${G_BSP_INFO}|grep -E 'PANGEA_V6R1C00_JBOD_SAS_EXP|PANGEA_V6R1C00_JBOF_PALM_EXP'`
    if [ -n "${data_node}" ];then
        echo "0"
        echo ""
        echo ""
        return 0
    fi
    if [ -f /OSM/conf/cpu_check_hook ]; then
        OS_echo "${UPGRADE_SH} : no_cpu_check flag exist, do not check cpu usage.">/dev/null
        echo "$(cat /OSM/conf/cpu_check_hook)"
        echo ""
        echo ""
        return 0
    fi

    #调用QOS的diagnose命令检查CPU利用率
    [ -f $G_CPU_USAGE ] && ${ROOT_PROXY} rm $G_CPU_USAGE
    local var=0
    for((i=0;i<10;i++))
    do
        OS_ExecDiagnose "*_12" "qos show cur_cpu_load" > $G_CPU_USAGE
        tmpvar=`cat $G_CPU_USAGE |grep "CPU load average" |${ROOT_PROXY} awk '{print $4}'`
        var=$(expr $var + $tmpvar)
        sleep 1
    done
    L_CUR_CPU_USING=$(expr $var / 10 )

    OS_echo "[$UPGRADE_SH]: the cpu usage is $L_CUR_CPU_USING." >/dev/null
    echo "${L_CUR_CPU_USING}"
    echo ""
    echo ""
    return 0
}
# **************************************************************************** #
# Function Name: OS_CHECK_CPU_USAGE
# Description: 检查 cpu 使用率
# Parameter:
# Return: 0表示成功，1表示失败
# **************************************************************************** #
OS_CHECK_CPU_USAGE()
{
    OS_echo "[${UPGRADE_SH}]: begin to check the cpu usage." >/dev/null

    if [ ${CUR_VERSION} -lt ${QOS_CHECK_CPU_USAGE_VERSION} ];then
        OS_CHECK_CPU_USAGE_QOS
    else
        OS_CHECK_CPU_USAGE_EAM
    fi

    return 0
}
# **************************************************************************** #
# Function Name: OS_CHECK_MGR_CPU_USAGE
# Description: 检查管理核 cpu 使用率
# Parameter:
# Return: 0表示成功，1表示失败
# **************************************************************************** #
OS_CHECK_MGR_CPU_USAGE()
{
    local data_node=`cat ${G_BSP_INFO}|grep -E 'PANGEA_V6R1C00_JBOD_SAS_EXP|PANGEA_V6R1C00_JBOF_PALM_EXP'`
    if [ -n "${data_node}" ];then
        OS_echo "[${UPGRADE_SH}]: data node do not check manager cpu load." >/dev/null
        OS_CHECK_CPU_USAGE
        return 0
    fi
    local cpu_load=$(python "${MGR_CPU_USAGE}" 2>/dev/null)
    OS_echo "[${UPGRADE_SH}]: manager cpu load is ${cpu_load}%." >/dev/null
    if [[ $(echo "${cpu_load} >= 98" | bc) -eq 1 ]];then
        OS_echo "[${UPGRADE_SH}]: manager cpu load is greater than 98%." >/dev/null
        echo "${cpu_load}"
        echo ""
        echo ""
        return 0
    else
        OS_CHECK_CPU_USAGE
    fi
    return 0
}
# **************************************************************************** #
# Function Name: OS_CheckIOState
# Description: check if IO exist which leads failure reset.
# Parameter:
# Return:
# **************************************************************************** #
OS_CHECK_IO_STATE()
{
    OS_echo "[$UPGRADE_SH]: begin to check the IO state."

    # check if IO exists using TGT diagnsoe cmd.
    [ -f $G_IO_STATE ] && ${ROOT_PROXY} rm $G_IO_STATE
    for((i=0;i<10;i++))
    do
        OS_ExecDiagnose "*_12" "tgt showiostate" $G_IO_STATE
        tmpvar=`cat $G_IO_STATE |grep "TRUE"`
        if [ ! -z $tmpvar ];then
            OS_echo "[$UPGRADE_SH]: IO exists in 10s."
            echo "False"
            echo ""
            echo ""
            return 0
        fi
        sleep 1
    done

	  OS_echo "[$UPGRADE_SH]: There is no IO durning 10s."
    echo "True"
    echo ""
    echo ""
    return 0
}

# **************************************************************************** #
# Function Name: OS_CheckDiskUsage
# Description: check if the disk usage is more than threshold
# Parameter:
# Return:
# **************************************************************************** #
GetDiskUsage()
{
    OS_ExecDiagnose "*_12" "sddebug userate 60" $G_DISK_UTIL
}

OS_CheckDiskUsage()
{
    local -i L_OneDiskUsage=0
    local -i L_DiskUtilUsageResult=0
    local -i L_DiskCount=0
    local -i L_DiskAverage=0
    local -i L_DiskSum=0
    local -i L_DiskUsageIgnore=10
    
    #检查前降低auto_tiering速度，协议在V5R7C20增加，不关心执行结果,sleep 60s后进行cpu检查（注：检查磁盘利用率是获取前60秒的数据，所有睡眠60秒）
    nas_proto_pum -m auto_tiering -c set_need_control_speed -v 1 &>/dev/null
    OS_echo " this func to check disk usage"
    sleep 60
    [ -f $G_DISK_UTIL ] && ${ROOT_PROXY} rm $G_DISK_UTIL
    GetDiskUsage
    while read sdid util type; 
    do
        if [ -n "$(echo $sdid | grep [a-zA-Z])" -o -z "$sdid" ]; then
            OS_echo "${UPGRADE_SH} : sdid=$sdid"
            continue;
        fi
        L_OneDiskUsage=$util
        OS_echo "${UPGRADE_SH} : sdid=$sdid  util=${L_OneDiskUsage}"
        #将利用率小于等于10的盘排除开，不计算平均值和最大值，可以避免长期利用率为零或者很低的盘对平均值计算的影响
        if [ ${L_DiskUsageIgnore} -gt ${L_OneDiskUsage} ]; then
            OS_echo "${UPGRADE_SH} : sdid=$sdid diskusage less than 10, continue"
            continue;
        fi
        [ ${L_OneDiskUsage} -gt ${L_DiskUtilUsageResult} ] && L_DiskUtilUsageResult=${L_OneDiskUsage}
        L_DiskCount=$(( $L_DiskCount + 1 ))
        L_DiskSum=$(($L_DiskSum + $L_OneDiskUsage ))
    done < $G_DISK_UTIL &>/dev/null
    
    #获取到的磁盘数不为0的时候，计算磁盘平均利用率
    if [ 0 -ne $L_DiskCount ]; then
        L_DiskAverage=$(( $L_DiskSum / $L_DiskCount ))
    fi
    #检查后恢复auto_tiering速度，协议在V5R7C20增加，不关心执行结果
    nas_proto_pum -m auto_tiering -c set_need_control_speed -v 0 &>/dev/null
    echo "upd diskusagecheck"
    echo "${L_DiskUtilUsageResult},${L_DiskAverage}"

    return 0
}

# **************************************************************************** #
# Function Name: OS_CheckSSDAndSASDiskUsage
# Description: check SSD and SAS disk usage
# Parameter:
# Return:
# **************************************************************************** #
OS_CheckSSDAndSASDiskUsage()
{
    local -i L_OneDiskUsage=0
    local -i L_SSDDiskUtilUsageResult=0
    local -i L_SSDDiskCount=0
    local -i L_SSDDiskAverage=0
    local -i L_SSDDiskSum=0
    
    local -i L_SASDiskUtilUsageResult=0
    local -i L_SASDiskCount=0
    local -i L_SASDiskAverage=0
    local -i L_SASDiskSum=0
    L_DiskType=""
    
    [ -f $G_DISK_UTIL ] && ${ROOT_PROXY} rm $G_DISK_UTIL
    GetDiskUsage
    while read sdid util type; 
    do
        if [ -n "$(echo $sdid | grep [a-zA-Z])" -o -z "$sdid" ]; then
            OS_echo "${UPGRADE_SH} : sdid=$sdid"
            continue;
        fi
        L_OneDiskUsage=$util
        L_DiskType=$type
        OS_echo "${UPGRADE_SH} : sdid=$sdid  util=${L_OneDiskUsage} type=${L_DiskType}"
        if [ "${L_DiskType}" == "SSD" ] || [ "${L_DiskType}" == "SLC_SSD" ] || [ "${L_DiskType}" == "MLC_SSD" ]; then
            [ ${L_OneDiskUsage} -gt ${L_SSDDiskUtilUsageResult} ] && L_SSDDiskUtilUsageResult=${L_OneDiskUsage}
            L_SSDDiskCount=$(( $L_SSDDiskCount + 1 ))
            L_SSDDiskSum=$(($L_SSDDiskSum + $L_OneDiskUsage ))
        else
            [ ${L_OneDiskUsage} -gt ${L_SASDiskUtilUsageResult} ] && L_SASDiskUtilUsageResult=${L_OneDiskUsage}
            L_SASDiskCount=$(( $L_SASDiskCount + 1 ))
            L_SASDiskSum=$(($L_SASDiskSum + $L_OneDiskUsage ))

        fi
    done < $G_DISK_UTIL &>/dev/null
    
    OS_echo "${UPGRADE_SH} : L_SSDDiskCount=$L_SSDDiskCount  L_SASDiskCount=$L_SASDiskCount"
    #计算磁盘平均利用率
    if [ "0" != "$L_SSDDiskCount" ];then
        L_SSDDiskAverage=$(( $L_SSDDiskSum / $L_SSDDiskCount ))
    fi
    if [ "0" != "$L_SASDiskCount" ];then
        L_SASDiskAverage=$(( $L_SASDiskSum / $L_SASDiskCount ))
    fi
    
    if [ ${L_SASDiskUtilUsageResult} -gt 80 -o ${L_SASDiskAverage} -gt 60 -o ${L_SSDDiskUtilUsageResult} -gt 30 -o ${L_SSDDiskAverage} -gt 20 ]; then
        echo "False"
        echo "diskusagehigh"
        echo ""
        return 1
    fi
    echo "True"
    echo ""
    echo ""
    return 0
}
 
#判断当前阵列版本号是否在目标版本号的黑名单中，如果在黑名单中则返回0，即不兼容；不在黑名单则都兼容，返回1
UPGRADE_SYS_NOT_COMPATIBLE()
{
    OS_echo "${UPGRADE_SH} : UPGRADE_SYS_COMPATIBLE">/dev/null
    
    L_CUR_VERSION_STR=`cat $G_CUR_YML_FILE |grep -m 1 Version |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`
    OS_echo "${UPGRADE_SH} : cur version $L_CUR_VERSION_STR">/dev/null

    declare -i L_CUR_VERSION=0
    declare L_VERSION_RANGE=""
    L_PRODUCT_NAME=`cat /proc/osp/bsp |grep -m 1 "Model of products is" |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`
    if [[ "${L_PRODUCT_NAME}" =~ .*_S$ || "${L_PRODUCT_NAME}" =~ .*_S_H$ ]];then
        L_VERSION_RANGE=`cat $G_PKG_YML_FILE |grep -m 1 A8000BlackVersionList |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/[[:space:]]//g' | ${ROOT_PROXY} sed -e 's/^["]*//g' -e 's/["]*$//g'`
    else
        L_VERSION_RANGE=`cat $G_PKG_YML_FILE |grep -m 1 BlackVersionList |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/[[:space:]]//g' | ${ROOT_PROXY} sed -e 's/^["]*//g' -e 's/["]*$//g'`
    fi
    OS_echo "${UPGRADE_SH} : black list $L_VERSION_RANGE">/dev/null

    #将获取的版本号中的.和T去掉, 看做整数, 进行比较
    L_CUR_VERSION=`echo "$L_CUR_VERSION_STR" | ${ROOT_PROXY} sed 's/[.T]//g' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`


    declare -i Index=1
    ComFirst=`echo $L_VERSION_RANGE|${ROOT_PROXY} awk -F";" '{print $'$Index'}'`
    OS_echo "upgrade ComFirst $ComFirst">/dev/null
    while [ -n "$ComFirst" ] ;do
        declare -i ComVerlow=`echo $ComFirst|${ROOT_PROXY} awk -F"-" '{print $1}'`
        OS_echo "upgrade ComVerlow $ComVerlow">/dev/null
        declare -i ComVerhig=`echo $ComFirst|${ROOT_PROXY} awk -F"-" '{print $2}'`
        OS_echo "upgrade ComVerhig $ComVerhig">/dev/null
        #在黑名单中，返回0
        if [ $L_CUR_VERSION -ge $ComVerlow ] && [ $L_CUR_VERSION -le $ComVerhig ];then
                return 0
        fi
        #以封号分割，判断下一个黑名单区间
        Index=Index+1
        ComFirst=`echo $L_VERSION_RANGE|${ROOT_PROXY} awk -F";" '{print $'$Index'}'`
    done
	OS_echo "${UPGRADE_SH} : not in black list">/dev/null
    return 1
}

PACKAGE_TYPE_COMPATIBLE()
{
    L_TARGET_PKG_TYPE=$(grep -m 1 Product ${G_PKG_YML_FILE} |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/ //g')

    L_CUR_PKG_TYPE=$(grep -m 1 Product ${G_CUR_YML_FILE} |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/ //g')

    CHECK_IS_X8000_WITHOUT_PRODUCT ${G_CUR_YML_FILE}
    local ret=$?
    if [ ${ret} -eq 0 ];then
      L_CUR_PKG_TYPE="BackupStorage"
    fi

    CHECK_IS_X8000_WITHOUT_PRODUCT ${G_PKG_YML_FILE}
    local ret=$?
    if [ ${ret} -eq 0 ];then
      L_TARGET_PKG_TYPE="BackupStorage"
    fi

    # 1.0.0/1.1.0/1.2.0 不能往 1.2.1升级
    CHECK_CAN_UPGRADE_TO_X8000_121
    local ret=$?
    if [ ${ret} -eq 1 ]; then
      OS_echo "${UPGRADE_SH} : Cur pkg type is not allowed to upgrade to X8000 1.2.1"
      return 1
    fi

    if [ "$L_TARGET_PKG_TYPE" != "" -a "$L_CUR_PKG_TYPE" != "" -a "${L_TARGET_PKG_TYPE}" != "${L_CUR_PKG_TYPE}" ]; then
        OS_echo "${UPGRADE_SH} : Cur product type ${L_CUR_PKG_TYPE} not match pkg type ${L_TARGET_PKG_TYPE}"
        return 1
    elif [ "$L_CUR_PKG_TYPE" == "" -a "${L_TARGET_PKG_TYPE}" != "" -a "${L_TARGET_PKG_TYPE}" != "Dorado" ]; then
        OS_echo "${UPGRADE_SH} : Dst pkg type is not dorado, It is ${L_TARGET_PKG_TYPE}"
        return 1
    fi

    local is_cur_d2000_str=`cat $G_CUR_YML_FILE | grep -m 1 "PackageName" | grep -o "Dorado_2000"`
    local is_upd_d2000_str=`cat $G_PKG_YML_FILE | grep -m 1 "PackageName" | grep -o "Dorado_2000"`

    L_CUR_VERSION_STR=`cat $G_CUR_YML_FILE |grep -m 1 Version |${ROOT_PROXY} awk -F":" '{print $2}'`
    OS_echo "${UPGRADE_SH} : upgrade:cur version is $L_CUR_VERSION_STR">/dev/null
    declare -i L_CUR_VERSION=0
    L_CUR_VERSION=`echo "$L_CUR_VERSION_STR" | ${ROOT_PROXY} sed 's/[.T]//g' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`

    L_PKG_VERSION_STR=`cat $G_PKG_YML_FILE |grep -m 1 Version |${ROOT_PROXY} awk -F":" '{print $2}'`
    OS_echo "${UPGRADE_SH} : upgrade:pkg version is $L_PKG_VERSION_STR">/dev/null
    declare -i L_PKG_VERSION=0
    L_PKG_VERSION=`echo "$L_PKG_VERSION_STR" | ${ROOT_PROXY} sed 's/[.T]//g' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`

    # 环境包为D2000,目标包为Dorado,目标包的版本高于等于共包时版本：Dorado支持升级到共包
    if [ "${is_cur_d2000_str}" == "Dorado_2000" -a "${L_TARGET_PKG_TYPE}" == "Dorado" -a "${is_upd_d2000_str}" != "Dorado_2000" ]; then
        if [ "${L_PKG_VERSION}" -ge "${D2000_ONE_PACKAGE_FIRST_VERSION}" ]; then
            OS_echo "${UPGRADE_SH} : ${L_PKG_SPC_VERSION_STR} support Dorado_2000 upgrade to Dorado.">/dev/null
            return 0
        fi
    fi
    # 环境包为Dorado,目标包为Dorado_2000, 环境包版本高于等于共包时版本,产品型号为D2000：共包支持降级到D2000
    if [ "${L_CUR_PKG_TYPE}" == "Dorado" -a "${is_cur_d2000_str}" != "Dorado_2000" -a "${is_upd_d2000_str}" == "Dorado_2000" ]; then
        if [ "${L_CUR_VERSION}" -ge "${D2000_ONE_PACKAGE_FIRST_VERSION}" ]; then
            # 共包后才有产品型号脚本
            CUR_PRODUCT_MODEL=$(python "${PRODUCT_MODEL_SCRIPT}")
            if [ "${CUR_PRODUCT_MODEL}" == "D2000" ]; then
                OS_echo "${UPGRADE_SH} : ${L_PKG_SPC_VERSION_STR} version support Dorado rollback to D2000.">/dev/null
            return 0
            fi
        fi
    fi
    # dorado与D2000产品软件包manifest.yml配置中Product均为dorado，通过是否存在PackageName字段差异进行判断dorado和D2000产品型号
    if [ "${is_cur_d2000_str}" != "${is_upd_d2000_str}" ]; then
      OS_echo "${UPGRADE_SH} : D2000 pkg type is not dorado, please check and retry."
      return 1
    fi

    return 0
}

# **************************************************************************** #
# Function Name: CHECK_IS_X8000_WITHOUT_PRODUCT
# Description: X8000 的历史C版本，以及1.2.x的在研B版本，没有Product参数的需添加Product
# Parameter:
# Return:
# **************************************************************************** #
CHECK_IS_X8000_WITHOUT_PRODUCT()
{
  local manifest_yml=$1
  local product_type
  product_type=$(grep -m 1 Product "${manifest_yml}" |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/ //g')

  # 当前版本存在Product或者目标版本不存在Product，直接返回，不做处理
  if [[ -n "${product_type}" ]];then
    return 1
  fi

  # X8000 的历史C版本，以及1.2.x的在研B版本，没有Product参数的需添加Product
  CHECK_HIS_C_VER_IS_X8000 "${manifest_yml}"
  local x8000_ret=$?
  if [[  ${x8000_ret} -eq 0 ]]; then
    return 0
  fi

  CHECK_X8000_120_BY_SPCVER "${manifest_yml}"
  local x8000_120_ret=$?
  CHECK_X8000_121_BY_SPCVER "${manifest_yml}"
  local x8000_121_ret=$?
  if [[ ${x8000_120_ret} -eq 0 || ${x8000_121_ret} -eq 0 ]]; then
    return 0
  fi

  return 1
}

# **************************************************************************** #
# Function Name: CHECK_CAN_UPGRADE_TO_X8000_121
# Description: X8000 的历史C版本，以及1.2.0的在研B版本(1.0.0/1.1.0/1.2.0)，不能升级到1.2.1版本
# Parameter:
# Return:
# **************************************************************************** #
CHECK_CAN_UPGRADE_TO_X8000_121()
{
  # X8000 的历史C版本，以及1.2.0的在研B版本，不能升级到1.2.1版本
  CHECK_HIS_C_VER_IS_X8000 ${G_CUR_YML_FILE}
  local x8000_ret=$?
  CHECK_X8000_120_BY_SPCVER ${G_CUR_YML_FILE}
  local x8000_120_ret=$?

  local spc_version
  spc_version=$(grep -m 1 SpcVersion ${G_PKG_YML_FILE} |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/ //g')

  if [[ (${x8000_ret} -eq 0 || ${x8000_120_ret} -eq 0) && ${spc_version} =~ 1.2.1 ]]; then
    return 1
  fi
  return 0
}

# **************************************************************************** #
# Function Name: CHECK_HIS_C_VER_IS_X8000
# Description: 判断是否为X8000 的历史C版本
# Parameter:
# Return:
# **************************************************************************** #
CHECK_HIS_C_VER_IS_X8000()
{
  local manifest_yml=$1
  local x8000_100=7600506216
  local x8000_110RC1=7600507261
  local x8000_110=7600508181
  local x8000_120RC1=7600509120
  local cur_version
  cur_version=$(grep -m 1 Version "${manifest_yml}" |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/[[:space:]]//g')
  # X8000 的历史C版本，没有Product参数的需添加Product
  if [[ "${cur_version}" == "${x8000_100}" || "${cur_version}" == "${x8000_110RC1}"
  || "${cur_version}" == "${x8000_110}" || "${cur_version}" == "${x8000_120RC1}" ]]; then
    return 0
  fi
  return 1
}

# **************************************************************************** #
# Function Name: CHECK_X8000_120_BY_SPCVER
# Description: 通过SPCVersion判断是否为X8000 1.2.0版本
# Parameter:
# Return:
# **************************************************************************** #
CHECK_X8000_120_BY_SPCVER()
{
  local manifest_yml=$1
  #1.2.0版本
  local spc_ver
  spc_ver=$(grep -m 1 SpcVersion "${manifest_yml}" |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/ //g')
  if [[ ${spc_ver} =~ 1.2.[^1] ]]; then
    return 0
  fi
  return 1
}

# **************************************************************************** #
# Function Name: CHECK_X8000_121_BY_SPCVER
# Description: 通过SPCVersion判断是否为X8000 1.2.1版本
# Parameter:
# Return:
# **************************************************************************** #
CHECK_X8000_121_BY_SPCVER()
{
  local manifest_yml=$1
  ##1.2.1版本
  local spc_ver
  spc_ver=$(grep -m 1 SpcVersion "${manifest_yml}" |${ROOT_PROXY} awk -F":" '{print $2}' | ${ROOT_PROXY} sed 's/ //g')
  if [[ ${spc_ver} =~ 1.2.1 ]]; then
    return 0
  fi
  return 1
}

# **************************************************************************** #
# Function Name: UPGRADE_SYS_VER_BLACK
# Description: 比较版本兼容性
# Parameter:
# Return:
# **************************************************************************** #
UPGRADE_SYS_VER_BLACK()
{
    OS_echo "${UPGRADE_SH} : Get Black List">/dev/null

    if [ ! -f $G_PKG_YML_FILE ] ; then
        echo "False"
        echo "pkgerror"
        return 0
    fi
    
    L_PKG_VERSION_STR=`cat $G_PKG_YML_FILE |grep -m 1 Version |${ROOT_PROXY} awk -F":" '{print $2}'`
    L_PKG_VERSION=`echo "$L_PKG_VERSION_STR" | ${ROOT_PROXY} sed 's/[.T]//g' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`
    
    if [ ! -d "/startup_disk/image/$L_PKG_VERSION" ];then
        echo "False"
        echo "10003"
        return 0
    fi
    
    PACKAGE_TYPE_COMPATIBLE
    if [ "$?" == "1" ] ; then
        echo "False"
        return 0
    fi

    UPGRADE_SYS_NOT_COMPATIBLE
    if [ "$?" == "0" ] ; then
        echo "False"
        return 0
    else
        echo "True"
        return 0
    fi

}

#升级包与当前版本一致时检查当前固件的兼容性
UPD_COMPARE_SYS_VER()
{
    L_CUR_VERSION_STR=`cat $G_CUR_YML_FILE |grep -m 1 Version |${ROOT_PROXY} awk -F":" '{print $2}'`
    OS_echo "${UPGRADE_SH} : upgrade:cur version $L_CUR_VERSION_STR">/dev/null
    declare -i L_CUR_VERSION=0
    L_CUR_VERSION=`echo "$L_CUR_VERSION_STR" | ${ROOT_PROXY} sed 's/[.T]//g' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`

    L_PKG_VERSION_STR=`cat $G_PKG_YML_FILE |grep -m 1 Version |${ROOT_PROXY} awk -F":" '{print $2}'`
    OS_echo "${UPGRADE_SH} : upgrade:pkg version $L_PKG_VERSION_STR">/dev/null
    declare -i L_PKG_VERSION=0
    L_PKG_VERSION=`echo "$L_PKG_VERSION_STR" | ${ROOT_PROXY} sed 's/[.T]//g' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`

    #获取SYS版本是否相同
    if [ $L_CUR_VERSION -eq $L_PKG_VERSION ];then
        return 1
    fi

    return 0
}

#升级前检查系统版本，如果相同则不允许升级
UPD_CHECK_SYS_VER()
{
    OS_echo "${UPGRADE_SH} : Check sys version">/dev/null

    UPD_COMPARE_SYS_VER
    ret=$?

    #版本相同，返回False
    if [ $ret -eq 1 ];then
        echo "False"
    else
        echo "True"
    fi

    return 0
}

#判断是否为降级
CHECK_IS_DOWNGRADE()
{
    if [ -f /startup_disk/conf/conf_local/no_check_downgrade ]; then
        OS_echo "${UPGRADE_SH} : No check downgrade flag exist.">/dev/null
        return 0
    fi
    OS_echo "${UPGRADE_SH} : Check whether the downgrade">/dev/null

    L_CUR_VERSION_STR=`cat $G_CUR_YML_FILE |grep -m 1 Version |${ROOT_PROXY} awk -F":" '{print $2}'`
    OS_echo "${UPGRADE_SH} : upgrade:cur version is $L_CUR_VERSION_STR">/dev/null
    declare -i L_CUR_VERSION=0
    L_CUR_VERSION=`echo "$L_CUR_VERSION_STR" | ${ROOT_PROXY} sed 's/[.T]//g' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`

    L_PKG_VERSION_STR=`cat $G_PKG_YML_FILE |grep -m 1 Version |${ROOT_PROXY} awk -F":" '{print $2}'`
    OS_echo "${UPGRADE_SH} : upgrade:pkg version is $L_PKG_VERSION_STR">/dev/null
    declare -i L_PKG_VERSION=0
    L_PKG_VERSION=`echo "$L_PKG_VERSION_STR" | ${ROOT_PROXY} sed 's/[.T]//g' | ${ROOT_PROXY} sed 's/[[:space:]]//g'`

    #降级，返回1
    if [ $L_CUR_VERSION -gt $L_PKG_VERSION ];then
        return 1
    fi

    return 0
}

#在线升级前检查系统版本，如果版本相同或者是降级则不允许
UPD_CHECK_SYS_VER_WHEN_ONLINE_UPGRADE()
{
    OS_echo "${UPGRADE_SH} : Check sys version when online upgrade">/dev/null

    UPD_COMPARE_SYS_VER
    ret=$?

    #版本相同,不允许升级
    if [ $ret -eq 1 ];then
        echo "False"
    else
        CHECK_IS_DOWNGRADE
        ret=$?
        if [ $ret -eq 0 ];then
            echo "True"
        else
            echo "False"
        fi
    fi

    return 0
}

#检查告警是否存在，存在则返回1，不存在则返回0
UPGRADE_CHECK_ALARM_STATE()
{
    OS_echo "${UPGRADE_SH} : EXEC diagnose" >/dev/null
    if [ -z "$3" ]; then
        OS_ExecDiagnose "*_666" "alarm querybyid $1 1 $2" /OSM/update/upgradealarminfo
    elif [ "$3" == "delay" ]; then
        OS_ExecDiagnose "*_666" "alarm queryallflagbyid $1 1 $2" /OSM/update/upgradealarminfo
    fi

    #检查告警是否存在，存在则返回1，不存在则返回0
    local L_ALARM_NUMBER=$(grep "Alarm Record Number is" /OSM/update/upgradealarminfo | grep -oE "[0-9]+")

    if [ -z "$L_ALARM_NUMBER" ];then
        OS_echo "${UPGRADE_SH} : Alarm $1 not exist !" >/dev/null
        return 0
    fi

    if [ "$L_ALARM_NUMBER" != "0" ];then
        OS_echo "${UPGRADE_SH} : Alarm $1 exist !" >/dev/null
        return 1
    fi

    OS_echo "${UPGRADE_SH} : Alarm $1 not exist !" >/dev/null

    return 0
}

CheckThirdUltraPath()
{
    OS_ExecDiagnose "*_12" "tgt show ultrapath" $G_ULTRA_CHECK
}

#检查多路径告警
UPGRADE_CHECK_ULTRA_PATH()
{
    local L_RetryTimes=24
    local L_SleepTime=0
    local L_WaitTime=0

    ##等待多路径告警恢复，最长等6分钟
    while [ true ]
    do
        if [ ${L_RetryTimes} -eq 0 ];then
            OS_echo "${UPGRADE_SH} : Sleep 6min and Alarm 64425885721 exist." >/dev/null
            return 1;
        fi
        #发送调试命令查询多路径故障告警，存在则返回1，不存在则返回0
        UPGRADE_CHECK_ALARM_STATE "64425885721" "1" "$1"

        #如果有告警则等6分钟等待链路恢复
        if [ 1 == $? ];then
            sleep 15
        else
            break;
        fi

        let "L_RetryTimes-=1"
        let "L_SleepTime+=15"
    done
    let L_WaitTime=360-${L_SleepTime}
    OS_echo "${UPGRADE_SH} : Wait time is ${L_WaitTime}." >/dev/null

    CheckThirdUltraPath
    diagnoseResult=$(grep -i "All hosts have" $G_ULTRA_CHECK)
    OS_echo "${UPGRADE_SH} : Check result is: (${diagnoseResult})." >/dev/null

    ${ROOT_PROXY} rm $G_ULTRA_CHECK >/dev/null
    checkResult=$(echo "${diagnoseResult}" | grep yes)
    #返回yes，代表全都安装了多路径，可以直接返回，否则等待
    if [ -n "${checkResult}" ];then
        OS_echo "${UPGRADE_SH} : Check result is yes." >/dev/null
        #待tgt合入后删除下面sleep的这一行
        sleep ${L_WaitTime}
        return 0
    fi

    # 在测试场景下，没有第三方多路径可以设置此标记减少升级时长
    [ -f /startup_disk/conf/conf_local/no_third_ultra_path ] || sleep ${L_WaitTime}
    return 0
}

#离线升级前检查告警
CHECK_ALARM_BEFORE_OFFLINE_UPGRADE()
{
    local MML_PARA=$1
    local ALARM_ID=$(echo $MML_PARA | grep -oE "[0-9]+")
    OS_echo "${UPGRADE_SH} : The para is $MML_PARA" >/dev/null
    #查询未清除未恢复的告警
    UPGRADE_CHECK_ALARM_STATE "$ALARM_ID" "1"
    if [ "1" == "$?" ]; then
        echo "$MML_PARA"
        echo "True"
        return 0
    fi

    echo "$MML_PARA"
    echo "False"
    return 0
}

UPGRADE_CHECK_DISK_USE_RATE()
{
    if [ -f $G_DISK_INFO ];then
        param=$(cat "$G_DISK_INFO" |${ROOT_PROXY} awk -F ' ' '{print $3}')
        if [ "" != "$param" ];then
            OS_echo "$param"
            unset G_DISK_ID_LIST
            UPGRADE_GET_DISKID $param
            if [ 0 != ${#G_DISK_ID_LIST} ];then
                OS_CalcDiskUseRateWithList ${G_DISK_ID_LIST[*]}
                return 0
            fi
        fi
    fi
    OS_CheckSSDAndSASDiskUsage
    return 0
}

UPGRADE_GET_DISKID()
{
    local str_location=$1
    local list_location=$(echo $str_location|tr "[;,]" "\n")
    local L_location=""
    local L_sdid=0
    OS_echo "$list_location"
    local num=0
    local L_CMD="ld showsds"


    [ -f $G_DISK_ID ] && ${ROOT_PROXY} rm $G_DISK_ID
    OS_ExecDiagnose "*_12" "$L_CMD" "$G_DISK_ID"

    while read location ldid sdid others;
    do
        if [ -n "$(echo $sdid | grep '[0-9]')" ]; then
            L_location=$location
            L_sdid=$sdid

            for l in $list_location;do
                if [ "$L_location" == "$l" ];then
                    G_DISK_ID_LIST[$num]=$L_sdid
                    num=$(( num + 1 ))
                fi
            done
        fi

    done < $G_DISK_ID
    return 0
}

OS_CalcDiskUseRateWithList()
{
    local i_sdid="$@"
    local -i L_OneDiskUsage=0
    local -i L_SSDDiskUtilUsageResult=0
    local -i L_SSDDiskCount=0
    local -i L_SSDDiskAverage=0
    local -i L_SSDDiskSum=0

    local -i L_SASDiskUtilUsageResult=0
    local -i L_SASDiskCount=0
    local -i L_SASDiskAverage=0
    local -i L_SASDiskSum=0
    local L_DiskType=""

    [ -f $G_DISK_UTIL ] && ${ROOT_PROXY} rm $G_DISK_UTIL
    GetDiskUsage

    while read sdid util type;
    do
        if [ -n "$(echo $sdid | grep "[a-zA-Z]")" -o -z "$sdid" ]; then
            continue;
        fi
        L_OneDiskUsage=$util
        L_DiskType=$type
        for i in $i_sdid;do
            if [ "$sdid" == "$i" ];then
                OS_echo "get it ${L_OneDiskUsage}"
                if [ "${L_DiskType}" == "SSD" ] || [ "${L_DiskType}" == "SLC_SSD" ] || [ "${L_DiskType}" == "MLC_SSD" ]; then
                    [ ${L_OneDiskUsage} -gt ${L_SSDDiskUtilUsageResult} ] && L_SSDDiskUtilUsageResult=${L_OneDiskUsage}
                    L_SSDDiskCount=$(( L_SSDDiskCount + 1 ))
                    L_SSDDiskSum=$((L_SSDDiskSum + L_OneDiskUsage ))
                else
                    [ ${L_OneDiskUsage} -gt ${L_SASDiskUtilUsageResult} ] && L_SASDiskUtilUsageResult=${L_OneDiskUsage}
                    L_SASDiskCount=$(( L_SASDiskCount + 1 ))
                    L_SASDiskSum=$((L_SASDiskSum + L_OneDiskUsage ))
                fi
            fi
        done
    done < $G_DISK_UTIL

    OS_echo "${UPGRADE_SH} : L_SSDDiskCount=$L_SSDDiskCount  L_SASDiskCount=$L_SASDiskCount"
    #计算磁盘平均利用率
    if [ "0" != "$L_SSDDiskCount" ];then
        L_SSDDiskAverage=$(( $L_SSDDiskSum / $L_SSDDiskCount ))
    fi
    if [ "0" != "$L_SASDiskCount" ];then
        L_SASDiskAverage=$(( $L_SASDiskSum / $L_SASDiskCount ))
    fi

    echo "upd diskupgradeusagecheck"
    echo "${L_SSDDiskUtilUsageResult},${L_SSDDiskAverage};${L_SASDiskUtilUsageResult},${L_SASDiskAverage}"

    return 0
}
# --------------------------  main  ------------------------------------------ #

#为了调用OS_SET_DISK_RW
OS_GetDriverID

L_INPUT_PARA_NUM=$#

if [ $L_INPUT_PARA_NUM -lt 1 ] ; then
    OS_echo "${UPGRADE_SH} : [os_system_update.sh]wrong input num: $#." >/dev/null
else
        L_TMP=`echo $1 | grep '\-[a-zA-Z0-9]'`
    if [ -z "$L_TMP" ] ; then
        OS_echo "${UPGRADE_SH} : [os_system_update.sh]wrong input para $1." >/dev/null
    fi
fi

while getopts dcCbkmuA:KIU options 2>/dev/null
do
    case $options in
        b)  OS_echo "${UPGRADE_SH} [-b] : Get Black List" >/dev/null
            UPGRADE_SYS_VER_BLACK
            exit $?
            ;;
        c)  OS_echo "${UPGRADE_SH} [-c] : Check CPU usage" >/dev/null
            OS_CHECK_CPU_USAGE
            exit $?
            ;;
        C)  OS_echo "${UPGRADE_SH} [-C] : Check Manager CPU usage" >/dev/null
            OS_CHECK_MGR_CPU_USAGE
            exit $?
            ;;
        d)  OS_echo "${UPGRADE_SH} [-D] : Check SSD and SAS usage" >/dev/null
            UPGRADE_CHECK_DISK_USE_RATE
            exit $?
            ;;
        k)  OS_echo "${UPGRADE_SH} [-k] : Check version " >/dev/null
            UPD_CHECK_SYS_VER
            exit $?
            ;;
        K)  OS_echo "${UPGRADE_SH} [-K] : Check version when online upgrade " >/dev/null
            UPD_CHECK_SYS_VER_WHEN_ONLINE_UPGRADE
            exit $?
            ;;
        u)  OS_echo "${UPGRADE_SH} [-u] : Check Ultra Path" >/dev/null
            UPGRADE_CHECK_ULTRA_PATH 
            exit $?
            ;;
        U)  OS_echo "${UPGRADE_SH} [-U] : Check delay Ultra Path" >/dev/null
            UPGRADE_CHECK_ULTRA_PATH "delay"
            exit $?
            ;;
        A)  OS_echo "${UPGRADE_SH} [-A] : Check alarm before offline upgrade" >/dev/null
            CHECK_ALARM_BEFORE_OFFLINE_UPGRADE "$OPTARG" 
            exit $?
            ;;
        I)  OS_echo "${UPGRADE_SH} [-I] : Check if IO exist" >/dev/null
            OS_CHECK_IO_STATE
            exit $?
            ;;
        *) OS_echo "${UPGRADE_SH} : invalid options" >/dev/null
           ;;
    esac
done

exit 0

