#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2022. All rights reserved.

import sys
import os
import time
from ctypes import CDLL, CFUNCTYPE, RTLD_GLOBAL, c_bool, c_char_p, c_int32, c_uint32, c_uint64, c_void_p
import ctypes
import re
import platform
import tempfile
import logging as log
try:
    import ConfigParser as configparser
except ImportError:
    import configparser
import subprocess

log.basicConfig(level=log.INFO,
                filename="/OSM/log/cur_debug/messages",
                format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                datefmt='%Y-%m-%d %H:%M:%S')

PID_RSF = 648
SWM_MSG = 437
PID_UPGRADE_CHECK = 644  # 升级检查注册om消息使用,使用复制utility pid注册
CHECK_RSF_STATUS = 59594907855
SIMU = (os.path.exists('/.dockerenv') or os.path.exists('/crash/permitdir')
        or os.path.exists('/.swm_llt_stub'))
ARM = (platform.machine() == 'aarch64')
BOARD_TYPE = os.getenv('BOARD_TYPE', '')
PYTHON_2 = '2.7' in sys.version
NODE_TYPE = os.getenv('NODE_TYPE', None)
DATA_NODE = (NODE_TYPE == "DATA_NODE")

BAY_CONFIG = '/OSM/conf/bay_config.ini'

MSG_DEFAULT_TIMEOUT = 60
RETURN_OK = 0
RETURN_ERROR = -1
CHECK_STAGE_RSF = 0
MSG_BID_LOCAL = 0xFFFF
MSG_RET_CMO = 0xFFF
MSG_OP_MMT_TLV_REQUEST = 0x103
MSG_ERR_ARG_CMO = 0xFFE
OM_OPCODE_GET_BACTH_NEXT = 0xD0000C000

LIB_DIR = "/OSM/component/swm/lib"
_SO_CONF = os.path.join(LIB_DIR,
                        'swm.so.conf' if not SIMU or os.path.exists(
                            '/crash/permitdir') else 'swm.so.simu.conf')
_SO_FILE = os.path.join(LIB_DIR, 'swm.so')


class MsgType(object):
    REQUEST = 1
    RESPONSE = 2
    REPORT = 3


class MsgState(object):
    """
    Msg state enum
    """
    NORMAL = 0
    ERROR = 1
    TIMEOUT = 2
    UNREACHABLE = 3


class ItemType(object):
    """
    Tlv msg item type
    """
    INVALID = 0
    STRING = 2
    UNSIGNED = 3
    BOOL = 5
    ARRAY = 7


class EnumItem(object):
    def __init__(self, name, value):
        self.name = name
        self.value = value
        return

    def __int__(self):
        return self.value

    def __str__(self):
        return self.name

    def __eq__(self, other):
        if other is self:
            return True
        if isinstance(other, int):
            return other == self.value
        if PYTHON_2:
            # noinspection PyUnresolvedReferences
            if isinstance(other, str) or isinstance(other, unicode):  # noqa: F821
                return str(other) == self.name
        else:
            if isinstance(other, str):
                return str(other) == self.name
        if isinstance(other, dict):
            return other.get('value', None) == self.value
        return False

    def __hash__(self):
        return hash(self.name)

    def __ne__(self, other):
        return False if self == other else True


class MsgObj(object):
    """
    Msg object
    """
    _PY_TYPE_2_ItemType = {
        int: ItemType.UNSIGNED, bool: ItemType.BOOL, str: ItemType.STRING,
        EnumItem: ItemType.UNSIGNED, list: ItemType.ARRAY
    }

    def __init__(self, rec_num, item_num, records, opcode=0):
        self.obj_addr = self.create_obj(rec_num, item_num, records, opcode)
        return

    def __del__(self):
        OMA.put_msg_obj(self.obj_addr)
        return

    def create_obj(self, rec_num, item_num, records, opcode):
        obj_addr = OMA.create_msg_obj(opcode, rec_num, item_num)
        for rec_id, rec in enumerate(records):
            for item_id, item_value in list(rec.items()):
                item_type = self._PY_TYPE_2_ItemType.get(type(item_value), ItemType.INVALID)
                self.get_obj_info(item_value, obj_addr, rec_id, item_id, item_type)
        return obj_addr

    def get_obj_info(self, item_value, obj_addr, rec_id, item_id, item_type):
        if PYTHON_2:
            if isinstance(item_value, unicode):  # noqa: F821
                item_type = ItemType.STRING
        real_value = int(item_value) if isinstance(item_value, EnumItem) or isinstance(item_value, bool) \
            else item_value
        if PYTHON_2 and isinstance(real_value, unicode):  # noqa: F821
            real_value = real_value.encode("utf-8")
            OMA.set_msg_item(obj_addr, int(item_type), rec_id, item_id, real_value)
        else:
            real_str = str(real_value).encode()
            OMA.set_msg_item(obj_addr, int(item_type), rec_id, item_id, real_str)


class SendMsg(object):
    def __init__(self, opcode, recv_pid, recv_bid, rec_num, item_num, records,
                 timeout=MSG_DEFAULT_TIMEOUT, send_pid=SWM_MSG,
                 send_bid=MSG_BID_LOCAL, private_data=None,
                 msg_type=MsgType.REQUEST,
                 msg_info_opcode=MSG_OP_MMT_TLV_REQUEST):
        self.opcode = opcode
        self.msg_info = MsgInfo(send_pid=send_pid, send_bid=send_bid,
                                recv_pid=recv_pid, recv_bid=recv_bid,
                                msg_type=msg_type, timeout=timeout,
                                opcode=msg_info_opcode)
        self.msg_obj = MsgObj(rec_num, item_num, records, opcode)
        self.private_data = private_data
        return


class OmMsg(object):
    """
    OM Msg class
    """
    _CLIENT_NAME = "SWM_MSG_CLIENT"
    _THREAD_NUM = 4
    _ACK_MSG_SIZE = 1024
    _rsp_priv_map = {}

    def __init__(self):
        self.rsp_data_map = {}
        self.msg_transfer = None
        self.msg_dispatcher = {}
        super(OmMsg, self).__init__()
        return

    @staticmethod
    def _msg_handler(msg_addr):
        msg = RecvMsg(msg_addr)
        return RETURN_OK

    @classmethod
    def send_sync_msg(cls, msg, send_err_log=True):
        if not isinstance(msg, SendMsg):
            log.error("OM_MSG: The send sync msg must be a instance of MsgInfo.")
            return False, None

        try:
            ack_msg_addr = ctypes.c_void_p()
            ret = OMA.send_sync_msg(msg.msg_obj.obj_addr, ctypes.pointer(msg.msg_info),
                                    ctypes.pointer(ack_msg_addr))
            if ret:
                log_intf = log.error if send_err_log else log.debug
                log_intf("OM_MSG: Send msg to c failed, msg(%s), ret(%d).", str(msg), ret)
                return False, None
            ack_msg = RecvMsg(ack_msg_addr, sync_ack=True)
            log.debug("OM_MSG: Get ack msg (%s).", str(ack_msg))
            return True, ack_msg
        except Exception:
            log.exception("OM_MSG: Send msg to c excepted, msg(%s).", str(msg))
            return False, None


class CheckRsfStatusCmo(object):
    """
    增值状态检查CMO
    """
    # 升级阶段 用于区分检查内容。 0：涉及增值业务所有模块检查 1:涉及增值业务中链路相关检查。
    CHECK_RSF_STATUS_STAGE = 64
    # 检查结果 0：检查通过，非0表示检查失败。
    CHECK_RSF_STATUS_RESULT = 65
    # 不在升级批次的位图
    CHECK_RSF_STATUS_NOT_UPGRADE_BITMAP = 66
    # 当前升级批次编号
    CUR_UPD_BATCH_ID = 67
    # 升级总批次
    UPD_BTACH_TOTAL = 68
    # 升级批次编号
    UPD_BTACH_ID = 69
    # 升级节点bit map
    UPD_NODE_BIT_MAP = 70


class OmAdapter(object):
    _SO_CONF = os.path.join(LIB_DIR,
                            'swm.so.conf' if not SIMU or os.path.exists(
                                '/crash/permitdir') else 'swm.so.simu.conf')
    _SO_FILE = os.path.join(LIB_DIR, 'swm.so')

    def __init__(self):
        self.dll = None
        self.msg_converter = CFUNCTYPE(c_bool, c_void_p)
        self.log_converter = CFUNCTYPE(None, c_uint32, c_int32,
                                       c_char_p, c_int32, c_char_p)
        self.msg_receiver = None
        self.c_log_handler = self.log_converter(self._c_log_handler)
        super(OmAdapter, self).__init__()
        return

    @staticmethod
    def open_relate_so(so_conf):
        with open(so_conf, 'r') as file_handle:
            so_list = [line.rstrip('\r\n') for line in file_handle.readlines()]
            for so_file in so_list:
                CDLL(so_file, mode=RTLD_GLOBAL)

    @staticmethod
    def _c_log_handler(pid, level, func, line, log_info):
        level_2_log_intf = {
            1: log.critical,
            2: log.error,
            3: log.warn,
            4: log.info,
            5: log.debug
        }

        log_intf = level_2_log_intf.get(level, log.debug)
        log_intf("%s(func: %s, line: %d).", log_info, func, line)
        return

    def start(self, msg_client_name, msg_handler, msg_pid_list):
        self.open_relate_so(self._SO_CONF)
        self.dll = CDLL(self._SO_FILE)
        if not (self._init_msg(msg_client_name)
                and self._register_msg_handler(msg_handler)):
            log.error("OM_ADAPTER: Start om adapter failed.")
            return False
        for pid in msg_pid_list:
            if not self._register_msg_pid(pid):
                log.error("OM_ADAPTER: Start om adapter failed.")
                return False
        log.info("OM_ADAPTER: Start om adapter successfully.")
        return True

    def create_msg_obj(self, opcode, rec_num, item_num):
        func_name = 'c_create_msg_obj'
        func = getattr(self, func_name, None)
        if func is None:
            func = self.dll.SWMMSG_CreateMsgObj
            func.restype = c_void_p
            func.argtypes = [c_uint64, c_uint32, c_uint32]
            setattr(self, func_name, func)
        return func(opcode, rec_num, item_num)

    def get_msg_obj(self, msg_addr):
        func_name = 'c_get_msg_obj'
        func = getattr(self, func_name, None)
        if func is None:
            func = self.dll.SWMMSG_GetMsgObj
            func.restype = c_void_p
            func.argtypes = [c_void_p]
            setattr(self, func_name, func)
        return func(msg_addr)

    def put_msg_obj(self, obj_addr):
        func_name = 'c_put_msg_obj'
        func = getattr(self, func_name, None)
        if func is None:
            func = self.dll.SWMMSG_PutMsgObj
            func.restype = None
            func.argtypes = [c_void_p]
            setattr(self, func_name, func)
        return func(obj_addr)

    def put_msg(self, msg_addr):
        func_name = 'c_put_msg'
        func = getattr(self, func_name, None)
        if func is None:
            func = self.dll.SWMMSG_PutMsg
            func.restype = None
            func.argtypes = [c_void_p]
            setattr(self, func_name, func)
        return func(msg_addr)

    def get_msg_info(self, msg_addr, msg_info):
        func_name = 'c_get_msg_info'
        func = getattr(self, func_name, None)
        if func is None:
            func = self.dll.SWMMSG_GetMsgInfo
            func.restype = c_int32
            func.argtypes = [c_void_p, c_void_p]
            setattr(self, func_name, func)
        return func(msg_addr, msg_info)

    def get_rec_opcode(self, obj_addr, rec_id):
        func_name = 'c_get_rec_opcode'
        func = getattr(self, func_name, None)
        if func is None:
            func = self.dll.SWMMSG_GetRecOpcode
            func.restype = c_uint64
            func.argtypes = [c_void_p, c_void_p]
            setattr(self, func_name, func)
        return func(obj_addr, rec_id)

    def get_msg_item(self, obj_addr, rec_id, item_id, buff, buff_len):
        func_name = 'c_get_msg_item'
        func = getattr(self, func_name, None)
        if func is None:
            func = self.dll.SWMMSG_GetItem
            func.restype = c_int32
            func.argtypes = [c_void_p, c_uint32, c_uint32, c_char_p, c_uint32]
            setattr(self, func_name, func)
        return func(obj_addr, rec_id, item_id, buff, buff_len)

    def set_msg_item(self, obj_addr, item_type, rec_id, item_id, value):
        func_name = 'c_set_msg_item'
        func = getattr(self, func_name, None)
        if func is None:
            func = self.dll.SWMMSG_SetItem
            func.restype = None
            func.argtypes = [c_void_p, c_int32, c_uint32, c_uint32, c_char_p]
            setattr(self, func_name, func)
        return func(obj_addr, item_type, rec_id, item_id, value)

    def send_sync_msg(self, obj_addr, msg_info, ack_addr):
        func_name = 'c_send_sync_msg'
        func = getattr(self, func_name, None)
        if func is None:
            func = self.dll.SWMMSG_SendSyncMsg
            func.restype = c_int32
            func.argtypes = [c_void_p, c_void_p, c_void_p]
            setattr(self, func_name, func)
        return func(obj_addr, msg_info, ack_addr)

    def _register_msg_handler(self, msg_handler):
        try:
            self.msg_receiver = self.msg_converter(msg_handler)
            func = self.dll.SWMMSG_RegisterHandle
            func.argtypes = [self.msg_converter]
            func.restype = None
            func(self.msg_receiver)
            return True
        except Exception:
            log.exception("OM_ADAPTER: Register msg handler excepted.")
            return False
    
    def _init_msg(self, client_name):
        try:
            ret = self.dll.SWMMSG_InitMsg(0, client_name)
            if ret:
                log.error("OM_ADAPTER: Init msg failed, ret(%d).", ret)
                return False
            log.info("OM_ADAPTER: Init msg successfully.")
            return True
        except Exception:
            log.exception("OM_ADAPTER: Init msg excepted.")
            return False

    def _register_msg_pid(self, msg_pid):
        try:
            ret = self.dll.SWMMSG_RegisterMsgPid(msg_pid)
            if ret:
                log.error("OM_ADAPTER: Register msg pid(%d) failed, ret(%d).", msg_pid, ret)
                return False
            log.info("OM_ADAPTER: Register msg pid(%d) successfully.", msg_pid)
            return True
        except Exception:
            log.exception("OM_ADAPTER: Register msg pid（%d) excepted.", msg_pid)
            return False


OMA = OmAdapter()


class MsgInfo(ctypes.Structure):
    """
    Msg info class, refer to msg info of om c code define
    """
    _fields_ = [
        ('send_bid', ctypes.c_uint16),
        ('send_pid', ctypes.c_uint16),
        ('recv_bid', ctypes.c_uint16),
        ('recv_pid', ctypes.c_uint16),
        ('msg_len', ctypes.c_uint32),
        ('msg_type', ctypes.c_uint32),
        ('opcode', ctypes.c_uint32),
        ('msg_state', ctypes.c_int32),
        ('timeout', ctypes.c_uint32),
        ('session_key', ctypes.c_uint32),
    ]

    def __init__(self, send_bid=MSG_BID_LOCAL, send_pid=SWM_MSG, recv_bid=MSG_BID_LOCAL,
                 recv_pid=SWM_MSG, msg_len=0, msg_type=MsgType.REQUEST, opcode=0,
                 msg_state=MsgState.NORMAL, timeout=MSG_DEFAULT_TIMEOUT, session_key=0):
        self.send_bid = send_bid
        self.send_pid = send_pid
        self.recv_bid = recv_bid
        self.recv_pid = recv_pid
        self.msg_len = msg_len
        self.msg_type = msg_type
        self.opcode = opcode
        self.msg_state = msg_state
        self.timeout = timeout
        self.session_key = session_key
        super(MsgInfo, self).__init__()
        return


def decode_str(value):
    if not PYTHON_2 and isinstance(value, bytes):
        return value.decode()
    return value


class RecvMsg(object):
    """
    Define the msg class of python receive msg
    """
    MAX_ITEM_LEN = 65535  # 与CHS一致64K

    def __init__(self, msg_addr, sync_ack=False):
        """

        :param msg_addr:
        :param sync_ack:
        """
        self.sync_ack = sync_ack
        self.msg_addr = msg_addr
        self.obj_addr = OMA.get_msg_obj(self.msg_addr)
        self._ret = None
        self._opcode = ctypes.c_uint64(0)
        self.msg_info = MsgInfo()
        msg_info_p = ctypes.pointer(self.msg_info)
        OMA.get_msg_info(self.msg_addr, msg_info_p)
        return

    def __del__(self):
        OMA.put_msg_obj(self.obj_addr)
        OMA.put_msg(self.msg_addr)
        return

    @property
    def ret(self):
        """
        Get the msg result
        :return:
        """
        if not self._ret:
            self._ret = self.get_item(int, 0, MSG_RET_CMO)
        return self._ret

    @property
    def opcode(self):
        """
        Get the msg opcode
        :return:
        """
        if not self._opcode:
            self._opcode = OMA.get_rec_opcode(self.obj_addr, 0)
        return self._opcode

    def get_item(self, item_type, rec_id, item_id):
        item_str = ctypes.create_string_buffer(self.MAX_ITEM_LEN)
        ret = OMA.get_msg_item(self.obj_addr, rec_id, item_id, item_str, self.MAX_ITEM_LEN)
        if ret:
            log.debug("SWM_MSG: Get msg item(rec: %d, item: %d, type: %s) failed, ret(%d).",
                      rec_id, item_id, item_type, ret)
            return None
        str_value = ctypes.string_at(item_str)
        return item_type(decode_str(str_value))

    def get_items(self, item_dict):
        result = dict()
        try:
            for item_name, item_info in list(item_dict.items()):
                item_value = self.get_item(*item_info)
                result[item_name] = item_value
            return result
        except Exception as e:
            log.warning("SWM_MSG: Exception:(%s)", e)
            return {}


def init_om_msg_interface():
    try:
        ret_code = OMA.start("RSF_CHECK_UPGRADE", OmMsg._msg_handler, [PID_UPGRADE_CHECK])
        if not ret_code:
            log.error("RSF_UPGRADE_CHECK: interface init failed.")
            return False
        return True
    except Exception as e:
        log.exception("Rsf_check: init fail Exception happen (%s).", str(e))
        return False


def send_sync_msg_with_retry(msg, retry_times=10, interval=10):
    ret_val = RETURN_ERROR
    _ack_msg = None
    for i in range(retry_times):
        ret, _ack_msg = OmMsg.send_sync_msg(msg)
        if not ret:
            log.warning("Rsf_Check: Send msg failed.")
            return False, _ack_msg, None
        _msgobj_ret_value_cmo = dict(
            ret_val=(int, 0, CheckRsfStatusCmo.CHECK_RSF_STATUS_RESULT), )
        ret_val = _ack_msg.get_items(_msgobj_ret_value_cmo)["ret_val"]
        if ret_val != RETURN_OK:
            log.warning("Rsf_Check: Check busy, retry %s times, ret(%s).",
                        i + 1, ret_val)
            time.sleep(interval)
            continue
        else:
            break
    if ret_val != RETURN_OK:
        log.error("Rsf_Check: Check failed, result(%s).", ret_val)
        return False, _ack_msg, ret_val
    return True, _ack_msg, None


class Diagnose(object):
    """
    Encapsulate kernel diagnose
    """
    _RET_ID = 1
    _CODE_ID = 2
    _PARAM_ID = 3
    _RETRY_TIMES = 3
    # diagnose 字符
    CMD_FMT = 'diagnose --auto <<EOF\n{cmd:s}\nquit\nEOF'
    OUTPUT_START_LINE = 0

    def __init__(self, timeout=15):
        self.timeout = timeout

    def exec_diagnose(self, cmd):
        try:
            self.timeout = int(self.timeout)
        except Exception:
            log.exception("timeout config error!")
            return RETURN_ERROR, None
        command = self._format_command(cmd)
        log.info('DIAGNOSE: Execute command (%s).', command)
        output, ret_code = [], 1

        process = subprocess.Popen(['diagsh', '--attach=*_12', '--cmd=sys showcls'],
                                   stdout=subprocess.PIPE, shell=False)

        while True:
            if process.poll() is not None:
                break

        ret_code = process.returncode
        output = process.stdout.readlines()
        if SIMU:
            ret_code = 0
            output = []

        if ret_code != 0:
            log.warning('DIAGNOSE: Command failed, code (%d), info(%s).',
                        ret_code, output)
            return RETURN_ERROR, None

        # diagnose登录时，可能多显示一行：connect to server fail,try again
        output = [decode_str(line) for line in output if "connect to server fail" not in decode_str(line)]
        return ret_code, output

    def exec_cmd(self, cmd):
        ret_code, output = self.exec_diagnose(cmd)

        if not output or ret_code:
            return RETURN_ERROR, output

        return RETURN_OK, output

    def exec_cmd_and_get_res(self, cmd):
        ret, output = self.exec_cmd(cmd)
        node_id_list = []
        sour_node, des_node = RETURN_ERROR, RETURN_ERROR
        if ret != RETURN_OK or not output:
            return RETURN_ERROR, sour_node, des_node, node_id_list
        for line in output:
            if "master" in line:
                des_node = int(line.split(" ")[1])
            if "local node" in line:
                sour_str = line.split(" ")[-1]
                sour_node = int(sour_str.split()[0])
            if ("master" in line or "slave" in line) and int(line.split(" ")[1]) not in node_id_list:
                node_id_list.append(int(line.split(" ")[1]))
        if sour_node != RETURN_ERROR and des_node != RETURN_ERROR:
            return RETURN_OK, sour_node, des_node, node_id_list
        return RETURN_ERROR, sour_node, des_node, node_id_list

    def _format_command(self, cmd):
        return self.CMD_FMT.format(cmd=cmd)


class DiagnoseUser(Diagnose):
    """
    Example: diagsh --attach="12" --cmd="sys showcls"
    """
    CMD_FMT = 'diagsh --attach=*_{process_id:s} --cmd="{cmd:s}"'
    OUTPUT_START_LINE = 0

    def __init__(self, process_id=12, timeout=15):
        self.process_id = str(process_id)
        super(DiagnoseUser, self).__init__(timeout=timeout)

    def _format_command(self, cmd):
        return self.CMD_FMT.format(process_id=self.process_id, cmd=cmd)


def get_sys_info():
    diagnose_obj = DiagnoseUser("12")
    tmp_cmd = 'sys showcls'
    ret, sour_node, des_node, node_id_list = diagnose_obj.exec_cmd_and_get_res(tmp_cmd)
    log.info("sys cls master sour_node %s des_node %s node_id_list %s.", str(sour_node), str(des_node), node_id_list)
    if ret != RETURN_OK or not len(node_id_list):
        return False, sour_node, des_node, node_id_list
    return True, sour_node, des_node, node_id_list


def get_rsf_bitmap(host_id_list, bitmap=0):
    """
    Get bitmap by host id list
    :param host_id_list:host id list, for example: [0,1,2,3]
    :param bitmap:
    :return:
    """
    if isinstance(host_id_list, int):
        host_id_list = [host_id_list]
    for node in host_id_list:
        bitmap += (node >> 5)
        mask = 1 << (node & 31)
        bitmap |= mask
    return bitmap


def get_rsf_upgrade_records(check_stage, node_id_list):
    cur_batch = 0
    non_succ_bit_map = 0
    batch_count = 2
    records = [
        {
            CheckRsfStatusCmo.CHECK_RSF_STATUS_STAGE: int(check_stage),
            CheckRsfStatusCmo.CHECK_RSF_STATUS_NOT_UPGRADE_BITMAP: non_succ_bit_map,
            CheckRsfStatusCmo.CUR_UPD_BATCH_ID: cur_batch,
            CheckRsfStatusCmo.UPD_BTACH_TOTAL: batch_count,
        }
    ]
    #  参考diagnose命令 分两个record
    node_len = len(node_id_list)
    node_id_list_fir, node_id_list_sec = [], []
    for node in range(node_len):
        if node % 2 == 1:
            node_id_list_sec.append(node_id_list[node])
        else:
            node_id_list_fir.append(node_id_list[node])
    upd_bit_map = get_rsf_bitmap(node_id_list_fir)
    log.info("node_id_list_fir %s, node_id_list_sec %s, upd_bit_map %s.",
             node_id_list_fir, node_id_list_sec, upd_bit_map)
    if len(node_id_list_fir):
        records.append(
            {
                CheckRsfStatusCmo.UPD_BTACH_ID: 0,
                CheckRsfStatusCmo.UPD_NODE_BIT_MAP: upd_bit_map
            }
        )
    if len(node_id_list_sec):
        upd_bit_map = get_rsf_bitmap(node_id_list_sec)
        records.append(
            {
                CheckRsfStatusCmo.UPD_BTACH_ID: 1,
                CheckRsfStatusCmo.UPD_NODE_BIT_MAP: upd_bit_map
            }
        )
    return records


def check_rsf_status_new(check_stage):
    if SIMU:
        log.info("Rsf_Check: It is docker env, return true directly.")
        return True, ""
    if DATA_NODE:
        log.info("Rsf_Check: It is data node, no need to check rsf status.")
        return True, ""
    try:
        log.info("Rsf_Check: begin to check(check stage=%s) rsf status.", check_stage)
        ret, sour_node, des_node, node_id_list = get_sys_info()
        if not ret:
            return False, "send msg failed"
        records = get_rsf_upgrade_records(check_stage, node_id_list)

        msg = SendMsg(CHECK_RSF_STATUS, PID_RSF, int(des_node), len(records),
                      len(records[0]), records, MSG_DEFAULT_TIMEOUT, PID_UPGRADE_CHECK, int(sour_node))
        if not msg:
            log.error("Rsf_Check: Create msg for checking rsf status failed.")
            return False, "send msg failed"
        ret, ack_msg, err_code = send_sync_msg_with_retry(msg)

        ack_tlv_map = dict(
            ret_val=(int, 0, CheckRsfStatusCmo.CHECK_RSF_STATUS_RESULT),
        )
        if not ret:
            log.info("Rsf_Check: Send msg fail(result:%s, err_code(%s)).",
                     ret, err_code)
            return False, str(err_code)
        log.info("Rsf_Check: records(%s), ack(%s), ret(%s), ack(%s)", str(records), str(ack_tlv_map), ret,
                 ack_msg.get_items(ack_tlv_map))
        res_temp = ack_msg.get_items(ack_tlv_map)
        res = res_temp.get('ret_val', 1)
        if res or res is None:
            log.error("Rsf_Check: rsf check fail (ret=%s, err code=%s).",
                      res, err_code)
            return False, str(err_code)
        log.info("Rsf_Check: rsf check success(ret=%s).", res)
        return True, ""
    except Exception as e:
        log.exception("Rsf_Check: Exception happen (%s).", str(e))
        return False, str(e)


def check_rsf_upgrade():
    log.info("RSF_UPGRADE_CHECK: begin.")
    rsf_init = init_om_msg_interface()
    if not rsf_init:
        print("False")
        print(str(1077949019))
        print("")
        return 0

    ret, errcode = check_rsf_status_new(CHECK_STAGE_RSF)
    if ret:
        print("True")
        print("")
    else:
        print("False")
        if(str(errcode).isdigit()):
            print(str(errcode))
        else:
            print(str(1077949019))
    print("")
    log.info("RSF_UPGRADE_CHECK: success res_str(%s) errcode(%s).", str(ret), str(errcode))
    return 0


if __name__ == '__main__':
    sys.exit(check_rsf_upgrade())