#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2018-2024. All rights reserved.
import os.path
import re
import sys

from check_item.check_util.sftp import SftpClient
from check_item.check_util.util import get_product_name, ProductName, call_system_cmd_list, remote_execute_cmd, \
    DATA_NODE_FLAG

from check_item.check_util.ip_util import IpInfoManager
from infra.debug.log import swm_logger as log


DISK_LOCATION_TMP_PATH = '/tmp/diskinformation'
EXTEND_PARAM_FILE = '/tmp/check_extend_param'
LOG_TAG = 'CheckPrepare'
INVALID_NID = 65535
RETURN_OK = 0
IS_A800 = ProductName.ASeries == get_product_name()

IP_MGR = IpInfoManager(is_global=IS_A800)


def get_file_src_id():
    """
    文件保存在业务主节点，或者全局业务主节点。
    :return:
    """
    if IS_A800:
        cmd_list = [['swm_diagnose', '-c', 'global_cls show'],
                    ['grep', '-E', '^global_master_id'],
                    ['awk', '{print $3}']]
    else:
        cmd_list = [['swm_diagnose', '-c', 'cls show'],
                    ['grep', '-E', '^master_id'],
                    ['awk', '{print $3}']]
    ret, output = call_system_cmd_list(cmd_list)
    if ret:
        log.error('%s: Get the source file host id failed(ret_code: %s, output: %s).', LOG_TAG, ret, output)
        return INVALID_NID
    return int(output.strip())


def get_file_src_id_in_nof():
    master_id = INVALID_NID
    # ip框
    cmd_list = [['/OSM/bin/diagsh', '--attach=*_12', '--cmd=sysnof showclsinfo'],
                ['grep', 'master'],
                ['awk', '{print $1}']]
    ret, output = call_system_cmd_list(cmd_list)
    if ret != RETURN_OK:
        return master_id
    mid = int(output.strip())
    log.info('%s: Get master id is %s.', LOG_TAG, mid)
    return mid


def get_local_id():
    if not IP_MGR.local_node:
        return INVALID_NID
    nid = IP_MGR.local_node.get_host_id()
    log.info('%s: Get local host id is %s.', LOG_TAG, nid)
    return nid


def get_node_host_list(node_id):
    if not IP_MGR.all_nodes:
        return []
    ip_list = []
    for node in IP_MGR.all_nodes:
        if node.get_host_id() == int(node_id):
            ip_list = node.addr_list
            break
    log.info('%s: Get node %s ip list is %s.', LOG_TAG, node_id, ip_list)
    return ip_list


def check_remote_file_exists(remote_ip, file_path):
    cmd = 'ls {0}'.format(file_path)
    ret, output = remote_execute_cmd(remote_ip, cmd, timeout=5)
    log.info('%s: Check remote file exists result is %s, output is %s.', LOG_TAG, ret, output)
    if ret:
        cmd_exec_ret = True if 'No such file or directory' in output else False
        return cmd_exec_ret, False
    return True, True


def get_file_from_master():
    master_id = get_file_src_id() if not os.path.exists(DATA_NODE_FLAG) else get_file_src_id_in_nof()
    if master_id == get_local_id():
        log.info('%s: The file source node(%s), do nothing.', LOG_TAG, master_id)
        return True
    host_ip_list = get_node_host_list(master_id)
    for ip in host_ip_list:
        if not _get_file_from_host(ip, DISK_LOCATION_TMP_PATH):
            continue
        if not _get_file_from_host(ip, EXTEND_PARAM_FILE):
            continue

        log.info('%s: Get file from %s success.', LOG_TAG, ip)
        return True
    log.error('%s: Get file failed.', LOG_TAG)
    return False


def _get_file_from_host(ip, filepath):
    exec_ret, cmd_ret = check_remote_file_exists(ip, filepath)
    if not exec_ret:
        return False
    if cmd_ret:
        os.path.isfile(filepath) and os.remove(filepath)
        sftp = SftpClient(ip)
        if not sftp.get_file(filepath, filepath):
            log.error('%s: Get file(%s) from %s failed.', LOG_TAG, filepath, ip)
            return False
    return True


def main():
    try:
        if get_file_from_master():
            print("True")
        else:
            print("False")
    except Exception:
        log.exception('%s: exception.', LOG_TAG)
        print("False")
    return 0


if __name__ == '__main__':
    sys.exit(main())
