#!/usr/bin/env python
# _*_ coding:utf-8 _*_
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

# 本脚本由容器提供给swm调用
# 场景：滚动升级、离线升级、内核升级、内核离线升级
# 在bios升级前，调用clean方法
# swm需要传入的参数有：1.升级类型(快速升级/离线升级...) 2.运行类型(升级/重试/回滚...) 2.原软件版本 3.目标软件版本 4.原阿波罗版本 5.目标阿波罗版本

import logging
import os
import re
import subprocess
import sys

from pkg_resources import parse_version

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')
LOG_NAME = "CONTAINER_DEL_IOMMU"
BSP_DIR = "/proc/osp/bsp"
CMD_GET_INFO = "ipmitool raw 0x32 0xbb 0x0 0x0 0x80 0x0 0x5c"
CMD_WRITE_INFO = "ipmitool raw 0x32 0xbc 0x0 0x0 0x80 0x0 0x4f {}"
STR_LEN = 64
# Upgrade run type 定义 START = 0 UPD = 1 RETRY = 2 ROLLBACK = 3 TERMINATE = 4 REPAIR = 5
RUN_TYPE = ["0", "1", "2", "3"]


def _call_system_cmd(cmd):
    cmd_ret = 1
    output = ""
    try:
        process = subprocess.Popen(cmd.split(), stdin=subprocess.PIPE, stdout=subprocess.PIPE)
        output, _ = process.communicate()
        cmd_ret = process.returncode
        if process.returncode:
            logging.warning("[%s] Exec cmd(%s) result abnormal, retCode(%d), "
                            "result(%s).", LOG_NAME, cmd, cmd_ret, output)
        else:
            logging.info("[%s] Exec cmd(%s) successfully.", LOG_NAME, cmd)
        return cmd_ret, output
    except Exception:
        logging.error("[%s] Exec cmd(%s) except, retCode(%d), result(%s).", LOG_NAME, cmd, cmd_ret, output)
        return cmd_ret, output


def _byte_to_string(bytes_):
    string_list = [''] * STR_LEN
    for i, hex_code in enumerate(bytes_.split()):
        if i == STR_LEN:
            break
        string_list[i] = chr(int(hex_code, 16))
    return "".join(string_list)


def _string_to_ascii(string):
    ascii_list = ['0x0'] * STR_LEN
    for i, char in enumerate(string):
        if i == STR_LEN:
            break
        ascii_list[i] = hex(ord(char))
    return " ".join(ascii_list)


# 将iommu写为0
def _del_iommu():
    get_ret, old_bsp = _call_system_cmd(CMD_GET_INFO)
    if get_ret:
        return get_ret
    old_str = _byte_to_string(old_bsp)
    if "iommu:1" not in old_str:
        return 0
    new_str = old_str.replace("iommu:1", "iommu:0")
    new_bsp = _string_to_ascii(new_str)
    write_ret, output = _call_system_cmd(CMD_WRITE_INFO.format(new_bsp))
    return write_ret


def container_enabled():
    with open(BSP_DIR, "rb") as f:
        lines = f.readlines()
    for line in lines:
        try:
            line = line.decode("utf-8")
        except Exception:
            continue
        pos = line.find("conta:")
        if pos == -1:
            continue
        info = line[pos:len(line)]
        tmp = re.split(":|;", info)
        level = tmp[1].strip()
        if level != "0":
            return True
    return False


def _need_remain(dst_apollo_version):
    # 目标阿波罗版本>=1.1.7 iommu特性预埋，不清理
    if parse_version(dst_apollo_version) >= parse_version("1.1.7.0"):
        logging.info("[%s] target apollo version is %s, will not del iommu.", LOG_NAME, dst_apollo_version)
        return True
    # 目标阿波罗版本为[1.1.2.0, 1.1.7.0), iommu特性不预埋，但如果此时已经开启容器，不清理
    if container_enabled() and (parse_version(dst_apollo_version) >= parse_version("1.1.2.0")):
        logging.info("[%s] container is enable, will not del iommu.", LOG_NAME)
        return True
    return False


def clean(upg_type, run_type, src_num_version, dst_num_version, src_apollo_version, dst_apollo_version):
    if _need_remain(dst_apollo_version):
        return 0
    return _del_iommu()


def exec_cmd():
    node_type = os.getenv('NODE_TYPE', None)
    # IP框不执行
    if node_type == "DATA_NODE":
        return 0
    upg_type = os.getenv('upg_type', None)
    run_type = os.getenv('run_type', None)
    src_num_version = os.getenv('src_num_version', None)
    dst_num_version = os.getenv('dst_num_version', None)
    src_apollo_version = os.getenv('src_apollo_version', None)
    dst_apollo_version = os.getenv('dst_apollo_version', None)
    if run_type in RUN_TYPE:
        return clean(upg_type, run_type, src_num_version, dst_num_version, src_apollo_version, dst_apollo_version)
    return 0


if __name__ == "__main__":
    ret = exec_cmd()
    sys.exit(ret)
