# -*- coding: UTF-8 -*-
"""
@desc: NetLogonEX check
@date: 2021.1.26
"""


from cbb.frame.checkitem.base_dsl_check import (
    BaseCheckItem,
    CheckStatus
)
from cbb.frame.adapter import replace_adapter
from cbb.frame.dsl import fault_mode
from cbb.frame.ruleConfig.common import getMsg
from cbb.frame.dsl.adapter import NEED_NEW_CTRL_CONN
from cbb.frame.cli.cliUtil import (
    isNoneLicense,
    hasCliExecPrivilege,
    getHorizontalCliRet,
)


class NetLogEnableCheck(BaseCheckItem):
    """
    NetLogonEX开关检查
    """
    SPECIAL_VERSION_V5 = "V500R007C60SPC300"
    SPECIAL_VERSION_V3 = "V300R006C61"
    # 只有白名单的大版本，及小范围的补丁版本才涉及此问题。
    risk_version_later_patch = {
        "V300R006C20": "V300R006C20SPH026",
        "V300R006C20SPC100": "V300R006C20SPH119",
        "V300R006C50SPC100": "V300R006C50SPH117",
        "V300R006C60": "V300R006C60SPH006",
        "V500R007C10": "V500R007C10SPH026",
        "V500R007C10SPC100": "V500R007C10SPH119",
        "V500R007C30SPC100": "V500R007C30SPH117",
        "V500R007C50": "V500R007C50SPH006",
    }
    # 只有白名单的版本才涉及此问题。
    risk_patch = {
        "V300R006C61": [
            "V300R006C61SPH009",
            "V300R006C61SPH011",
            "V300R006C61SPH012",
        ],
        "V500R007C61": [
            "V500R007C61SPH009",
            "V500R007C61SPH011",
            "V500R007C61SPH012",
        ],
        "V500R007C60SPC300 Kunpeng": ["V500R007C60SPH309"],
        "V500R007C70SPC200 Kunpeng": ["V500R007C70SPH201"],
        "V500R007C70SPC200": ["V500R007C70SPH201"],
    }

    def __init__(self, tool_context):
        super(NetLogEnableCheck, self).__init__(tool_context)
        self.dev_version = tool_context.get("mem_version")
        self.dev_type = tool_context.get("product_mode")
        self.patch_version = ""
        self.context[NEED_NEW_CTRL_CONN] = False
        self.error_list = []

    def execute(self):
        """
        主函数
        :return:
        """
        if not self.__is_risk_version():
            return CheckStatus.NO_SUPPORT, "Not a risk version."

        if not self.__has_file_system():
            return CheckStatus.PASS, ""

        vstore_ids = self.dsl(
            "exec_cli 'show vstore |filterColumn include "
            "columnList=ID' | horizontal_parser",
        )
        if any([
            self.dev_version.startswith("V5") and
            self.dev_version < self.SPECIAL_VERSION_V5,
            self.dev_version.startswith("V3") and
            self.dev_version < self.SPECIAL_VERSION_V3
        ]):
            self.__check_limit_version_before(vstore_ids)
        else:
            self.__check_limit_version_after(vstore_ids)

        if self.error_list:
            return (
                CheckStatus.NOTPASS,
                getMsg(
                    self.lang,
                    "netlog.ex.enable.not.pass",
                    ", ".join(self.error_list),
                ),
            )

        return CheckStatus.PASS, ""

    def __has_file_system(self):
        """
        是否有文件系统
        :return:
        """
        self.dsl(
            "exec_cli 'show file_system general'",
            return_if={fault_mode.FindException: ""},
        )
        ret_list = self.last_dsl.origin_info
        ret = ""

        if ret_list:
            ret = ret_list[-1]

        cli_ret = ret.lower()

        if isNoneLicense(cli_ret) or not hasCliExecPrivilege(cli_ret):
            return False

        if "error:" in cli_ret:
            raise Exception("Error Response")

        return bool(getHorizontalCliRet(cli_ret))

    def __check_limit_version_before(self, vstore_ids):
        """
        检查之前的版本
        :param vstore_ids:
        :return:
        """
        for vstore_info in vstore_ids:
            vstore_id = vstore_info.get("ID")
            bit_info = self.dsl(
                "exec_developer 'show cifs reserve_bit "
                "tenantid={}' | vertical_parser".format(vstore_id),
            )

            if not bit_info:
                continue

            bit_value = bit_info.get("Reservebit Value", "")

            if not bit_value.isdigit():
                continue

            if int(bit_value) >> 25 & 1 != 1:
                continue

            self.dsl("exec_cli 'change vstore view id={}'".format(vstore_id),)
            ad_info_dict = self.dsl(
                "exec_cli 'show domain ad' | vertical_parser",
            )
            self.dsl("exec_cli 'exit'")

            if not ad_info_dict:
                continue

            if ad_info_dict.get("Domain Status") != "Joined":
                continue

            self.error_list.append(vstore_id)

    def __check_limit_version_after(self, vstore_ids):
        """
        检查之后的版本
        :param vstore_ids:
        :return:
        """
        for vstore_info in vstore_ids:
            vstore_id = vstore_info.get("ID")
            self.dsl("exec_cli 'change vstore view id={}'".format(vstore_id),)
            ad_info_dict = self.dsl(
                "exec_cli 'show domain ad' | vertical_parser",
            )

            if not ad_info_dict:
                self.dsl("exec_cli 'exit'")
                continue

            if ad_info_dict.get("Domain Status") != "Joined":
                self.dsl("exec_cli 'exit'")
                continue

            service_file_system_info = self.dsl(
                "exec_developer 'show service cifs' | vertical_parser"
            )

            if service_file_system_info.get("NTLM Logonex Enable") == "No":
                self.error_list.append(vstore_id)

            self.dsl("exec_cli 'exit'")

        return True, ""

    def __is_risk_version(self):
        """
        是否风险版本
        :return:
        """
        (flag, __, self.patch_version,) = replace_adapter.get_system_version(
            self.context
        )

        if (
            self.dev_type == "Dorado NAS"
            and self.dev_version == "V300R002C10SPC100"
        ):
            return self.patch_version >= "V300R002C10SPH118"

        return any(
            [
                self.dev_version in self.risk_version_later_patch
                and self.patch_version
                >= self.risk_version_later_patch.get(self.dev_version),
                self.dev_version in self.risk_patch
                and self.patch_version
                in self.risk_patch.get(self.dev_version),
            ]
        )
