# -*- coding: UTF-8 -*-

from cbb.business.operate.expansion import common
from cbb.frame.checkitem.base_dsl_check import CheckStatus
from cbb.frame.cli import cliUtil
from cbb.frame.tlv import adTlvUtil
from cbb.frame.tlv import tlvData
from cbb.frame.context import contextUtil
from cbb.frame.cli import cli_con_mgr
from cbb.frame.base.exception import UnCheckException
from cbb.frame.base.config import DORADO_DEVS_V6_LOW
from cbb.frame.base.config import BOM_INFO_OF_MEMORY_TYPE_B
from cbb.frame.base.config import BOM_INFO_OF_MEMORY_CXMT
from java.lang import Exception as JException

# 内存类型
MEMORY_TYPE_A = "B21-S"
MEMORY_TYPE_B = "B21-P"
MEMORY_TYPE_CXMT = "CXMT"

# 电子标签中包含了“,FSM”，则为B21-P,否则为B21-S
MEMORY_TYPE_FLAG = ",FSM"


def execute(context):
    """
    检查内存类型
    :param context: context
    :return: 检查结果
    """
    check_memory_type = CheckMemoryType(context)
    check_memory_type.execute_check()

    return check_memory_type.check_result, "\n".join(
        check_memory_type.cli_ret_list)


class CheckMemoryType:
    """
    检查内存类型
    """

    def __init__(self, context):
        self.context = context
        self.cli = cli_con_mgr.get_available_conn(context, context.get("cli"))
        self.logger = context.get("logger")
        self.lang = contextUtil.getLang(context)
        self.cli_ret_list = []
        self.is_exist_cxmt = False
        self.check_result = {"flag": CheckStatus.PASS, "errMsg": "",
                             "suggestion": "", "ignoreFlag": False}

    def execute_check(self):
        try:
            # 低端不涉及
            if not self.check_product_model():
                self.check_result["flag"] = CheckStatus.NO_SUPPORT
                return

            if self.check_memory_type_compatible():
                return

            # 不通过时设置错误信息
            if not self.is_exist_cxmt:
                self.check_result["ignoreFlag"] = True
                self.set_result(CheckStatus.NOTPASS, "memory.type.not.compatible")
            else:
                self.set_result(CheckStatus.NOTPASS, "memory.type.not.ignore")

        except (Exception, JException) as e:
            self.logger.error(str(e))
            self.set_result(CheckStatus.NOCHECK, "system.abnormal")

    def set_result(self, check_status, msg_key):
        """
        设置错误信息
        @param check_status: 检查结果
        @param msg_key: 错误信息的key
        @return:
        """
        self.check_result["flag"] = check_status
        self.check_result["errMsg"], self.check_result["suggestion"] = \
            common.getMsg(self.lang, msg_key)

    def check_product_model(self):
        """
        检查设备型号
        """
        # 查询设备型号
        flag, product_model, err_msg, cli_ret = \
            cliUtil.get_product_model_with_ret(self.cli, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException("failed to query the product_model.")
        self.logger.info("product_model: {}".format(product_model))

        return product_model not in DORADO_DEVS_V6_LOW

    def check_memory_type_compatible(self):
        """
        检查内存类型是否兼容
        @return: True: 内存类型兼容；False：内存类型不兼容
        """
        origin_memory_type = self.get_origin_memory_type()
        new_memory_type = self.get_new_memory_type()
        if origin_memory_type == MEMORY_TYPE_CXMT or new_memory_type == MEMORY_TYPE_CXMT:
            self.is_exist_cxmt = True
            return new_memory_type == origin_memory_type
        return not (origin_memory_type == MEMORY_TYPE_A and new_memory_type == MEMORY_TYPE_B)

    def get_origin_memory_type(self):
        """
        获取原集群的内存类型
        @return: 内存类型
        """
        cmd = r"show controller general |filterColumn include columnList=Electronic\sLabel"
        flag, cli_ret, err_msg = \
            cliUtil.excuteCmdInCliMode(self.cli, cmd, True, self.lang)
        self.cli_ret_list.append(cli_ret)
        if flag is not True:
            raise UnCheckException(err_msg)
        for bom_info in BOM_INFO_OF_MEMORY_CXMT:
            if bom_info in cli_ret:
                return MEMORY_TYPE_CXMT
        return MEMORY_TYPE_B if MEMORY_TYPE_FLAG in cli_ret else MEMORY_TYPE_A

    def get_new_memory_type(self):
        """
        获取待扩集群的内存类型
        @return: 内存类型
        """
        self.logger.info("tool_type: {}".format(self.context.get("tool_type")))
        # 扩容评估场景
        if self.context.get("tool_type") == "inspect":
            return self.get_new_memory_type_in_inspect_scene()

        new_boards_list = contextUtil.getItem(self.context, "newBoardsList")
        memory_type_list = list()
        for board in new_boards_list:
            memory_type_list.extend(
                self.get_new_memory_type_in_expansion_scene(board))

        if MEMORY_TYPE_CXMT in memory_type_list:
            return MEMORY_TYPE_CXMT
        return MEMORY_TYPE_B if MEMORY_TYPE_B in memory_type_list \
            else MEMORY_TYPE_A

    def get_new_memory_type_in_inspect_scene(self):
        """
        扩容评估场景下获取待扩集群内存类型
        @return: 内存类型
        """

        bom_list = self.context.get("newEncBom").upper().split(",")
        bom_list.extend(self.context.get("newCtrlBom").upper().split(","))
        self.logger.info("bom_list: {}".format(bom_list))
        for bom in bom_list:
            if bom in BOM_INFO_OF_MEMORY_CXMT:
                return MEMORY_TYPE_CXMT
        for bom in bom_list:
            if bom in BOM_INFO_OF_MEMORY_TYPE_B:
                return MEMORY_TYPE_B
        return MEMORY_TYPE_A

    def get_new_memory_type_in_expansion_scene(self, board):
        """
        扩容场景下获取待扩集群内存类型
        @param board: 节点
        @return: 内存类型列表
        """
        memory_type_list = list()
        tlv = contextUtil.getTlv(self.context)
        records = adTlvUtil.getControllerRecords(tlv, board)
        if records is None:
            return memory_type_list

        for record in records:
            electronic_label = adTlvUtil.getRecordValue(
                record, tlvData.CONTROLLER['elabel'])
            if any(bom_info in electronic_label for bom_info in BOM_INFO_OF_MEMORY_CXMT):
                memory_type_list.append(MEMORY_TYPE_CXMT)
            elif MEMORY_TYPE_FLAG in electronic_label:
                memory_type_list.append(MEMORY_TYPE_B)
            else:
                memory_type_list.append(MEMORY_TYPE_A)

        return memory_type_list
