# -*- coding: UTF-8 -*-

import common
import config
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest.restUtil import CommonRest

'''
根据SAS线缆告警检查连线正确性
'''

SAS_ALARMS = {
    "64438009898": {"errKey": "sas.cable.board.error", "errParamIndex": [1, 2], "type": "EXPBOARD",
                    "errMsgArgs": ["location"]},  # 单链路：0x000F00CE002A
    "64438140950": {"errKey": "sas.cable.port.error", "errParamIndex": [1, 3, 4], "type": "SAS_PORT",
                    "errMsgArgs": ["location"]},  # SAS线缆连接错误告警：0x000F00D00016
    "64424902671": {"errKey": "sas.cable.port.error", "errParamIndex": [1, 3, 4], "type": "SAS_PORT",
                    "errMsgArgs": ["location"]},
    # 级联端口线缆连接错误：0xF0006000F
    "64424902672": {"errKey": "sas.cable.port.error", "errParamIndex": [1, 4, 5], "type": "SAS_PORT",
                    "errMsgArgs": ["location"]},  # SAS线缆连接错误告警：0xF00060010
    "64438206534": {"errKey": "sas.cable.port.error", "errParamIndex": [1, 3, 4], "type": "SAS_PORT",
                    "errMsgArgs": ["location"]},  # SAS线缆连接错误告警：0xF00D10046
    "64424902673": {"errKey": "sas.cable.pair.port.error", "errParamIndex": [(17, 21, 22), (25, 29, 30)],
                    "type": "SAS_PORT", "errMsgArgs": ["location", "suggestPeerLocation"]},
    # SAS端口连接不正确：0xF00060011
    "64424902678": {"errKey": "sas.cable.pair.port.incompatible", "errParamIndex": [2, 4, 5], "type": "PCIE_PORT",
                    "errMsgArgs": ["location"]},  # 线缆规格与端口不兼容：F00060016
}


class SasCableAlarm():
    def __init__(self, context):
        self.context = context
        self.logger = common.getLogger(context.get("logger"), __file__)
        self.lang = contextUtil.getLang(context)

    def execute(self):
        try:
            baseUtil.safeSleep(config.Overtime.SAS_CABLE_ALARM_TIMEOUT)
            rest = contextUtil.getRest(self.context)

            # 获取告警
            sasAlarms = self.getValidAlarms(rest)
            if not sasAlarms:
                self.logger.logPass()
                return []

            # 提示错误信息
            return self.getErrInfo(sasAlarms)

        except Exception as exception:
            self.logger.logException(exception)
            infoDict = {
                "location": "",
                "suggestPeerLocation": "",
                "type": "",
                "errMsgKey": "dev.conn.failure",
                "errMsgArgs": []
            }
            return [infoDict.copy()]

    def getValidAlarms(self, rest):
        '''
        :param records:
        :return:
        '''
        sasAlarms = []
        records = CommonRest.getAlarms(rest)
        if not records:
            return sasAlarms

        self.logger.logInfo("get alarm id records:%s" % records)
        for record in records:
            alarmId = str(CommonRest.getRecordValue(record, restData.RestCfg.Alarm.EVENTID))
            self.logger.logInfo("get alarm id:%s" % alarmId)
            if alarmId in SAS_ALARMS:
                sasAlarms.append(record)
        return sasAlarms

    def getSingleLocation(self, params, errParamIndex):
        if len(params) < errParamIndex[-1]:
            return None

        values = []
        for index in errParamIndex:
            values.append(params[index])

        return ".".join(values)

    def getPairLocation(self, params, errParamIndex):
        if len(params) < errParamIndex[-1][-1]:
            return None

        pair = []
        for indexList in errParamIndex:
            values = []
            for index in indexList:
                values.append(params[index])
            pair.append(".".join(values))

        return tuple(pair)

    def getErrInfo(self, sasAlarms):

        errInfo = []
        for alarm in sasAlarms:
            alarmId = str(CommonRest.getRecordValue(alarm, restData.RestCfg.Alarm.EVENTID))
            paramInfo = CommonRest.getRecordValue(alarm, restData.RestCfg.Alarm.EVENTPARAM)
            params = paramInfo.split(",")
            errElements = SAS_ALARMS.get(alarmId)
            errParamIndex = errElements.get("errParamIndex")
            errType = errElements.get("type")

            suggestPeerLocation = ""
            if alarmId not in ["64424902673"]:
                location = self.getSingleLocation(params, errParamIndex)
            else:
                location, suggestPeerLocation = self.getPairLocation(params, errParamIndex)

            if not location:
                continue

            infoDict = {
                "location": location,
                "suggestPeerLocation": suggestPeerLocation,
                "type": errType,
                "errMsgKey": errElements.get("errKey"),
                "errMsgArgs": errElements.get("errMsgArgs")
            }
            errInfo.append(infoDict.copy())
        return errInfo
