﻿# -*- coding: UTF-8 -*-
import ast
import datetime
import os
import re
import sys
import time
import traceback

from cbb.frame.base import config


class const:
    """
    工具总体常量定义类
    """
    RET_ERROR = -1
    FRU_PERSIST = "persist_fru_info"
    DATA_PERSIST = "persist_tool_data_dict"
    INVALID_SAS_PORTID = "4294967295"  # 此处必须为字符串
    INVALID_PCIE_PORTID = "4294967295"  # 此处必须为字符串
    INVALID_IPSCALEOUT_DSWID = "4294967295"  # 此处必须为字符串

    HELP_DESC_ZH = u"请联系技术支持工程师协助处理。"
    HELP_DESC_EN = "Contact technical support engineers for help."

    # 信息获取失败时，统一处理
    SUG_DESC_ZH = u'请检查网络连接或设备状态是否正常。'
    SUG_DESC_EN = 'Check the whether the network connection or device ' \
                  'status is normal.'
    ERR_DESC_ZH = u'设备信息获取失败。'
    ERR_DESC_EN = 'Failed to obtain device information.'

    # 错误状态
    ERROR_STATUS = -1

    # 系统产品型号
    SYSMODEL = {"ERROR": ERROR_STATUS,  # 错误
                "S2600T": 0,  # 26T控制框
                "S5500T": 1,  # 55T控制框
                "T_ENGINE": 2,  # T独立机头
                "V3_3U_ENGINE": 3,  # 3U独立机头
                "V3_6U_ENGINE": 4,  # 6U独立机头
                "5X00V3": 5,  # 53V3/55V3控制框(Fan in ctrl)
                "2600V3": 6,  # 26V3控制框
                "V3_18000": 7,  # V3 18000控制框
                "2800V3": 8,  # 2800V3控制框
                "V5_V6_2U_ENGINE": 9,  # 2U控制框5X10V5（5310 5510 5610 5810）
                "V5_V6_4U_ENGINE": 10,  # 4U独立机头5X10V5（6810 18510）
                }
    EXPENCMODEL = {"ERROR": ERROR_STATUS,  # 错误
                   "T_2U": 0,  # T-2U硬盘框
                   "T_4U": 1,  # T-4U硬盘框
                   "T_4U75": 2,  # T-4U 75盘 硬盘框
                   "V3_2U": 3,  # V3-2U硬盘框
                   "V3_4U": 4  # V3-4U硬盘框
                   }


class returnVal:
    """
    通用返回值类：实现统一的字典返回格式
    """

    @staticmethod
    def dict2(succ, info):
        """构造两个返回值的字典

        :param succ: 执行结果是否成功
        :param info: 需要回传的信息
        :return:
        """
        return {"succ": succ, "info": info}

    @staticmethod
    def dict3(succ, reason, suggestion):
        """构造三个返回值的字典

        :param succ: 执行结果是否成功
        :param reason: 原因
        :param suggestion: 建议
        :return:
        """
        return {"succ": succ, "reason": reason, "suggestion": suggestion}


class globalVar:
    """
    全局变量类：用于定义和存放所有脚本中均可访问的全局变量
    """

    # 工具上下文。此处定义，避免在函数间不断传递，且使得整个脚本均对其有访问权限
    # 请通过getContext获取变量并使用，禁止直接访问该变量，方便为None时返回合理的异常信息
    context = None

    @staticmethod
    def getContext():
        """获取全局可访问的工具上下文

        :return:
        """
        if globalVar.context is None:
            raise exception.newToolException(u"全局工具上下文已失效或未设置，"
                                             u"工具无法运行。",
                                             const.HELP_DESC_ZH,
                                             "The global context of the tool "
                                             "has become invalid or has not "
                                             "been configured. The tool "
                                             "cannot work.",
                                             const.HELP_DESC_EN)
        else:
            return globalVar.context

    @staticmethod
    def setContext(context):
        """设置全局可访问的工具上下文

        :param context: 工具上下文
        :return:
        """
        if context is None:
            raise exception.newToolException(u"无效的工具上下文，工具无法运行。",
                                             const.HELP_DESC_ZH,
                                             "Invalid tool context. "
                                             "The tool cannot work.",
                                             const.HELP_DESC_EN)
        else:
            globalVar.context = context


class exception:
    """
    异常处理类：用于定义脚本框架中能识别的异常和处理机制
    """
    # 异常信息字段索引，可通过args[id]获取各字段描述信息
    argsId = {
        "FLAG": 0,  # 自定义异常标识
        "MSG_ZH": 1,  # 中文错误信息
        "SUG_ZH": 2,  # 中文处理建议
        "MSG_EN": 3,  # 英文错误信息
        "SUG_EN": 4,  # 英文处理建议
        "ERRID": 5  # 错误ID
    }

    # 默认工具异常名称：该类异常由工具脚本生成，包含中英文的异常描述和处理建议
    ALL_LANG_EXCEPTION = "ALL_LANG_EXCEPTION"
    # 框架异常
    FRAMEWORK_EXCEPTION = "FRAMEWORK_EXCEPTION"

    @staticmethod
    def newToolException(err_msg_zh, suggestion_zh, err_msg_en, suggestion_en,
                         err_code=-1):
        """生成新的工具异常对象

        :param err_msg_zh: 中文异常描述
        :param suggestion_zh: 中文处理建议
        :param err_msg_en: 英文异常描述
        :param suggestion_en: 英文处理建议
        :param err_code: 错误码
        :return:
        """
        return Exception(exception.ALL_LANG_EXCEPTION,
                         err_msg_zh, suggestion_zh,
                         err_msg_en, suggestion_en,
                         str(err_code))

    @staticmethod
    def newToolExceptionByKeys(context, err_key, sug_key, err_param_tuple=(),
                               sug_param_tuple=(), err_code=-1):
        """生成新的工具异常对象

        :param context: 工具上下文
        :param err_key: 错误描述Key。为None 或  ""时，该字段返回结果为""
        :param sug_key: 建议描述Key。为None 或  ""时，该字段返回结果为""
        :param err_param_tuple: 错误信息参数列表、元组 或单个字符串
        :param sug_param_tuple: 建议信息参数列表、元组 或单个字符串
        :param err_code: 错误码。自定义异常可采用默认-1
        :return:
        """
        err_msg, sug_msg = lang.getErrAndSugDesc(context, err_key, sug_key,
                                                 err_param_tuple,
                                                 sug_param_tuple)
        if lang.isChinese(context):
            return exception.newToolException(err_msg, sug_msg, "", "",
                                              err_code)
        else:
            return exception.newToolException("", "", err_msg, sug_msg,
                                              err_code)

    @staticmethod
    def newUnexpectedToolException(str_err_info_en=""):
        """生成新的工具异常对象：不可预知异常(包括进入原本不可能进入的逻辑条件分支等)

        :param str_err_info_en: 英文错误信息，默认为空
        :return:
        """
        return exception.newToolException(u"未知工具异常。" + str_err_info_en,
                                          const.HELP_DESC_ZH,
                                          "Unknown tool "
                                          "exception." + str_err_info_en,
                                          const.HELP_DESC_EN)

    @staticmethod
    def getToolExceptionByErrId(ex):
        """通过错误编号生成新的工具异常对象
            Params：框架抛出的ToolException中getErrorId()的值，接口定义的参数
            详细信息如下：。
            1 ----鉴权错误（如：密码错误）
            2 ----连接异常（如：超时异常,IP 不通）
            3 ----用户状态异常（只适用于TLV连接，如：未初始化）
            4 ----设备状态异常（只适用于TLV连接，如：开工失败）
            9 ----其他异常
        :param ex:
        :return:
        """
        # 容错：保证输入参数为整数
        error_id = 0
        # noinspection PyBroadException
        try:
            error_id = int(str(ex.getErrorId()))
        except Exception:
            return exception.newUnexpectedToolException("non-integer Framework"
                                                        " Exception errorId="
                                                        + str(error_id))

        # 据错误编号设置异常信息
        if 1 == error_id:  # 鉴权错误（如：密码错误）
            err_msg_zh = u"用户名或密码错误。"
            suggestion_zh = u"请更正用户名和密码。"
            err_msg_en = "Incorrect username or password."
            suggestion_en = "Enter correct username and password."
        elif 2 == error_id:  # 连接异常（如：超时异常,IP 不通）
            err_msg_zh = u"设备连接异常。"
            suggestion_zh = u"请确保设备和网络状态正常。"
            err_msg_en = "Abnormal device connection."
            suggestion_en = "Ensure that the status of the device and " \
                            "network is normal."
        elif 3 == error_id:  # 用户状态异常（只适用于TLV连接，如：未初始化）
            err_msg_zh = ex.getErrMsg()
            suggestion_zh = ""
            err_msg_en = ex.getErrMsg()
            suggestion_en = ""
        elif 4 == error_id:  # 设备状态异常（只适用于TLV连接，如：开工失败）
            err_msg_zh = u"设备状态异常。"
            suggestion_zh = u"请确保设备状态正常。"
            err_msg_en = "Abnormal device status."
            suggestion_en = "Ensure that the device status is normal."
        elif 9 == error_id:  # 工具框架返回的其他异常类型
            err_msg_zh = ex.getErrMsg()
            suggestion_zh = ""
            err_msg_en = ex.getErrMsg()
            suggestion_en = ""
        elif 1077949002 == error_id:
            err_msg_zh = u"获取系统信息失败。"
            suggestion_zh = const.HELP_DESC_ZH
            err_msg_en = "Cannot get the system infomation."
            suggestion_en = const.HELP_DESC_EN
        else:  # 其他异常
            err_msg_zh = u"工具框架运行异常。"
            suggestion_zh = const.HELP_DESC_ZH
            err_msg_en = "The tool works abnormally."
            suggestion_en = const.HELP_DESC_EN

        # 生成工具异常
        return exception.newToolException(err_msg_zh, suggestion_zh,
                                          err_msg_en, suggestion_en, error_id)

    @staticmethod
    def newFrameworkException(err_msg, suggestion):
        """生成新的框架异常对象

        :param err_msg: 异常描述
        :param suggestion: 处理建议
        :return:
        """
        return Exception(exception.FRAMEWORK_EXCEPTION, err_msg, suggestion)

    @staticmethod
    def handler(context, ex):
        """工具通用异常处理器

        :param context: 工具上下文
        :param ex: 工具通用异常
        :return:
        """
        # 参数检测，若入口参数无效，则抛出异常
        # 若工具上下文无效，则抛出新异常，让工具终止运行
        if context is None:
            raise Exception(u"无效的工具上下文。", const.HELP_DESC_ZH,
                            "Invalid tool context.", const.HELP_DESC_EN)

        # 若ex不为有效异常，则报错
        elif not isinstance(ex, Exception) or len(ex.args) == 0:
            log.error(context,
                      "the param ex is not a invalid exception. "
                      "ex=%s" % unicode(ex))
            try:
                log.error(context,
                          "the call trace of exception as follows:"
                          "\n%s" % traceback.format_exc())
            except Exception as exx:
                log.error(context,
                          "exception procce failed, exx=%s" % unicode(exx))
            result.setResultFail(context,
                                 u"系统处理消息失败。", const.HELP_DESC_ZH,
                                 "Failed to process the message.",
                                 const.HELP_DESC_EN)
            return False

        # 按异常的类型设置context
        if exception.ALL_LANG_EXCEPTION == ex.args[0]:  # 标准工具异常
            result.setResultFail(context, ex.args[1], ex.args[2], ex.args[3],
                                 ex.args[4])
            is_succ = True

        elif exception.FRAMEWORK_EXCEPTION == ex.args[0]:  # 封装的框架异常
            result.setResultFailByDesc(context, ex.args[1], ex.args[2])
            is_succ = True

        elif ex.args is not None and len(ex.args) == 2:
            exp_args = ex.args
            err_code = str(exp_args[0]).strip()
            # 设备未连接
            if err_code in config.ERR_CODE_FAIL_COMMUNICATION:
                err_msg_zh = u"与设备通信异常。"
                suggestion_zh = u"请检查网络连接或设备状态是否正常。如果有任何" \
                                u"疑问，请联系技术支持工程师协助处理。"
                err_msg_en = "Communicating with the device failed."
                suggestion_en = "Please check that the network connection " \
                                "or the system is normal. If you have any " \
                                "problems, please contact technical support" \
                                " engineers for help."
                result.setResultFail(context, err_msg_zh, suggestion_zh,
                                     err_msg_en, suggestion_en)
                is_succ = True

            # 引擎无法监控
            elif err_code in config.ERR_CODE_ENGINE_CAN_NOT_MONITOR:
                err_msg_zh = u"引擎无法监控。"
                suggestion_zh = u"请查看系统中告警并按照告警修复建议进行处理。"
                err_msg_en = "The engine can not be monitored."
                suggestion_en = "Please rectify the faults by referring " \
                                "to recommended actions."
                result.setResultFail(context, err_msg_zh, suggestion_zh,
                                     err_msg_en, suggestion_en)
                is_succ = True

            # 命令执行失败
            elif err_code in config.ERR_CODE_CAN_NOT_EXECUTE:
                err_msg_zh = u"命令执行失败。"
                suggestion_zh = u""
                err_msg_en = "Command execute failed."
                suggestion_en = ""
                result.setResultFail(context, err_msg_zh, suggestion_zh,
                                     err_msg_en, suggestion_en)
                is_succ = True

            elif err_code == config.ERR_CODE_UPGRADE_ERROR_CODE:
                err_msg_zh = u"系统正在升级。"
                suggestion_zh = u""
                err_msg_en = "The system is being upgraded."
                suggestion_en = ""
                result.setResultFail(context, err_msg_zh, suggestion_zh,
                                     err_msg_en, suggestion_en)
                is_succ = True

            else:
                log.error(context,
                          "the param ex is not a standard exception. "
                          "ex=%s" % unicode(ex))
                err_msg_zh = u"系统处理消息失败。"
                suggestion_zh = u"请检查系统状态或稍后重试。如果有任何疑问，" \
                                u"请联系技术支持工程师协助处理。"
                err_msg_en = "Failed to process the message."
                suggestion_en = "Please check the system status or try " \
                                "again later. If you have any problems, " \
                                "please contact technical support engineers" \
                                " for help."
                result.setResultFail(context, err_msg_zh, suggestion_zh,
                                     err_msg_en, suggestion_en)
                is_succ = False
        else:
            log.error(context,
                      "the param ex is not a standard exception."
                      " ex=%s" % unicode(ex))
            err_msg_zh = u"系统处理消息失败。"
            suggestion_zh = u"请检查系统状态或稍后重试。如果有任何疑问，" \
                            u"请联系技术支持工程师协助处理。"
            err_msg_en = "Failed to process the message."
            suggestion_en = "Please check the system status or try again " \
                            "later. If you have any problems, please " \
                            "contact technical support engineers for help."
            result.setResultFail(context, err_msg_zh, suggestion_zh,
                                 err_msg_en, suggestion_en)
            is_succ = False

        # 先行记录异常的函数调用路径信息
        log.error(context,
                  "***********************************"
                  "GENERAL EXCEPTION HANDLER BEGIN"
                  "***********************************")
        log.error(context,
                  "the call trace of exception as follows:"
                  "\n%s" % traceback.format_exc())
        # 记录错误信息
        log.error(context, result.getResultInfo(context))
        log.error(context,
                  "***********************************"
                  "GENERAL EXCEPTION HANDLER END"
                  "***********************************")
        return is_succ


class log:
    """
    日志记录类：提供各种级别的日志记录入口，并打印日志记录函数信息，方便问题定位
    """

    @staticmethod
    def getCallerInfo(max_caller_level=5, skip_last_level=True):
        """日志记录函数的子函数，用于获取调用函数和调用行号

        :param max_caller_level: 最大调用关系层数
        :param skip_last_level: 是否忽略最近的调用关系
        :return:
        """
        # 初始化参数
        func_back = sys._getframe().f_back
        if skip_last_level:
            func_back = func_back.f_back
            max_caller_level -= 1

        # 生成函数调用关系
        caller_info = ""
        info_format = " Level:%s [%s:%s]%s"
        for i in range(0, max_caller_level):
            # 获取该级调用函数和行号
            if hasattr(func_back, "f_code") and hasattr(func_back, "f_lineno"):
                func_name = func_back.f_code.co_name
                linen_umber = func_back.f_lineno
                caller_info = info_format % (str(i), str(func_name),
                                             str(linen_umber), caller_info)
            else:
                break
            # 刷新Back函数
            if hasattr(func_back, "f_back"):
                func_back = func_back.f_back
            else:
                break

        # 返回函数调用关系
        return caller_info

    @staticmethod
    def debug(context, info):
        """记录调试信息info到工具日志中

        :param context: 工具上下文
        :param info: 要记录的信息
        :return:
        """
        log_info = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").debug('[ToolLog]:%s' % log_info)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. "
                            "info=%s" % log_info)

    @staticmethod
    def error(context, info):
        """记录错误信息info到工具日志中

        :param context: 工具上下文
        :param info: 要记录的信息
        :return:
        """
        log_info = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").error('[ToolLog]:%s' % log_info)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. "
                            "info=%s" % log_info)

    @staticmethod
    def info(context, info):
        """记录普通信息info到工具日志中

        :param context: 工具上下文
        :param info: 要记录的信息
        :return:
        """
        log_info = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").info('[ToolLog]:%s' % log_info)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. "
                            "info=%s" % log_info)

    @staticmethod
    def warn(context, info):
        """记录警告信息info到工具日志中

        :param context: 工具上下文
        :param info: 要记录的信息
        :return:
        """
        log_info = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").warn('[ToolLog]:%s' % log_info)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. "
                            "info=%s" % log_info)

    @staticmethod
    def test(info):
        """记录调试信息

        :param info: 要记录的信息
        :return:
        """
        context = globalVar.getContext()
        log_info = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").info('[ToolLog]:%s' % log_info)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. "
                            "info=%s" % log_info)


class json:
    """
    JSON字符串相关函数类：实现JSON字符串和字典（或字典列表）间的相互转换，
    以及其他相关子函数
    """

    @staticmethod
    def decodeUtf8ToUnicode(dict_obj):
        """将utf-8编码的字典中的所有元素转换为unicode

        :param dict_obj: 字典对象
        :return:
        """
        new_dict = None
        if isinstance(dict_obj, dict):
            # 逐一将字典的key和value转换为unicode
            new_dict = {}
            for (key, value) in dict_obj.items():
                new_key = unicode(key, "utf-8")
                new_value = unicode(value, "utf-8")
                new_dict[new_key] = new_value
        return new_dict

    @staticmethod
    def dictToStr(dict_obj):
        """将字典转换为JSON字符串
            字典的元素必须全为unicodea编码，若为其他编码格式，转换时将出现异常
        :param dict_obj:
        :return:
        """
        final_str = None
        if isinstance(dict_obj, dict):
            str_dict_list = []
            for (key, value) in dict_obj.items():
                # 转换为unicode格式的str
                str_key = formatUtil.getUnicodeStr(key)
                str_value = formatUtil.getUnicodeStr(value)
                # 构建字典字符串
                str_dict = '\"%s\":\"%s\"' % (str_key, str_value)
                str_dict_list.append(str_dict)
            final_str = '{%s}' % ','.join(str_dict_list)
        return final_str

    @staticmethod
    def toStr(py_obj):
        """生成显示需要的字符串时使用：将字典或字典列表转换为JSON字符串
            UI显示支持的含中文的Json字符串格式:
            界面上能够正常显示的json字符串，返回的字符串（特别是中文）必须为下面的
                格式才行。
            即中文字符必须使用unicode编码，且直接与其他部分组合成字符串，unicode
                编码的中文字符无需转str，强转时会报错。
        :param py_obj: 字典或字典列表
        :return:
        """
        # 若输入为字典
        if isinstance(py_obj, dict):
            return json.dictToStr(py_obj)
        # 若输入为字典列表
        elif isinstance(py_obj, list):
            str_obj_list = []
            for index in range(0, len(py_obj)):
                str_dict = json.dictToStr(py_obj[index])
                if str_dict is not None:
                    str_obj_list.append(str_dict)
                else:
                    return "Err: invalid sub dict: index=%s, dict=%s" % (
                        str(index), str(py_obj[index]))
            str_obj = '[%s]' % ','.join(str_obj_list)
            return str_obj
        # 其他情况
        else:
            return "Err: invalid pyobj=%s" % str(py_obj)

    @staticmethod
    def toDict(str_json):
        """将JSON字符串转换为字典
            字典的数据中不能带有单引号
        :param str_json: JSON字符串
        :return:
        """
        return ast.literal_eval(str_json) if str_json else None


class result:
    """
    处理结果设置类：在此处统一设置脚本执行后的结果，修复建议，和错误提示信息到context中
    设置统一的结果返回设置函数
    """

    @staticmethod
    def test(context):
        """接口测试

        :param context: 上下文
        :return:
        """
        # 单Key语言设置接口测试
        result.setResultFailByKey(context, "EXEC_TLV_CMD_FAIL")
        log.error(context, "1_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        result.setResultFailByKey(context, "EXEC_TLV_CMD_FAIL", None, None)
        log.error(context, "2_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        # 双Key语言设置接口测试
        # 无参
        result.setResultFailByKeys(context, "EXEC_TLV_CMD_FAIL", "KEY_TEST")
        log.error(context, "3_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        # 字符串
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST",
                                   "[err]", "[sug]")
        log.error(context, "4_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        # 中英文列表，半空
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST",
                                   ["[err]", u"[参数]"], None)
        log.error(context, "5_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        # 半空，英中文列表
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST",
                                   None, [u"[错误]", "[param]"])
        log.error(context, "6_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        # 元组
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST",
                                   ("[err]", "[param]"), None)
        log.error(context, "7_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        # 元组
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST",
                                   None, ("[err]", "[param]"))
        log.error(context, "8_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        # 空元组，空列表
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST", (), ())
        log.error(context, "9_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        # 过多参数元组&列表
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST",
                                   ("[err]", "[param]", "123"),
                                   ["[err]", "[param]", "123"])
        log.error(context, "10_succ=%s" % str(context["succ"]))
        log.error(context, "errMsg=%s" % context["errMsg"])
        log.error(context, "suggestion=%s" % context["suggestion"])

        return

    # 判断结果
    @staticmethod
    def isResultFail(context):
        """判断结果是否为失败

        :param context: 上下文
        :return:
        """
        return context.get("succ") is False

    @staticmethod
    def setResult(context, succ, err_msg_zh, suggestion_zh, err_msg_en,
                  suggestion_en):
        """设置结果信息

        :param context: 上下文
        :param succ: 结果标记
        :param err_msg_zh: 错误消息中文
        :param suggestion_zh: 修复建议中文
        :param err_msg_en: 错误消息英文
        :param suggestion_en: 修复建议英文
        :return:
        """
        context["succ"] = succ
        lan = context.get("lan")
        if lan == "zh":
            context["errMsg"] = err_msg_zh
            context["suggestion"] = suggestion_zh
        else:
            context["errMsg"] = err_msg_en
            context["suggestion"] = suggestion_en
        return

    @staticmethod
    def setResultFail(context, err_msg_zh, suggestion_zh, err_msg_en,
                      suggestion_en):
        """设置结果为失败

        :param context: 上下文
        :param err_msg_zh: 错误消息中文
        :param suggestion_zh: 修复建议中文
        :param err_msg_en: 错误消息英文
        :param suggestion_en: 修复建议英文
        :return:
        """
        result.setResult(context, False, err_msg_zh, suggestion_zh,
                         err_msg_en, suggestion_en)
        log.error(context, "set result to fail.")
        return

    @staticmethod
    def setResultPass(context):
        """设置结果为通过

        :param context: 上下文
        :return:
        """
        result.setResult(context, True, "", "", "", "")
        log.info(context, "set result to pass.")
        return

    @staticmethod
    def setResultFailByKey(context, msg_key, errParamTuple=(),
                           sugParamTuple=()):
        """使用相同的key设置错误和建议语言内容。
            用于独立拥有错误和建议信息的语言描述项。
        :param context: 上下文
        :param msg_key: 错误消息key
        :param errParamTuple: 错误消息参数元组
        :param sugParamTuple: 修复建议参数元组
        :return:
        """
        result.setResultFailByKeys(context, msg_key, msg_key,
                                   errParamTuple, sugParamTuple)
        return

    @staticmethod
    def set_result_fail_by_err_keys(context, key_list, err_param_tuple=(), sug_param_tuple=()):
        """
        用于处理多个key拼接的情况，具体格式如下
        1、错误信息1； 2、错误信息2； ...
        1、修复建议
        """
        # 设置结果为不通过
        context["succ"] = False

        total_err_msg = ""
        total_sug_msg = ""
        # 循环遍历key_list,获取并设置错误和建议提示信息
        for index, key in enumerate(key_list):
            cur_err_msg, cur_sug_msg = lang.getErrAndSugDesc(context, key, key,
                                                             err_param_tuple,
                                                             sug_param_tuple)
            total_err_msg += u"{}、{}; ".format(index + 1, cur_err_msg)
            if not total_sug_msg:
                total_sug_msg = cur_sug_msg

        context["errMsg"] = total_err_msg
        context["suggestion"] = total_sug_msg
        log.error(context, "set result to fail.")

    @staticmethod
    def setResultFailByKeys(context, err_key, sug_key, err_param_tuple=(),
                            sug_param_tuple=()):
        """直接填写错误和建议的语言描述key，内部自动从lang类获取语言并填充参数
            用于精简语言描述内容，错误和建议的Key不同的情况下。
        :param context: 工具上下文
        :param err_key: 错误描述Key
        :param sug_key: 建议描述Key
        :param err_param_tuple: 错误信息参数列表、元组 或单个字符串
        :param sug_param_tuple: 建议信息参数列表、元组 或单个字符串
        :return:
        """
        # 设置结果为不通过
        context["succ"] = False

        # 获取并设置错误和建议提示信息
        err_msg, sug_msg = lang.getErrAndSugDesc(context, err_key, sug_key,
                                                 err_param_tuple,
                                                 sug_param_tuple)
        context["errMsg"] = err_msg
        context["suggestion"] = sug_msg

        log.error(context, "set result to fail.")
        return

    @staticmethod
    def getMsg(context, err_key, sug_key, err_param_tuple=(),
               sug_param_tuple=()):
        """获取资源描述

        :param context: 上下文
        :param err_key: 错误描述Key
        :param sug_key: 建议描述Key
        :param err_param_tuple: 错误信息参数列表、元组 或单个字符串
        :param sug_param_tuple: 建议信息参数列表、元组 或单个字符串
        :return:
        """
        # 获取并设置错误和建议提示信息
        return lang.getErrAndSugDesc(context, err_key, sug_key,
                                     err_param_tuple, sug_param_tuple)

    @staticmethod
    def setResultFailByDesc(context, err_msg, suggestion):
        """直接填写语言描述，用于和lang类配合使用(初级)

        :param context: 上下文
        :param err_msg: 错误消息
        :param suggestion: 修复建议
        :return:
        """
        context["succ"] = False
        context["errMsg"] = err_msg
        context["suggestion"] = suggestion
        return

    @staticmethod
    def getResultInfo(context):
        """获取context中的返回值描述信息

        :param context: 上下文
        :return:
        """
        err_msg = unicode(context.get("errMsg", ""), "utf-8")
        sug_msg = unicode(context.get("suggestion", ""), "utf-8")
        if context.get("succ") is True:  # 执行成功
            rst_info = "CONTEXT_RESULT_INFO: succ=True"

        elif context.get("succ") is False:  # 执行失败
            rst_info = "CONTEXT_RESULT_INFO: succ=False, errMsg=%s, " \
                       "suggestion=%s" % (err_msg, sug_msg)

        else:  # 未定义状态
            rst_info = "CONTEXT_RESULT_INFO: succ=Undefined Value:[%s], " \
                       "errMsg=%s, suggestion=%s" % (str(context.get("succ")),
                                                     err_msg, sug_msg)
        return rst_info


class lang:
    """
    语言信息类：实现工具中自定义语言描述的自动获取
    实现方式：将语言文件中的信息初始化并保存到context中，后续获取仅从context中的
        对象获取即可，且语言文件内容变动时自动刷新信息
    """
    # 信息保持项ID定义
    # DESC模块
    PERSIST_MODULE_DESC = "TOOL_DESC_MODULE"
    # 英文语言项列表
    PERSIST_ITEM_ENLIST = "DESC_LIST_ENGLISH"
    # 中文语言项列表
    PERSIST_ITEM_ZHLIST = "DESC_LIST_CHINESE"
    # 语言文件的修改时间。当语言文件的修改时间晚于context中记录的时间时，
    # 自动刷新context中的语言描述
    PERSIST_ITEM_UPDATE_TIME = "LANGUAGE_UPDATE_TIME"

    # 语言描述Key头
    HEAD_ERR = "ERR@"
    HEAD_SUG = "SUG@"

    @staticmethod
    def isChinese(context):
        """判断UI语言是否为中文

        :param context: 上下文
        :return:
        """
        return context.get("lan") == "zh"

    @staticmethod
    def isEnglish(context):
        """判断UI语言是否为英文
        :param context: 上下文
        :return:
        """
        return context.get("lan") == "en"

    @staticmethod
    def getErrAndSugDesc(context, err_key, sug_key, err_param_tuple=(),
                         sug_param_tuple=()):
        """据语言KEY获取错误和建议描述

        :param context: 工具上下文
        :param err_key: 错误描述Key
        :param sug_key: 建议描述Key
        :param err_param_tuple: 错误信息参数列表、元组 或单个字符串
        :param sug_param_tuple: 建议信息参数列表、元组 或单个字符串
        :return:
        """
        # 设置错误提示信息
        err_msg = ""  # 默认为空
        if err_key:
            if err_param_tuple is None:  # 传入参数为None
                err_msg = lang.getDesc(context, lang.HEAD_ERR + err_key)
            # 传入参数为元组或列表
            elif isinstance(err_param_tuple, (tuple, list)):
                err_msg = lang.getDesc(context,
                                       lang.HEAD_ERR + err_key,
                                       *err_param_tuple)
            # 传入参数为单个字符串或其他类型
            else:
                err_msg = lang.getDesc(context,
                                       lang.HEAD_ERR + err_key,
                                       formatUtil.getUnicodeStr(
                                           err_param_tuple))

        # 设置错误建议信息
        # noinspection PyBroadException
        try:
            sug_msg = ""  # 默认为空
            if sug_key:
                if sug_param_tuple is None:  # 传入参数为None
                    sug_msg = lang.getDesc(context, lang.HEAD_SUG + sug_key)
                # 传入参数为元组或列表
                elif isinstance(sug_param_tuple, (tuple, list)):
                    sug_msg = lang.getDesc(context,
                                           lang.HEAD_SUG + sug_key,
                                           *sug_param_tuple)
                # 传入参数为单个字符串
                else:
                    sug_msg = lang.getDesc(context,
                                           lang.HEAD_SUG + sug_key,
                                           formatUtil.getUnicodeStr(
                                               sug_param_tuple))
        except Exception:
            if lang.isChinese(context):
                sug_msg = const.SUG_DESC_ZH
            else:
                sug_msg = const.SUG_DESC_EN
        return err_msg, sug_msg

    @staticmethod
    def getDesc(context, desc_key, *params):
        """据Context中的当前语言类型获取描述

        :param context: 上下文
        :param desc_key: 描述的key
        :param params: 参数
        :return:
        """
        return lang.getDescByLang(context, context.get("lan"), desc_key,
                                  *params)

    @staticmethod
    def getDescZh(context, desc_key, *params):
        """获取中文语言描述

        :param context: 上下文
        :param desc_key: 描述的key
        :param params: 参数
        :return:
        """
        return lang.getDescByLang(context, "zh", desc_key, *params)

    @staticmethod
    def getDescEn(context, desc_key, *params):
        """获取英文语言描述

        :param context: 上下文
        :param desc_key: 描述的key
        :param params: 参数
        :return:
        """
        return lang.getDescByLang(context, "en", desc_key, *params)

    @staticmethod
    def getDescByLang(context, lang_type, desc_key, *params):
        """获取语言描述（自动据context中的语言类型来区分中英文）

        :param context: 工具上下文
        :param lang_type: 语言类型
        :param desc_key: 语言描述Key
        :param params: 语言描述参数
        :return:
        """
        # 获取语言描述模块
        desc_module = lang.getDescModule(context)
        if desc_module is None:
            log.error(context,
                      "fail to get the desc of key [%s] since desc module is "
                      "inexisted." % desc_key)
            lang.raiseErrorInfo()
        # 获取语言描述列表
        if lang_type == "zh":  # 中文
            desc_list = desc_module.get(lang.PERSIST_ITEM_ZHLIST)
        else:  # 英文
            desc_list = desc_module.get(lang.PERSIST_ITEM_ENLIST)
        if desc_list is None:
            log.error(context,
                      "fail to get the desc of key [%s] since desc list is "
                      "inexisted." % desc_key)
            lang.raiseErrorInfo()

        # 获取并返回语言描述信息
        str_desc = ""
        if desc_key in desc_list:
            str_desc = desc_list.get(desc_key)
        else:
            log.error(context,
                      "the key [%s] is not existed in desc list." % desc_key)
            lang.raiseErrorInfo()

        # 替换所有参数并返回结果
        # 注意：描述行中的参数使用{index}进行标记，其中index为参数的位置，从0开始
        param_num = len(params)
        for index in range(0, param_num):
            str_desc = str_desc.replace("{%s}" % str(index), params[index])
        return str_desc

    @staticmethod
    def initDescList(context):
        """从语言文件生成语言描述列表，并保存在context中

        :param context: 工具上下文
        :return:
        """
        lang_file_path = context.get("languageTxtFile")
        # 判断语言文件是否存在
        if not os.path.isfile(lang_file_path):
            log.error(context,
                      "the lang file %s is not existed." % lang_file_path)
            raise exception.newToolException(u"工具语言文件不存在。",
                                             const.HELP_DESC_ZH,
                                             "Non-existent tool language "
                                             "file.",
                                             const.HELP_DESC_EN)

        # 读取语言文件内容
        desc_file = None
        # noinspection PyBroadException
        try:
            # 打开并读回所有描述文件信息
            desc_file = open(lang_file_path)
            all_desc_info = desc_file.read()
        except Exception:
            log.error(context, "fail to read lang file's contents.")
            all_desc_info = None
        finally:
            if desc_file:
                desc_file.close()
        if all_desc_info is None:
            raise exception.newToolException(u"读取工具语言文件失败。",
                                             const.HELP_DESC_ZH,
                                             "Failed to read the tool "
                                             "language file.",
                                             const.HELP_DESC_EN)

        # 语言文件格式标识定义
        zh_line_prefix = "ZH>"  # 中文语言行描述前缀
        en_line_prefix = "EN>"  # 英文语言行描述前缀
        sec_separator = "="  # KEY和语言描述间的分隔符

        # 解析描述文件信息到中英文语言字典
        zh_desc_dict = {}
        en_desc_dict = {}
        all_desc_info = all_desc_info.splitlines()
        line_num = len(all_desc_info)
        for index in range(0, line_num):
            desc_line = all_desc_info[index]
            desc_line = desc_line.lstrip()  # 去除左边的空格

            if "" == desc_line.strip(" "):
                # 跳过空行
                continue
            elif desc_line.startswith("#"):
                # 跳过注释行
                continue
            elif desc_line.startswith(zh_line_prefix):
                # 为中文语言描述行
                # 去除语言描述前缀
                desc_line = desc_line[len(zh_line_prefix):].lstrip()
                # 获取语言key和内容
                eq_index = desc_line.index(sec_separator)
                desc_key = desc_line[:eq_index].rstrip()
                desc_value = desc_line[eq_index + 1:].decode("utf-8")
                # 设置到语言字典
                zh_desc_dict[desc_key] = desc_value
                continue
            elif desc_line.startswith(en_line_prefix):
                # 为英文语言描述行
                # 去除语言描述前缀
                desc_line = desc_line[len(en_line_prefix):].lstrip()
                # 获取语言key和内容
                eq_index = desc_line.index(sec_separator)
                desc_key = desc_line[:eq_index].rstrip()
                desc_value = desc_line[eq_index + 1:].decode("utf-8")
                # 设置到语言字典
                en_desc_dict[desc_key] = desc_value
                continue
            else:
                # 记录错误并记录无效行
                log.error(context,
                          "invalid line(%s): %s" % (str(index + 1), desc_line))
                continue

        # 保存语言字典到context中，并返回初始化成功
        persist.setObject(context, lang.PERSIST_MODULE_DESC,
                          lang.PERSIST_ITEM_ZHLIST, zh_desc_dict)
        persist.setObject(context, lang.PERSIST_MODULE_DESC,
                          lang.PERSIST_ITEM_ENLIST, en_desc_dict)
        # 初始化语言文件修改时间
        cur_update_time = time.ctime(os.stat(lang_file_path).st_mtime)
        log.info(context, "curUpdateTime=%s" % str(cur_update_time))
        persist.setObject(context, lang.PERSIST_MODULE_DESC,
                          lang.PERSIST_ITEM_UPDATE_TIME, cur_update_time)
        log.info(context, "succeed to init the language desc module.")
        return True

    @staticmethod
    def getDescModule(context):
        """获取语言描述模块

        :param context: 工具上下文
        :return:
        """
        # 从context中获取已保存的语言模块，若存在且未修改，则直接返回
        desc_module = persist.getModule(context, lang.PERSIST_MODULE_DESC)
        lang_file_path = context.get("languageTxtFile")
        if desc_module:
            last_update_time = persist.getObject(context,
                                                 lang.PERSIST_MODULE_DESC,
                                                 lang.PERSIST_ITEM_UPDATE_TIME)
            # 语言文件的当前修改时间
            cur_update_time = time.ctime(os.stat(lang_file_path).st_mtime)
            # 如果初始化时记录的文件修改时间和当前查出的文件修改时间相等，
            # 则认为未更新过。否则删除已有语言描述，重新初始化
            if cur_update_time == last_update_time:
                return desc_module
            else:
                persist.delModule(context, lang.PERSIST_MODULE_DESC)

        # 不存在时，则从语言文件创建语言模块并保存在context中
        if lang.initDescList(context) is not True:
            log.error(context, "fail to init lang desc list.")
            return None

        # 再次获取已保存的语言模块
        return persist.getModule(context, lang.PERSIST_MODULE_DESC)

    @staticmethod
    def raiseErrorInfo():
        """封装异常错误信息

        :return:
        """
        raise exception.newToolException(const.ERR_DESC_ZH,
                                         const.SUG_DESC_ZH,
                                         const.ERR_DESC_EN,
                                         const.SUG_DESC_EN)


class ttime:
    """
    工具时间类（tool time）：用于实现工具时间的相关转换
    """

    @staticmethod
    def getTimeStrNow(str_format='%Y-%m-%d %H:%M:%S'):
        """获取当前时间的字符串

        :param str_format: 时间字符串格式
        :return:
        """
        return time.strftime(str_format)

    @staticmethod
    def getTimeStrFromObj(obj_time, str_format='%Y-%m-%d %H:%M:%S'):
        """从时间对象生成时间字符串

        :param obj_time: 时间对象
        :param str_format: 时间字符串格式
        :return:
        """
        return datetime.datetime.strftime(obj_time, str_format)

    @staticmethod
    def getTimeStrByTimeStamp(int_stmp_value, str_format='%Y-%m-%d %H:%M:%S'):
        """从时间戳（数值）生成时间字符串

        :param int_stmp_value: 时间戳（数值）
        :param str_format: 时间字符串格式
        :return:
        """
        # 转换为时间结构体。传入的时间戳必须为整形
        tmp_value = time.localtime(int_stmp_value)
        # 转为正常日期格式
        str_time = time.strftime(str_format, tmp_value)
        return str_time

    @staticmethod
    def getTimeStampByTimeStr(str_time, str_format='%Y-%m-%d %H:%M:%S'):
        """从时间字符串生成对应的时间戳

        :param str_time: 时间字符串
        :param str_format: 时间字符串格式
        :return:
        """
        stmp = time.mktime(time.strptime(str_time, str_format))
        return int(stmp)

    @staticmethod
    def getTimeObjNow():
        """获取当前时间对象

        :return:
        """
        return datetime.datetime.now()

    @staticmethod
    def getTimeObjFromStr(str_time, str_format='%Y-%m-%d %H:%M:%S'):
        """从时间字符串生成时间对象

        :param str_time: 时间字符串
        :param str_format: 时间字符串格式
        :return:
        """
        return datetime.datetime.strptime(str_time, str_format)

    @staticmethod
    def getTimeObjBeforeSeconds(obj_time, long_seconds):
        """获取指定时间之前longSeconds的时间对象

        :param obj_time: 指定时间
        :param long_seconds: 时长（秒）
        :return:
        """
        return obj_time - datetime.timedelta(seconds=long_seconds)

    @staticmethod
    def getTimeObjAfterSeconds(obj_time, long_seconds):
        """获取指定时间之后longSeconds的时间对象

        :param obj_time: 指定时间
        :param long_seconds: 时长（秒）
        :return:
        """
        return obj_time + datetime.timedelta(seconds=long_seconds)


class formatUtil:
    """
    数据格式类：将信息保存在上下文中，方便无调用关系的脚本中分享信息
    """

    @staticmethod
    def isInteger(str_obj):
        """是否为整数格式

        :param str_obj: 字符串对象
        :return:
        """
        return re.search("^[1-9][0-9]*$", str(str_obj)) is not None

    @staticmethod
    def getUnicodeStr(obj):
        """将obj转换为字符串

        :param obj: 对象
        :return:
        """
        # 常规字符串或unicode字符串，直接返回
        if isinstance(obj, (str, unicode)):
            return obj
        # 正数、字典、元组、列表
        elif isinstance(obj, (int, float, bool, dict, tuple, list, long)):
            return str(obj)
        elif str(type(obj)) == "<type 'instance'>":
            return str(obj)
        else:  # 含有非英文字符
            return obj.decode("utf-8")

    @staticmethod
    def getUnicodeStrItemList(obj_list):
        """将objList中的所有元素转换为字符串，并重新构造为列表

        :param obj_list: 对象列表
        :return:
        """
        if not isinstance(obj_list, (list, tuple)):
            return None

        str_list = []
        for obj in obj_list:
            str_list.append(formatUtil.getUnicodeStr(obj))

        return str_list


class persist:
    """
    持久化信息类：将信息保存在上下文中，方便无调用关系的脚本中分享信息
    """
    # 持久化信息在context中的工具数据对象KEY
    ROOT = "tool_persist_data"

    @staticmethod
    def isObjectExisted(context, mdl_name, obj_name):
        """判断指定模块中的指定对象是否存在

        :param context: 上下文
        :param mdl_name: 指定模块
        :param obj_name: 指定对象
        :return:
        """
        # noinspection PyBroadException
        try:
            return persist.getObject(context, mdl_name, obj_name) is not None
        except Exception:
            return False

    @staticmethod
    def getObject(context, mdl_name, obj_name):
        """获取指定模块中的指定对象值

        :param context: 上下文
        :param mdl_name: 指定模块
        :param obj_name: 指定对象
        :return:
        """
        # 入口参数检查
        if context is None or not mdl_name or not obj_name:
            return None
        # 获取context中的工具数据字典
        data_dict = context.get(persist.ROOT, {})
        # 获取工具数据中相应module的字典
        mdl_dict = data_dict.get(mdl_name, {})

        # 获取模块数据中相应对象的信息
        return mdl_dict.get(obj_name)

    @staticmethod
    def setObject(context, mdl_name, obj_name, obj_value):
        """设置指定模块中的指定对象值

        :param context: 上下文
        :param mdl_name: 指定模块
        :param obj_name: 指定对象
        :param obj_value: 值
        :return:
        """
        # 入口参数检查
        if context is None or not mdl_name or not obj_name:
            return False

        # 获取context中的工具数据字典，不存在则创建
        data_dict = context.setdefault(persist.ROOT, {})
        # 获取工具数据中相应module的字典，不存在则创建
        mdl_dict = data_dict.setdefault(mdl_name, {})

        # 设置模块数据中相应对象的信息
        mdl_dict[obj_name] = obj_value
        # 刷新dataDict对象：保险措施，防备dataDict对象值传不到context中
        context[persist.ROOT] = data_dict
        # 返回设置结果
        log.info(context,
                 "succeed to set tool object data of context. mdlName=%s, "
                 "objName=%s" % (mdl_name, obj_name))
        return True

    @staticmethod
    def delObject(context, mdl_name, obj_name):
        """删除指定模块下的指定对象数据

        :param context: 上下文
        :param mdl_name: 指定模块
        :param obj_name: 指定对象
        :return:
        """
        # 入口参数检查
        if context is None or not mdl_name or not obj_name:
            return False

        # 获取context中的工具数据字典
        data_dict = context.get(persist.ROOT, {})
        # 获取工具数据中相应module的字典
        mdl_dict = data_dict.get(mdl_name, {})
        # 删除模块数据中相应对象的信息
        if obj_name in mdl_dict:
            del mdl_dict[obj_name]

        # 返回删除结果
        log.info(context,
                 "succeed to delete tool object data of context. "
                 "mdlName=%s, objName=%s" % (mdl_name, obj_name))
        return True

    @staticmethod
    def isModuleExisted(context, mdl_name):
        """指定模块是否存在

        :param context: 上下文
        :param mdl_name: 指定模块
        :return:
        """
        # noinspection PyBroadException
        try:
            return persist.getModule(context, mdl_name) is not None
        except Exception:
            return False

    @staticmethod
    def getModule(context, mdl_name):
        """获取指定的模块对象

        :param context: 上下文
        :param mdl_name: 指定模块
        :return:
        """
        # 入口参数检查
        if not context or not mdl_name:
            return None
        # 获取context中的工具数据字典
        data_dict = context.get(persist.ROOT, {})
        # 获取工具数据中相应module的字典
        mdl_dict = data_dict.get(mdl_name)
        return mdl_dict

    @staticmethod
    def setModule(context, mdl_name, mdl_value):
        """设置指定的模块

        :param context: 上下文
        :param mdl_name: 指定模块
        :param mdl_value: 模块的值
        :return:
        """
        # 入口参数检查
        if context is None or not mdl_name:
            return False
        # 获取context中的工具数据字典，不存在则创建
        data_dict = context.setdefault(persist.ROOT, {})

        # 设置工具数据中相应module的字典
        data_dict[mdl_name] = mdl_value
        # 刷新dataDict对象：保险措施，防备dataDict对象值传不到context中
        context[persist.ROOT] = data_dict
        # 返回设置结果
        log.info(context,
                 "succeed to set tool module data of context. "
                 "mdlName=%s" % mdl_name)
        return True

    @staticmethod
    def delModule(context, mdl_name):
        """删除指定的模块数据

        :param context: 上下文
        :param mdl_name: 指定模块
        :return:
        """
        # 入口参数检查
        if context is None or not mdl_name:
            return False

        if persist.ROOT in context:
            # 获取context中的工具数据字典
            data_dict = context.get(persist.ROOT, {})
            # 删除工具数据中相应module的字典
            if mdl_name in data_dict:
                del data_dict[mdl_name]
            # 刷新dataDict对象：保险措施，防备dataDict对象值传不到context中
            context[persist.ROOT] = data_dict

        # 返回删除结果
        log.info(context,
                 "succeed to delete tool module data of context. "
                 "mdlName=%s" % mdl_name)
        return True

    @staticmethod
    def delAll(context):
        """删除所有工具的持久化数据

        :param context: 上下文
        :return:
        """
        # 入口参数检查
        if context is None:
            return False
        # 删除context中的工具数据字典
        if persist.ROOT in context:
            del context[persist.ROOT]
        # 返回删除结果
        log.info(context, "succeed to delete all tool data of context.")
        return True

    @staticmethod
    def getInnerIps(context):
        """获取内部IP列表

        :param context: 上下文
        :return:
        """
        return context.get("DEV_INNER_IP_LIST")
