﻿# -*- coding: UTF-8 -*-
import ast
import os
import re
import codecs

from common import BaseFactory

'''
模块测试入口，供TestFactory调用
'''


def testStatEntrance(context):
    # 清除代码中的所有签名文件
    cleanAllSignFiles(context)

    # 从代码文件中搜索指定指定的信息
    findAllKeyInfoToFile(context)

    return


'''
清除代码中的所有签名文件
'''


def cleanAllSignFiles(context):
    # 工具脚本目录：
    TOOL_PATH = os.path.dirname(__file__) + "\\..\\..\\"  # S5000TV200R002C10
    # 保存命令信息列表的文件
    LOG_FILE_PATH = os.path.dirname(__file__) + "\\..\\..\\res\\TOOL_CLEAN_SIGN_LOG.txt"

    # 遍历所有的代码文件，获取其中所有使用TLV命令的地方
    allInfoLines = traverse(context, TOOL_PATH, cleanSignFile)
    isSucc = appendFileContents(LOG_FILE_PATH, allInfoLines)
    BaseFactory.log.info(context, "write succ=" + str(isSucc) + ", cmd infos as follows:\n" + allInfoLines)
    return


'''
Function: 若文件为签名文件，则删除之
Return: 删除文件名称 或 ""
Attention: 下面的函数千万不能动，否则有可能删除所有文件
'''


def cleanSignFile(context, filePath):
    # 签名文件扩展名列表
    fileExtList = [".sign"]

    # 与文件扩展名进行签名文件删除
    delFileInfo = ""
    curFileExt = os.path.splitext(filePath)[1]
    if curFileExt in fileExtList:
        os.remove(filePath)
        delFileInfo = filePath
    else:
        delFileInfo = ""

    return delFileInfo


'''
Function: 文件的递归遍历操作
遍历filePath下所有子目录中的文件，然后调用strFileOptFunc对每个文件进行处理并返回结果
Attention: filePath中不能包含非英文字符，如中文字符
'''


def traverse(context, filePath, strFileOptFunc, skipFileList=None):
    # 设置本次遍历的结果信息为空
    strInfos = ""

    # 若filePath为一个文件，则直接对其进行处理并返回结果
    if os.path.isfile(filePath):
        strInfos = strFileOptFunc(context, filePath)
        return strInfos

    # filePath为目录时，遍历当前文件目录下的所有文件和子目录下的文件
    for item in os.listdir(filePath):
        subpath = os.path.join(filePath, item)
        if None != skipFileList:
            needSkip = False
            for skipFile in skipFileList:
                if None != re.search(skipFile, subpath):
                    needSkip = True
                    break
            if True == needSkip:
                BaseFactory.log.info(context, "the file: %s is skipped." % subpath)
                continue

        tmpInfo = traverse(context, subpath, strFileOptFunc, skipFileList)  # 文件和目录均进入遍历
        if "" != tmpInfo:
            tmpInfo = tmpInfo.rstrip("\n")
            strInfos = strInfos + tmpInfo + "\n"

    # 返回遍历处理结果
    return strInfos


'''
Function: 获取文件中的所有内容
Return: 文件中的所有内容 或 None(文件不存在 或 获取失败)
'''


def getFileContents(filePath):
    # 参数检测
    if not os.path.exists(filePath):
        return None
    if not os.path.isfile(filePath):
        return None

    # 读取文件内容
    tmpFile = None
    allLines = ""
    try:  # 打开并读回所有描述文件信息
        tmpFile = open(filePath)
        allLines = tmpFile.read()
    except Exception:
        allLines = None
    finally:
        if None != tmpFile:
            tmpFile.close()

    # 返回获取到的文件内容（获取失败时为None）
    return allLines


'''
Function: 追加信息到文件（文件不存在则创建）
Return: True=成功；False=失败
'''


def appendFileContents(filePath, infos):
    # 参数检测: 文件所在目录必须存在
    if not os.path.exists(os.path.dirname(filePath)):
        return False

    # 写入文件内容
    isSucc = True
    try:
        with codecs.open(filePath, 'w+', encoding='utf-8') as f:
            f.write(str(infos))
    except Exception:
        isSucc = False

    # 返回写入结果
    return isSucc


'''
Function: 从指定文件中搜索包含任一KEY的行
Params:
    context=工具上下文，供调测使用，无法实际意义
    filePath=待搜索的文件的路径
    searchKeyList=搜索的KEY列表，不能为空
    fileExtList=需要搜索的文件类型（扩展名）列表，当为[]时，表示搜索全部类型文件
    skipCommentLine=是否跳过文件中的注释行，注释行以#开头
    skipFileWrnInfo=是否跳过警告信息，当文件不是指定类型的文件时，将提示告警信息
Return: 文件中的所有内容 或 None(文件不存在 或 获取失败)
Attention
'''


def searchStrInFile(context, filePath, searchKeyList=["Tlv.execmd"], fileExtList=[".py"],
                    skipCommentLine=False, skipFileWrnInfo=True, resultFunc=None):
    # 当前文件不用搜索，直接返回搜索结果为空（abspath会自动转换路径中的..信息）
    if os.path.abspath(__file__) == os.path.abspath(filePath):
        return ""

    # 参数检测
    if type(searchKeyList) != list or type(fileExtList) != list:
        return "[Error]: the searchKeyList(" + str(searchKeyList) \
               + ") or fileExtList(" + str(fileExtList) + ") is not list."
    if not os.path.exists(filePath):
        return "[Error]: the File(" + filePath + ") is not existed."
    if not os.path.isfile(filePath):
        return "[Error]: the File(" + filePath + ") is not a file."

    # 若对文件类型有要求，则当文件不为指定扩展名的文件，则直接返回
    curFileExt = os.path.splitext(filePath)[1]
    if len(fileExtList) > 0 and curFileExt not in fileExtList:
        if False == skipFileWrnInfo:
            return "[Warning]: the File(" + filePath + ") is not a requirred file."
        else:
            return ""

    # 获取文件中的命令内容
    allFileLines = getFileContents(filePath)
    if None == allFileLines:
        return "[Error]: fail to get the contents of file(" + filePath + ")."
    allFileLines = allFileLines.splitlines()

    # 遍历所有代码行，搜索含有cmdKey的行
    cmdInfoLines = ""
    lineNum = len(allFileLines)
    for index in range(0, lineNum):
        codeLine = allFileLines[index]
        codeLine = codeLine.lstrip()
        # 按要求跳过单注释行
        if True == skipCommentLine and codeLine.startswith("#"):
            continue
        # 搜索当前代码行中是否存在cmdKey，若存在则记录该行信息
        for item in searchKeyList:
            if None != re.search(item, codeLine):
                strTailer = "\n"
                # 如果结果处理函数不为空，则加上结果处理函数 的处理结果
                if None != resultFunc:
                    BaseFactory.log.error(context, "codeLine=" + codeLine)
                    strTailer = "|" + resultFunc(codeLine, item) + "\n"
                # 构造打印行
                cmdInfoLines = cmdInfoLines + os.path.abspath(filePath) + "|" \
                               + str(index + 1) + "L|" + codeLine + strTailer
                break

    # 返回搜索结果
    return cmdInfoLines


# 对搜索出的TLV信息行进行处理
def tlvCmdLineHandler(codeLine, matchKey):
    # 据关键字进行信息拆分
    tmpInfos = codeLine.split(matchKey)
    tmpInfos = tmpInfos[1].replace("\'", "\"").split("\"")
    if len(tmpInfos) > 1:
        return tmpInfos[1]
    else:
        return ""


'''
Function: 从代码文件中搜索使用到的TLV命令
Return: 文件中的所有内容 或 None(文件不存在 或 获取失败)
'''


def searchTlvCmd(context, filePath):
    # 命令信息列表
    searchKeyList = ["Tlv.execmd"]
    # 代码文件扩展名列表
    fileExtList = [".py"]
    # 是否忽略注释行
    skipCommentLine = False
    # 是否屏蔽文件错误信息
    skipFileWrnInfo = True

    # 执行搜索并返回结果
    return searchStrInFile(context, filePath, searchKeyList,
                           fileExtList, skipCommentLine,
                           skipFileWrnInfo, tlvCmdLineHandler)


# 对搜索出的TLV信息行进行处理
def tlvObjLineHandler(codeLine, matchKey):
    # 据关键字进行信息拆分
    tmpInfos = codeLine.split(matchKey)
    tmpInfos = tmpInfos[1].replace("\'", "\"").split("\"")
    if len(tmpInfos) > 1:
        return tmpInfos[1]  # ETH_PORT
    else:
        return ""


'''
Function: 从代码文件中搜索使用到的TLV对象
Return: 文件中的所有内容 或 None(文件不存在 或 获取失败)
'''


def searchTlvObj(context, filePath):
    # 命令信息列表
    searchKeyList = ["TlvEnum.Enum.OM_OBJ_E"]
    # 代码文件扩展名列表
    fileExtList = [".py"]
    # 是否忽略注释行
    skipCommentLine = False
    # 是否屏蔽文件错误信息
    skipFileWrnInfo = True

    # 执行搜索并返回结果
    return searchStrInFile(context, filePath, searchKeyList,
                           fileExtList, skipCommentLine,
                           skipFileWrnInfo, tlvObjLineHandler)


# 对搜索出的字符串信息行进行处理
def tmpInfoLineHandler(codeLine, matchKey):
    # 据关键字进行信息拆分
    tmpInfos = codeLine.split(matchKey)
    tmpInfos = tmpInfos[1].replace("\'", "\"").split("\"")
    if len(tmpInfos) > 1:
        return tmpInfos[1]
    else:
        return ""


'''
Function: 从代码文件中搜索指定的字符串
Return: 文件中的所有内容 或 None(文件不存在 或 获取失败)
'''


def searchTmpInfo(context, filePath):
    # 命令信息列表
    # 注意：特殊字符（如“(”）前必须加转义字符，否则re会报 error: unbalanced parenthesis错误
    searchKeyList = ["setResultFail", "setResult\(context, False", "setResult\(context,False",
                     "result.setResultFailByKey\(", "result.setResultFailByKeys\("]
    # 代码文件扩展名列表
    fileExtList = [".py"]
    # 是否忽略注释行
    skipCommentLine = True
    # 是否屏蔽文件错误信息
    skipFileWrnInfo = True

    # 执行搜索并返回结果
    return searchStrInFile(context, filePath, searchKeyList,
                           fileExtList, skipCommentLine,
                           skipFileWrnInfo, tmpInfoLineHandler)


def findCmdInfoToFile(context):
    # 代码目录：process
    CODE_PATH = os.path.dirname(__file__) + "\\..\\..\\process"
    # 保存命令信息列表的文件
    CMD_FILE_PATH = os.path.dirname(__file__) + "\\..\\..\\res\\TOOL_TLV_CMD_LIST.txt"
    # 保存TLV对象信息列表的文件
    OBJ_FILE_PATH = os.path.dirname(__file__) + "\\..\\..\\res\\TOOL_TLV_OBJ_LIST.txt"
    # 保存指定对象信息列表的文件
    TMP_FILE_PATH = os.path.dirname(__file__) + "\\..\\..\\res\\TOOL_TMP_OBJ_LIST.txt"

    # 遍历所有的代码文件，获取其中所有使用TLV命令的地方
    allCmdInfoLines = traverse(context, CODE_PATH, searchTlvCmd)
    isSucc = appendFileContents(CMD_FILE_PATH, allCmdInfoLines)
    BaseFactory.log.info(context, "write succ=" + str(isSucc) + ", cmd infos as follows:\n" + allCmdInfoLines)

    # 遍历所有的代码文件，获取其中所有访问TLV对象的地方
    allObjInfoLines = traverse(context, CODE_PATH, searchTlvObj)
    isSucc = appendFileContents(OBJ_FILE_PATH, allObjInfoLines)
    BaseFactory.log.info(context, "write succ=" + str(isSucc) + ", obj infos as follows:\n" + allObjInfoLines)

    # 遍历所有的代码文件，获取其中所有出现指定对象的地方
    tmpObjInfoLines = traverse(context, CODE_PATH, searchTmpInfo)
    isSucc = appendFileContents(TMP_FILE_PATH, tmpObjInfoLines)
    BaseFactory.log.info(context, "write succ=" + str(isSucc) + ", obj infos as follows:\n" + tmpObjInfoLines)


# 关键信息筛选入口
def findAllKeyInfoToFile(context):
    findCmdInfoToFile(context)


# 对搜索出的TLV信息行进行处理
allTlvEnumList = {}


def tlvEnumLineHandler(codeLine, matchKey):
    tmpLine = codeLine.replace("\"", "\'")
    enumInfos = ""
    while True:
        try:
            # 查找下一个matchKey字符串
            keyPos = tmpLine.index(matchKey)
            tmpLine = tmpLine[keyPos:]
            endPos = tmpLine.index("]")
            curEnumInfo = tmpLine[:endPos + 1]
            enumInfos = enumInfos + curEnumInfo + ","

            # 全局TlvEnum信息统计和计数
            if curEnumInfo not in allTlvEnumList.keys():
                allTlvEnumList[curEnumInfo] = 1
            else:
                allTlvEnumList[curEnumInfo] = allTlvEnumList[curEnumInfo] + 1

            # 重新调整剩余字符串，准备下次搜索
            tmpLine = tmpLine[endPos:]

        except:  # 当index无法再查找到matchKey时，将抛出异常
            break

    return enumInfos


# 从代码文件中搜索使用到的TlvEnum字段
def searchTlvEnumInfo(context, filePath):
    # 命令信息列表
    searchKeyList = ["TlvEnum."]
    # 代码文件扩展名列表
    fileExtList = [".py"]
    # 是否忽略注释行
    skipCommentLine = False
    # 是否屏蔽文件错误信息
    skipFileWrnInfo = True

    # 执行搜索并返回结果
    return searchStrInFile(context, filePath, searchKeyList,
                           fileExtList, skipCommentLine,
                           skipFileWrnInfo, tlvEnumLineHandler)


def findTlvEnumInfoToFile(context):
    # 代码目录：process
    CODE_PATH = os.path.dirname(__file__) + "\\..\\..\\process"
    # 保存ENUM详细信息列表的文件
    ENUM_DTLS_FILE_PATH = os.path.dirname(__file__) + "\\..\\..\\res\\TOOL_TLV_ENUM_DETAILS.txt"
    # 保存ENUM统计信息列表的文件
    ENUM_STAT_FILE_PATH = os.path.dirname(__file__) + "\\..\\..\\res\\TOOL_TLV_ENUM_RESULT.txt"

    # 遍历所有的代码文件，获取其中所有使用TlvEnum的地方
    skipFileList = ["StatisticFactory.py"]
    allTlvEnumDetailLines = traverse(context, CODE_PATH, searchTlvEnumInfo, skipFileList)
    isSucc = appendFileContents(ENUM_DTLS_FILE_PATH, allTlvEnumDetailLines)
    BaseFactory.log.info(context,
                         "write succ=" + str(isSucc) + ", TlvEnum details as follows:\n" + allTlvEnumDetailLines)

    # 输出TlvEnum统计结果文件
    allEnumInfos = sorted(allTlvEnumList.keys(), cmp=lambda x, y: cmp(x, y))
    allTlvEnumStatisticLines = "TlvEnum_Field_Description TlvEnum_Field_ID Usage_Times\n"
    errKeys = ""
    for key in allEnumInfos:
        # 识别key及其错误
        keyId = 0
        try:
            keyId = ast.literal_eval(key)
        except:
            errKeys = errKeys + key + ","
            continue
        # 合并key信息
        allTlvEnumStatisticLines += "%s %s %d\n" \
                                    % (key, keyId, allTlvEnumList[key])
    isSucc = appendFileContents(ENUM_STAT_FILE_PATH, allTlvEnumStatisticLines)
    BaseFactory.log.info(context,
                         "write succ=" + str(isSucc) + ", TlvEnum results as follows:\n" + allTlvEnumStatisticLines)
    if "" != errKeys:
        BaseFactory.log.error(context, "the keys:%s is error, pls check it." % errKeys)
        raise BaseFactory.exception.newToolException("meet key error, check it.", "", "meet key error, check it.", "")
