# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

import json
import collections

import checkEncTemperature
from cbb.frame.context import contextUtil
from cbb.frame.rest import restUtil
from cbb.frame.rest import restData
from cbb.frame.base import baseUtil
from cbb.frame.base import jsonUtil
from cbb.business.operate.fru.common import BaseFactory
from cbb.business.operate.fru.common import FuncFactory

from com.huawei.ism.tool.obase.entity import EntityUtils
from com.huawei.ism.tool.protocol.rest import RestConnectionManager

# 通过FCV查询的FRU
FRU_TYPE_FROM_FCV_API = [restData.Enum.ObjEnum.POWER, restData.Enum.ObjEnum.FAN]
ALARM_LEVEL = ["major", "critical"]

# 健康状态和运行状态
HEALTH_STATUS_ADN_RUNNING_STATUS_DICT = {
    "normal": "正常",
    "abnormal": "不正常",
    "Online": "在线",
    "Offline": "离线"
}
# 管理板信息
ManageBoardInfo = collections.namedtuple('ManageBoardInfo', ['ip', 'port', 'name', 'password'])


def need_get_info_from_fcv_rest(context, fru_type):
    """
    是否需要通过fcv接口查询信息
    @param context: 上下文
    @param fru_type: fru_type
    @return: True:需要，False:不需要
    """
    # 非计算型存储设备，不需要
    if not baseUtil.is_computing_dev(contextUtil.getProductModel(context)):
        return False

    return fru_type in FRU_TYPE_FROM_FCV_API


def get_select_ctrl_enc_overview_uri(context):
    """
    根据选择框获取对应的组件信息
    :param context: 上下文
    :return: overview 的 uri
    """
    dev_node = EntityUtils.toOldDev(context.get("devNode0"))
    chassis_num = 1
    if dev_node.getComputeStorageDevNode():
        chassis_num = dev_node.getComputeStorageDevNode().getChassisNum()
    return "/fce/device-manager/v1/device/devices/overview?chassisNum={}".format(chassis_num)


def get_fru_list_info(context, fru_type):
    """
    获取Fru硬件列表信息
    @param context: 上下文
    @param fru_type: fru类型
    @return: fru硬件列表
    """
    records = get_fcv_rest_record(context, get_select_ctrl_enc_overview_uri(context), fru_type)
    BaseFactory.log.info(context, "records={}".format(records))
    return records


def get_fcv_device_id(context):
    """
    获取fcv device id
    @param context: 上下文
    @return: device ip
    """
    record = get_system_overview_data(context)
    compute_nodes = record.get("data", {}).get("computeNodes")
    return compute_nodes[0].get("deviceId")


def get_fcv_ctrl_enc_manage_board_ips(context):
    """
    查询当前框上的BMC管理板ip
    :param context: 上下文
    :return: 管理板ip
    """
    record = get_system_overview_data(context)
    return [board.get("deviceIp") for board in record.get("data", {}).get("manageBorads", [])]


def get_system_overview_data(context):
    """
    获取当前框的system overview 全部组件信息
    :param context: 上下文
    :return: 组件信息
    """
    dev_node = EntityUtils.toOldDev(context.get("devNode0"))
    rest_connection = get_compute_storage_connection(dev_node)
    uri = get_select_ctrl_enc_overview_uri(context)
    response_info = rest_connection.execGet(rest_connection.getBaseUrl() + uri)
    data_string = response_info.getContent()
    record = json.loads(data_string)
    if not is_right_response(record):
        raise Exception("error response.")
    return record


def get_compute_storage_alarms(context, begin_timestamp=None, end_timestamp=None):
    """
    获取某个时间段内产生的告警
    :param context: 上下文
    :param begin_timestamp: 开始时间戳
    :param end_timestamp: 结束时间戳
    :return: 告警信息
    """

    if begin_timestamp is None or end_timestamp is None:
        uri = "/fce/monitor/v1/events?eventType=Uncleared"
    else:
        uri = "/fce/monitor/v1/events?eventType=Uncleared&startTime={}&endTime={}".format(int(begin_timestamp * 1000),
                                                                      int(end_timestamp * 1000))
    records = get_fcv_rest_record(context, uri, "events")
    BaseFactory.log.info(context, "alarms records={}".format(records))
    if not records:
        return []
    alarms = list()
    for record in records:
        level = record.get("severity")
        alarm_id = record.get("eventId")
        alarm_parma = record.get("eventDescriptionArgs")
        if level in ALARM_LEVEL:
            alarms.append((alarm_id, alarm_parma))
    return alarms


def get_compute_storage_connection(dev):
    """
    获取计算型存储的rest连接
    @param dev: 存储dev
    @return: rest连接
    """
    compute_node = EntityUtils.convertToComputeStorageDevNode(dev)
    return RestConnectionManager.getFceRestConnection(compute_node)


def get_compute_storage_fcv_connection(context):
    """
    获取计算型存储的rest连接
    @param context: 上下文
    @return: rest连接
    """
    dev_node = EntityUtils.toOldDev(context.get("devNode0"))
    compute_node = EntityUtils.convertToComputeStorageDevNode(dev_node)
    return RestConnectionManager.getFceRestConnection(compute_node)


def get_fcv_rest_record(context, uri, fru_type):
    """
    获取Fcv接口的返回数据
    @param context: 上下文
    @param uri: uri路径
    @param fru_type: fru_type
    @return: 接口返回的数据
    """
    dev_node = EntityUtils.toOldDev(context.get("devNode0"))
    rest_connection = get_compute_storage_connection(dev_node)

    response_info = rest_connection.execGet(rest_connection.getBaseUrl() + uri)
    data_string = response_info.getContent()
    record = json.loads(data_string)

    if not is_right_response(record):
        raise Exception("error response.")
    return record.get("data", {}).get(fru_type)


def is_right_response(record):
    """
    校验响应数据是否正确。
    :param record: 响应数据
    :return:
    """
    return record.get("code") == "0"


def check_redundant_power(context):
    """
    检查电源冗余
    @param context: 上下文
    """
    sel_row = BaseFactory.json.toDict(context.get("input_selectfru_frutab"))
    location = sel_row.get("location")
    fru_id = sel_row.get("id")
    enc_type = sel_row.get("type")
    enc_id = sel_row.get("parentID")
    BaseFactory.log.info(context, "the replacing power info: before update id={}, location={}".format(fru_id, location))
    if is_fru_fault(sel_row):
        BaseFactory.result.setResultPass(context)
        return

    fru_infos = get_fru_list_info(context, "powers")

    redundant_fru_num = 0
    for fru_info in fru_infos:
        if all([fru_info.get("parentId") == enc_id, fru_info.get("id") != fru_id,
                fru_info.get("healthStatus") == "normal"]):
            redundant_fru_num += 1
    # 计算型存储控制框冗余策略2+2
    if redundant_fru_num >= 2:
        BaseFactory.result.setResultPass(context)
        return
    BaseFactory.result.setResultFailByKey(context, FuncFactory.LangKey.POWER_X_EXIST_NONE_REDT_POWER, location)


def is_fru_fault(sel_row):
    """
    判断选择的fru是否故障
    @param sel_row: 选择列表
    @return: True: 故障， False: 非故障
    """
    return sel_row.get("healthStatus") not in ["normal", HEALTH_STATUS_ADN_RUNNING_STATUS_DICT.get("normal")]


def post_check_power_health_status(context):
    """
    检查更换后电源的健康状态
    @param context: 上下文
    """
    sel_row = BaseFactory.json.toDict(context.get("input_selectfru_frutab"))
    location = sel_row.get("location")
    fru_id = sel_row.get("id")

    fru_infos = get_fru_list_info(context, "powers")

    for fru_info in fru_infos:
        if fru_info.get("id") == fru_id and fru_info.get("healthStatus") == "normal":
            BaseFactory.result.setResultPass(context)
            return
    BaseFactory.result.setResultFailByKey(context, FuncFactory.LangKey.FUNC_REPLEACED_FRU_X_ABNORMAL, location)


def post_check_fan_health_status(context):
    """
    检查更换后风扇的健康状态
    @param context: 上下文
    """
    cache_data = BaseFactory.persist.getModule(context, FuncFactory.PERSIST_MULTI_REPLACE_FAN)
    location = cache_data.get_first_loc()

    fru_infos = get_fru_list_info(context, "fans")
    fan_infos = update_fan_info(fru_infos, "fans")
    for fan_info in fan_infos:
        if fan_info.get("location") == location and fan_info.get("healthStatus") == "normal" and fan_info.get(
                "runningStatus") == "Online":
            BaseFactory.result.setResultPass(context)
            return
    BaseFactory.result.setResultFailByKey(context, FuncFactory.LangKey.FUNC_REPLEACED_FRU_X_ABNORMAL, location)


def check_redundant_bbu(context):
    """
    检查BBU冗余
    @param context: 上下文
    """
    sel_row = BaseFactory.json.toDict(context.get("input_selectfru_frutab"))
    location = sel_row.get("location")
    fru_id = sel_row.get("id")
    parent_id = sel_row.get("parentID")
    BaseFactory.log.info(context, "the replacing power info: before update id={}, location={}".format(fru_id, location))
    if FuncFactory.isFruFault(context, restData.Enum.ObjEnum.BACKUP_POWER, fru_id):
        BaseFactory.result.setResultPass(context)
        return
    fru_infos = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.BACKUP_POWER)

    for fru_info in fru_infos:
        if all([fru_info.get("parentID") == parent_id, fru_info.get("id") != fru_id,
                fru_info.get("healthStatus") == "normal", fru_info.get("runningStatus") == "Online"]):
            BaseFactory.result.setResultPass(context)
            return
    BaseFactory.result.setResultFailByKey(context, FuncFactory.LangKey.BBU_X_EXISTED_NONE_REDT_BBU, None, location)


def init_fru_data_for_compute_storage(context, fru_type, fru_tab_key):
    """
    计算型存储初始化备件信息
    @param context:上下文
    @param fru_type:备件类型
    @param fru_tab_key:构造显示数据的key、
    """
    fru_infos = get_fru_list_info(context, fru_type)
    if not fru_infos:
        raise Exception("Failed to get fru list")
    if fru_type == "fans":
        _init_fans_infos(context, fru_type, fru_infos, fru_tab_key)
        return
    # 构造列表显示数据
    ui_tab = list()
    for fru_info in fru_infos:
        ui_row = get_ui_row(fru_info, fru_type)
        update_ui_row(context, ui_row)
        ui_tab.append(ui_row)
    BaseFactory.log.info(context, "ui_tab={}".format(ui_tab))
    context[fru_tab_key] = BaseFactory.json.toStr(ui_tab)
    BaseFactory.result.setResultPass(context)


def _init_fans_infos(context, fru_type, fru_infos, fru_tab_key):
    """
    刷新数据
    @param context: 上下文
    @param fru_type: 备件类型
    @param fru_infos: 备件信息
    @param fru_tab_key: 构造显示数据的key
    """
    ui_tab = update_fan_info(fru_infos, fru_type)
    for ui_row in ui_tab:
        update_ui_row(context, ui_row)
    BaseFactory.log.info(context, "ui_tab={}".format(ui_tab))
    context[fru_tab_key] = BaseFactory.json.toStr(ui_tab)
    BaseFactory.result.setResultPass(context)


def update_fan_info(fru_infos, fru_type):
    """
    更新风扇数据（综合前转轮和后转轮为一条信息）
    @param fru_infos:
    @param fru_type:
    @return:
    """
    ui_tab = list()
    for fru_info in fru_infos:
        ui_row = get_ui_row(fru_info, fru_type)
        name = ui_row.get("location")
        if name not in [fru_row.get("location") for fru_row in ui_tab]:
            ui_tab.append(ui_row)
            continue
        for fru_row in ui_tab:
            if fru_row.get("location") != name:
                continue
            if ui_row.get("healthStatus") != fru_row.get("healthStatus"):
                fru_row["healthStatus"] = "abnormal"
                fru_row["visible"] = "true"
            if ui_row.get("runningStatus") != fru_row.get("runningStatus"):
                fru_row["runningStatus"] = "Offline"
    return ui_tab


def update_ui_row(context, ui_row):
    """
    更新行数据
    @param context: 上下文
    @param ui_row: 行信息
    """
    if contextUtil.getLang(context) == "en":
        return
    ui_row["healthStatus"] = HEALTH_STATUS_ADN_RUNNING_STATUS_DICT.get(ui_row.get("healthStatus"), "--")
    ui_row["runningStatus"] = HEALTH_STATUS_ADN_RUNNING_STATUS_DICT.get(ui_row.get("runningStatus", "--"))


def get_ui_row(fru_info, fru_type):
    """
    获取行信息
    @param fru_info: fru信息
    @param fru_type: fru类型
    @return: 行信息
    """
    ui_row = dict()
    ui_row["id"] = get_fru_info(fru_info, "id")
    fan_type = get_fru_info(fru_info, "type")
    ui_row["type"] = fan_type
    ui_row["location"] = get_fru_info(fru_info, "name").split()[0] if fan_type == '0' \
        else get_fru_info(fru_info, "name").replace(" ", "")
    ui_row["parentID"] = get_fru_info(fru_info, "parentId")
    ui_row["healthStatus"] = get_fru_info(fru_info, "healthStatus")
    ui_row["runningStatus"] = get_fru_info(fru_info, "runningStatus")
    if fru_type == "powers":
        ui_row["powerType"] = get_fru_info(fru_info, "powerMode")
    if fru_info.get("healthStatus") != "normal":
        ui_row["visible"] = "true"
    return ui_row


def get_fru_info(fru_info, fru_key):
    """
    获取备件字段信息
    @param fru_info: 备件信息
    @param fru_key: 备件字段
    @return: 备件信息
    """
    return fru_info.get(fru_key) if fru_info.get(fru_key) else "--"


def check_fan_temperature(context):
    """
    检查风扇所在框的温度
    @param context:
    @return:
    """
    cache_data = BaseFactory.persist.getModule(context,
                                               FuncFactory.PERSIST_MULTI_REPLACE_FAN)
    if is_fru_fault(cache_data.sel_rows[0]):
        BaseFactory.result.setResultPass(context)
        return

    enc_id = cache_data.parent_id
    # 检测风扇所在框是否超温
    checkEncTemperature.execute(context, alarmIds=FuncFactory.ENC_TEMPERATURE_ALM_IDS, encId=enc_id)


def get_disk_drawer_info(context):
    """
    获取硬盘抽屉信息
    @param context: 上下文
    @return: 硬盘抽屉信息
    """
    uri = "/redfish/v1/chassis/enclosure/hdddrawer"
    return get_info_by_redfish_rest(context, uri)


def get_info_by_redfish_rest(context, uri):
    """
    通过redfish接口获取信息
    @param context: 上下文
    @param uri: uri地址
    @return: 接口信息
    """
    rest_connection = get_redfish_connection(context)
    result = rest_connection.executeGet(uri)
    res_json = result.getResult()
    if not result.isSuccess():
        raise Exception("error response.")
    return jsonUtil.jsonStr2Dict(res_json)


def get_info_by_redfish_rest_conn(redfish_conn, uri):
    """
    通过redfish接口获取信息
    @param redfish_conn: redfish 连接
    @param uri: uri地址
    @return: 接口信息
    """
    result = redfish_conn.executeGet(uri)
    res_json = result.getResult()
    if not result.isSuccess():
        raise Exception("error response.")
    return jsonUtil.jsonStr2Dict(res_json)


def get_redfish_connection(context):
    """
    获取redfish连接
    @param context: 上下文
    @return: redfish连接
    """
    from com.huawei.uMate.common.rest import BmcRedfishConnManagement
    manage_board_infos = get_all_bmc_manage_boards(context)
    if not manage_board_infos:
        BaseFactory.log.info(context, "fail to manage_board_info")
        return None

    # 增加过滤当前框下的管理板
    board_ips = get_fcv_ctrl_enc_manage_board_ips(context)
    manage_board_info_list = [board_info for board_info in manage_board_infos if board_info.ip in board_ips]
    for manage_board_info in manage_board_info_list:
        cluster_node = BmcRedfishConnManagement.createClusterNode(
            manage_board_info.ip, manage_board_info.port, manage_board_info.name, manage_board_info.password
        )
        return BmcRedfishConnManagement.getRedFishConnection(cluster_node)


def get_redfish_connection_by_bmc_manager_board(context, manage_board_info):
    """
    获取redfish连接
    @param context: 上下文
    @param manage_board_info: BMC管理板
    @return: redfish连接
    """
    from com.huawei.uMate.common.rest import BmcRedfishConnManagement
    if not manage_board_info:
        BaseFactory.log.info(context, "fail to manage_board_info")
        return None
    cluster_node = BmcRedfishConnManagement.createClusterNode(manage_board_info.ip, manage_board_info.port,
                                                              manage_board_info.name, manage_board_info.password)
    return BmcRedfishConnManagement.getRedFishConnection(cluster_node)


def get_all_bmc_manage_boards(context):
    """
    获取管理版信息
    @param context: 上下文
    @return: 管理版信息
    """
    uri = "/fce/device-manager/v1/device/info/list"
    record = get_record_by_fcv_rest(context, uri, is_show_log=False)
    manage_node_infos = record.get("data", {}).get("manageBoards")
    board_list = []
    for manage_node_info in manage_node_infos:
        if all([manage_node_info.get("ip"), manage_node_info.get("port"), manage_node_info.get("username"),
                manage_node_info.get("password")]):
            board_list.append(ManageBoardInfo(manage_node_info.get("ip"), str(manage_node_info.get("port")),
                                              manage_node_info.get("username"), manage_node_info.get("password")))
    return board_list


def check_power_off(context):
    """
    计算型存储系统下电检查
    @param context: 上下文
    @return: True: 检查通过 False: 检查失败
    """
    dev_node = EntityUtils.toOldDev(context.get("devNode0"))
    compute_connection = get_compute_storage_connection(dev_node)

    # 检查下电状态
    uri = "/fce/fcc-business/v1/servers/power/process"
    response_info = compute_connection.execGet(compute_connection.getBaseUrl() + uri)
    data_string = response_info.getContent()
    record = json.loads(data_string)
    step_infos = record.get("data", {}).get("stepInfo", [])
    for step_info in step_infos:
        # 如果codeResult等于103，则表示下电失败，返回False
        if str(step_info.get("codeResult")) == "103":
            return False
    return True


def power_off_compute_storage_system(context):
    """
    计算型存储系统下电
    @param context: 上下文
    """
    dev_node = EntityUtils.toOldDev(context.get("devNode0"))
    compute_connection = get_compute_storage_connection(dev_node)
    # 下电前校验
    if not verify_before_power_off(context, compute_connection):
        raise Exception("verify before power off failed.")
    # 执行下电
    uri = "/fce/fcc-business/v1/servers/power/off"
    response_info = compute_connection.execPost(compute_connection.getBaseUrl() + uri, "")
    data_string = response_info.getContent()
    record = json.loads(data_string)
    if not is_right_response(record):
        raise Exception(record.get("code"), record.get("msg"))


def verify_before_power_off(context, compute_connection):
    """
    下电前校验
    @param context: 上下文
    @param compute_connection: rest连接
    @return: True： 通过 False： 不通过
    """
    dev_node = EntityUtils.toOldDev(context.get("devNode0"))
    user_info = dev_node.getComputeStorageDevNode().getOpenapiUser()
    uri = "/fce/framework/v1/iam/auth/authenticate/secondary/power-off"
    param_str = jsonUtil.dict2JsonStrByDumps({"userName": user_info.getUserName(), "password": user_info.getPassword()})
    response_info = compute_connection.execPost(compute_connection.getBaseUrl() + uri, param_str)
    data_string = response_info.getContent()
    record = json.loads(data_string)
    return is_right_response(record)


def check_compute_node_running_status(context):
    """
    检查计算节点运行状态
    @param context: 上下文
    @return: True：正常， False：不正常
    """
    uri = get_select_ctrl_enc_overview_uri(context)
    record = get_record_by_fcv_rest(context, uri)
    compute_info = record.get("data", {}).get("computeNodes")
    if not compute_info:
        return False
    for node_info in compute_info:
        if node_info.get("deviceOnlineStatus") != "Online":
            return False
    return True


def get_record_by_fcv_rest(context, uri, is_show_log=True):
    """
    获取fcv接口的信息
    @param context: 上下文
    @param uri: uri地址
    @param is_show_log: 是否打印日志
    @return: record
    """
    dev_node = EntityUtils.toOldDev(context.get("devNode0"))
    compute_connection = get_compute_storage_connection(dev_node)
    if is_show_log:
        response_info = compute_connection.execGet(compute_connection.getBaseUrl() + uri)
    else:
        response_info = compute_connection.execGetWithNoLog(compute_connection.getBaseUrl() + uri, None)
    data_string = response_info.getContent()
    record = json.loads(data_string)
    if not is_right_response(record):
        raise Exception("error response.")
    return record


def get_manage_port_info(context):
    """
    获取管理口信息（LinkUp）
    @param context: 上下文
    @return: 管理口信息
    """
    manage_port_info = get_all_manage_port_info(context)
    linkup_port_info = dict()
    for port_id, status in manage_port_info.items():
        if status in ("LinkUp", "Standby"):
            linkup_port_info[port_id] = "LinkUp"

    return linkup_port_info


def get_all_manage_port_info(context):
    """
    获取所有的管理口信息
    @param context: 上下文
    @return: 管理口信息
    """
    boards = get_all_bmc_manage_boards(context)
    uri = "/redfish/v1/Chassis/Enclosure/Switches"
    manage_port_info = dict()
    for board in boards:
        redfish_conn = get_redfish_connection_by_bmc_manager_board(context, board)
        record = get_info_by_redfish_rest_conn(redfish_conn, uri)
        smm_uri = _get_smm_member(record.get("Members", [{}]))
        smm_id = smm_uri.split("/")[-1]

        manage_port_record = get_info_by_redfish_rest_conn(redfish_conn, "{}/Ports/Mgmt".format(smm_uri))
        port_id = "{}_{}".format(manage_port_record.get("PortId"), smm_id)
        for link_status_dict in manage_port_record.get("Oem", {}).values():
            if "LinkStatus" in link_status_dict:
                manage_port_info[port_id] = link_status_dict.get('LinkStatus')
    return manage_port_info


def _get_smm_member(members):
    for member in members:
        if "Smm" in member.get("@odata.id", ""):
            return member.get("@odata.id", "")
    return ""
