# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2012-2024. All rights reserved.
from cbb.frame.base import baseUtil
from cbb.frame.base.config import CTRL_6U_PRODUCTS
from cbb.frame.base.config import DORADO_DEVS_V6_HIGH
from cbb.frame.base.config import ARM_HIGH_END_NEW
from cbb.frame.context import contextUtil
from cbb.frame.cli import cliUtil
from cbb.common.query.hardware import controller
from cbb.business.operate.fru.common import common
from cbb.business.operate.fru.common.controller_patch_function import \
    ControllerPatchFunction, LangKey
from cbb.business.operate.fru.common.BaseFactory import result, log

TIME_OUT = 60

TCP_CMD_MAP = {
    "TCP_MEM": "tcpconfig change TCP_MEM",
    "TCP_RMEM": "tcpconfig change TCP_RMEM",
    "TCP_WMEM": "tcpconfig change TCP_WMEM",
}


def execute(context, ctrl_id):
    """恢复TCP缓存设置

    :param context: 上下文
    :param ctrl_id: 待更换的控制器
    :return:
    """
    try:
        ctrl_patch_func = ControllerPatchFunction(context)
        is_support_patch_scene = contextUtil.getItem(context, "is_patch_scene")
        init_tcp_mem_process(context, is_support_patch_scene)
        is_patch_pass = False
        is_tcp_pass = False
        if is_support_patch_scene:
            log.info(context, "6u4c patch final operation.")
            is_patch_pass = ctrl_patch_func.patch_operation_final_v2()

        # 从上下文中获取保存的TCP缓存设置
        tcp_mem_setting = contextUtil.getItem(context, "tcp_mem_setting", {})
        if tcp_mem_setting:
            is_tcp_pass = set_tcp_mem_setting(
                context, ctrl_id, tcp_mem_setting)
            if not is_tcp_pass:
                result.setResultFailByKey(
                    context, "CNTR_RESTORE_TCP_MEM_SETTING")

        # 补丁场景与TCP缓存设置均失败
        if all([not is_patch_pass, is_support_patch_scene,
                tcp_mem_setting, not is_tcp_pass]):
            result.setResultFailByKey(context,
                                      LangKey.TCP_AND_PATCH_BOTH_FAILED)
            return False
        # 任一失败
        if (not is_patch_pass and is_support_patch_scene) \
                or (tcp_mem_setting and not is_tcp_pass):
            return False
        return True
    finally:
        common.finishProcess(context)


def init_tcp_mem_process(context, is_support_patch_scene):
    """离线控制器的进度条设置

    :param context: 上下文
    :param is_support_patch_scene: 是否为补丁场景
    :return:
    """
    limit_time = TIME_OUT + (5 * 60 if is_support_patch_scene else 0)
    interval_time = 5
    common.threadUpProcess(context, limit_time, interval_time)


def set_tcp_mem_setting(context, ctrl_id, tcp_mem_setting):
    """设置TCP缓存

    :param context: 上下文
    :param ctrl_id: 待更换的控制器
    :param tcp_mem_setting: TCP缓存参数
    :return:
    """
    dev_model = contextUtil.getProductModel(context)
    is_high_end = dev_model in (CTRL_6U_PRODUCTS + DORADO_DEVS_V6_HIGH +
                                ARM_HIGH_END_NEW)
    lang = contextUtil.getLang(context)
    dev_obj = contextUtil.getDevObj(context)
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    cli = cliUtil.get_cli_for_svp(context, logger)

    # 获取目标节点
    flag, target_node = controller.get_node_by_ctrl(is_high_end, ctrl_id)
    logger.logInfo("target node is:{}".format(target_node))
    if not flag:
        return False

    try:
        # 心跳到目标节点
        heart_cmd = "sshtoremoteExt {}".format(target_node)
        heart_ret = cliUtil.sshToRemoteContr(cli, heart_cmd,
                                             dev_obj.get("pawd"), lang)
        # 心跳失败
        if "You have accessed the system." not in heart_ret[1]:
            logger.logInfo("Failed to heartbeat to node:{}".format(
                target_node))
            return False
        logger.logInfo("Succeed to heartbeat to node:{}".format(target_node))

        # 在目标节点小系统下执行查询命令
        sh_name = "eam.sh"
        if baseUtil.isDoradoV6Dev(dev_model) or baseUtil.is_big_clusters_ai_dev(dev_model):
            sh_name = "ifm.sh"

        query_flag = True
        cliUtil.excuteCmdInMinisystemModel(cli, sh_name, lang)
        for tcp_cmd in TCP_CMD_MAP:
            # 执行命令
            params = tcp_mem_setting.get(tcp_cmd)
            cmd = "{} {} {}".format(sh_name, TCP_CMD_MAP.get(tcp_cmd), params)
            cli_ret = cliUtil.execCliCmd(cli, cmd, True)
            cmd = "y"
            cli_ret = cliUtil.execCliCmd(cli, cmd, True)
            cli_ret = cliUtil.execCliCmd(cli, cmd, True)
            if "success: change tcp config" not in cli_ret.lower():
                query_flag = False
                break

        # 退出心跳
        cliUtil.exitHeartbeatCli(cli, lang)
        # 退出minisystem
        cliUtil.exitMinisystemToCliMode(cli)
        if not query_flag:
            return False

        return True
    except Exception as ex:
        logger.logException(ex)
        # 退出心跳
        cliUtil.exitHeartbeatCli(cli, lang)
        # 退出minisystem
        cliUtil.exitMinisystemToCliMode(cli)
        return False
