#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@version: Toolkit V200R006C00
@time: 2020/07/28
@file: expand_encrypt_disk_domain_check.py
@function:
@modify:
"""
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
import traceback
from java.lang import Exception as JException


def execute(context):
    """手动扩容加密硬盘域流程检查
    @OMRPId expand_encrypt_disk_domain_check

    :param
        context
    :return: (是否通过， 回显， 错误信息)
    """

    logger = context.get("logger")
    try:
        return _check_expand_disk_domain(context, logger)
    except (Exception, JException) as ex:
        logger.error(traceback.format_exc())
        raise Exception(ex)


def _check_expand_disk_domain(context, logger):
    """
    检查待扩硬盘域
    :param context:
    :param logger:
    :return: (是否通过， 回显， 错误信息)
    """
    lang = contextUtil.getLang(context)
    cli = context.get("CLI_CONNECTION")

    # 执行获取设备硬盘信息指令
    cmd = "show disk general"
    res, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if not res:
        return res, cli_ret, err_msg

    res, disk_info = _get_disk_info(cli_ret, logger)
    if not res:
        # 获取硬盘信息失败
        err_msg = cliUtil.getMsg(lang, "cannot.get.disk.info")
        return cliUtil.RESULT_NOCHECK, cli_ret, err_msg
    if disk_info == "":
        # 设备无硬盘
        return True, cli_ret, ""

    # 检查待扩硬盘域
    exp_disk_list = context.get("expDiskList")
    if not exp_disk_list:
        return True, cli_ret, ""

    for disk_cfg in exp_disk_list:
        logger.info("ExpdiskDomain:"+str(disk_cfg.get("diskDomain")))
        disk_domain = disk_cfg.get("diskDomain")
        if disk_domain == "none":
            # 如果是新扩硬盘域，则不检查
            continue
        # 如果是在原硬盘域上扩盘，执行检查
        ret = _is_domain_encrypt(disk_domain, cli_ret,
                                 disk_info, lang, logger)
        if ret[0] == cliUtil.RESULT_WARNING:
            # 检查到该硬盘域为加密硬盘域，提出修复建议
            return ret

    return True, cli_ret, ""


def _get_disk_info(cli_ret, logger):
    """
    获取设备硬盘信息
    :param cli_ret:
    :param logger:
    :return: (是否成功获取， 硬盘信息列表)
    """
    if cliUtil.queryResultWithNoRecord(cli_ret):
        # 表示没硬盘
        logger.info("There is no disk")
        return True, ""

    cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)
    if len(cli_ret_lines_list) == 0:
        # 错误，无法获取硬盘信息
        logger.info("Cannot get information about disk status")
        return False, ""

    return True, cli_ret_lines_list


def _is_domain_encrypt(disk_domain, cli_ret, disk_info, lang, logger):
    """
    检查硬盘域是否为加密硬盘域
    :param disk_domain:
    :param cli_ret:
    :param disk_info:
    :param lang:
    :param logger:
    :return: （检查结果， 回显， 错误信息）
    """
    for info_dict in disk_info:
        if info_dict.get("Disk Domain ID") == disk_domain and \
                "SED" in info_dict.get("Type"):
            # 存在风险，弹出建议优化信息
            logger.info("Expanding SED disk domain may have a risk")
            err_msg = cliUtil.getMsg(lang, "expand.encrypt.check.nopass")
            return cliUtil.RESULT_WARNING, cli_ret, err_msg

    return True, "", ""
