# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.
"""
@version: Smartkit 22.0.0
@time: 2022/01/16
@file: tar_util.py
@function: 提供tar解压的通用接口和方法
@modify:
"""
import os
import tarfile

from cbb.frame.adapter.applicationContext import import_application_context

APPLICATION_CONTEXT = import_application_context()
MAX_ZIP_FILE_SIZE = APPLICATION_CONTEXT.getMaxUnzipFileSize()
MAX_ZIP_FILE_NUM = APPLICATION_CONTEXT.getMaxUnzipFileNum()


class UnZipLimit(object):
    """
    主要针对压缩包嵌套压缩包场景，用此对象跟踪判断解压出来的文件数量或大小是否超过阈值
    """

    def __init__(self, max_file_size=MAX_ZIP_FILE_SIZE,
                 max_file_num=MAX_ZIP_FILE_NUM):
        # 最大可解压的文件大小，单位为byte
        self.__max_file_size = max_file_size
        # 最大可解压的文件数量
        self.__max_file_num = max_file_num
        # 当前已解压的文件大小，单位为byte
        self.__file_size = 0
        # 当前已解压的文件数量
        self.__file_num = 0

    def increase_one_file_num(self):
        """
        当前文件数量统计加1
        :return: 无返回
        """
        self.__file_num += 1

    def increase_file_size(self, file_size):
        """
        前文件大小统计新增指定大小
        :param file_size: 新增大小
        :return:
        """
        self.__file_size += file_size

    def is_file_size_over_limit(self):
        """
        判断是否大于或等于文件大小阈值
        :return: 是否大于或等于文件大小阈值
        """
        return (self.__max_file_size != 0 and
                self.__file_size >= self.__max_file_size)

    def is_file_num_over_limit(self):
        """
        判断是否大于或等于文件数量阈值
        :return: 是否大于或等于文件数量阈值
        """
        return (self.__max_file_num != 0 and
                self.__file_num >= self.__max_file_num)

    def is_file_stat_over_limit(self):
        """
        判断文件统计过程中是否有超过阈值的情况
        :return: 文件大小或文件数量是否存在超过阈值的情况
        """
        return self.is_file_size_over_limit() or self.is_file_num_over_limit()

    def get_max_file_size(self):
        """
        获取文件允许的最大文件大小
        :return: 文件大小的最大值
        """
        return self.__max_file_size

    def get_max_file_num(self):
        """
        获取文件允许的最大文件数量
        :return: 文件数量的最大值
        """
        return self.__max_file_num


def get_safe_entry_name(tar_name):
    """
    转换压缩对象的名称，过滤目录攻击的情况
    :param tar_name: 压缩实体名称
    :return: 校验后的实体名称
    """
    if "\\.\\.\\" in tar_name or "\\.\\./" in tar_name:
        return tar_name.replace("\\.\\.\\", "").replace("\\.\\./", "")
    return tar_name


def decompress_tar_all_file(tar_file, dest_path, open_type="r:gz",
                            max_size=MAX_ZIP_FILE_SIZE,
                            max_num=MAX_ZIP_FILE_NUM):
    """
    解压所有的文件

    :param tar_file: 待解压的文件
    :param dest_path: 解压路径
    :param open_type: 压缩文件类型
    :param max_size: 压缩包的最大文件大小
    :param max_num: 压缩包的文件最大文件个数
    :return: 是否解压成功和异常信息
    """
    is_over_limit = False
    with tarfile.open(tar_file, open_type) as tar_obj:
        limit = UnZipLimit(max_size, max_num)
        for tar in tar_obj:
            limit.increase_file_size(tar.size)
            limit.increase_one_file_num()
            if limit.is_file_stat_over_limit():
                is_over_limit = True
                break
            tar_obj.extract(get_safe_entry_name(tar.name), dest_path)

    return not is_over_limit


def decompress_tar_all_file_with_detail(tar_file, dest_path,
                                        open_type="r:gz",
                                        max_size=MAX_ZIP_FILE_SIZE,
                                        max_num=MAX_ZIP_FILE_NUM):
    """
    解压所有文件，并返回解压结果和文件名称以及文件全路径+文件名的列表
    :param tar_file: 待解压的文件
    :param dest_path: 解压路径
    :param open_type: 压缩文件类型
    :param max_size: 压缩包的最大文件大小
    :param max_num: 压缩包的文件最大文件个数
    :return: (是否解压成功，解压的文件名称列表，文件路径+文件名称的列表)
    """
    is_over_limit = False
    file_names = []
    file_path_names = []
    with tarfile.open(tar_file, open_type) as tar_obj:
        limit = UnZipLimit(max_size, max_num)
        for tar in tar_obj:
            limit.increase_file_size(tar.size)
            limit.increase_one_file_num()
            if limit.is_file_stat_over_limit():
                is_over_limit = True
                break
            tar_safe_name = get_safe_entry_name(tar.name)
            tar_obj.extract(tar_safe_name, dest_path)
            file_names.append(tar_safe_name)
            file_path_names.append(os.path.join(dest_path, tar_safe_name))
    return not is_over_limit, file_names, file_path_names


def decompress_tar_special_file(tar_file, dest_path, fileName, open_type="r:gz",
                                max_size=MAX_ZIP_FILE_SIZE):
    """
    解压指定文件
    :param tar_file: 压缩包
    :param dest_path: 解压路径
    :param fileName: 文件名称或待解压的文件名列表或元组
    :param open_type: 压缩文件类型
    :param max_size: 最大压缩包大小
    :return: 是否解压成功和异常信息
    """

    def decompress_special_file(tar_obj, file_list, limit):
        is_decompress_success = False
        for tar in tar_obj:
            if tar.name not in file_list:
                continue
            limit.increase_file_size(tar.size)
            if limit.is_file_size_over_limit():
                return False
            tar_obj.extract(get_safe_entry_name(tar.name), dest_path)
            is_decompress_success = True
        return is_decompress_success

    def wrapper_file_name_to_list():
        file_list = []
        if isinstance(fileName, list):
            file_list = fileName
        elif isinstance(fileName, tuple):
            file_list = list(fileName)
        else:
            # 其他情况就当做单个文件处理
            file_list.append(fileName)
        return file_list

    # 包装一下外部传递的文件名称，可能是列表，元祖、或字符串
    file_list = wrapper_file_name_to_list()
    # 打开文件，开始处理解压
    with tarfile.open(tar_file, open_type) as tar_obj:
        limit = UnZipLimit(max_file_size=max_size)
        return decompress_special_file(tar_obj, file_list, limit), None


def decompress_tar_special_file_with_exception(tar_file, dest_path, file_name):
    """
    解压单个文件，如果解压失败，抛异常
    :param tar_file: 压缩包
    :param dest_path: 目标路径
    :param file_name: 解压文件
    :return:
    """
    is_success, _ = decompress_tar_special_file(tar_file, dest_path, file_name)
    if not is_success:
        raise tarfile.ExtractError


def check_tar_special_file(tar_file, fileter_callback, open_type="r:gz"):
    """
    检查某个过滤规则是否满足
    :param tar_file: 压缩文件
    :param fileter_callback: 压缩对象回调
    :param open_type: 压缩文件类型
    :return: 是否找到满足filter的条件
    """
    with tarfile.open(tar_file, open_type) as tar_obj:
        for tar in tar_obj:
            if fileter_callback(tar):
                return True
    return False


def decompress_tar_custom_file(tar_file, dest_path, fileter_callback,
                               open_type="r:gz", max_size=MAX_ZIP_FILE_SIZE,
                               max_num=MAX_ZIP_FILE_NUM):
    """
    根据过滤条件判断是否解压文件
    :param tar_file: 压缩文件
    :param dest_path: 解压路径
    :param fileter_callback: 过滤回调
    :param open_type: 压缩文件类型
    :param max_size: 最大压缩包大小
    :param max_num: 压缩包的文件最大文件个数
    :return: 是否解压成功
    """

    def decompress_custom_file(tar_obj, limit):
        is_decompress_success = False
        for tar in tar_obj:
            if not fileter_callback(tar):
                continue
            limit.increase_file_size(tar.size)
            limit.increase_one_file_num()
            if limit.is_file_stat_over_limit():
                return False
            tar_obj.extract(get_safe_entry_name(tar.name), dest_path)
            is_decompress_success = True
        return is_decompress_success

    limit = UnZipLimit(max_size, max_num)
    with tarfile.open(tar_file, open_type) as tar_obj:
        return decompress_custom_file(tar_obj, limit)


def decompress_one_file_guess_path_by_end_str(target_name, source_file_path, target_end=None, open_type="r:gz"):
    """
    解压单独的文件到目录，如果不指定target_end,则尝试两个路径

    :param target_name: 目标文件
    :param source_file_path: 源压缩包
    :param target_end: 根据目标字符串在源压缩包路径中寻找目标路径
    :param open_type: 压缩包解压类型
    :return:
    """
    with tarfile.open(source_file_path, open_type) as tar_obj:
        for tarinfo in tar_obj:
            if tarinfo.name != target_name:
                continue
            if tarinfo.size > MAX_ZIP_FILE_SIZE:
                raise ValueError('file size over limit')
            return _tar_one_file_to_path(source_file_path, tar_obj, target_end, tarinfo)
        return False


def _tar_one_file_to_path(source_file_path, tar_obj, target_end, tarinfo):
    if target_end:
        tar_obj.extract(tarinfo.name, source_file_path[0: source_file_path.index(target_end)])
    else:
        try:
            tar_obj.extract(tarinfo.name, source_file_path[0: source_file_path.index('.tgz')])
        except (tarfile.ExtractError, EnvironmentError):
            tar_obj.extract(tarinfo.name, source_file_path[0: source_file_path.index('.tar.gz')])
    return True
