#!/usr/bin/python
# -*- coding: utf-8 -*-
import logging
import json
import os
import re
import sys
import time
from infra.util import shell
from check_item.check_util.cli_executor import CliExecutor

_LOG_PREFIX = "NTP_SWITCH_CHECK"

logging.basicConfig(level=logging.INFO,
                    filename="/OSM/log/cur_debug/messages",
                    format='[%(asctime)s][%(levelname)s][%(message)s][%(filename)s, %(lineno)d]',
                    datefmt='%Y-%m-%d %H:%M:%S')


class CmdExecError(Exception):
    """
    命令执行异常
    """
    def __init__(self, cmd):
        self.cmd = cmd

    def __str__(self):
        return "An error occurred when running the command(%s)." % self.cmd


class UnExpectValueError(Exception):
    """
    返回值为不期望的值
    """
    def __str__(self):
        return "The return value is not within the expected range."


class CheckNtpSwitch(object):
    _SHOW_NTP_SWITCH_CLI_CMD = "show ntp_server general|filterColumn include columnList=NTP\\sSwitch"
    _SHOW_UPGRADE_PACKAGE_CLI_CMD = "show upgrade package"
    _SHOW_NTP_STATUS_OFFSET_CLI_CMD = "show ntp status|filterColumn include columnList=Offset"
    _SHOW_RTC_SYS_TIME_SHELL_CMD = "sudo hwclock -r; date +\'%Y-%m-%d %H:%M:%S\';"
    _MAX_DEVIATION = 3  # 允许的时间最大差值为3秒

    def __init__(self):
        # 命令返回值
        self.cmd_results = {}
    
    def execute(self):
        # ntp开关关闭，检查通过
        if not self.check_ntp_switch_opened():
            return True

        # 打过符合条件的补丁的版本，检查通过
        if self.check_version():
            return True

        # 检查NTP服务器时间与系统时间的差值，绝对值大于3秒，检查不通过
        if not self.check_ntp_time_deviation():
            return False

        # 若SystemTime与RTCTime差值的绝对值大于3秒，检查不通过
        if not self.check_system_and_rtc_time():
            return False

        # 其余情况检查通过
        return True

    def get_ntp_system_interval(self):
        """
        获取配置的NTP服务器和系统的时间差值
        """
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(self._SHOW_NTP_STATUS_OFFSET_CLI_CMD)
        if not res:
            logging.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX, self._SHOW_NTP_STATUS_OFFSET_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_NTP_STATUS_OFFSET_CLI_CMD)

        self.get_cmd_results(self._SHOW_NTP_STATUS_OFFSET_CLI_CMD, cli_exec_tmp_file)

        cmd_result = self.cmd_results.get(self._SHOW_NTP_STATUS_OFFSET_CLI_CMD)
        if not cmd_result:
            return None

        for line in cmd_result.splitlines():
            result = re.findall(r"Offset\s:\s(.*)\ssecond\(s\)", line)  # 示例：Offset : +0.000527 second(s)
            if result:
                return result[0]
            result = re.findall(r"Offset\s:\s(.*)", line)  # 示例：  Offset : --
            if result and result[0].strip() == "--":
                return 0  # 未查询到取默认值0
        return None

    def check_ntp_time_deviation(self):
        """
        通过获取NTP服务器上的时间差检查，NTP服务器时间与系统时间的差值绝对值大于3秒，检查不通过
        """
        for _ in range(16):
            interval = self.get_ntp_system_interval()
            if interval is None:
                time.sleep(1)  # 执行命令失败，等待2秒后重试一次
                continue
            if abs(float(interval)) > float(self._MAX_DEVIATION):
                logging.error("%s: not pass, deviation time(%s).", _LOG_PREFIX, interval)
                return False
            return True
        logging.error("%s: get ntp status failed.", _LOG_PREFIX)
        raise UnExpectValueError()

    def check_system_and_rtc_time(self):
        """
        通过比较系统时间和硬件时间检查，若SystemTime与RTCTime差值的绝对值大于3秒，检查不通过
        """
        ret_code, output = shell.call_system_cmd(self._SHOW_RTC_SYS_TIME_SHELL_CMD)
        self.cmd_results[self._SHOW_RTC_SYS_TIME_SHELL_CMD] = output
        if ret_code != 0:
            logging.error("%s: exec cmd failed(%d). cmd output (%s)", _LOG_PREFIX, ret_code, output)
            raise CmdExecError(self._SHOW_RTC_SYS_TIME_SHELL_CMD)
        time_list = output.strip().split('\n')
        if len(time_list) != 2: # 正常情况有且只有2个时间
            logging.error("%s: parse time failed. (%s)", _LOG_PREFIX, time_list)
            raise UnExpectValueError()
        
        rtc_time = int(time.mktime(time.strptime(time_list[0][:19], '%Y-%m-%d %H:%M:%S')))  # 时间字符串精确到秒共19字符
        sys_time = int(time.mktime(time.strptime(time_list[1][:19], '%Y-%m-%d %H:%M:%S')))  # 时间字符串精确到秒共19字符
        if abs(rtc_time - sys_time) > self._MAX_DEVIATION:
            logging.error("%s: not pass. RTC time (%s) is not equal to system time (%s).",
                _LOG_PREFIX, time_list[0], time_list[1])
            return False
        return True

    def get_cmd_results(self, cmd, file_name):
        """
        从文件中读取命令返回值
        """
        with open(file_name, 'r') as f:
            self.cmd_results[cmd] = f.read().strip()
        self._try_to_del_file(file_name)

    def get_ntp_switch_status(self):
        """
        获取NTP开关状态
        """
        cmd_result = self.cmd_results.get(self._SHOW_NTP_SWITCH_CLI_CMD)
        if not cmd_result:
            return None

        for line in cmd_result.splitlines():
            result = re.findall(r"NTP\sSwitch\s:(.*)", line)
            if result:
                return result[0].strip()
        return None

    def get_soft_version(self):
        """
        获取软件版本
        :return:string，软件版本
        """
        cmd_result = self.cmd_results.get(self._SHOW_UPGRADE_PACKAGE_CLI_CMD)
        if not cmd_result:
            return None

        result = re.findall(r"Software\sVersion([\s\S]*?)HotPatch\sVersion", cmd_result)
        if not result:
            return None
        
        lines = result[0].strip().splitlines()
        if len(lines) > 2: # 如果有控制器版本，回显行数大于2
            soft_ver = re.findall(r"[^\s]+", lines[-1].strip())[2] # 如果没有IP，则在第3个位置
            if soft_ver.startswith("V500R007"):
                return soft_ver
            return re.findall(r"[^\s]+", lines[-1].strip())[3] # 当前版本在分割数组中第4个位置
        return None
    
    def get_patch_version(self):
        """
        获取补丁版本
        :return:string，补丁版本
        """
        cmd_result = self.cmd_results.get(self._SHOW_UPGRADE_PACKAGE_CLI_CMD)
        if not cmd_result:
            return None

        result = re.findall(r"HotPatch\sVersion([\s\S]*)", cmd_result)
        if not result:
            return None
        
        lines = result[0].strip().splitlines()
        if len(lines) > 2:  # 如果有补丁，回显行数大于2
            patch_ver = re.findall(r"[^\s]+", lines[-1].strip())[2] # 如果没有IP，则在第3个位置
            if patch_ver.startswith("V500R007"):
                return patch_ver
            return re.findall(r"[^\s]+", lines[-1].strip())[3] # 补丁版本在分割数组中第4个位置
        return None

    def check_ntp_switch_opened(self):
        """
        检查ntp开关是否打开
        :return:True-NTP开关打开，False-NTP开关关闭
        """
        # 检查方法：查询NTP开关状态
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(self._SHOW_NTP_SWITCH_CLI_CMD)
        if not res:
            logging.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX, self._SHOW_NTP_SWITCH_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_NTP_SWITCH_CLI_CMD)

        self.get_cmd_results(self._SHOW_NTP_SWITCH_CLI_CMD, cli_exec_tmp_file)
        
        ntp_switch = self.get_ntp_switch_status()
        if not ntp_switch:
            logging.error("%s: get ntp from temp file failed.", _LOG_PREFIX)
            raise UnExpectValueError()
        
        return ntp_switch == "Enable"

    def check_version(self):
        """
        检查系统软件版本和补丁版本是否满足条件：
            1) 若软件版本为V500R007C71SPC100，补丁版本为V500R007C71SPH105或之后版本；
            2) 若软件版本为V500R007C60SPC300，补丁版本为V500R007C60SPH322或之后版本；
        :return:True-满足条件，False-不满足条件
        """
        # 检查方法：查询版本信息
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(self._SHOW_UPGRADE_PACKAGE_CLI_CMD)
        if not res:
            logging.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX, self._SHOW_UPGRADE_PACKAGE_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_UPGRADE_PACKAGE_CLI_CMD)

        self.get_cmd_results(self._SHOW_UPGRADE_PACKAGE_CLI_CMD, cli_exec_tmp_file)           
 
        soft_ver = self.get_soft_version()        
        if not soft_ver:
            logging.error("%s: get Software Version failed.", _LOG_PREFIX)
            raise UnExpectValueError()

        if soft_ver != "V500R007C71SPC100" and soft_ver != "V500R007C60SPC300":
            return False

        patch_ver = self.get_patch_version()
        if not patch_ver:
            logging.error("%s: get HotPatch Version failed.", _LOG_PREFIX)
            raise UnExpectValueError()
        
        # 当未打补丁时, patch_ver="--", 值小于"V500R007XXXX", 可以兼用以下判断
        return (soft_ver == "V500R007C71SPC100" and patch_ver >= "V500R007C71SPH105") or \
            (soft_ver == "V500R007C60SPC300" and patch_ver >= "V500R007C60SPH322")
    
    @staticmethod
    def _try_to_del_file(cli_exec_tmp_file):
        if os.path.exists(cli_exec_tmp_file):
            os.remove(cli_exec_tmp_file)
        return


def main(args):
    checker = CheckNtpSwitch()
    try:
        result = checker.execute()
        if not result:
            # 检查不通过时打印命令回显，便于日志定位原因
            logging.info("%s: The checking returned string is(%s).", _LOG_PREFIX, json.dumps(checker.cmd_results))
        print('True' if result else 'False')
    except Exception as e:
        logging.exception(e)
        print_str = json.dumps(checker.cmd_results)
        logging.info("%s: The checking returned string is(%s).", _LOG_PREFIX, print_str)
    return 0


if __name__ == '__main__':
    sys.exit(main(sys.argv))
