#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
# @brief   : 2021/12/20
# @File    : check_sysdisk.py
# @Software: Software management infrastructure
"""
import gc
import getopt
import os
import random
import sys
import time

from check_item.check_util.util import check_mem_is_enough, get_pkg_version, get_patch_version, exec_diagnose
from check_item.check_util.util import shell
from infra.debug.log import swm_logger as log
from plat.host.host_mgr import HostMgr


class SysDiskCheck(object):
    RAID_CMD = "mdadm -D /dev/md1"
    MOUNT_CMD = "mount -l | grep {path:s}"
    CHECK_PATH = ["/startup_disk/image", "/OSM/coffer_data"]
    CHECK_DIR = "/startup_disk/image"
    CUR_PKG_MANIFEST = "/startup_disk/image/pkg_cur/manifest.yml"
    CUR_PATCH_YML = "/startup_disk/image/pkg_cur/../hotpatch/patch_cur/patch.yml"
    DATA_NODE_FLAG = "/startup_disk/conf/conf_local/data_node"
    NO_NEED_CHECK_MIN_PATCH_DICT = {
        6200815132: 150,  # V5R7C71SPH150
        6200819073: 120,  # V5R7C73SPH120
    }

    @classmethod
    def _check_raid(cls):
        # exec cmd
        ret, outputs = shell.call_system_cmd(cls.RAID_CMD)
        if ret != 0:
            log.warning("SysDiskCheck: Execute cmd failed, ret(%s), output(%s).", ret, outputs)
            return False
        outputs = outputs.split("\n")
        # check raid
        state_line = None
        for line in outputs:
            line = line.strip()
            if "State :" in line:
                state_line = line
                break
        if not state_line:
            log.error("SysDiskCheck: The echo is error, output(%s).", outputs)
            return False

        if "degraded" in state_line:
            log.error("SysDiskCheck: The raid state is wrong, state(%s).", state_line)
            return False

        log.info("SysDiskCheck: Check raid success, the disks are active, info(%s).", state_line)
        return True

    @classmethod
    def _check_mount(cls):
        for path in cls.CHECK_PATH:
            mount_cmd = cls.MOUNT_CMD.format(path=path)
            ret, outputs = shell.call_system_cmd(mount_cmd)
            if ret != 0:
                log.error("SysDiskCheck: Check mount failed, ret(%s), path(%s), outputs(%s).", ret, path, outputs)
                return False
        log.info("SysDiskCheck: Check mount success.")
        return True

    @classmethod
    @shell.remount_image
    def _check_write(cls):
        for path in cls.CHECK_PATH:
            random_str = str(random.Random(os.urandom(8)).randint(0, 999999))
            write_test_file = "".join(("write_test", str(time.time()), random_str))
            test_file = os.path.join(path, write_test_file)
            try:
                if not os.path.exists(path):
                    log.error("SysDiskCheck: The dir is not existed, %s.", path)
                    return False

                if os.path.exists(test_file):
                    os.remove(test_file)
                with open(test_file, "w") as fp:
                    fp.write("SysDiskCheck: WriteTest.")
            except Exception:
                log.exception("SysDiskCheck: Unknown error.")
                return False
            finally:
                if os.path.exists(test_file):
                    os.remove(test_file)
        log.info("SysDiskCheck: Check write success.")
        return True

    @classmethod
    def get_master_id(cls):
        master_id = -1
        ret, output = exec_diagnose('sys showcls')
        if ret:
            return master_id
        try:
            for line in output.splitlines():
                if "master" in line:
                    master_id = int(line.split()[0])
                    break
        except Exception:
            log.exception("SysDiskCheck: get node id exception")
            return -1
        return master_id

    @classmethod
    def local_node_need_check(cls, master_batch=True):
        """
        奇偶分批，控制器主控所在的批次先执行，IP框A板和主控制在同一批次
        @param master_batch: 主节点所在批次
        @return:
        """
        local_id = HostMgr.get_local_id()
        if os.path.exists(cls.DATA_NODE_FLAG):
            if master_batch and local_id % 2 != 0:
                log.info('SysDiskCheck: current node not check in master batch.')
                return False
            if not master_batch and local_id % 2 == 0:
                log.info('SysDiskCheck: current node not check in slave batch.')
                return False
        else:
            master_id = cls.get_master_id()
            if master_id < 0:
                return not master_batch

            if master_batch and (local_id % 2 != master_id % 2):
                log.info('SysDiskCheck: current node not check in master batch.')
                return False

            if not master_batch and (local_id % 2 == master_id % 2):
                log.info('SysDiskCheck: current node not check in slave batch.')
                return False
        return True

    @classmethod
    def check(cls, master_batch=True):
        if cls.version_match():
            return True

        if not cls.local_node_need_check(master_batch):
            return True

        if not check_mem_is_enough(500):
            return True

        return cls._check_raid() and cls._check_mount() and cls._check_write() and cls._check_read()

    @classmethod
    def version_match(cls):
        """
        # 补丁版本如果是已经合入了整改问题的版本，就不执行检查了
        C71SPH150/C73SPH120
        """
        sys_version = get_pkg_version(cls.CUR_PKG_MANIFEST)
        patch_version = get_patch_version(cls.CUR_PATCH_YML)
        if not sys_version:
            return False

        if not patch_version:
            patch_version = 0
        else:
            patch_version = int(patch_version.replace('SPH', ''))
        if int(sys_version) in cls.NO_NEED_CHECK_MIN_PATCH_DICT and \
                patch_version >= cls.NO_NEED_CHECK_MIN_PATCH_DICT.get(int(sys_version)):
            log.info('SysDiskCheck: cur base:%s, cur patch: %s, no need to check image file.',
                     sys_version, patch_version)
            return True
        return False

    @classmethod
    def _check_read(cls):
        log.info('SysDiskCheck: Start check image file.')
        for root, _, files in os.walk(cls.CHECK_DIR):
            file_paths = [os.path.join(root, filename) for filename in files]
            for file_path in file_paths:
                if not cls._check_file_can_be_read(file_path):
                    return False
        log.info('SysDiskCheck: check result success.')
        return True

    @classmethod
    def _check_file_can_be_read(cls, filepath):
        max_read_bytes = 10 * 1024 * 1024
        try:
            with open(filepath, 'rb') as file_handler:
                while file_handler.read(max_read_bytes):
                    gc.collect(0)
                    time.sleep(0.01)
        except Exception:
            log.exception("SysDiskCheck: Read %s exception.", filepath)
            return False
        return True


def show_help():
    usage = (
        "This is used for upgrade pre-check.\n"
        "  -h, --help show        the cmd help info.\n"
        "  -s, --check_sys_disk   check system disk.\n"
    ).format(os.path.basename(__file__))
    print(usage)


def check_sys_disk(master_batch=True):
    print('True' if SysDiskCheck.check(master_batch=master_batch) else 'False')
    return 0


def main(argv=None):
    if argv is None:
        argv = sys.argv
    try:
        opts, args = getopt.getopt(argv[1:], "hsm", ["check_sys_disk_slave_batch", "check_sys_disk_master_batch"])
        for o, a in opts:
            if o in ("-h", "--help"):
                show_help()
                return 0

            if o in ['-s', '--check_sys_disk_slave_batch']:
                return check_sys_disk(master_batch=False)
            if o in ['-m', '--check_sys_disk_master_batch']:
                return check_sys_disk()

    except Exception as e:
        print('False')
        log.exception(e)
        return 1
    return 0


if __name__ == "__main__":
    sys.exit(main())
