#!/usr/bin/python
# -*- coding: utf-8 -*-

import json
import getopt
import os
import re
import shutil
import sys
import tarfile
import threading

try:
    from Queue import Queue
except:
    from queue import Queue

from check_item.host_check.check_exception import CmdExecError, \
    UnExpectValueError
from infra.debug.log import swm_logger as log
from check_item.host_check.host_instance_mgr import HostGenerator, CheckResultType, \
    CheckResultInfo, FailedReason, HostOnArray
from check_item.check_util.cli_executor import CliExecutor
from check_item.check_util.extend_param_mgr import ExtendParamMgr

_LOG_PREFIX = "HOST_COMPATIBLE_CHECK"


class MyThread(threading.Thread):
    def __init__(self, func):
        super(MyThread, self).__init__()
        self.func = func
        self.result = CheckResultInfo(0, "")
        self.host_id = ""

    def run(self):
        self.result = self.func()

    def get_result(self):
        return self.result


class HostCompatibleChecker(object):
    """
    Host compatibility check policy manager
    """
    _PYTHON_2 = '2.7' in sys.version
    _ERROR_CODE_HAVE_NFS_SHARE = '0x01b5000d'
    _IPV4_REGEX = r"((25[0-5]|2[0-4]\d|1\d{2}|[1-9]?\d)\.)" \
                  r"{3}(25[0-5]|2[0-4]\d|1\d{2}|[1-9]?\d)"
    _IPV6_REGEX = r'([(::)|:])'
    _HOST_INFO_PKG_DECOMPRESS_DIR = "/OSM/coffer_data/update/host_info"
    _TEMP_TAR_PKG_DIR = "/OSM/coffer_data/update/host_info_tmp/"
    _CHECK_RETURN_DICT = {"hostCompaEvaluCheck": "True",
                          "pass_hosts": [],
                          "not_pass_hosts": [],
                          "can_not_check_hosts": []}
    _SUPPORT_OS_TYPE = ('Linux', 'SUSE', 'CentOS', 'RedHat', 'AIX',
                        'Solaris', 'VMware', 'Windows')
    _SHOW_HOST_CLI_CMD = "show host general"
    _SHOW_INITIATOR_CLI_CMD = "show initiator"
    _SHOW_LOGIC_PORT_CLI_CMD = "show logical_port general"
    _SHOW_SYSTEM_CLI_CMD = "show system general"
    _SHOW_LUN_CLI_CMD = "show lun general |filterColumn include columnList=ID,WWN"
    _SHOW_NFS_SHARE_CLI_CMD = "show share nfs"
    _SHOW_VSTORE_CLI_CMD = "show vstore"
    _SHOW_VSTORE_NFS_SHARE_CLI_CMD = "change vstore view id={0};show share nfs"
    _PARALLEL_CHECK_THREAD_NUM = 10

    def __init__(self):
        self.host_info_pkg_path = ""
        self.host_with_ultra_path_list = []
        self.pass_host_check_result_list = []
        self.not_pass_host_check_result_list = []
        self.can_not_check_host_result_list = []
        self.all_host_id_list = []
        self.array_name = ""
        self.array_hosts = []
        self.array_logic_ports = []
        self.array_lun_wwns = []
        self.array_nfs_shares = []
        self.smartkit_manage_ip = ""

    def check(self):
        self.all_host_id_list = self._get_all_host_id_list()
        self.array_logic_ports = self._get_array_logic_ports()
        self.array_name = self._get_array_name()
        self.array_hosts = self._get_array_hosts()
        self.array_lun_wwns = self._get_array_lun_wwns()
        self.array_nfs_shares = self._get_array_nfs_shares()

        nas_lun = [self.array_lun_wwns, self.all_host_id_list, self.array_hosts]
        nfs_shares = [self.array_nfs_shares, self.array_logic_ports]

        lun_or_nfs = True if all(nas_lun) or all(nfs_shares) else False
        if not lun_or_nfs:
            log.info("{0}: There are no lun and nfs on current device.".format(_LOG_PREFIX))
            self._print_error_info_when_all_pass()
            self._print_host_infos()
            return True

        res = self._get_host_info_pkg_path()
        if not self.host_info_pkg_path:
            # 1. 如果没有获取到主机信息文件路径时，我们无法评估，只能通过给工具返一个-1让工具不灰化兼容性评估，继续提示客户排查
            # 2. 前端共享卡支持热升级，也会返回-1，工具会灰化
            self.can_not_check_host_result_list.append(CheckResultInfo(-1, -1,
                                                                       failed_reason=''))
            self._print_error_info_when_all_pass()
            self._print_host_infos()
            return True

        if not res:
            log.error("%s: Get host info pkg path failed.", _LOG_PREFIX)
            return self._print_all_host_cannot_check()

        if not self._decompress():
            log.error("%s: decompress host info pkg failed.", _LOG_PREFIX)
            return self._print_all_host_cannot_check()

        if not self._generate_hosts():
            log.error("%s: Generate host objects failed.", _LOG_PREFIX)
            return self._print_all_host_cannot_check()

        if not self._check_all_host():
            return self._deal_check_failed()
        else:
            return self._deal_check_success()

    def _deal_check_success(self):
        if len(self.all_host_id_list) == \
                len(self.pass_host_check_result_list):
            log.info("%s: Good, all host compatible checking are passed.",
                     _LOG_PREFIX)
            self._print_error_info_when_all_pass()
            self._print_host_infos()
            return True
        else:
            log.warning("%s: Some host compatible checking are not passed "
                        "or can not check because the host information "
                        "is incomplete.",
                        _LOG_PREFIX)
            result = "False"
            need_return_err_code_list = \
                self.can_not_check_host_result_list + \
                self.not_pass_host_check_result_list
            if self.array_nfs_shares and not \
                    self.can_not_check_host_result_list and not \
                    self.not_pass_host_check_result_list:
                log.warning("%s: The current storage array has NFS shares, "
                            "need to manually check whether a Linux host whose"
                            " kernel version is 7.6 is associated with the "
                            "storage array.", _LOG_PREFIX)
                need_return_err_code_list.append(
                    CheckResultInfo(-1, -1,
                                    failed_reason=self.
                                    _ERROR_CODE_HAVE_NFS_SHARE))
            error_code = "|".join([check_res.failed_reason for
                                   check_res in need_return_err_code_list])
            detail_info = ";".join([str(check_res.detail_info) for
                                    check_res in need_return_err_code_list])
            log.warning("%s: result: %s, {'errorCode': '%s', "
                        "'param': '%s'}", _LOG_PREFIX, result,
                        error_code, detail_info)
            print(result)
            print(error_code)
            print(detail_info)
            self._print_host_infos()
            return True

    def _deal_check_failed(self):
        result = "False"
        error_code = FailedReason.INNER_ERROR
        err_param = ""
        log.warning("%s: Check host compatible error, result: %s, "
                    "{'errorCode': '(%s)', 'param': '%s'}",
                    _LOG_PREFIX, result, error_code, err_param)
        print(result)
        print(error_code)
        print(err_param)
        self._print_host_infos()
        return False

    def _print_error_info_when_all_pass(self):
        result = "True"
        if self.array_nfs_shares and not \
                self.can_not_check_host_result_list and not \
                self.not_pass_host_check_result_list:
            err_code = self._ERROR_CODE_HAVE_NFS_SHARE
            err_param = "-1"
            log.warning("%s: The current storage array has NFS shares, "
                        "need to manually check whether a Linux host whose "
                        "kernel version is 7.6 is associated with the "
                        "storage array.", _LOG_PREFIX)
        else:
            err_code = ""
            err_param = ""
        log.info("%s: result: %s, {'errorCode': '%s', 'param': '%s'}",
                 _LOG_PREFIX, result, err_code, err_param)
        print(result)
        print(err_code)
        print(err_param)

    def _get_array_name(self):
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(
            self._SHOW_SYSTEM_CLI_CMD)
        if not res:
            log.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX,
                      self._SHOW_SYSTEM_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_SYSTEM_CLI_CMD)

        with open(cli_exec_tmp_file) as lines:
            lines_array = lines.readlines()
            for line in lines_array:
                line = self._decode_str(line)
                if re.findall("System Name", line):
                    array_name = line.split(":")[1].strip()
                    log.debug("%s: Array name(%s).", _LOG_PREFIX,
                              array_name)
                    self._try_to_del_file(cli_exec_tmp_file)
                    return array_name
        self._try_to_del_file(cli_exec_tmp_file)
        raise UnExpectValueError()

    def _decode_str(self, value):
        if not self._PYTHON_2 and isinstance(value, bytes):
            return value.decode()
        return value

    @staticmethod
    def _try_to_del_file(cli_exec_tmp_file):
        if os.path.exists(cli_exec_tmp_file):
            os.remove(cli_exec_tmp_file)
        return

    def _print_host_infos(self):
        pass_host_ids = list(set([check_res.host_id for check_res in
                                  self.pass_host_check_result_list])
                             )
        not_pass_host_ids = list(set([check_res.host_id for check_res in
                                      self.not_pass_host_check_result_list])
                                 )
        can_not_check_host_ids = list(set([check_res.host_id for check_res in
                                           self.can_not_check_host_result_list])
                                      )
        # 如果有NFS文件系统共享时，因为可能存在只关联的文件系统的主机
        # 我们没有评估，只能通过给工具返一个-1让工具不灰化兼容性评估，
        # 继续提示客户排查
        if self.array_nfs_shares and not \
                self.can_not_check_host_result_list and not \
                self.not_pass_host_check_result_list:
            can_not_check_host_ids.append(-1)

        self._CHECK_RETURN_DICT["pass_hosts"] = str(pass_host_ids)
        self._CHECK_RETURN_DICT["not_pass_hosts"] = str(not_pass_host_ids)
        self._CHECK_RETURN_DICT["can_not_check_hosts"] = \
            str(can_not_check_host_ids)
        print_str = json.dumps(self._CHECK_RETURN_DICT)
        print(print_str)
        log.info("%s: The checking returned string is(%s).",
                 _LOG_PREFIX, print_str)
        return

    @property
    def check_return_dict(self):
        return self._CHECK_RETURN_DICT

    def _print_all_host_cannot_check(self):
        print("False")
        print(FailedReason.INNER_ERROR)
        print("")
        self._CHECK_RETURN_DICT["pass_hosts"] = "[]"
        self._CHECK_RETURN_DICT["not_pass_hosts"] = "[]"
        self._CHECK_RETURN_DICT["can_not_check_hosts"] = \
            str(self.all_host_id_list)
        print_str = json.dumps(self._CHECK_RETURN_DICT)
        print(print_str)
        log.info("%s: The checking returned string is(%s).",
                 _LOG_PREFIX, print_str)
        return False

    def _get_host_info_pkg_path(self):
        res, path = ExtendParamMgr.get_extend_param("hostInfoPkgPath")
        if not res:
            return res

        if path.find(':') != -1:
            path_params = path.split(':')
            self.smartkit_manage_ip = ':'.join(path_params[:-1])
            self.host_info_pkg_path = path_params[-1]
        else:
            self.host_info_pkg_path = path

        return True

    def _decompress(self):
        """
        Decompress data file.
        The Path of the decompressed data file:
        self._HOST_INFO_PKG_DECOMPRESS_DIR/xxx.data
        return True or False
        """
        if os.path.exists(self._TEMP_TAR_PKG_DIR):
            shutil.rmtree(self._TEMP_TAR_PKG_DIR)
        os.mkdir(self._TEMP_TAR_PKG_DIR)

        if os.path.exists(self._HOST_INFO_PKG_DECOMPRESS_DIR):
            shutil.rmtree(self._HOST_INFO_PKG_DECOMPRESS_DIR)
        os.mkdir(self._HOST_INFO_PKG_DECOMPRESS_DIR)

        try:
            if not os.path.exists(self.host_info_pkg_path) and \
                    self.smartkit_manage_ip:
                # 如果本控没有包，尝试从其他控制器获取，应对工具未连接到主控的场景
                if not ExtendParamMgr.get_file_from_management_ip(self.smartkit_manage_ip,
                                                                  self.host_info_pkg_path):
                    log.error("%s: Get host info file from (%s) failed.",
                              _LOG_PREFIX, self.smartkit_manage_ip)
                    return False

                if not os.path.exists(self.host_info_pkg_path):
                    log.error("%s: Host info pkg path(%s) is not exist.",
                              _LOG_PREFIX, self.host_info_pkg_path)
                    return False
            # 压缩包中的路径做黑白名单校验
            if not self.tarfile_check(self.host_info_pkg_path):
                log.error("%s:The compressed package(%s) startswith '/' or exists '../'.",
                          _LOG_PREFIX, self.host_info_pkg_path)
                return False

            with tarfile.open(self.host_info_pkg_path, 'r') as tar_f:
                tar_f.extractall(self._TEMP_TAR_PKG_DIR)
            self._tar_data_files(self._TEMP_TAR_PKG_DIR)
        except Exception as e:
            log.error('{0}: decompress failed: {1}, reason: {2}'.
                      format(_LOG_PREFIX, self.host_info_pkg_path, e))
            return False

        if os.path.exists(self._TEMP_TAR_PKG_DIR):
            shutil.rmtree(self._TEMP_TAR_PKG_DIR)

        return True

    @staticmethod
    def tarfile_check(tarfile_path):
        """
        白名单校验的字符为“../”和名字的首字符不能为“/”
        :paramtarfile_path:
        :return:
        """
        try:
            with tarfile.open(tarfile_path) as tar:
                illegal_names = list(filter(lambda s: s.startswith('/') or s.find('../') != -1, tar.getnames()))
                return len(illegal_names) == 0
        except Exception:
            log.exception('%s: Open tar file failed.', _LOG_PREFIX)
            return False

    @staticmethod
    def _extract_tar_files(tar_files, root_path, dst_path):
        tar_post_fix = '.tar.gz'
        for file_name in tar_files:
            if file_name.endswith(tar_post_fix):
                host_tar_path = os.path.join(root_path, file_name)
                if not HostCompatibleChecker.tarfile_check(host_tar_path):
                    log.error("%s:The compressed package(%s) startswith '/' or exists '../'.",
                              _LOG_PREFIX, host_tar_path)
                    return
                with tarfile.open(host_tar_path, 'r') as tar_f:
                    tar_f.extractall(os.path.join(dst_path, file_name[:-len(tar_post_fix)]))

        return

    def _tar_data_files(self, root_path):
        if not os.path.isdir(root_path):
            return

        for root, dirs, tar_files in os.walk(root_path):
            self._extract_tar_files(tar_files, root_path, self._HOST_INFO_PKG_DECOMPRESS_DIR)
            if not dirs:
                return
            for dir_tmp in dirs:
                sub_root_dir = os.path.join(root_path, dir_tmp)
                return self._tar_data_files(sub_root_dir)
        return

    def _generate_hosts(self):
        if not self._gen_host_with_ultra_path():
            return False

        if not self._gen_host_without_ultra_path():
            return False
        return True

    def _gen_host_with_ultra_path(self):
        try:
            if not os.path.isdir(self._HOST_INFO_PKG_DECOMPRESS_DIR):
                log.info("%s: Host info decompress dir(%s) is not exist, "
                         "no need to generate host with ultra path.",
                         _LOG_PREFIX, self._HOST_INFO_PKG_DECOMPRESS_DIR)
                return True
            files_in_decompress_dir = {}
            for root, dirs, tar_files in os.walk(self._HOST_INFO_PKG_DECOMPRESS_DIR):

                for file in tar_files:
                    if file.endswith('.data'):
                        files_in_decompress_dir[root] = file
            if not files_in_decompress_dir:
                log.warning("%s: No any host info file in dir(%s), "
                            "no need to generate host with ultra path.",
                            _LOG_PREFIX, self._HOST_INFO_PKG_DECOMPRESS_DIR)
                return True

            for file_dir, host_info_file in files_in_decompress_dir.items():
                host_os_type = host_info_file.split('_')[0]
                if not host_os_type or host_os_type not in \
                        self._SUPPORT_OS_TYPE:
                    log.warning("%s: Host type(%s) is empty or "
                                "not supported currently.",
                                _LOG_PREFIX, host_os_type)
                    continue
                host_instance = HostGenerator.gen_host_instance(host_os_type,
                                                                os.path.join(
                                                                    file_dir,
                                                                    host_info_file),
                                                                self.array_name,
                                                                self.array_hosts,
                                                                self.array_logic_ports,
                                                                self.array_lun_wwns)
                if host_instance:
                    self.host_with_ultra_path_list.append(host_instance)
        except Exception as e:
            log.exception(e)
            return False
        return True

    def _gen_host_without_ultra_path(self):
        """

        :return:
        """
        if not self.host_with_ultra_path_list:
            host_ids_with_ultra_path = []
        else:
            host_ids_with_ultra_path = [host.host_id for host in
                                        self.host_with_ultra_path_list]

        for host_id in self.all_host_id_list:
            if host_id not in host_ids_with_ultra_path:
                log.warning("%s: Get a host without ultraPath, host id(%s).",
                            _LOG_PREFIX, host_id)
                # 未安装UltraPath的主机，视为无法检查的主机
                check_res = CheckResultInfo(host_id,
                                            host_id,
                                            CheckResultType.CAN_NOT_CHECK,
                                            FailedReason.ULTRA_PATH_NOT_INSTALL)
                self.can_not_check_host_result_list.append(check_res)
        return True

    def _get_array_hosts(self):
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(
            self._SHOW_INITIATOR_CLI_CMD)
        if not res:
            log.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX,
                      self._SHOW_INITIATOR_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_INITIATOR_CLI_CMD)

        # 根据阵列上的启动器和主机启动器的对应关系，获取主机ID
        # 阵列上的启动器信息格式：
        # WWN               Running Status  Free  Alias  Host ID
        # ----              --------------  ----  -----  -------
        # 100000109b17bff2  Online          No    --     0
        # 100000109b17bff3  Online          No    --     0
        # iSCSI IQN         Running Status  Free  Alias  Host ID
        # ----------        -------  ----  -----  -------
        # iqn.1994*        Offline         No    --     2

        array_hosts = set()
        with open(cli_exec_tmp_file) as lines:
            lines_array = lines.readlines()
            for line in lines_array:
                # 跳过不包含主机ID的行，用于跳过标题行
                if not re.findall(r"\d", line):
                    continue
                line = line.strip()
                log.debug("%s: Array host info(%s).",
                          _LOG_PREFIX, line)
                array_host = HostOnArray(line.split()[4],
                                         line.split()[0],
                                         line.split()[1])
                array_hosts.add(array_host)
        log.debug("%s: Array host(%s).", _LOG_PREFIX,
                  [str(array_host) for array_host in array_hosts])
        self._try_to_del_file(cli_exec_tmp_file)
        return list(array_hosts)

    def _get_all_host_id_list(self):
        """
        通过阵列获取所有主机ID
        :return:
        """
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(
            self._SHOW_HOST_CLI_CMD)
        if not res:
            log.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX,
                      self._SHOW_HOST_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_HOST_CLI_CMD)

        # 解析阵列上映射的主机ID
        # 阵列上的主机信息格式：
        # ID      Name       Health Status         Operating System
        # --      -----      -----------------     -------------
        # 0       host1      No redundant link     Linux
        # 1       host1      Normal                Linux
        with open(cli_exec_tmp_file) as lines:
            read_lines = lines.readlines()
            host_info_lines = [line for line in read_lines if
                               re.findall(r"^\d", line.strip())]

        if not host_info_lines:
            self._try_to_del_file(cli_exec_tmp_file)
            return []

        all_host_id_list = []
        for line in host_info_lines:
            log.debug("%s: host line info(%s).", _LOG_PREFIX, line.strip())
            line = line.lstrip()
            # 去掉标题行
            if not line[0].isdigit():
                continue
            host_id = line.split()[0]
            all_host_id_list.append(int(host_id))

        self._try_to_del_file(cli_exec_tmp_file)
        log.info("%s: All host id on array(%s) are(%s).", _LOG_PREFIX,
                 self.array_name, all_host_id_list)
        return all_host_id_list

    def _get_array_logic_ports(self):
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(
            self._SHOW_LOGIC_PORT_CLI_CMD)
        if not res:
            log.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX,
                      self._SHOW_LOGIC_PORT_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_LOGIC_PORT_CLI_CMD)

        # 根据阵列上的启动器和主机启动器的对应关系，获取主机ID
        # 阵列上的启动器信息格式：
        #  Logical Port Name         Running Status  IPv4 Address
        # ------------------         --------------  ------------
        # 8.47.14.73_nfs             Link Up         8.47.14.73
        logic_ips = set()
        with open(cli_exec_tmp_file) as lines:
            lines_array = lines.readlines()
            for line in lines_array:
                if re.findall(self._IPV4_REGEX, line):
                    log.debug("%s: Logic port info(%s).", _LOG_PREFIX, line)
                    logic_ips.add(line.split()[3])
                if re.findall(self._IPV6_REGEX, line):
                    log.debug("%s: Logic port info(%s).", _LOG_PREFIX, line)
                    logic_ips.add(line.split()[4])
        self._try_to_del_file(cli_exec_tmp_file)
        return list(logic_ips)

    def _check_all_host(self):
        if not self._parallel_check():
            log.error("%s:Host check failed.", _LOG_PREFIX)
            return False
        return True

    def _deal_queue(self):
        host_queue = Queue()
        for host in self.host_with_ultra_path_list:
            host_queue.put(host)
        while not host_queue.empty():
            t_pool = []
            # 每次并发检查PARALLEL_CHECK_THREAD_NUM台主机
            for _ in list(range(self._PARALLEL_CHECK_THREAD_NUM)):
                if host_queue.empty():
                    break
                host = host_queue.get()
                tmp = MyThread(host.check)
                t_pool.append(tmp)
                tmp.start()
            for t in t_pool:
                t.join()
                result = t.get_result()
                if result.check_res_type == CheckResultType.NOT_PASS:
                    self.not_pass_host_check_result_list.append(result)
                elif result.check_res_type == CheckResultType.CAN_NOT_CHECK:
                    self.can_not_check_host_result_list.append(result)
                else:
                    self.pass_host_check_result_list.append(result)

    def _parallel_check(self):
        if not self.host_with_ultra_path_list:
            log.warning("%s: There have no any host has been "
                        "installed ultra path, can not check.",
                        _LOG_PREFIX)
            return True
        try:
            self._deal_queue()
        except Exception as e:
            log.exception(e)
            return False
        return True

    def _get_array_lun_wwns(self):
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(
            self._SHOW_LUN_CLI_CMD)
        if not res:
            log.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX,
                      self._SHOW_LUN_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_LUN_CLI_CMD)

        luns = set()
        with open(cli_exec_tmp_file) as lines:
            lines_array = lines.readlines()
            for line in lines_array:
                if re.findall(r"^\d+", line.strip()):
                    log.debug("%s: Lun info(%s).", _LOG_PREFIX, line)
                    luns.add(line.split()[1])
        self._try_to_del_file(cli_exec_tmp_file)
        return list(luns)

    def _get_array_vstore(self):
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(self._SHOW_VSTORE_CLI_CMD)
        if not res:
            log.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX, self._SHOW_VSTORE_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_VSTORE_CLI_CMD)

        vstores = set()
        with open(cli_exec_tmp_file) as lines:
            lines_array = lines.readlines()
            for line in lines_array:
                if re.findall(r"^\d+", line.strip()):
                    log.debug("%s: Vstore info(%s).", _LOG_PREFIX, line)
                    vstores.add(line.split()[0])
        self._try_to_del_file(cli_exec_tmp_file)
        return list(vstores)

    def _get_nfs_info(self, nfs_file):
        """
        Get nfs information from temporary file.
        :param nfs_file:
        :return:
        """
        nfs_shares = set()
        with open(nfs_file) as lines:
            lines_array = lines.readlines()
            for line in lines_array:
                if re.findall(r"^\d+", line.strip()):
                    log.debug("%s: nfs share info(%s).", _LOG_PREFIX, line)
                    share_nfs = re.findall(r"/\S*", line)
                    nfs_shares.add(share_nfs[0])
        self._try_to_del_file(nfs_file)
        return list(nfs_shares)

    def _get_vstore_nsf_shares(self):
        """
        Get vstore share nfs information on the controller.
        information content as below:
        Share ID  Description  Local Path     File System ID
        --------  -----------  -------------  -------------
        1                    /file_system1  1
        :return:
        """
        nsf_shares = []
        vstore_id_list = self._get_array_vstore()
        log.info("%s: Vstore id list(%s).", _LOG_PREFIX, vstore_id_list)
        for vstore_id in vstore_id_list:
            cmd = self._SHOW_VSTORE_NFS_SHARE_CLI_CMD.format(vstore_id)
            res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(cmd)
            if not res:
                log.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX, cmd)
                continue
            nsf_shares.extend(self._get_nfs_info(cli_exec_tmp_file))
            if nsf_shares:
                return nsf_shares
        return nsf_shares

    def _get_array_nfs_shares(self):
        res, cli_exec_tmp_file = CliExecutor.exec_show_cmd_to_file(self._SHOW_NFS_SHARE_CLI_CMD)
        if not res:
            log.error("%s: Cmd(%s) exec failed.", _LOG_PREFIX, self._SHOW_NFS_SHARE_CLI_CMD)
            self._try_to_del_file(cli_exec_tmp_file)
            raise CmdExecError(self._SHOW_NFS_SHARE_CLI_CMD)
        system_nfs_shares = self._get_nfs_info(cli_exec_tmp_file)
        if not system_nfs_shares:
            return self._get_vstore_nsf_shares()
        return system_nfs_shares

    def get_decompress_dir(self):
        return self._HOST_INFO_PKG_DECOMPRESS_DIR

    def clear_path(self):
        file_dir_path = (self.host_info_pkg_path, self.get_decompress_dir())
        for path in file_dir_path:
            if not os.path.exists(path):
                continue
            if os.path.isfile(path):
                os.remove(path)
            else:
                shutil.rmtree(path)


def show_help():
    usage = (
        "This is used for upgrade pre-check.\n"
        "  -h, --help             show the cmd help info.\n"
        "  -c, --check            check host compatible.\n"
    )
    print(usage)


def _exec_check(host_checker):
    res = host_checker.check()
    if not res:
        log.error("%s: Host compatible check failed.", _LOG_PREFIX)


def main(args):
    host_checker = HostCompatibleChecker()
    try:
        opts, args = getopt.getopt(args[1:], "hc", ["help", "check"])
        for o, a in opts:
            if o in ("-h", "--help"):
                show_help()
                return 0

            if o in ['-c', '--check']:
                _exec_check(host_checker)

    except Exception as e:
        log.exception(e)
        result = "False"
        error_code = FailedReason.INNER_ERROR
        detail_info = ""
        print(result)
        print(error_code)
        print(detail_info)
        log.error("%s: result: %s, {'errorCode': '(%s)', "
                  "'param': '%s'}", _LOG_PREFIX, result,
                  error_code, detail_info)
        return_dict = host_checker.check_return_dict
        return_dict["pass_hosts"] = []
        return_dict["not_pass_hosts"] = []
        return_dict["can_not_check_hosts"] = host_checker.all_host_id_list
        print_str = json.dumps(return_dict)
        print(print_str)
        log.info("%s: The checking returned string is(%s).",
                 _LOG_PREFIX, print_str)

    try:
        host_checker.clear_path()
    except OSError:
        log.exception('{0}: clear path failed.'.format(_LOG_PREFIX))

    return 0


if __name__ == '__main__':
    sys.exit(main(sys.argv))
