# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

import json
import re
import time
from nvme.task_util import exec_host_cmd, get_ultra_path_version


def get_callback_params(status, custom_result=None):
    params = {"status": status, "customResult": custom_result}
    return json.dumps(params)


def execute(context):
    return FcTaskService(context).run()


class FcTaskService:

    def __init__(self, context):
        self.context = context
        self.logger = context.get("logger")
        self.lang = context.get("language")
        self.cli = context.get("SSH")
        self.call_back = context.get("call_back")
        self.base_path = self.call_back.getNvmeTaskPath().getDestBasePath()
        self.methods = context.get("methods")
        self.network_configs = context.get("network_configs")

    # 入口执行方法
    def run(self):
        # 需要执行的方法
        self.logger.info("----------- FcTaskService start ---------")
        self.logger.info("----------- methods ---------:{}".format(self.methods))
        methods = self.methods.split(",")
        is_success = True
        for method_name in methods:
            func = getattr(self, method_name, None)
            if func:
                begin_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 开始之前，调用一下回调的开始
                custom_result = {"beginTime": begin_time, "methodName": method_name}
                self.call_back.start(get_callback_params(CheckStatus.PASS, custom_result))
                return_value = func()
                status = return_value[0]
                err_msg = return_value[1]
                origin_info = return_value[2]
                end_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 组装回调的参数
                custom_result = {"beginTime": begin_time, "endTime": end_time, "detail": err_msg,
                                 "methodName": method_name, "originInfo": origin_info}
                self.call_back.update(get_callback_params(status, custom_result))
                if status != CheckStatus.PASS:
                    is_success = False
            else:
                return CheckStatus.NOT_PASS, "process.task.step.detail.step_not_exist", method_name
        if not is_success:
            return CheckStatus.NOT_PASS, "", ""
        return CheckStatus.PASS, "", ""

    def check_python_env(self):
        """
        网络配置
        """
        # 跳转到执行脚本的目录。
        cmd = "python --version"
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        if "Python" not in ret:
            return CheckStatus.NOT_PASS, "python.env.failed", ret
        return CheckStatus.PASS, "", ret

    def check_initiator_id(self):
        """
        检查启动器ID是否一致
        """
        self.logger.info("-----------check initiator id consistent.-----------")
        cmd = "cat /sys/class/fc_host/host*/port_name"
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        initiator_id_str = str(self.call_back.getInitiatorId())
        self.logger.info("initiator_id:{}".format(initiator_id_str))
        if not initiator_id_str:
            return CheckStatus.NOT_PASS, "check.initiator.id.not.allow.empty", ret
        initiator_ids = initiator_id_str.split(",")
        for initiator_id in initiator_ids:
            if initiator_id not in ret:
                return CheckStatus.NOT_PASS, "check.initiator.id.not.consistent", ret
        return CheckStatus.PASS, "", ret

    def check_ultra_path(self):
        """
        检查是否安装华为自研多路径
        """
        self.logger.info("Check whether UltraPath software is installed.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never UltraPath", True)
        if "UltraPath-" not in ret:
            return CheckStatus.NOT_PASS, "check.ultra.path.failed", ret
        self.call_back.setUltraPathVersion(get_ultra_path_version(ret))
        return CheckStatus.PASS, "", ret

    def check_dm_multi_path(self):
        """
        检查是否安装操作系统多路径
        """
        self.logger.info("Check whether UltraPath software is installed.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never multipath", True)
        if "device-mapper-multipath-" in ret or "multipath-tools-" in ret:
            return CheckStatus.PASS, "", ret
        return CheckStatus.NOT_PASS, "check.dm.multipath.not.install", ret

    def scan_namespace(self):
        """
        扫描NameSpace
        """
        self.logger.info("-----------scan_namespace start-----------")
        rets = []
        # 查看扫描前的盘符信息
        ret = exec_host_cmd(self.cli, "lsscsi", True)
        rets.append(ret)
        init_scsi_infos = ret.splitlines()
        ret = exec_host_cmd(self.cli, "ls /sys/class/fc_host/", True)
        rets.append(ret)
        pattern = r'(host\d+)'
        hosts = re.findall(pattern, ret)
        for host in hosts:
            ret = exec_host_cmd(self.cli, "echo \" - - -\" > /sys/class/scsi_host/{}/scan".format(host), True)
            rets.append(ret)
        # 查看扫描后的盘符信息
        scan_scsi_info = exec_host_cmd(self.cli, "lsscsi", True)
        rets.append(scan_scsi_info)
        scan_scsis = scan_scsi_info.splitlines()
        if len(scan_scsis) > len(init_scsi_infos):
            return CheckStatus.PASS, "", "\n".join(rets)
        return CheckStatus.NOT_PASS, "scan.namespace.failed", "\n".join(rets)

    def config_multi_path(self):
        """
        操作系统自带多路径配置
        """
        self.logger.info("----------- config multi_path-----------")
        rets = []
        # 检查是否安装多路径
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never multipath", True)
        rets.append(ret)
        if "device-mapper-multipath-" not in ret and "multipath-tools-" not in ret:
            return CheckStatus.NOT_PASS, "check.dm.multipath.not.install", "\n".join(rets)
        # 配置/etc/multipath.conf
        config_cmd = "echo \"devices {\n" + "   device {\n" \
                                            "                vendor                      \"HUAWEI\"     \n" \
                                            "                product                     \"XSG1\"  \n" \
                                            "                path_grouping_policy         multibus\n" \
                                            "                path_checker                tur\n" \
                                            "                prio                        const\n" \
                                            "                path_selector               \"service-time 0\"\n" \
                                            "                failback                    immediate\n" \
                                            "                dev_loss_tmo                30\n" \
                                            "                fast_io_fail_tmo            5\n" \
                                            "                no_path_retry               15\n" \
                                            "}\n" \
                                            "}\" > /etc/multipath.conf"
        ret = exec_host_cmd(self.cli, config_cmd, True)
        rets.append(ret)
        # 启用多路径
        ret = exec_host_cmd(self.cli, "systemctl start multipathd.service", True)
        rets.append(ret)
        # 配置多路径随系统启动
        ret = exec_host_cmd(self.cli, "systemctl enable multipathd.service", True)
        rets.append(ret)
        return CheckStatus.PASS, "", "\n".join(rets)

    def check_multi_path_config(self):
        """
            操作系统多路径配置检查
        """
        ret = exec_host_cmd(self.cli, "multipath -ll", True)
        if "size=" in ret:
            return CheckStatus.PASS, "", ret
        return CheckStatus.NOT_PASS, "check.config.multi.path.failed", ret


class CheckStatus:
    PASS = "PASS"
    NOT_PASS = "NOT_PASS"
    NOT_CHECK = "NOT_CHECK"
    NOT_SUPPORT = "NOT_SUPPORT"
    WARNING = "WARNING"
    UNKNOWN = "UNKNOWN"
