# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

import json
import time
from nvme.task_util import exec_host_cmd, get_err_msg, get_ultra_path_version


def get_callback_params(status, custom_result=None):
    params = {"status": status, "customResult": custom_result}
    return json.dumps(params)


def execute(context):
    return EntryLevelIscsiTaskService(context).run()


class EntryLevelIscsiTaskService:

    def __init__(self, context):
        self.context = context
        self.logger = context.get("logger")
        self.lang = context.get("language")
        self.cli = context.get("SSH")
        self.call_back = context.get("call_back")
        self.base_path = self.call_back.getNvmeTaskPath().getDestBasePath()
        self.methods = context.get("methods")
        self.network_configs = context.get("network_configs")

    @staticmethod
    def is_red_hat_6(model, version):
        return model == "Red Hat" and version == "6"

    @staticmethod
    def is_suse_11(model, version):
        return model == "SUSE" and version == "11"

    # 入口执行方法
    def run(self):
        # 需要执行的方法
        self.logger.info("----------- EntryLevelIscsiTaskService start ---------")
        is_success = True
        for method_name in self.methods.split(","):
            func = getattr(self, method_name, None)
            if func:
                begin_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 开始之前，调用一下回调的开始
                custom_result = {"beginTime": begin_time, "methodName": method_name}
                self.call_back.start(get_callback_params(CheckStatus.PASS, custom_result))
                return_value = func()
                status = return_value[0]
                err_msg = return_value[1]
                origin_info = return_value[2]
                end_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 组装回调的参数
                custom_result = {
                    "beginTime": begin_time, "endTime": end_time, "detail": err_msg, "methodName": method_name,
                    "originInfo": origin_info
                }
                self.call_back.update(get_callback_params(status, custom_result))
                if status != CheckStatus.PASS:
                    is_success = False
            else:
                return CheckStatus.NOT_PASS, "process.task.step.detail.step_not_exist", method_name
        if not is_success:
            return CheckStatus.NOT_PASS, "", ""
        return CheckStatus.PASS, "", ""

    def check_initiator_id(self):
        """
        检查启动器ID是否一致
        """
        self.logger.info("-----------check initiator id consistent.-----------")
        cmd = "cat /etc/iscsi/initiatorname.iscsi"
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        initiator_id = str(self.call_back.getInitiatorId())
        self.logger.info("initiator_id:{}".format(initiator_id))
        initiator_ids = initiator_id.split(",")
        for initiator_id in initiator_ids:
            if initiator_id not in ret:
                return CheckStatus.NOT_PASS, "check.initiator.id.not.consistent", ret
        return CheckStatus.PASS, "", ret

    def check_plan_ip_conflict(self):
        """
        检查主机配置网卡口是否与已有IP冲突
        """
        self.logger.info("Check whether network adapter port configured conflicts with IP address.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "ip a", True)
        self.logger.info("ssh_ret: {}".format(ret))
        # 获取规划的主机IP
        plan_ips = [str(network_config["plan_ip"]) for network_config in self.network_configs]
        conflict_ips = []
        for plan_ip in plan_ips:
            if plan_ip in ret:
                conflict_ips.append(plan_ip)
        self.logger.info("plan_ips:{}".format(plan_ips))
        if conflict_ips:
            return CheckStatus.NOT_PASS, "plan.ip.conflict.network.port.config.failed", ret
        return CheckStatus.PASS, "", ret

    def check_install_iscsi(self):
        """
        检查是否安装iSCSI软件包
        """
        self.logger.info("Check whether the iSCSI software package is installed.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never iscsi", True)
        # Red Hat:iscsi-initiator-utils-
        # SUSE:yast-iscsi-client-\yast2-iscsi-client-\yast-iscsi-server-\open-iscsi-
        if self.check_suse_iscsi(ret) or self.check_ret_hat_iscsi(ret):
            return CheckStatus.NOT_PASS, "check.install.iscsi.not.pass", ret
        return CheckStatus.PASS, "", ret

    def check_suse_iscsi(self, ret):
        return self.call_back.getHostOs() == "SUSE" and ("yast" not in ret or "open-iscsi-" not in ret)

    def check_ret_hat_iscsi(self, ret):
        return self.call_back.getHostOs() == "Red Hat" and "iscsi-initiator-utils-" not in ret

    def check_ultra_path(self):
        """
        检查是否安装华为自研多路径
        """
        self.logger.info("Check whether UltraPath software is installed.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never UltraPath", True)
        if "UltraPath-" not in ret:
            return CheckStatus.NOT_PASS, "check.ultra.path.failed", ret
        self.call_back.setUltraPathVersion(get_ultra_path_version(ret))
        return CheckStatus.PASS, "", ret

    def check_dm_multi_path(self):
        """
        检查是否安装操作系统多路径
        """
        self.logger.info("Check whether UltraPath software is installed.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never device-mapper", True)
        if "device-mapper-" in ret or "device-mapper-multipath-" in ret or "multipath-tools-" in ret:
            return CheckStatus.PASS, "", ret
        return CheckStatus.NOT_PASS, "check.dm.multipath.not.install", ret

    def config_network(self):
        """
        iSCSI网络配置种类
        1、vlan配置
        2、网口配置
        """
        rets = []
        # 网口配置。
        self.config_network_port(rets)
        # 重启网络服务
        ret = exec_host_cmd(self.cli, self.get_restart_network_cmd(), True)
        rets.append(ret)
        return CheckStatus.PASS, "", "\n".join(rets)

    def get_restart_network_cmd(self):
        # SUSE 11、Red Hat 6 : /etc/init.d/network restart
        # SUSE 12/15、Red Hat 7 : systemctl restart network.service
        # Red Hat 8: systemctl restart NetworkManager.service
        model = self.call_back.getHostOs()
        version = self.call_back.getHostVersion()
        if model == "Red Hat" and version == "8":
            return "systemctl restart NetworkManager.service"
        if self.is_suse_11(model, version) or self.is_red_hat_6(model, version):
            return "/etc/init.d/network restart"
        return "systemctl restart network.service"

    def config_network_port(self, rets):
        """
        配置网口的网络配置文件
        """
        if self.call_back.getHostOs() == "Red Hat":
            self.config_red_hat_network_port(rets)
        else:
            self.config_suse_network_port(rets)

    def config_suse_network_port(self, rets):
        for network_config in self.network_configs:
            port_name = network_config["port_name"]
            plan_ip = network_config["plan_ip"]
            subnet_mask = network_config["netmask"]
            ifcfg_path = "/etc/sysconfig/network/ifcfg-{}".format(port_name)
            config_info = "BOOTPROTO='static'\n" \
                          "BROADCAST=''\n" \
                          "ETHTOOL_OPTIONS=''\n" \
                          "IPADDR='{}'\n" \
                          "NETMASK='{}'\n" \
                          "MTU=''\n" \
                          "NAME=''\n" \
                          "NETWORK=''\n" \
                          "REMOTE_IPADDR=''\n" \
                          "STARTMODE='auto'".format(plan_ip, subnet_mask)
            config_network_interface_cmd = 'echo -e "{}"  > {}'.format(config_info, ifcfg_path)
            ret = exec_host_cmd(self.cli, config_network_interface_cmd, True)
            rets.append(ret)

    def config_red_hat_network_port(self, rets):
        for network_config in self.network_configs:
            port_name = network_config["port_name"]
            plan_ip = network_config["plan_ip"]
            subnet_mask = network_config["netmask"]
            ifcfg_path = "/etc/sysconfig/network-scripts/ifcfg-{}".format(port_name)
            config_network_interface_cmd = 'echo -e "TYPE=Ethernet\n' \
                                           'PROXY_METHOD=none\n' \
                                           'BROWSER_ONLY=no\n' \
                                           'BOOTPROTO=static\n' \
                                           'DEFROUTE=yes\n' \
                                           'IPV4_FAILURE_FATAL=no\n' \
                                           'IPV6INIT=yes\n' \
                                           'IPV6_AUTOCONF=no\n' \
                                           'IPV6_FAILURE_FATAL=no\n' \
                                           'IPV6_ADDR_GEN_MODE=stable-privacy\n' \
                                           'NAME={}\n' \
                                           'DEVICE={}\n' \
                                           'ONBOOT=yes\n' \
                                           'IPADDR={}\n' \
                                           'NETMASK={}"  > {}'.format(port_name, port_name, plan_ip, subnet_mask,
                                                                      ifcfg_path)
            ret = exec_host_cmd(self.cli, config_network_interface_cmd, True)
            rets.append(ret)

    def config_host_initiator(self):
        """
        主机启动器配置（使用iscsiadm）
        """
        rets = []
        # 启动iSCSI服务
        exec_host_cmd(self.cli, "systemctl start iscsi.service", True)
        for network_config in self.network_configs:
            # 确定对端IP是否能够ping通
            storage_ip = network_config["storage_ip"]
            ping_cmd = 'ping -c 3 {}'.format(storage_ip)
            ret = exec_host_cmd(self.cli, ping_cmd, True)
            rets.append(ret)
            if '3 received, 0% packet loss' not in ret:
                return CheckStatus.NOT_PASS, "ping.storage.ip.error", "\n".join(rets)
            # 查找目标器
            find_target_cmd = "iscsiadm -m discovery -t st -p {}".format(storage_ip)
            ret = exec_host_cmd(self.cli, find_target_cmd, True)
            rets.append(ret)
            # 登录目标器
            login_target_cmd = "iscsiadm -m node -p {} -l".format(storage_ip)
            ret = exec_host_cmd(self.cli, login_target_cmd, True)
            rets.append(ret)
            # 主机系统上修改iSCSI服务为主机启动后自动开启。
            ret = exec_host_cmd(self.cli, self.get_modify_iscsi_service_cmd(), True)
            rets.append(ret)

            # 主机系统上设置重启后自动连接目标器。
            ret = exec_host_cmd(self.cli, "iscsiadm -m node -o update -n node.startup -v automatic")
            rets.append(ret)

            # 重启iscsi服务
            ret = exec_host_cmd(self.cli, self.get_restart_iscsi_service_cmd(), True)
            rets.append(ret)
        return CheckStatus.PASS, "", rets

    def get_modify_iscsi_service_cmd(self):
        # SUSE 11: chkconfig open-iscsi on
        # SUSE 12/15、Red Hat 7/8: systemctl enable iscsi.service
        # Red Hat 6: chkconfig iscsi on
        model = self.call_back.getHostOs()
        version = self.call_back.getHostVersion()
        if model == "SUSE" and version == "11":
            return "chkconfig open-iscsi on"
        if model == "Red Hat" and version == "6":
            return "chkconfig iscsi on"
        return "systemctl enable iscsi.service"

    def get_restart_iscsi_service_cmd(self):
        # SUSE 11、Red Hat 6: /etc/init.d/iscsi restart
        # SUSE 12/15、Red Hat 7/8: systemctl restart iscsi.service
        model = self.call_back.getHostOs()
        version = self.call_back.getHostVersion()
        if self.is_suse_11(model, version) or self.is_red_hat_6(model, version):
            return "/etc/init.d/iscsi restart"
        return "systemctl restart iscsi.service"

    def scan_lun(self):
        """
        扫描lun
        """
        self.logger.info("-----------scan_lun start-----------")
        ret = exec_host_cmd(self.cli, "iscsiadm -m session --rescan", True)
        for network_config in self.network_configs:
            storage_ip = network_config["storage_ip"]
            if storage_ip not in ret:
                return CheckStatus.NOT_PASS, "scan.lun.failed", ret
        return CheckStatus.PASS, "", ret

    def config_multi_path(self):
        """
        操作系统自带多路径配置
        """
        self.logger.info("----------- config multi_path-----------")
        rets = []
        # 检查是否安装多路径
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never device-mapper", True)
        rets.append(ret)
        if "device-mapper-multipath-" not in ret and "device-mapper-" not in ret and "multipath-tools-" not in ret:
            return CheckStatus.NOT_PASS, "check.dm.multipath.not.install", "\n".join(rets)
        # 配置/etc/multipath.conf
        config_cmd = "echo \'devices {\n" + "   device {\n" \
                                            "                vendor                      \"HUAWEI\"\n" \
                                            "                product                     \"XSG1\"\n" \
                                            "                path_grouping_policy        multibus\n" \
                                            "                path_checker                tur\n" \
                                            "                prio                        const\n" \
                                            "                path_selector               \"round-robin 0\"\n" \
                                            "                failback                    immediate\n" \
                                            "                dev_loss_tmo                30\n" \
                                            "                fast_io_fail_tmo            5\n" \
                                            "                no_path_retry               15\n" \
                                            "}\n" \
                                            "}\' > /etc/multipath.conf"
        ret = exec_host_cmd(self.cli, config_cmd, True)
        rets.append(ret)
        model = self.call_back.getHostOs()
        version = self.call_back.getHostVersion()
        # 启用多路径
        ret = exec_host_cmd(self.cli, self.get_start_multipath_cmd(model, version), True)
        rets.append(ret)
        # 配置多路径随系统启动
        ret = exec_host_cmd(self.cli, "systemctl enable multipathd.service", True)
        rets.append(ret)
        # 重启多路径，修改multipath.conf后，需要执行命令重启多路径服务生效
        ret = exec_host_cmd(self.cli, self.get_restart_multipath_cmd(model, version), True)
        ret = self.cli.eliminatesResultColorCode(ret)
        rets.append(ret)
        # 重构initrd引导镜像，对于SuSE 12/15的系统，当配置启用或禁用多路径后，需要重构initrd，否则系统重启后有概率引导失败
        if model == "SUSE" and (version == "12" or version == "15"):
            ret = exec_host_cmd(self.cli, "dracut --force --add multipath", True)
            rets.append(ret)

        return CheckStatus.PASS, "", "\n".join(rets)

    def get_start_multipath_cmd(self, model, version):
        if self.is_suse_11(model, version) or self.is_red_hat_6(model, version):
            return "/etc/init.d/multipathd start"
        return "systemctl start multipathd.service"

    def get_restart_multipath_cmd(self, model, version):
        if self.is_suse_11(model, version) or self.is_red_hat_6(model, version):
            return "/etc/init.d/multipathd restart"
        return "systemctl restart multipathd.service"

    def check_multi_path_config(self):
        """
        操作系统多路径配置检查
        """
        ret = exec_host_cmd(self.cli, "multipath -ll", True)
        ret = self.cli.eliminatesResultColorCode(ret)
        if "size=" in ret:
            return CheckStatus.PASS, "", ret
        return CheckStatus.NOT_PASS, "check.config.multi.path.failed", ret


class CheckStatus:
    PASS = "PASS"
    NOT_PASS = "NOT_PASS"
    NOT_CHECK = "NOT_CHECK"
    NOT_SUPPORT = "NOT_SUPPORT"
    WARNING = "WARNING"
    UNKNOWN = "UNKNOWN"
