# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import json
import time
from nvme.task_util import exec_host_cmd, get_err_msg, get_ultra_path_version


def get_callback_params(status, custom_result=None):
    params = {"status": status, "customResult": custom_result}
    return json.dumps(params)


def execute(context):
    return IscsiTaskService(context).run()


class IscsiTaskService:

    def __init__(self, context):
        self.context = context
        self.logger = context.get("logger")
        self.lang = context.get("language")
        self.cli = context.get("SSH")
        self.call_back = context.get("call_back")
        self.base_path = self.call_back.getNvmeTaskPath().getDestBasePath()
        self.methods = context.get("methods")
        self.network_configs = context.get("network_configs")

    # 入口执行方法
    def run(self):
        # 需要执行的方法
        self.logger.info("----------- IscsiTaskService start ---------")
        is_success = True
        for method_name in self.methods.split(","):
            func = getattr(self, method_name, None)
            if func:
                begin_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 开始之前，调用一下回调的开始
                custom_result = {"beginTime": begin_time, "methodName": method_name}
                self.call_back.start(get_callback_params(CheckStatus.PASS, custom_result))
                return_value = func()
                status = return_value[0]
                err_msg = return_value[1]
                origin_info = return_value[2]
                end_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 组装回调的参数
                custom_result = {"beginTime": begin_time, "endTime": end_time, "detail": err_msg,
                                 "methodName": method_name, "originInfo": origin_info}
                self.call_back.update(get_callback_params(status, custom_result))
                if status != CheckStatus.PASS:
                    is_success = False
            else:
                return CheckStatus.NOT_PASS, "process.task.step.detail.step_not_exist", method_name
        if not is_success:
            return CheckStatus.NOT_PASS, "", ""
        return CheckStatus.PASS, "", ""

    def check_initiator_id(self):
        """
        检查启动器ID是否一致
        """
        self.logger.info("-----------check initiator id consistent.-----------")
        cmd = "cat /etc/iscsi/initiatorname.iscsi"
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        initiator_id_str = str(self.call_back.getInitiatorId())
        self.logger.info("initiator_id:{}".format(initiator_id_str))
        initiator_ids = initiator_id_str.split(",")
        for initiator_id in initiator_ids:
            if initiator_id not in ret:
                return CheckStatus.NOT_PASS, "check.initiator.id.not.consistent", ret
        return CheckStatus.PASS, "", ret

    def check_plan_ip_conflict(self):
        """
        检查主机配置网卡口是否与已有IP冲突
        """
        self.logger.info("Check whether network adapter port configured conflicts with IP address.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "ip a", True)
        self.logger.info("ssh_ret: {}".format(ret))
        # 获取规划的主机IP
        plan_ips = [str(network_config["plan_ip"]) for network_config in self.network_configs]
        conflict_ips = []
        for plan_ip in plan_ips:
            if plan_ip in ret:
                conflict_ips.append(plan_ip)
        self.logger.info("plan_ips:{}".format(plan_ips))
        if conflict_ips:
            return CheckStatus.NOT_PASS, "plan.ip.conflict.network.port.config.failed", ret
        return CheckStatus.PASS, "", ret

    def check_install_iscsi(self):
        """
            检查是否安装iSCSI软件包
        """
        self.logger.info("Check whether the iSCSI software package is installed.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never iscsi", True)
        if "iscsi-initiator-utils-" not in ret:
            return CheckStatus.NOT_PASS, "check.install.iscsi.not.pass", ret
        return CheckStatus.PASS, "", ret

    def check_ultra_path(self):
        """
        检查是否安装华为自研多路径
        """
        self.logger.info("Check whether UltraPath software is installed.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never UltraPath", True)
        if "UltraPath-" not in ret:
            return CheckStatus.NOT_PASS, "check.ultra.path.failed", ret
        self.call_back.setUltraPathVersion(get_ultra_path_version(ret))
        return CheckStatus.PASS, "", ret

    def check_dm_multi_path(self):
        """
        检查是否安装操作系统多路径
        """
        self.logger.info("Check whether UltraPath software is installed.")
        # 查询设备当前所用的IP配置。
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never multipath", True)
        if "device-mapper-multipath-" in ret or "multipath-tools-" in ret:
            return CheckStatus.PASS, "", ret
        return CheckStatus.NOT_PASS, "check.dm.multipath.not.install", ret

    def config_network(self):
        """
        iSCSI网络配置种类
        1、vlan配置
        2、网口配置
        """
        rets = []
        # 网口配置。
        vlan_id = self.network_configs[0].get("vlan_id", "")
        if not vlan_id:
            self.config_network_port(rets)
        else:
            self.config_vlan(rets)
        # 重启网络服务
        restart_network_cmd = self.get_restart_network_cmd(rets)

        ret = exec_host_cmd(self.cli, restart_network_cmd, True)
        rets.append(ret)
        return CheckStatus.PASS, "", "\n".join(rets)

    def get_restart_network_cmd(self, rets):
        ret = exec_host_cmd(self.cli, "cat /etc/centos-release", True)
        rets.append(ret)
        if "CentOS Linux release 7" in ret:
            return "systemctl restart network.service"
        else:
            return "systemctl restart NetworkManager.service"

    def config_network_port(self, rets):
        """
        配置网口的网络配置文件
        """
        for network_config in self.network_configs:
            port_name = network_config["port_name"]
            plan_ip = network_config["plan_ip"]
            subnet_mask = network_config["netmask"]
            ifcfg_path = "/etc/sysconfig/network-scripts/ifcfg-{}".format(port_name)
            config_network_interface_cmd = 'echo -e "TYPE=Ethernet\n' \
                                           'PROXY_METHOD=none\n' \
                                           'BROWSER_ONLY=no\n' \
                                           'BOOTPROTO=static\n' \
                                           'DEFROUTE=yes\n' \
                                           'IPV4_FAILURE_FATAL=no\n' \
                                           'IPV6INIT=yes\n' \
                                           'IPV6_AUTOCONF=no\n' \
                                           'IPV6_FAILURE_FATAL=no\n' \
                                           'IPV6_ADDR_GEN_MODE=stable-privacy\n' \
                                           'NAME={}\n' \
                                           'DEVICE={}\n' \
                                           'ONBOOT=yes\n' \
                                           'IPADDR={}\n' \
                                           'NETMASK={}"  > {}'.format(port_name, port_name, plan_ip, subnet_mask,
                                                                      ifcfg_path)
            ret = exec_host_cmd(self.cli, config_network_interface_cmd, True)
            rets.append(ret)

    def config_vlan(self, rets):
        for network_config in self.network_configs:
            port_name = network_config["port_name"]
            plan_ip = network_config["plan_ip"]
            subnet_mask = network_config["netmask"]
            # 配置网口
            ifcfg_path = "/etc/sysconfig/network-scripts/ifcfg-{}".format(port_name)
            config_network_interface_cmd = 'echo -e "TYPE=Ethernet\n' \
                                           'PROXY_METHOD=none\n' \
                                           'BROWSER_ONLY=no\n' \
                                           'BOOTPROTO=none\n' \
                                           'DEFROUTE=yes\n' \
                                           'IPV6INIT=no\n' \
                                           'IPV6_AUTOCONF=no\n' \
                                           'NAME={}\n' \
                                           'DEVICE={}\n' \
                                           'ONBOOT=yes"  > {}'.format(port_name, port_name, ifcfg_path)
            ret = exec_host_cmd(self.cli, config_network_interface_cmd, True)
            rets.append(ret)
            # 配置vlan
            vlan_id = network_config["vlan_id"]
            ifcfg_vlan_path = "/etc/sysconfig/network-scripts/ifcfg-{}.{}".format(port_name, vlan_id)
            vlan_name = "{}.{}".format(port_name, vlan_id)
            vlan_config_cmd = 'echo -e "BOOTPROTO=static\n' \
                              'IPV4_FAILURE_FATAL=no\n' \
                              'NAME={}\n' \
                              'DEVICE={}\n' \
                              'ONBOOT=yes\n' \
                              'VLAN=yes\n' \
                              'IPADDR={}\n' \
                              'NETMASK={}\n' \
                              'VLAN_EGRESS_PRIORITY_MAP=' \
                              '0:3,1:3,2:3,3:3,4:3,5:3,6:3,7:3" > {}' \
                .format(vlan_name, vlan_name, plan_ip, subnet_mask, ifcfg_vlan_path)
            ret = exec_host_cmd(self.cli, vlan_config_cmd, True)
            rets.append(ret)

    def config_host_initiator(self):
        """
        主机启动器配置（使用iscsiadm）
        """
        rets = []
        # 启动iSCSI服务
        exec_host_cmd(self.cli, "systemctl start iscsi.service", True)
        for network_config in self.network_configs:
            # 确定对端IP是否能够ping通
            storage_ip = network_config["storage_ip"]
            ping_cmd = 'ping -c 3 {}'.format(storage_ip)
            ret = exec_host_cmd(self.cli, ping_cmd, True)
            rets.append(ret)
            if '3 received, 0% packet loss' not in ret:
                return CheckStatus.NOT_PASS, "ping.storage.ip.error", "\n".join(rets)
            # 查找目标器
            find_target_cmd = "iscsiadm -m discovery -t st -p {}".format(storage_ip)
            ret = exec_host_cmd(self.cli, find_target_cmd, True)
            rets.append(ret)
            # 登录目标器
            login_target_cmd = "iscsiadm -m node -p {} -l".format(storage_ip)
            ret = exec_host_cmd(self.cli, login_target_cmd, True)
            rets.append(ret)
            # 主机系统上修改iSCSI服务为主机启动后自动开启。
            ret = exec_host_cmd(self.cli, "systemctl enable iscsi.service", True)
            rets.append(ret)
            # 主机系统上设置重启后自动连接目标器。
            ret = exec_host_cmd(self.cli, "iscsiadm -m node -o update -n node.startup -v automatic")
            rets.append(ret)
            # 重启iscsi服务
            ret = exec_host_cmd(self.cli, "systemctl restart iscsi.service", True)
            rets.append(ret)
        return CheckStatus.PASS, "", rets

    def scan_namespace(self):
        """
            扫描NameSpace
        """
        self.logger.info("-----------scan_namespace start-----------")
        ret = exec_host_cmd(self.cli, "iscsiadm -m session --rescan", True)
        for network_config in self.network_configs:
            storage_ip = network_config["storage_ip"]
            if storage_ip not in ret:
                return CheckStatus.NOT_PASS, "scan.namespace.failed", ret
        return CheckStatus.PASS, "", ret

    def config_multi_path(self):
        """
        操作系统自带多路径配置
        """
        self.logger.info("----------- config multi_path-----------")
        rets = []
        # 检查是否安装多路径
        ret = exec_host_cmd(self.cli, "rpm -qa | grep --color=never multipath", True)
        rets.append(ret)
        if "device-mapper-multipath-" not in ret and "multipath-tools-" not in ret:
            return CheckStatus.NOT_PASS, "check.dm.multipath.not.install", "\n".join(rets)
        # 配置/etc/multipath.conf
        config_cmd = "echo \"devices {\n" + "   device {\n" \
                                            "                vendor                      \"HUAWEI\"     \n" \
                                            "                product                     \"XSG1\"  \n" \
                                            "                path_grouping_policy         multibus\n" \
                                            "                path_checker                tur\n" \
                                            "                prio                        const\n" \
                                            "                path_selector               \"service-time 0\"\n" \
                                            "                failback                    immediate\n" \
                                            "                dev_loss_tmo                30\n" \
                                            "                fast_io_fail_tmo            5\n" \
                                            "                no_path_retry               15\n" \
                                            "}\n" \
                                            "}\" > /etc/multipath.conf"
        ret = exec_host_cmd(self.cli, config_cmd, True)
        rets.append(ret)
        # 启用多路径
        ret = exec_host_cmd(self.cli, "systemctl start multipathd.service", True)
        rets.append(ret)
        # 配置多路径随系统启动
        ret = exec_host_cmd(self.cli, "systemctl enable multipathd.service", True)
        rets.append(ret)
        return CheckStatus.PASS, "", "\n".join(rets)

    def check_multi_path_config(self):
        """
            操作系统多路径配置检查
        """
        ret = exec_host_cmd(self.cli, "multipath -ll", True)
        if "size=" in ret:
            return CheckStatus.PASS, "", ret
        return CheckStatus.NOT_PASS, "check.config.multi.path.failed", ret


class CheckStatus:
    PASS = "PASS"
    NOT_PASS = "NOT_PASS"
    NOT_CHECK = "NOT_CHECK"
    NOT_SUPPORT = "NOT_SUPPORT"
    WARNING = "WARNING"
    UNKNOWN = "UNKNOWN"
