# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import json
import time
from nvme.task_util import exec_host_cmd, get_err_msg

from com.huawei.ism.tool.storagedeploy.util.ProcessUtils import getResultFile


def get_callback_params(status, custom_result=None):
    params = {"status": status, "customResult": custom_result}
    return json.dumps(params)


def execute(context):
    return NvmeTaskService(context).run()


class NvmeTaskService:

    def __init__(self, context):
        self.context = context
        self.logger = context.get("logger")
        self.lang = context.get("language")
        self.cli = context.get("SSH")
        self.call_back = context.get("call_back")
        self.base_path = self.call_back.getNvmeTaskPath().getDestBasePath()
        self.methods = context.get("methods")
        self.python_cmd = "python" if str(self.call_back.getPythonType()).lower() == "python2" else "python3"

    # 入口执行方法
    def run(self):
        # 需要执行的方法
        self.logger.info("----------- NvmeTaskService start ---------")
        self.logger.info("----------- self.methods ---------:{}".format(self.methods))
        methods = self.methods.split(",")
        for method_name in methods:
            func = getattr(self, method_name, None)
            if func:
                begin_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 开始之前，调用一下回调的开始
                custom_result = {"beginTime": begin_time, "methodName": method_name}
                self.call_back.start(get_callback_params(CheckStatus.PASS, custom_result))
                return_value = func()
                status = return_value[0]
                err_msg = return_value[1]
                origin_info = return_value[2]
                end_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 组装回调的参数
                custom_result = {"beginTime": begin_time, "endTime": end_time, "detail": err_msg,
                                 "methodName": method_name, "originInfo": origin_info}
                if status == CheckStatus.NOT_PASS:
                    # 执行不通过也回调一下
                    self.call_back.update(get_callback_params(status, custom_result))
                    return status, err_msg, origin_info
                else:
                    # 执行通过，回调一下
                    self.call_back.update(get_callback_params(status, custom_result))
            else:
                return CheckStatus.NOT_PASS, "process.task.step.detail.step_not_exist", method_name
        return CheckStatus.PASS, "", ""

    def config_network(self):
        """
        网络配置
        """
        self.logger.info("-----------config_network start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} config_network.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "config_network failed.", ret
        result = json.loads(getResultFile(self.call_back, "config_network_result.json"))
        self.logger.info("origin_info:, {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.NOT_PASS, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, "", result.get("origin_info", "")

    def config_pfc(self):
        """
        配置端口pfc
        """
        self.logger.info("-----------config pfc start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} config_pfc.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "", ""
        result = json.loads(getResultFile(self.call_back, "config_pfc_result.json"))
        self.logger.info("origin_info:, {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.NOT_PASS, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, "", result.get("origin_info")

    def load_driver(self):
        """
        加载驱动
        """
        self.logger.info("-----------load_driver start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} load_driver.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "", ""
        result = json.loads(getResultFile(self.call_back, "load_driver_result.json"))
        self.logger.info("origin_info:, {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.NOT_PASS, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, "", result.get("origin_info")

    def connect_target(self):
        """
        连接目标服务器
        """
        self.logger.info("-----------connect_target start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} connect_target.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "", ""
        result = json.loads(getResultFile(self.call_back, "connect_target_result.json"))
        self.logger.info("origin_info:, {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.NOT_PASS, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, "", result.get("origin_info")

    def scan_namespace(self):
        """
        扫描NameSpace
        """
        self.logger.info("-----------scan_namespace start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} scan_namespace.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "", ""
        result = json.loads(getResultFile(self.call_back, "scan_namespace_result.json"))
        self.logger.info("origin_info:, {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.NOT_PASS, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, "", result.get("origin_info")


class CheckStatus:
    PASS = "PASS"
    NOT_PASS = "NOT_PASS"
    NOT_CHECK = "NOT_CHECK"
    NOT_SUPPORT = "NOT_SUPPORT"
    WARNING = "WARNING"
    UNKNOWN = "UNKNOWN"
