# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import json
import time
from nvme.task_util import exec_host_cmd, get_err_msg

from com.huawei.ism.tool.storagedeploy.util.ProcessUtils import getResultFile
from java.lang import Exception as JException

REBOOT_CONNECT_TIMEOUT = 3600


def get_callback_params(status, custom_result=None):
    params = {"status": status, "customResult": custom_result}
    return json.dumps(params)


def execute(context):
    return NvmeTaskService(context).run()


class NvmeTaskService:

    def __init__(self, context):
        self.context = context
        self.dev = context.get("dev")
        self.logger = context.get("logger")
        self.lang = context.get("language")
        self.cli = context.get("SSH")
        self.call_back = context.get("call_back")
        self.base_path = self.call_back.getNvmeTaskPath().getDestBasePath()
        self.methods = context.get("methods")
        self.python_cmd = "python" if str(self.call_back.getPythonType()).lower() == "python2" else "python3"

    # 入口执行方法
    def run(self):
        # 需要执行的方法
        self.logger.info("----------- NvmeTaskService start ---------")
        self.logger.info("----------- self.methods ---------:{}".format(self.methods))
        methods = self.methods.split(",")
        is_success = True
        for method_name in methods:
            func = getattr(self, method_name, None)
            if func:
                begin_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 开始之前，调用一下回调的开始
                custom_result = {"beginTime": begin_time, "methodName": method_name}
                self.call_back.start(get_callback_params(CheckStatus.PASS, custom_result))
                return_value = func()
                status = return_value[0]
                err_msg = return_value[1]
                origin_info = return_value[2]
                end_time = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())
                # 组装回调的参数
                custom_result = {"beginTime": begin_time, "endTime": end_time, "detail": err_msg,
                                 "methodName": method_name, "originInfo": origin_info}
                self.call_back.update(get_callback_params(status, custom_result))
                if status != CheckStatus.PASS:
                    is_success = False
            else:
                return CheckStatus.NOT_PASS, "process.task.step.detail.step_not_exist", method_name
        if not is_success:
            return CheckStatus.NOT_PASS, "", ""
        return CheckStatus.PASS, "", ""

    def install_nvme_cli(self):
        """
        网络配置
        """
        self.logger.info("-----------install_nvme_cli start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} install_nvme_cli.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "install_nvme_cli failed.", ret
        result = json.loads(getResultFile(self.call_back, "install_nvme_cli_result.json"))
        self.logger.info("origin_info: {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.NOT_PASS, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, "", result.get("origin_info")

    def install_oceanstor_nof(self):
        """
        配置端口pfc
        """
        self.logger.info("-----------install_oceanstor_nof start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} install_oceanstor_nof.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "", ""
        result = json.loads(getResultFile(self.call_back, "install_oceanstor_nof_result.json"))
        self.logger.info("origin_info: {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.NOT_PASS, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        # 重启主机：
        if result.get("error_code") == "host needs to be restarted":
            self.logger.info("start reboot host.")
            try:
                exec_host_cmd(self.cli, "reboot", True)
            except JException as e:
                self.logger.error("reboot host.", e)
                if not self.check_host_reboot():
                    return CheckStatus.NOT_PASS, get_err_msg("check.host.reboot.failed", self.lang), result.get(
                        "origin_info")
        return CheckStatus.PASS, "", result.get("origin_info")

    def check_host_reboot(self):
        """
        检查是否重启成功，超时时间为10分钟，间隔10秒重试一次
        """
        start_time = time.clock()
        while time.clock() - start_time < REBOOT_CONNECT_TIMEOUT:
            self.logger.info("-------reconnect ssh.-------")
            try:
                self.cli.reConnect()
                self.logger.info('reboot device success.')
                return True
            except (Exception, JException) as exep:
                self.logger.info('Connect host exception. used time:%s.' % (str(time.clock() - start_time)))
                time.sleep(10)
        self.logger.info('reboot device failed.')
        return False

    def install_mlnx_tools(self):
        """
        加载驱动
        """
        self.logger.info("-----------install_mlnx_tools start-----------")
        # 跳转到执行脚本的目录。
        cmd = "cd {}/script/".format(self.base_path)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret:, {}".format(ret))
        # 执行配置网络脚本。
        cmd = "{} install_mlnx_tools.py".format(self.python_cmd)
        ret = exec_host_cmd(self.cli, cmd, True)
        self.logger.info("ssh_ret: {}".format(ret))
        if "Execution completed" not in ret:
            return CheckStatus.NOT_PASS, "", ""
        result = json.loads(getResultFile(self.call_back, "install_mlnx_tools_result.json"))
        self.logger.info("origin_info: {}".format(result.get("origin_info")))
        if not result.get("status", ""):
            return CheckStatus.NOT_PASS, get_err_msg(result.get("error_code"), self.lang), result.get("origin_info")
        return CheckStatus.PASS, "", result.get("origin_info")


class CheckStatus:
    PASS = "PASS"
    NOT_PASS = "NOT_PASS"
    NOT_CHECK = "NOT_CHECK"
    NOT_SUPPORT = "NOT_SUPPORT"
    WARNING = "WARNING"
    UNKNOWN = "UNKNOWN"
